# This file is part of aiootp, an asynchronous pseudo one-time pad based
# crypto and anonymity library.
#
# Licensed under the AGPLv3: https://www.gnu.org/licenses/agpl-3.0.html
# Copyright © 2019-2021 Gonzo Investigative Journalism Agency, LLC
#            <gonzo.development@protonmail.ch>
#           © 2019-2021 Richard Machado <rmlibre@riseup.net>
# All rights reserved.
#


__all__ = [
    "primes",
    "Tables",
    "WORD_LIST",
    "PrimeGroups",
    "UniformPrimes",
]


__doc__ = (
    "A collection of conversion tables & precomputed values to standard"
    "ize, speed up, & clean up their usage in the package."
)


class Tables:
    """
    Conversion tables that are made available throughout the package.
    """

    __slots__ = []

    ASCII_95 = "".join(chr(val) for val in range(32, 127))
    ASCII_128 = "".join(chr(val) for val in range(128))
    BASE_36 = "0123456789abcdefghijklmnopqrstuvwxyz"
    BASE_38 = "-0123456789_abcdefghijklmnopqrstuvwxyz"
    BASE_64 = "+/0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    BASE_64 += "abcdefghijklmnopqrstuvwxyz"
    BYTES = [val.to_bytes(1, "big") for val in range(256)]
    ONION_CHARS = "234567abcdefghijklmnopqrstuvwxyz"
    URL_SAFE = "-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    URL_SAFE += "_abcdefghijklmnopqrstuvwxyz"


class UniformPrimes:
    """
    A collection of static randomly generated primes, organized by bit
    length, with no other expectation than starting with `0xff` &
    containing uniformly distributed bits.
    """

    __slots__ = []

    PRIME_32: int = 0xffe0a5b3
    PRIME_64: int = 0xff7a83e357b995c9
    PRIME_96: int = 0xff9669c7d66552972041fda5
    PRIME_128: int = 0xffc58ad53144285a641a0207fb844019
    PRIME_192: int = 0xff1ea3c024fabb588256055744450332f251264b7db2f3e7
    PRIME_256: int = 0xff0ad61bc46b8662b58c27f6ccafafb80802b9371dc0dab3fc0537f45a1712a3
    PRIME_384: int = 0xffff3a6471a198374d13c7b627f6695f82caf90d3063001ecaf3bf78267a27a3d30011133205fbc898758080ff4fa4df
    PRIME_512: int = 0xffffa2898b4ebc7563db2f07fce74444cdd6d38ca1fbe0932a433e099a77b3a5ea012e1e0bfbd73530d594c374f3f1d42f2c72a3b72317a11cc78897f62fb2a7


class PrimeGroups:
    """
    A collection of mostly small prime moduli & each of their respective
    smallest primitive roots organized by bit length.
    """

    __slots__ = []

    GENERATOR_32: int = 2
    GENERATOR_40: int = 3
    GENERATOR_48: int = 3
    GENERATOR_56: int = 2
    GENERATOR_64: int = 3
    GENERATOR_128: int = 11
    GENERATOR_192: int = 2
    GENERATOR_256: int = 2
    GENERATOR_384: int = 3
    GENERATOR_512: int = 7
    DH_GENERATOR_4096_GROUP_16: int = 2
    MOD_32: int = 0xfffffffb
    MOD_40: int = 0xfffffffc29
    MOD_48: int = 0xfffffffffab5
    MOD_56: int = 0xfffffffffff833
    MOD_64: int = 0xfffffffffffff9b9
    MOD_128: int = 0xffffffffffffffffffffffffffffe807
    MOD_192: int = 0xfffffffffffffffffffffffffffffffffffffffffffffa85
    MOD_256: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff8dcb
    MOD_384: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff5e31
    MOD_512: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff9203
    DH_MOD_4096_GROUP_16: int = 0xffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a637ed6b0bff5cb6f406b7edee386bfb5a899fa5ae9f24117c4b1fe649286651ece45b3dc2007cb8a163bf0598da48361c55d39a69163fa8fd24cf5f83655d23dca3ad961c62f356208552bb9ed529077096966d670c354e4abc9804f1746c08ca18217c32905e462e36ce3be39e772c180e86039b2783a2ec07a28fb5c55df06f4c52c9de2bcbf6955817183995497cea956ae515d2261898fa051015728e5a8aaac42dad33170d04507a33a85521abdf1cba64ecfb850458dbef0a8aea71575d060c7db3970f85a6e1e4c7abf5ae8cdb0933d71e8c94e04a25619dcee3d2261ad2ee6bf12ffa06d98a0864d87602733ec86a64521f2b18177b200cbbe117577a615d6c770988c0bad946e208e24fa074e5ab3143db5bfce0fd108e4b82d120a92108011a723c12a787e6d788719a10bdba5b2699c327186af4e23c1a946834b6150bda2583e9ca2ad44ce8dbbbc2db04de8ef92e8efc141fbecaa6287c59474e6bc05d99b2964fa090c3a2233ba186515be7ed1f612970cee2d7afb81bdd762170481cd0069127d5b05aa993b4ea988d8fddc186ffb7dc90a6c08f4df435c934063199ffffffffffffffff


primes = {
    # A static table of primes for efficient and/or deterministic
    # functionality requiring prime numbers. The primes are organized
    # so there's two primes in a list in each element of the dictionary.
    # The first & last element of each list contains the first & last
    # prime of bit length ``key``.
    2: [b"\x02", b"\x03"],
    3: [b"\x05", b"\x07"],
    4: [b"\x0b", b"\r"],
    5: [b"\x13", b"\x1f"],
    6: [b"%", b"="],
    7: [b"C", b"\x7f"],
    8: [b"\x83", b"\xfb"],
    9: [2 * b"\x01", b"\x01" + b"\xfd"],
    10: [b"\x02" + b"\t", b"\x03" + b"\xfd"],
    11: [b"\x04" + b"\x07", b"\x07" + b"\xf7"],
    12: [b"\x08" + b"\x05", b"\x0f" + b"\xfd"],
    13: [b"\x10" + b"\x03", b"\x1f" + b"\xff"],
    14: [b" " + b"\x11", b"?" + b"\xfd"],
    15: [b"@" + b"\x1b", b"\x7f" + b"\xed"],
    16: [b"\x80" + b"\x03", b"\xff" + b"\xf1"],
    17: [b"\x01" + b"\x00" + b"\x01", b"\x01" + 2 * b"\xff"],
    18: [b"\x02" + b"\x00" + b"\x1d", b"\x03" + b"\xff" + b"\xfb"],
    19: [b"\x04" + b"\x00" + b"\x03", b"\x07" + 2 * b"\xff"],
    20: [b"\x08" + b"\x00" + b"\x15", b"\x0f" + b"\xff" + b"\xfd"],
    21: [b"\x10" + b"\x00" + b"\x07", b"\x1f" + b"\xff" + b"\xf7"],
    22: [b" " + b"\x00" + b"\x11", b"?" + b"\xff" + b"\xfd"],
    23: [b"@" + b"\x00" + b"\x0f", b"\x7f" + b"\xff" + b"\xf1"],
    24: [b"\x80" + b"\x00" + b"\t", 2 * b"\xff" + b"\xfd"],
    25: [b"\x01" + 2 * b"\x00" + b"+", b"\x01" + 2 * b"\xff" + b"\xd9"],
    26: [b"\x02" + 2 * b"\x00" + b"#", b"\x03" + 2 * b"\xff" + b"\xfb"],
    27: [b"\x04" + 2 * b"\x00" + b"\x0f", b"\x07" + 2 * b"\xff" + b"\xd9"],
    28: [b"\x08" + 2 * b"\x00" + b"\x1d", b"\x0f" + 2 * b"\xff" + b"\xc7"],
    29: [b"\x10" + 2 * b"\x00" + b"\x03", b"\x1f" + 2 * b"\xff" + b"\xfd"],
    30: [b" " + 2 * b"\x00" + b"\x0b", b"?" + 2 * b"\xff" + b"\xdd"],
    31: [b"@" + 2 * b"\x00" + b"\x03", b"\x7f" + 3 * b"\xff"],
    32: [b"\x80" + 2 * b"\x00" + b"\x0b", 3 * b"\xff" + b"\xfb"],
    33: [b"\x01" + 3 * b"\x00" + b"\x0f", b"\x01" + 3 * b"\xff" + b"\xf7"],
    34: [b"\x02" + 3 * b"\x00" + b"\x11", b"\x03" + 3 * b"\xff" + b"\xd7"],
    35: [b"\x04" + 3 * b"\x00" + b"\x19", b"\x07" + 3 * b"\xff" + b"\xe1"],
    36: [b"\x08" + 3 * b"\x00" + b"5", b"\x0f" + 3 * b"\xff" + b"\xfb"],
    37: [b"\x10" + 3 * b"\x00" + b"\x1f", b"\x1f" + 3 * b"\xff" + b"\xe7"],
    38: [b" " + 3 * b"\x00" + b"\t", b"?" + 3 * b"\xff" + b"\xd3"],
    39: [b"@" + 3 * b"\x00" + b"\x07", b"\x7f" + 3 * b"\xff" + b"\xf9"],
    40: [b"\x80" + 3 * b"\x00" + b"\x17", 4 * b"\xff" + b"\xa9"],
    41: [b"\x01" + 4 * b"\x00" + b"\x0f", b"\x01" + 4 * b"\xff" + b"\xeb"],
    42: [b"\x02" + 4 * b"\x00" + b"\x1b", b"\x03" + 4 * b"\xff" + b"\xf5"],
    43: [b"\x04" + 4 * b"\x00" + b"\x0f", b"\x07" + 4 * b"\xff" + b"\xc7"],
    44: [b"\x08" + 4 * b"\x00" + b"\x1d", b"\x0f" + 4 * b"\xff" + b"\xef"],
    45: [b"\x10" + 4 * b"\x00" + b"\x07", b"\x1f" + 4 * b"\xff" + b"\xc9"],
    46: [b" " + 4 * b"\x00" + b";", b"?" + 4 * b"\xff" + b"\xeb"],
    47: [b"@" + 4 * b"\x00" + b"\x0f", b"\x7f" + 4 * b"\xff" + b"\x8d"],
    48: [b"\x80" + 4 * b"\x00" + b"\x05", 5 * b"\xff" + b"\xc5"],
    49: [b"\x01" + 5 * b"\x00" + b"\x15", b"\x01" + 5 * b"\xff" + b"\xaf"],
    50: [b"\x02" + 5 * b"\x00" + b"E", b"\x03" + 5 * b"\xff" + b"\xe5"],
    51: [b"\x04" + 5 * b"\x00" + b"7", b"\x07" + 5 * b"\xff" + b"\x7f"],
    52: [b"\x08" + 5 * b"\x00" + b"\x15", b"\x0f" + 5 * b"\xff" + b"\xd1"],
    53: [b"\x10" + 5 * b"\x00" + b"\x15", b"\x1f" + 5 * b"\xff" + b"\x91"],
    54: [b" " + 5 * b"\x00" + b"\x05", b"?" + 5 * b"\xff" + b"\xdf"],
    55: [b"@" + 5 * b"\x00" + b"\x9f", b"\x7f" + 5 * b"\xff" + b"\xc9"],
    56: [b"\x80" + 5 * b"\x00" + b"\x03", 6 * b"\xff" + b"\xfb"],
    57: [b"\x01" + 6 * b"\x00" + b"Q", b"\x01" + 6 * b"\xff" + b"\xf3"],
    58: [b"\x02" + 6 * b"\x00" + b"\t", b"\x03" + 6 * b"\xff" + b"\xe5"],
    59: [b"\x04" + 6 * b"\x00" + b"E", b"\x07" + 6 * b"\xff" + b"\xc9"],
    60: [b"\x08" + 6 * b"\x00" + b"\x83", b"\x0f" + 6 * b"\xff" + b"\xa3"],
    61: [b"\x10" + 6 * b"\x00" + b"!", b"\x1f" + 7 * b"\xff"],
    62: [b" " + 6 * b"\x00" + b"\x0f", b"?" + 6 * b"\xff" + b"\xc7"],
    63: [b"@" + 6 * b"\x00" + b"\x87", b"\x7f" + 6 * b"\xff" + b"\xe7"],
    64: [b"\x80" + 6 * b"\x00" + b"\x1d", 7 * b"\xff" + b"\xc5"],
    65: [b"\x01" + 7 * b"\x00" + b"\r", b"\x01" + 7 * b"\xff" + b"\xcf"],
    66: [b"\x02" + 7 * b"\x00" + b"\x83", b"\x03" + 7 * b"\xff" + b"\xfb"],
    67: [b"\x04" + 7 * b"\x00" + b"\t", b"\x07" + 7 * b"\xff" + b"\xed"],
    68: [b"\x08" + 7 * b"\x00" + b"\x03", b"\x0f" + 7 * b"\xff" + b"\xe9"],
    69: [b"\x10" + 7 * b"\x00" + b"!", b"\x1f" + 7 * b"\xff" + b"\xed"],
    70: [b" " + 7 * b"\x00" + b"\x1d", b"?" + 7 * b"\xff" + b"\xdd"],
    71: [b"@" + 7 * b"\x00" + b"\x19", b"\x7f" + 7 * b"\xff" + b"\x19"],
    72: [b"\x80" + 7 * b"\x00" + b"\x0b", 8 * b"\xff" + b"\xa3"],
    73: [b"\x01" + 8 * b"\x00" + b"\x0f", b"\x01" + 8 * b"\xff" + b"\xbb"],
    74: [b"\x02" + 8 * b"\x00" + b"\x1d", b"\x03" + 8 * b"\xff" + b"\xdd"],
    75: [b"\x04" + 8 * b"\x00" + b"%", b"\x07" + 8 * b"\xff" + b"\x9f"],
    76: [b"\x08" + 8 * b"\x00" + b"!", b"\x0f" + 8 * b"\xff" + b"\xf1"],
    77: [b"\x10" + 8 * b"\x00" + b"\x0f", b"\x1f" + 8 * b"\xff" + b"\xdf"],
    78: [b" " + 8 * b"\x00" + b"\x0b", b"?" + 8 * b"\xff" + b"\xf5"],
    79: [b"@" + 8 * b"\x00" + b"\x07", b"\x7f" + 8 * b"\xff" + b"\xbd"],
    80: [b"\x80" + 8 * b"\x00" + b"\x17", 9 * b"\xff" + b"\xbf"],
    81: [b"\x01" + 9 * b"\x00" + b"\r", b"\x01" + 9 * b"\xff" + b"\xcd"],
    82: [b"\x02" + 9 * b"\x00" + b"\x11", b"\x03" + 9 * b"\xff" + b"\xc7"],
    83: [b"\x04" + 9 * b"\x00" + b"\t", b"\x07" + 9 * b"\xff" + b"\xc9"],
    84: [b"\x08" + 9 * b"\x00" + b"K", b"\x0f" + 9 * b"\xff" + b"\xdd"],
    85: [b"\x10" + 9 * b"\x00" + b"\x03", b"\x1f" + 9 * b"\xff" + b"\xed"],
    86: [b" " + 9 * b"\x00" + b"\xab", b"?" + 9 * b"\xff" + b"\xdd"],
    87: [b"@" + 9 * b"\x00" + b"\x1b", b"\x7f" + 9 * b"\xff" + b"\xbd"],
    88: [b"\x80" + 9 * b"\x00" + b"'", 9 * b"\xff" + b"\xfe" + b"\xd5"],
    89: [b"\x01" + 10 * b"\x00" + b"\x07", b"\x01" + 11 * b"\xff"],
    90: [
        b"\x02" + 10 * b"\x00" + b"\x1d",
        b"\x03" + 10 * b"\xff" + b"\xdf",
    ],
    91: [
        b"\x04" + 10 * b"\x00" + b"\x85",
        b"\x07" + 10 * b"\xff" + b"\xd3",
    ],
    92: [b"\x08" + 10 * b"\x00" + b";", b"\x0f" + 10 * b"\xff" + b"\xad"],
    93: [
        b"\x10" + 10 * b"\x00" + b"\x19",
        b"\x1f" + 10 * b"\xff" + b"\xe7",
    ],
    94: [b" " + 10 * b"\x00" + b"i", b"?" + 10 * b"\xff" + b"\xfd"],
    95: [b"@" + 10 * b"\x00" + b"\x81", b"\x7f" + 10 * b"\xff" + b"\xf1"],
    96: [b"\x80" + 10 * b"\x00" + b"\t", 11 * b"\xff" + b"\xef"],
    97: [b"\x01" + 11 * b"\x00" + b"=", b"\x01" + 11 * b"\xff" + b"s"],
    98: [b"\x02" + 11 * b"\x00" + b"i", b"\x03" + 11 * b"\xff" + b"\xcd"],
    99: [
        b"\x04" + 11 * b"\x00" + b"\x07",
        b"\x07" + 11 * b"\xff" + b"\x8d",
    ],
    100: [
        b"\x08" + 11 * b"\x00" + b"\xff",
        b"\x0f" + 11 * b"\xff" + b"\xf1",
    ],
    101: [
        b"\x10" + 10 * b"\x00" + b"\x01" + b"\x15",
        b"\x1f" + 11 * b"\xff" + b"\xbb",
    ],
    102: [b" " + 11 * b"\x00" + b"Q", b"?" + 11 * b"\xff" + b"\xdf"],
    103: [
        b"@" + 10 * b"\x00" + b"\x01" + b"\x0b",
        b"\x7f" + 11 * b"\xff" + b"\x9f",
    ],
    104: [b"\x80" + 11 * b"\x00" + b"Q", 12 * b"\xff" + b"\xef"],
    105: [b"\x01" + 12 * b"\x00" + b"o", b"\x01" + 12 * b"\xff" + b"\xf3"],
    106: [b"\x02" + 12 * b"\x00" + b"'", b"\x03" + 12 * b"\xff" + b"\x8b"],
    107: [b"\x04" + 12 * b"\x00" + b"c", b"\x07" + 13 * b"\xff"],
    108: [b"\x08" + 12 * b"\x00" + b"'", b"\x0f" + 12 * b"\xff" + b"\xc5"],
    109: [b"\x10" + 12 * b"\x00" + b"!", b"\x1f" + 12 * b"\xff" + b"\xe1"],
    110: [b" " + 12 * b"\x00" + b"\x93", b"?" + 12 * b"\xff" + b"\xeb"],
    111: [b"@" + 12 * b"\x00" + b"\x1b", b"\x7f" + 12 * b"\xff" + b"\xdb"],
    112: [b"\x80" + 12 * b"\x00" + b"3", 13 * b"\xff" + b"\xb5"],
    113: [b"\x01" + 13 * b"\x00" + b"\x19", b"\x01" + 13 * b"\xff" + b"{"],
    114: [
        b"\x02" + 12 * b"\x00" + b"\x01" + b"\x19",
        b"\x03" + 13 * b"\xff" + b"\xf5",
    ],
    115: [b"\x04" + 13 * b"\x00" + b"+", b"\x07" + 13 * b"\xff" + b"\xbd"],
    116: [b"\x08" + 13 * b"\x00" + b"G", b"\x0f" + 13 * b"\xff" + b"\xfd"],
    117: [
        b"\x10" + 13 * b"\x00" + b"!",
        b"\x1f" + 12 * b"\xff" + b"\xfe" + b"\xe9",
    ],
    118: [b" " + 13 * b"\x00" + b"\x1d", b"?" + 13 * b"\xff" + b"\xfb"],
    119: [b"@" + 13 * b"\x00" + b"\x19", b"\x7f" + 13 * b"\xff" + b"\xbb"],
    120: [b"\x80" + 13 * b"\x00" + b"\t", 14 * b"\xff" + b"\x89"],
    121: [
        b"\x01" + 13 * b"\x00" + b"\x01" + b"\xc3",
        b"\x01" + 14 * b"\xff" + b"\xb7",
    ],
    122: [b"\x02" + 14 * b"\x00" + b")", b"\x03" + 14 * b"\xff" + b"\xfd"],
    123: [
        b"\x04" + 13 * b"\x00" + b"\x01" + b"\x15",
        b"\x07" + 14 * b"\xff" + b"\xbd",
    ],
    124: [
        b"\x08" + 14 * b"\x00" + b"\xa5",
        b"\x0f" + 14 * b"\xff" + b"\xc5",
    ],
    125: [b"\x10" + 14 * b"\x00" + b"C", b"\x1f" + 14 * b"\xff" + b"\xf7"],
    126: [b" " + 14 * b"\x00" + b"\x1b", b"?" + 14 * b"\xff" + b"w"],
    127: [b"@" + 14 * b"\x00" + b"\x07", b"\x7f" + 15 * b"\xff"],
    128: [b"\x80" + 14 * b"\x00" + b"\x1d", 15 * b"\xff" + b"a"],
    129: [b"\x01" + 15 * b"\x00" + b"3", b"\x01" + 15 * b"\xff" + b"\xe7"],
    130: [
        b"\x02" + 15 * b"\x00" + b"\x11",
        b"\x03" + 15 * b"\xff" + b"\xfb",
    ],
    131: [
        b"\x04" + 15 * b"\x00" + b"\xa9",
        b"\x07" + 15 * b"\xff" + b"\xbb",
    ],
    132: [
        b"\x08" + 15 * b"\x00" + b"'",
        b"\x0f" + 14 * b"\xff" + b"\xfe" + b"\xa5",
    ],
    133: [b"\x10" + 15 * b"\x00" + b"C", b"\x1f" + 15 * b"\xff" + b"\x9d"],
    134: [b" " + 15 * b"\x00" + b"\x1b", b"?" + 15 * b"\xff" + b"\xd3"],
    135: [b"@" + 15 * b"\x00" + b"\x1b", b"\x7f" + 15 * b"\xff" + b"\xd3"],
    136: [b"\x80" + 15 * b"\x00" + b"!", 16 * b"\xff" + b"\x8f"],
    137: [b"\x01" + 16 * b"\x00" + b"U", b"\x01" + 16 * b"\xff" + b"\xf3"],
    138: [
        b"\x02" + 16 * b"\x00" + b"\x9b",
        b"\x03" + 16 * b"\xff" + b"\x97",
    ],
    139: [b"\x04" + 16 * b"\x00" + b"W", b"\x07" + 16 * b"\xff" + b"E"],
    140: [
        b"\x08" + 16 * b"\x00" + b"\x9b",
        b"\x0f" + 16 * b"\xff" + b"\xe5",
    ],
    141: [b"\x10" + 16 * b"\x00" + b"%", b"\x1f" + 16 * b"\xff" + b"\xf7"],
    142: [b" " + 16 * b"\x00" + b"\x05", b"?" + 16 * b"\xff" + b"\x91"],
    143: [b"@" + 16 * b"\x00" + b"\xd9", b"\x7f" + 16 * b"\xff" + b"\xbb"],
    144: [b"\x80" + 16 * b"\x00" + b"\x05", 17 * b"\xff" + b"\xad"],
    145: [b"\x01" + 17 * b"\x00" + b"\xaf", b"\x01" + 17 * b"\xff" + b"i"],
    146: [b"\x02" + 17 * b"\x00" + b"\x1b", b"\x03" + 17 * b"\xff" + b"g"],
    147: [b"\x04" + 17 * b"\x00" + b"U", b"\x07" + 17 * b"\xff" + b"o"],
    148: [b"\x08" + 17 * b"\x00" + b"3", b"\x0f" + 17 * b"\xff" + b"Y"],
    149: [b"\x10" + 17 * b"\x00" + b"[", b"\x1f" + 17 * b"\xff" + b"\xe1"],
    150: [b" " + 17 * b"\x00" + b"E", b"?" + 17 * b"\xff" + b"\xfd"],
    151: [b"@" + 17 * b"\x00" + b"\x93", b"\x7f" + 17 * b"\xff" + b"="],
    152: [b"\x80" + 17 * b"\x00" + b"-", 18 * b"\xff" + b"\xef"],
    153: [
        b"\x01" + 18 * b"\x00" + b"\xfd",
        b"\x01" + 18 * b"\xff" + b"\xbb",
    ],
    154: [b"\x02" + 18 * b"\x00" + b"_", b"\x03" + 18 * b"\xff" + b"\r"],
    155: [
        b"\x04" + 18 * b"\x00" + b"\x1b",
        b"\x07" + 18 * b"\xff" + b"\xe1",
    ],
    156: [b"\x08" + 18 * b"\x00" + b"\x0f", b"\x0f" + 18 * b"\xff" + b"q"],
    157: [b"\x10" + 18 * b"\x00" + b"-", b"\x1f" + 18 * b"\xff" + b"\xed"],
    158: [b" " + 18 * b"\x00" + b"E", b"?" + 18 * b"\xff" + b"\xf1"],
    159: [b"@" + 18 * b"\x00" + b"a", b"\x7f" + 18 * b"\xff" + b"\xa5"],
    160: [b"\x80" + 17 * b"\x00" + b"\x01" + b"+", 19 * b"\xff" + b"\xd1"],
    161: [b"\x01" + 19 * b"\x00" + b"\x07", b"\x01" + 19 * b"\xff" + b"a"],
    162: [b"\x02" + 19 * b"\x00" + b"k", b"\x03" + 19 * b"\xff" + b"\x9b"],
    163: [
        b"\x04" + 19 * b"\x00" + b"\x13",
        b"\x07" + 19 * b"\xff" + b"\xc9",
    ],
    164: [
        b"\x08" + 19 * b"\x00" + b"\x15",
        b"\x0f" + 19 * b"\xff" + b"\xc1",
    ],
    165: [b"\x10" + 19 * b"\x00" + b"u", b"\x1f" + 19 * b"\xff" + b"\xe7"],
    166: [b" " + 19 * b"\x00" + b"\x8d", b"?" + 19 * b"\xff" + b"\xfb"],
    167: [b"@" + 19 * b"\x00" + b"U", b"\x7f" + 19 * b"\xff" + b"y"],
    168: [b"\x80" + 19 * b"\x00" + b"S", 19 * b"\xff" + b"\xfe" + b"\xff"],
    169: [
        b"\x01" + 20 * b"\x00" + b"W",
        b"\x01" + 19 * b"\xff" + b"\xfd" + b"}",
    ],
    170: [b"\x02" + 20 * b"\x00" + b"\x93", b"\x03" + 20 * b"\xff" + b"q"],
    171: [b"\x04" + 20 * b"\x00" + b"1", b"\x07" + 20 * b"\xff" + b"\xed"],
    172: [
        b"\x08" + 20 * b"\x00" + b"\x81",
        b"\x0f" + 20 * b"\xff" + b"\xa1",
    ],
    173: [b"\x10" + 20 * b"\x00" + b"i", b"\x1f" + 20 * b"\xff" + b"\xc9"],
    174: [b" " + 20 * b"\x00" + b"M", b"?" + 20 * b"\xff" + b"\xfd"],
    175: [b"@" + 20 * b"\x00" + b"\x07", b"\x7f" + 20 * b"\xff" + b"\x1b"],
    176: [b"\x80" + 20 * b"\x00" + b"\t", 21 * b"\xff" + b"\x17"],
    177: [
        b"\x01" + 20 * b"\x00" + b"\x01" + b"\xab",
        b"\x01" + 20 * b"\xff" + b"\xfe" + b"\xad",
    ],
    178: [b"\x02" + 21 * b"\x00" + b"K", b"\x03" + 21 * b"\xff" + b"\xd7"],
    179: [b"\x04" + 21 * b"\x00" + b"W", b"\x07" + 21 * b"\xff" + b"\xcf"],
    180: [
        b"\x08" + 20 * b"\x00" + b"\x01" + b"5",
        b"\x0f" + 21 * b"\xff" + b"\xd1",
    ],
    181: [b"\x10" + 21 * b"\x00" + b"\x0f", b"\x1f" + 21 * b"\xff" + b"["],
    182: [b" " + 21 * b"\x00" + b"\xa5", b"?" + 21 * b"\xff" + b"_"],
    183: [b"@" + 21 * b"\x00" + b"1", b"\x7f" + 21 * b"\xff" + b"m"],
    184: [b"\x80" + 21 * b"\x00" + b"\xd7", 22 * b"\xff" + b"\xdf"],
    185: [
        b"\x01" + 22 * b"\x00" + b"\x1b",
        b"\x01" + 21 * b"\xff" + b"\xfe" + b"\xd1",
    ],
    186: [
        b"\x02" + 22 * b"\x00" + b"\x9f",
        b"\x03" + 21 * b"\xff" + b"\xfe" + b"\x8d",
    ],
    187: [
        b"\x04" + 22 * b"\x00" + b"\xcd",
        b"\x07" + 22 * b"\xff" + b"\xab",
    ],
    188: [
        b"\x08" + 21 * b"\x00" + b"\x01" + b"/",
        b"\x0f" + 22 * b"\xff" + b"\x83",
    ],
    189: [b"\x10" + 22 * b"\x00" + b"9", b"\x1f" + 22 * b"\xff" + b"\xe7"],
    190: [b" " + 22 * b"\x00" + b"#", b"?" + 22 * b"\xff" + b"\xf5"],
    191: [b"@" + 22 * b"\x00" + b"\x81", b"\x7f" + 22 * b"\xff" + b"\xed"],
    192: [b"\x80" + 22 * b"\x00" + b"\x05", 23 * b"\xff" + b"\x13"],
    193: [
        b"\x01" + 23 * b"\x00" + b"\x85",
        b"\x01" + 23 * b"\xff" + b"\xe1",
    ],
    194: [b"\x02" + 23 * b"\x00" + b"A", b"\x03" + 23 * b"\xff" + b"\xdf"],
    195: [b"\x04" + 23 * b"\x00" + b"\x1b", b"\x07" + 23 * b"\xff" + b"y"],
    196: [b"\x08" + 23 * b"\x00" + b"#", b"\x0f" + 23 * b"\xff" + b"\xf1"],
    197: [
        b"\x10" + 23 * b"\x00" + b"\x15",
        b"\x1f" + 23 * b"\xff" + b"\xb5",
    ],
    198: [b" " + 23 * b"\x00" + b"k", b"?" + 23 * b"\xff" + b"\xef"],
    199: [b"@" + 23 * b"\x00" + b"\x0f", b"\x7f" + 23 * b"\xff" + b"\xcf"],
    200: [b"\x80" + 23 * b"\x00" + b"e", 24 * b"\xff" + b"\xb5"],
    201: [
        b"\x01" + 24 * b"\x00" + b"\xeb",
        b"\x01" + 24 * b"\xff" + b"\xc9",
    ],
    202: [
        b"\x02" + 23 * b"\x00" + b"\x01" + b"_",
        b"\x03" + 24 * b"\xff" + b"I",
    ],
    203: [b"\x04" + 24 * b"\x00" + b"C", b"\x07" + 24 * b"\xff" + b"a"],
    204: [b"\x08" + 24 * b"\x00" + b"\x0f", b"\x0f" + 24 * b"\xff" + b"Y"],
    205: [
        b"\x10" + 24 * b"\x00" + b"\x07",
        b"\x1f" + 24 * b"\xff" + b"\xaf",
    ],
    206: [
        b" " + 23 * b"\x00" + b"\x02" + b"E",
        b"?" + 24 * b"\xff" + b"\xfb",
    ],
    207: [b"@" + 24 * b"\x00" + b"!", b"\x7f" + 24 * b"\xff" + b"\xa5"],
    208: [
        b"\x80" + 24 * b"\x00" + b"\xcb",
        24 * b"\xff" + b"\xfe" + b"\xd5",
    ],
    209: [
        b"\x01" + 24 * b"\x00" + b"\x01" + b"w",
        b"\x01" + 25 * b"\xff" + b"\xdf",
    ],
    210: [b"\x02" + 25 * b"\x00" + b"/", b"\x03" + 25 * b"\xff" + b"\xd1"],
    211: [b"\x04" + 25 * b"\x00" + b"!", b"\x07" + 25 * b"\xff" + b"Q"],
    212: [b"\x08" + 25 * b"\x00" + b"G", b"\x0f" + 25 * b"\xff" + b"\xe9"],
    213: [b"\x10" + 25 * b"\x00" + b"9", b"\x1f" + 25 * b"\xff" + b"\xfd"],
    214: [b" " + 25 * b"\x00" + b"K", b"?" + 25 * b"\xff" + b"G"],
    215: [b"@" + 25 * b"\x00" + b"\x07", b"\x7f" + 25 * b"\xff" + b"c"],
    216: [
        b"\x80" + 25 * b"\x00" + b"\xfb",
        25 * b"\xff" + b"\xfe" + b"\x87",
    ],
    217: [
        b"\x01" + 25 * b"\x00" + b"\x01" + b"\xa7",
        b"\x01" + 26 * b"\xff" + b"\xc3",
    ],
    218: [
        b"\x02" + 26 * b"\x00" + b"\x81",
        b"\x03" + 26 * b"\xff" + b"\xdf",
    ],
    219: [
        b"\x04" + 26 * b"\x00" + b"\xa3",
        b"\x07" + 26 * b"\xff" + b"\x87",
    ],
    220: [
        b"\x08" + 26 * b"\x00" + b"\xb9",
        b"\x0f" + 26 * b"\xff" + b"\xb3",
    ],
    221: [
        b"\x10" + 26 * b"\x00" + b"\xd9",
        b"\x1f" + 26 * b"\xff" + b"\xfd",
    ],
    222: [b" " + 26 * b"\x00" + b"Q", b"?" + 26 * b"\xff" + b"\x8b"],
    223: [b"@" + 26 * b"\x00" + b"1", b"\x7f" + 26 * b"\xff" + b"\x15"],
    224: [b"\x80" + 26 * b"\x00" + b"\xbd", 27 * b"\xff" + b"\xc1"],
    225: [
        b"\x01" + 26 * b"\x00" + b"\x02" + b"\xdf",
        b"\x01" + 27 * b"\xff" + b"\xcf",
    ],
    226: [b"\x02" + 27 * b"\x00" + b"w", b"\x03" + 27 * b"\xff" + b"\xfb"],
    227: [
        b"\x04" + 26 * b"\x00" + b"\x02" + b"\xdf",
        b"\x07" + 26 * b"\xff" + b"\xfe" + b"k",
    ],
    228: [
        b"\x08" + 26 * b"\x00" + b"\x01" + b"\xe3",
        b"\x0f" + 27 * b"\xff" + b"\xa3",
    ],
    229: [
        b"\x10" + 27 * b"\x00" + b"\x03",
        b"\x1f" + 27 * b"\xff" + b"\xa5",
    ],
    230: [b" " + 27 * b"\x00" + b"\xf9", b"?" + 27 * b"\xff" + b"\xe5"],
    231: [b"@" + 27 * b"\x00" + b"C", b"\x7f" + 27 * b"\xff" + b"["],
    232: [b"\x80" + 27 * b"\x00" + b"i", 27 * b"\xff" + b"\xfd" + b"\xc9"],
    233: [
        b"\x01" + 27 * b"\x00" + b"\x01" + b"e",
        b"\x01" + 28 * b"\xff" + b"\xfd",
    ],
    234: [
        b"\x02" + 27 * b"\x00" + b"\x01" + b"\xaf",
        b"\x03" + 28 * b"\xff" + b"\xad",
    ],
    235: [b"\x04" + 28 * b"\x00" + b"+", b"\x07" + 28 * b"\xff" + b"\xf1"],
    236: [b"\x08" + 28 * b"\x00" + b"Q", b"\x0f" + 28 * b"\xff" + b"/"],
    237: [b"\x10" + 28 * b"\x00" + b"\x19", b"\x1f" + 28 * b"\xff" + b"K"],
    238: [b" " + 28 * b"\x00" + b"\xf9", b"?" + 28 * b"\xff" + b"_"],
    239: [b"@" + 28 * b"\x00" + b"C", b"\x7f" + 28 * b"\xff" + b"\xa9"],
    240: [b"\x80" + 28 * b"\x00" + b"\x1d", 28 * b"\xff" + b"\xfe" + b"-"],
    241: [b"\x01" + 29 * b"\x00" + b"s", b"\x01" + 29 * b"\xff" + b"\xd9"],
    242: [
        b"\x02" + 28 * b"\x00" + b"\x01" + b"\x05",
        b"\x03" + 29 * b"\xff" + b"\xc1",
    ],
    243: [b"\x04" + 29 * b"\x00" + b"E", b"\x07" + 29 * b"\xff" + b"\xf7"],
    244: [b"\x08" + 29 * b"\x00" + b";", b"\x0f" + 29 * b"\xff" + b"C"],
    245: [b"\x10" + 29 * b"\x00" + b"\x85", b"\x1f" + 29 * b"\xff" + b"]"],
    246: [
        b" " + 28 * b"\x00" + b"\x01" + b";",
        b"?" + 29 * b"\xff" + b"\x95",
    ],
    247: [
        b"@" + 28 * b"\x00" + b"\x01" + b"Q",
        b"\x7f" + 29 * b"\xff" + b"\xaf",
    ],
    248: [b"\x80" + 29 * b"\x00" + b"?", 30 * b"\xff" + b"\x13"],
    249: [b"\x01" + 30 * b"\x00" + b"Q", b"\x01" + 30 * b"\xff" + b"\xb5"],
    250: [b"\x02" + 30 * b"\x00" + b"w", b"\x03" + 30 * b"\xff" + b"1"],
    251: [
        b"\x04" + 30 * b"\x00" + b"\x19",
        b"\x07" + 30 * b"\xff" + b"\xf7",
    ],
    252: [b"\x08" + 30 * b"\x00" + b"A", b"\x0f" + 30 * b"\xff" + b"\x7f"],
    253: [
        b"\x10" + 29 * b"\x00" + b"\x01" + b"\xa5",
        b"\x1f" + 29 * b"\xff" + b"\xfe" + b"\xef",
    ],
    254: [b" " + 30 * b"\x00" + b"'", b"?" + 30 * b"\xff" + b"\x0b"],
    255: [b"@" + 30 * b"\x00" + b"O", b"\x7f" + 30 * b"\xff" + b"\xed"],
    256: [b"\x80" + 30 * b"\x00" + b"_", 31 * b"\xff" + b"C"],
    257: [
        b"\x01" + 30 * b"\x00" + b"\x01" + b")",
        b"\x01" + 31 * b"\xff" + b"\xa3",
    ],
    383: [b"@" + 46 * b"\x00" + b"\xff", b"\x7f" + 46 * b"\xff" + b"\xe1"],
    384: [
        b"\x80" + 45 * b"\x00" + b"\x01" + b"q",
        46 * b"\xff" + b"\xfe" + b"\xc3",
    ],
    385: [
        b"\x01" + 47 * b"\x00" + b"\xe7",
        b"\x01" + 46 * b"\xff" + b"\xfe" + b"\xf7",
    ],
    511: [b"@" + 62 * b"\x00" + b"\x0f", b"\x7f" + 62 * b"\xff" + b"E"],
    512: [b"\x80" + 62 * b"\x00" + b"o", 62 * b"\xff" + b"\xfd" + b"\xc7"],
    513: [
        b"\x01" + 63 * b"\x00" + b"K",
        b"\x01" + 62 * b"\xff" + b"\xfe" + b"C",
    ],
    1023: [
        b"@" + 125 * b"\x00" + b"\x05" + b"\xa3",
        b"\x7f" + 125 * b"\xff" + b"\xfe" + b"\x97",
    ],
    1024: [
        b"\x80" + 125 * b"\x00" + b"\x04" + b"\x83",
        127 * b"\xff" + b"\x97",
    ],
    1025: [
        b"\x01" + 126 * b"\x00" + b"\x02" + b"\x83",
        b"\x01" + 126 * b"\xff" + b"\xf5" + b"\x8f",
    ],
    1535: [
        b"@" + 189 * b"\x00" + b"\x01" + b"\t",
        b"\x7f" + 189 * b"\xff" + b"\xfa" + b"\xdb",
    ],
    1536: [
        b"\x80" + 189 * b"\x00" + b"\x02" + b"\xbb",
        190 * b"\xff" + b"\xf2" + b"\x83",
    ],
    1537: [b"\x01" + 191 * b"\x00" + b"K", b"\x01" + 191 * b"\xff" + b"Q"],
    2047: [
        b"@" + 253 * b"\x00" + b"\x10" + b"3",
        b"\x7f" + 254 * b"\xff" + b"\xab",
    ],
    2048: [
        b"\x80" + 253 * b"\x00" + b"\x07" + b"\x7f",
        254 * b"\xff" + b"\xf9" + b"\xeb",
    ],
    2049: [
        b"\x01" + 254 * b"\x00" + b"\x03" + b"\xd5",
        b"\x01" + 254 * b"\xff" + b"\xfa" + b"W",
    ],
    2559: [
        b"@" + 317 * b"\x00" + b"\x03" + b"\x87",
        b"\x7f" + 317 * b"\xff" + b"\xfd" + b";",
    ],
    2560: [
        b"\x80" + 317 * b"\x00" + b"\x08" + b"\xeb",
        319 * b"\xff" + b"\xb5",
    ],
    2561: [
        b"\x01" + 318 * b"\x00" + b"\x03" + b"\x87",
        b"\x01" + 318 * b"\xff" + b"\xfd" + b"\x99",
    ],
    3071: [b"@" + 382 * b"\x00" + b"\xc3", b"\x7f" + 382 * b"\xff" + b"E"],
    3072: [
        b"\x80" + 381 * b"\x00" + b"\x08" + b"\xf3",
        383 * b"\xff" + b"\xd1",
    ],
    3073: [
        b"\x01" + 382 * b"\x00" + b"\x03" + b"-",
        b"\x01" + 382 * b"\xff" + b"\xfc" + b"\xb7",
    ],
    3583: [
        b"@" + 445 * b"\x00" + b"\t" + b"'",
        b"\x7f" + 445 * b"\xff" + b"\xf3" + b"\xe7",
    ],
    3584: [
        b"\x80" + 445 * b"\x00" + b"\x10" + b"\xa3",
        446 * b"\xff" + b"\xfe" + b"S",
    ],
    3585: [
        b"\x01" + 447 * b"\x00" + b"\x15",
        b"\x01" + 446 * b"\xff" + b"\xfe" + b"e",
    ],
    4095: [
        b"@" + 509 * b"\x00" + b"\x01" + b"\xe9",
        b"\x7f" + 509 * b"\xff" + b"\xf9" + b"\xb1",
    ],
    4096: [
        b"\x80" + 509 * b"\x00" + b"\x02" + b"C",
        510 * b"\xff" + b"\xf6" + b"\x0b",
    ],
    4097: [
        b"\x01" + 510 * b"\x00" + b"\x06" + b"\xe1",
        b"\x01" + 510 * b"\xff" + b"\xfc" + b"\xf1",
    ],
}


for level in primes.values():
    level[0] = int.from_bytes(level[0], "big")
    level[1] = int.from_bytes(level[1], "big")


WORD_LIST = [
    "abandon",
    "ability",
    "able",
    "about",
    "above",
    "absent",
    "absorb",
    "abstract",
    "absurd",
    "abuse",
    "access",
    "accident",
    "account",
    "accuse",
    "achieve",
    "acid",
    "acoustic",
    "acquire",
    "across",
    "act",
    "action",
    "actor",
    "actress",
    "actual",
    "adapt",
    "add",
    "addict",
    "address",
    "adjust",
    "admit",
    "adult",
    "advance",
    "advice",
    "aerobic",
    "affair",
    "afford",
    "afraid",
    "again",
    "age",
    "agent",
    "agree",
    "ahead",
    "aim",
    "air",
    "airport",
    "aisle",
    "alarm",
    "album",
    "alcohol",
    "alert",
    "alien",
    "all",
    "alley",
    "allow",
    "almost",
    "alone",
    "alpha",
    "already",
    "also",
    "alter",
    "always",
    "amateur",
    "amazing",
    "among",
    "amount",
    "amused",
    "analyst",
    "anchor",
    "ancient",
    "anger",
    "angle",
    "angry",
    "animal",
    "ankle",
    "announce",
    "annual",
    "another",
    "answer",
    "antenna",
    "antique",
    "anxiety",
    "any",
    "apart",
    "apology",
    "appear",
    "apple",
    "approve",
    "april",
    "arch",
    "arctic",
    "area",
    "arena",
    "argue",
    "arm",
    "armed",
    "armor",
    "army",
    "around",
    "arrange",
    "arrest",
    "arrive",
    "arrow",
    "art",
    "artefact",
    "artist",
    "artwork",
    "ask",
    "aspect",
    "assault",
    "asset",
    "assist",
    "assume",
    "asthma",
    "athlete",
    "atom",
    "attack",
    "attend",
    "attitude",
    "attract",
    "auction",
    "audit",
    "august",
    "aunt",
    "author",
    "auto",
    "autumn",
    "average",
    "avocado",
    "avoid",
    "awake",
    "aware",
    "away",
    "awesome",
    "awful",
    "awkward",
    "axis",
    "baby",
    "bachelor",
    "bacon",
    "badge",
    "bag",
    "balance",
    "balcony",
    "ball",
    "bamboo",
    "banana",
    "banner",
    "bar",
    "barely",
    "bargain",
    "barrel",
    "base",
    "basic",
    "basket",
    "battle",
    "beach",
    "bean",
    "beauty",
    "because",
    "become",
    "beef",
    "before",
    "begin",
    "behave",
    "behind",
    "believe",
    "below",
    "belt",
    "bench",
    "benefit",
    "best",
    "betray",
    "better",
    "between",
    "beyond",
    "bicycle",
    "bid",
    "bike",
    "bind",
    "biology",
    "bird",
    "birth",
    "bitter",
    "black",
    "blade",
    "blame",
    "blanket",
    "blast",
    "bleak",
    "bless",
    "blind",
    "blood",
    "blossom",
    "blouse",
    "blue",
    "blur",
    "blush",
    "board",
    "boat",
    "body",
    "boil",
    "bomb",
    "bone",
    "bonus",
    "book",
    "boost",
    "border",
    "boring",
    "borrow",
    "boss",
    "bottom",
    "bounce",
    "box",
    "boy",
    "bracket",
    "brain",
    "brand",
    "brass",
    "brave",
    "bread",
    "breeze",
    "brick",
    "bridge",
    "brief",
    "bright",
    "bring",
    "brisk",
    "broccoli",
    "broken",
    "bronze",
    "broom",
    "brother",
    "brown",
    "brush",
    "bubble",
    "buddy",
    "budget",
    "buffalo",
    "build",
    "bulb",
    "bulk",
    "bullet",
    "bundle",
    "bunker",
    "burden",
    "burger",
    "burst",
    "bus",
    "business",
    "busy",
    "butter",
    "buyer",
    "buzz",
    "cabbage",
    "cabin",
    "cable",
    "cactus",
    "cage",
    "cake",
    "call",
    "calm",
    "camera",
    "camp",
    "can",
    "canal",
    "cancel",
    "candy",
    "cannon",
    "canoe",
    "canvas",
    "canyon",
    "capable",
    "capital",
    "captain",
    "car",
    "carbon",
    "card",
    "cargo",
    "carpet",
    "carry",
    "cart",
    "case",
    "cash",
    "casino",
    "castle",
    "casual",
    "cat",
    "catalog",
    "catch",
    "category",
    "cattle",
    "caught",
    "cause",
    "caution",
    "cave",
    "ceiling",
    "celery",
    "cement",
    "census",
    "century",
    "cereal",
    "certain",
    "chair",
    "chalk",
    "champion",
    "change",
    "chaos",
    "chapter",
    "charge",
    "chase",
    "chat",
    "cheap",
    "check",
    "cheese",
    "chef",
    "cherry",
    "chest",
    "chicken",
    "chief",
    "child",
    "chimney",
    "choice",
    "choose",
    "chronic",
    "chuckle",
    "chunk",
    "churn",
    "cigar",
    "cinnamon",
    "circle",
    "citizen",
    "city",
    "civil",
    "claim",
    "clap",
    "clarify",
    "claw",
    "clay",
    "clean",
    "clerk",
    "clever",
    "click",
    "client",
    "cliff",
    "climb",
    "clinic",
    "clip",
    "clock",
    "clog",
    "close",
    "cloth",
    "cloud",
    "clown",
    "club",
    "clump",
    "cluster",
    "clutch",
    "coach",
    "coast",
    "coconut",
    "code",
    "coffee",
    "coil",
    "coin",
    "collect",
    "color",
    "column",
    "combine",
    "come",
    "comfort",
    "comic",
    "common",
    "company",
    "concert",
    "conduct",
    "confirm",
    "congress",
    "connect",
    "consider",
    "control",
    "convince",
    "cook",
    "cool",
    "copper",
    "copy",
    "coral",
    "core",
    "corn",
    "correct",
    "cost",
    "cotton",
    "couch",
    "country",
    "couple",
    "course",
    "cousin",
    "cover",
    "coyote",
    "crack",
    "cradle",
    "craft",
    "cram",
    "crane",
    "crash",
    "crater",
    "crawl",
    "crazy",
    "cream",
    "credit",
    "creek",
    "crew",
    "cricket",
    "crime",
    "crisp",
    "critic",
    "crop",
    "cross",
    "crouch",
    "crowd",
    "crucial",
    "cruel",
    "cruise",
    "crumble",
    "crunch",
    "crush",
    "cry",
    "crystal",
    "cube",
    "culture",
    "cup",
    "cupboard",
    "curious",
    "current",
    "curtain",
    "curve",
    "cushion",
    "custom",
    "cute",
    "cycle",
    "dad",
    "damage",
    "damp",
    "dance",
    "danger",
    "daring",
    "dash",
    "daughter",
    "dawn",
    "day",
    "deal",
    "debate",
    "debris",
    "decade",
    "december",
    "decide",
    "decline",
    "decorate",
    "decrease",
    "deer",
    "defense",
    "define",
    "defy",
    "degree",
    "delay",
    "deliver",
    "demand",
    "demise",
    "denial",
    "dentist",
    "deny",
    "depart",
    "depend",
    "deposit",
    "depth",
    "deputy",
    "derive",
    "describe",
    "desert",
    "design",
    "desk",
    "despair",
    "destroy",
    "detail",
    "detect",
    "develop",
    "device",
    "devote",
    "diagram",
    "dial",
    "diamond",
    "diary",
    "dice",
    "diesel",
    "diet",
    "differ",
    "digital",
    "dignity",
    "dilemma",
    "dinner",
    "dinosaur",
    "direct",
    "dirt",
    "disagree",
    "discover",
    "disease",
    "dish",
    "dismiss",
    "disorder",
    "display",
    "distance",
    "divert",
    "divide",
    "divorce",
    "dizzy",
    "doctor",
    "document",
    "dog",
    "doll",
    "dolphin",
    "domain",
    "donate",
    "donkey",
    "donor",
    "door",
    "dose",
    "double",
    "dove",
    "draft",
    "dragon",
    "drama",
    "drastic",
    "draw",
    "dream",
    "dress",
    "drift",
    "drill",
    "drink",
    "drip",
    "drive",
    "drop",
    "drum",
    "dry",
    "duck",
    "dumb",
    "dune",
    "during",
    "dust",
    "dutch",
    "duty",
    "dwarf",
    "dynamic",
    "eager",
    "eagle",
    "early",
    "earn",
    "earth",
    "easily",
    "east",
    "easy",
    "echo",
    "ecology",
    "economy",
    "edge",
    "edit",
    "educate",
    "effort",
    "egg",
    "eight",
    "either",
    "elbow",
    "elder",
    "electric",
    "elegant",
    "element",
    "elephant",
    "elevator",
    "elite",
    "else",
    "embark",
    "embody",
    "embrace",
    "emerge",
    "emotion",
    "employ",
    "empower",
    "empty",
    "enable",
    "enact",
    "end",
    "endless",
    "endorse",
    "enemy",
    "energy",
    "enforce",
    "engage",
    "engine",
    "enhance",
    "enjoy",
    "enlist",
    "enough",
    "enrich",
    "enroll",
    "ensure",
    "enter",
    "entire",
    "entry",
    "envelope",
    "episode",
    "equal",
    "equip",
    "era",
    "erase",
    "erode",
    "erosion",
    "error",
    "erupt",
    "escape",
    "essay",
    "essence",
    "estate",
    "eternal",
    "ethics",
    "evidence",
    "evil",
    "evoke",
    "evolve",
    "exact",
    "example",
    "excess",
    "exchange",
    "excite",
    "exclude",
    "excuse",
    "execute",
    "exercise",
    "exhaust",
    "exhibit",
    "exile",
    "exist",
    "exit",
    "exotic",
    "expand",
    "expect",
    "expire",
    "explain",
    "expose",
    "express",
    "extend",
    "extra",
    "eye",
    "eyebrow",
    "fabric",
    "face",
    "faculty",
    "fade",
    "faint",
    "faith",
    "fall",
    "false",
    "fame",
    "family",
    "famous",
    "fan",
    "fancy",
    "fantasy",
    "farm",
    "fashion",
    "fat",
    "fatal",
    "father",
    "fatigue",
    "fault",
    "favorite",
    "feature",
    "february",
    "federal",
    "fee",
    "feed",
    "feel",
    "female",
    "fence",
    "festival",
    "fetch",
    "fever",
    "few",
    "fiber",
    "fiction",
    "field",
    "figure",
    "file",
    "film",
    "filter",
    "final",
    "find",
    "fine",
    "finger",
    "finish",
    "fire",
    "firm",
    "first",
    "fiscal",
    "fish",
    "fit",
    "fitness",
    "fix",
    "flag",
    "flame",
    "flash",
    "flat",
    "flavor",
    "flee",
    "flight",
    "flip",
    "float",
    "flock",
    "floor",
    "flower",
    "fluid",
    "flush",
    "fly",
    "foam",
    "focus",
    "fog",
    "foil",
    "fold",
    "follow",
    "food",
    "foot",
    "force",
    "forest",
    "forget",
    "fork",
    "fortune",
    "forum",
    "forward",
    "fossil",
    "foster",
    "found",
    "fox",
    "fragile",
    "frame",
    "frequent",
    "fresh",
    "friend",
    "fringe",
    "frog",
    "front",
    "frost",
    "frown",
    "frozen",
    "fruit",
    "fuel",
    "fun",
    "funny",
    "furnace",
    "fury",
    "future",
    "gadget",
    "gain",
    "galaxy",
    "gallery",
    "game",
    "gap",
    "garage",
    "garbage",
    "garden",
    "garlic",
    "garment",
    "gas",
    "gasp",
    "gate",
    "gather",
    "gauge",
    "gaze",
    "general",
    "genius",
    "genre",
    "gentle",
    "genuine",
    "gesture",
    "ghost",
    "giant",
    "gift",
    "giggle",
    "ginger",
    "giraffe",
    "girl",
    "give",
    "glad",
    "glance",
    "glare",
    "glass",
    "glide",
    "glimpse",
    "globe",
    "gloom",
    "glory",
    "glove",
    "glow",
    "glue",
    "goat",
    "goddess",
    "gold",
    "good",
    "goose",
    "gorilla",
    "gospel",
    "gossip",
    "govern",
    "gown",
    "grab",
    "grace",
    "grain",
    "grant",
    "grape",
    "grass",
    "gravity",
    "great",
    "green",
    "grid",
    "grief",
    "grit",
    "grocery",
    "group",
    "grow",
    "grunt",
    "guard",
    "guess",
    "guide",
    "guilt",
    "guitar",
    "gun",
    "gym",
    "habit",
    "hair",
    "half",
    "hammer",
    "hamster",
    "hand",
    "happy",
    "harbor",
    "hard",
    "harsh",
    "harvest",
    "hat",
    "have",
    "hawk",
    "hazard",
    "head",
    "health",
    "heart",
    "heavy",
    "hedgehog",
    "height",
    "hello",
    "helmet",
    "help",
    "hen",
    "hero",
    "hidden",
    "high",
    "hill",
    "hint",
    "hip",
    "hire",
    "history",
    "hobby",
    "hockey",
    "hold",
    "hole",
    "holiday",
    "hollow",
    "home",
    "honey",
    "hood",
    "hope",
    "horn",
    "horror",
    "horse",
    "hospital",
    "host",
    "hotel",
    "hour",
    "hover",
    "hub",
    "huge",
    "human",
    "humble",
    "humor",
    "hundred",
    "hungry",
    "hunt",
    "hurdle",
    "hurry",
    "hurt",
    "husband",
    "hybrid",
    "ice",
    "icon",
    "idea",
    "identify",
    "idle",
    "ignore",
    "ill",
    "illegal",
    "illness",
    "image",
    "imitate",
    "immense",
    "immune",
    "impact",
    "impose",
    "improve",
    "impulse",
    "inch",
    "include",
    "income",
    "increase",
    "index",
    "indicate",
    "indoor",
    "industry",
    "infant",
    "inflict",
    "inform",
    "inhale",
    "inherit",
    "initial",
    "inject",
    "injury",
    "inmate",
    "inner",
    "innocent",
    "input",
    "inquiry",
    "insane",
    "insect",
    "inside",
    "inspire",
    "install",
    "intact",
    "interest",
    "into",
    "invest",
    "invite",
    "involve",
    "iron",
    "island",
    "isolate",
    "issue",
    "item",
    "ivory",
    "jacket",
    "jaguar",
    "jar",
    "jazz",
    "jealous",
    "jeans",
    "jelly",
    "jewel",
    "job",
    "join",
    "joke",
    "journey",
    "joy",
    "judge",
    "juice",
    "jump",
    "jungle",
    "junior",
    "junk",
    "just",
    "kangaroo",
    "keen",
    "keep",
    "ketchup",
    "key",
    "kick",
    "kid",
    "kidney",
    "kind",
    "kingdom",
    "kiss",
    "kit",
    "kitchen",
    "kite",
    "kitten",
    "kiwi",
    "knee",
    "knife",
    "knock",
    "know",
    "lab",
    "label",
    "labor",
    "ladder",
    "lady",
    "lake",
    "lamp",
    "language",
    "laptop",
    "large",
    "later",
    "latin",
    "laugh",
    "laundry",
    "lava",
    "law",
    "lawn",
    "lawsuit",
    "layer",
    "lazy",
    "leader",
    "leaf",
    "learn",
    "leave",
    "lecture",
    "left",
    "leg",
    "legal",
    "legend",
    "leisure",
    "lemon",
    "lend",
    "length",
    "lens",
    "leopard",
    "lesson",
    "letter",
    "level",
    "liar",
    "liberty",
    "library",
    "license",
    "life",
    "lift",
    "light",
    "like",
    "limb",
    "limit",
    "link",
    "lion",
    "liquid",
    "list",
    "little",
    "live",
    "lizard",
    "load",
    "loan",
    "lobster",
    "local",
    "lock",
    "logic",
    "lonely",
    "long",
    "loop",
    "lottery",
    "loud",
    "lounge",
    "love",
    "loyal",
    "lucky",
    "luggage",
    "lumber",
    "lunar",
    "lunch",
    "luxury",
    "lyrics",
    "machine",
    "mad",
    "magic",
    "magnet",
    "maid",
    "mail",
    "main",
    "major",
    "make",
    "mammal",
    "man",
    "manage",
    "mandate",
    "mango",
    "mansion",
    "manual",
    "maple",
    "marble",
    "march",
    "margin",
    "marine",
    "market",
    "marriage",
    "mask",
    "mass",
    "master",
    "match",
    "material",
    "math",
    "matrix",
    "matter",
    "maximum",
    "maze",
    "meadow",
    "mean",
    "measure",
    "meat",
    "mechanic",
    "medal",
    "media",
    "melody",
    "melt",
    "member",
    "memory",
    "mention",
    "menu",
    "mercy",
    "merge",
    "merit",
    "merry",
    "mesh",
    "message",
    "metal",
    "method",
    "middle",
    "midnight",
    "milk",
    "million",
    "mimic",
    "mind",
    "minimum",
    "minor",
    "minute",
    "miracle",
    "mirror",
    "misery",
    "miss",
    "mistake",
    "mix",
    "mixed",
    "mixture",
    "mobile",
    "model",
    "modify",
    "mom",
    "moment",
    "monitor",
    "monkey",
    "monster",
    "month",
    "moon",
    "moral",
    "more",
    "morning",
    "mosquito",
    "mother",
    "motion",
    "motor",
    "mountain",
    "mouse",
    "move",
    "movie",
    "much",
    "muffin",
    "mule",
    "multiply",
    "muscle",
    "museum",
    "mushroom",
    "music",
    "must",
    "mutual",
    "myself",
    "mystery",
    "myth",
    "naive",
    "name",
    "napkin",
    "narrow",
    "nasty",
    "nation",
    "nature",
    "near",
    "neck",
    "need",
    "negative",
    "neglect",
    "neither",
    "nephew",
    "nerve",
    "nest",
    "net",
    "network",
    "neutral",
    "never",
    "news",
    "next",
    "nice",
    "night",
    "noble",
    "noise",
    "nominee",
    "noodle",
    "normal",
    "north",
    "nose",
    "notable",
    "note",
    "nothing",
    "notice",
    "novel",
    "now",
    "nuclear",
    "number",
    "nurse",
    "nut",
    "oak",
    "obey",
    "object",
    "oblige",
    "obscure",
    "observe",
    "obtain",
    "obvious",
    "occur",
    "ocean",
    "october",
    "odor",
    "off",
    "offer",
    "office",
    "often",
    "oil",
    "okay",
    "old",
    "olive",
    "olympic",
    "omit",
    "once",
    "one",
    "onion",
    "online",
    "only",
    "open",
    "opera",
    "opinion",
    "oppose",
    "option",
    "orange",
    "orbit",
    "orchard",
    "order",
    "ordinary",
    "organ",
    "orient",
    "original",
    "orphan",
    "ostrich",
    "other",
    "outdoor",
    "outer",
    "output",
    "outside",
    "oval",
    "oven",
    "over",
    "own",
    "owner",
    "oxygen",
    "oyster",
    "ozone",
    "pact",
    "paddle",
    "page",
    "pair",
    "palace",
    "palm",
    "panda",
    "panel",
    "panic",
    "panther",
    "paper",
    "parade",
    "parent",
    "park",
    "parrot",
    "party",
    "pass",
    "patch",
    "path",
    "patient",
    "patrol",
    "pattern",
    "pause",
    "pave",
    "payment",
    "peace",
    "peanut",
    "pear",
    "peasant",
    "pelican",
    "pen",
    "penalty",
    "pencil",
    "people",
    "pepper",
    "perfect",
    "permit",
    "person",
    "pet",
    "phone",
    "photo",
    "phrase",
    "physical",
    "piano",
    "picnic",
    "picture",
    "piece",
    "pig",
    "pigeon",
    "pill",
    "pilot",
    "pink",
    "pioneer",
    "pipe",
    "pistol",
    "pitch",
    "pizza",
    "place",
    "planet",
    "plastic",
    "plate",
    "play",
    "please",
    "pledge",
    "pluck",
    "plug",
    "plunge",
    "poem",
    "poet",
    "point",
    "polar",
    "pole",
    "police",
    "pond",
    "pony",
    "pool",
    "popular",
    "portion",
    "position",
    "possible",
    "post",
    "potato",
    "pottery",
    "poverty",
    "powder",
    "power",
    "practice",
    "praise",
    "predict",
    "prefer",
    "prepare",
    "present",
    "pretty",
    "prevent",
    "price",
    "pride",
    "primary",
    "print",
    "priority",
    "prison",
    "private",
    "prize",
    "problem",
    "process",
    "produce",
    "profit",
    "program",
    "project",
    "promote",
    "proof",
    "property",
    "prosper",
    "protect",
    "proud",
    "provide",
    "public",
    "pudding",
    "pull",
    "pulp",
    "pulse",
    "pumpkin",
    "punch",
    "pupil",
    "puppy",
    "purchase",
    "purity",
    "purpose",
    "purse",
    "push",
    "put",
    "puzzle",
    "pyramid",
    "quality",
    "quantum",
    "quarter",
    "question",
    "quick",
    "quit",
    "quiz",
    "quote",
    "rabbit",
    "raccoon",
    "race",
    "rack",
    "radar",
    "radio",
    "rail",
    "rain",
    "raise",
    "rally",
    "ramp",
    "ranch",
    "random",
    "range",
    "raaad",
    "rare",
    "rate",
    "rather",
    "raven",
    "raw",
    "razor",
    "ready",
    "real",
    "reason",
    "rebel",
    "rebuild",
    "recall",
    "receive",
    "recipe",
    "record",
    "recycle",
    "reduce",
    "reflect",
    "reform",
    "refuse",
    "region",
    "regret",
    "regular",
    "reject",
    "relax",
    "release",
    "relief",
    "rely",
    "remain",
    "remember",
    "remind",
    "remove",
    "render",
    "renew",
    "rent",
    "reopen",
    "repair",
    "repeat",
    "replace",
    "report",
    "require",
    "rescue",
    "resemble",
    "resist",
    "resource",
    "response",
    "result",
    "retire",
    "retreat",
    "return",
    "reunion",
    "reveal",
    "review",
    "reward",
    "rhythm",
    "rib",
    "ribbon",
    "rice",
    "rich",
    "ride",
    "ridge",
    "rifle",
    "right",
    "rigid",
    "ring",
    "riot",
    "ripple",
    "risk",
    "ritual",
    "rival",
    "river",
    "road",
    "roast",
    "robot",
    "robust",
    "rocket",
    "romance",
    "roof",
    "rookie",
    "room",
    "rose",
    "rotate",
    "rough",
    "round",
    "route",
    "royal",
    "rubber",
    "rude",
    "rug",
    "rule",
    "run",
    "runway",
    "rural",
    "sad",
    "saddle",
    "sadness",
    "safe",
    "sail",
    "salad",
    "salmon",
    "salon",
    "salt",
    "salute",
    "same",
    "sample",
    "sand",
    "satisfy",
    "satoshi",
    "sauce",
    "sausage",
    "save",
    "say",
    "scale",
    "scan",
    "scare",
    "scatter",
    "scene",
    "scheme",
    "school",
    "science",
    "scissors",
    "scorpion",
    "scout",
    "scrap",
    "screen",
    "script",
    "scrub",
    "sea",
    "search",
    "season",
    "seat",
    "second",
    "secret",
    "section",
    "security",
    "seed",
    "seek",
    "segment",
    "select",
    "sell",
    "seminar",
    "senior",
    "sense",
    "sentence",
    "series",
    "service",
    "session",
    "settle",
    "setup",
    "seven",
    "shadow",
    "shaft",
    "shallow",
    "share",
    "shed",
    "shell",
    "sheriff",
    "shield",
    "shift",
    "shine",
    "ship",
    "shiver",
    "shock",
    "shoe",
    "shoot",
    "shop",
    "short",
    "shoulder",
    "shove",
    "shrimp",
    "shrug",
    "shuffle",
    "shy",
    "sibling",
    "sick",
    "side",
    "siege",
    "sight",
    "sign",
    "silent",
    "silk",
    "silly",
    "silver",
    "similar",
    "simple",
    "since",
    "sing",
    "siren",
    "sister",
    "situate",
    "six",
    "size",
    "skate",
    "sketch",
    "ski",
    "skill",
    "skin",
    "skirt",
    "skull",
    "slab",
    "slam",
    "sleep",
    "slender",
    "slice",
    "slide",
    "slight",
    "slim",
    "slogan",
    "slot",
    "slow",
    "slush",
    "small",
    "smart",
    "smile",
    "smoke",
    "smooth",
    "snack",
    "snake",
    "snap",
    "sniff",
    "snow",
    "soap",
    "soccer",
    "social",
    "sock",
    "soda",
    "soft",
    "solar",
    "soldier",
    "solid",
    "solution",
    "solve",
    "someone",
    "song",
    "soon",
    "sorry",
    "sort",
    "soul",
    "sound",
    "soup",
    "source",
    "south",
    "space",
    "spare",
    "spatial",
    "spawn",
    "speak",
    "special",
    "speed",
    "spell",
    "spend",
    "sphere",
    "spice",
    "saader",
    "spike",
    "spin",
    "spirit",
    "split",
    "spoil",
    "sponsor",
    "spoon",
    "sport",
    "spot",
    "spray",
    "spread",
    "spring",
    "spy",
    "square",
    "squeeze",
    "squirrel",
    "stable",
    "stadium",
    "staff",
    "stage",
    "stairs",
    "stamp",
    "stand",
    "start",
    "state",
    "stay",
    "steak",
    "steel",
    "stem",
    "step",
    "stereo",
    "stick",
    "still",
    "sting",
    "stock",
    "stomach",
    "stone",
    "stool",
    "story",
    "stove",
    "strategy",
    "street",
    "strike",
    "strong",
    "struggle",
    "student",
    "stuff",
    "stumble",
    "style",
    "subject",
    "submit",
    "subway",
    "success",
    "such",
    "sudden",
    "suffer",
    "sugar",
    "suggest",
    "suit",
    "summer",
    "sun",
    "sunny",
    "sunset",
    "super",
    "supply",
    "supreme",
    "sure",
    "surface",
    "surge",
    "surprise",
    "surround",
    "survey",
    "suspect",
    "sustain",
    "swallow",
    "swamp",
    "swap",
    "swarm",
    "swear",
    "sweet",
    "swift",
    "swim",
    "swing",
    "switch",
    "sword",
    "symbol",
    "symptom",
    "syrup",
    "system",
    "table",
    "tackle",
    "tag",
    "tail",
    "talent",
    "talk",
    "tank",
    "tape",
    "target",
    "task",
    "taste",
    "tattoo",
    "taxi",
    "teach",
    "team",
    "tell",
    "ten",
    "tenant",
    "tennis",
    "tent",
    "term",
    "test",
    "text",
    "thank",
    "that",
    "theme",
    "then",
    "theory",
    "there",
    "they",
    "thing",
    "this",
    "thought",
    "three",
    "thrive",
    "throw",
    "thumb",
    "thunder",
    "ticket",
    "tide",
    "tiger",
    "tilt",
    "timber",
    "time",
    "tiny",
    "tip",
    "tired",
    "tissue",
    "title",
    "toast",
    "tobacco",
    "today",
    "toddler",
    "toe",
    "together",
    "toilet",
    "token",
    "tomato",
    "tomorrow",
    "tone",
    "tongue",
    "tonight",
    "tool",
    "tooth",
    "top",
    "topic",
    "topple",
    "torch",
    "tornado",
    "tortoise",
    "toss",
    "total",
    "tourist",
    "toward",
    "tower",
    "town",
    "toy",
    "track",
    "trade",
    "traffic",
    "tragic",
    "train",
    "transfer",
    "trap",
    "trash",
    "travel",
    "tray",
    "treat",
    "tree",
    "trend",
    "trial",
    "tribe",
    "trick",
    "trigger",
    "trim",
    "trip",
    "trophy",
    "trouble",
    "truck",
    "true",
    "truly",
    "trumpet",
    "trust",
    "truth",
    "try",
    "tube",
    "tuition",
    "tumble",
    "tuna",
    "tunnel",
    "turkey",
    "turn",
    "turtle",
    "twelve",
    "twenty",
    "twice",
    "twin",
    "twist",
    "two",
    "type",
    "typical",
    "ugly",
    "umbrella",
    "unable",
    "unaware",
    "uncle",
    "uncover",
    "under",
    "undo",
    "unfair",
    "unfold",
    "unhappy",
    "uniform",
    "unique",
    "unit",
    "universe",
    "unknown",
    "unlock",
    "until",
    "unusual",
    "unveil",
    "update",
    "upgrade",
    "uphold",
    "upon",
    "upper",
    "upset",
    "urban",
    "urge",
    "usage",
    "use",
    "used",
    "useful",
    "useless",
    "usual",
    "utility",
    "vacant",
    "vacuum",
    "vague",
    "valid",
    "valley",
    "valve",
    "van",
    "vanish",
    "vapor",
    "various",
    "vast",
    "vault",
    "vehicle",
    "velvet",
    "vendor",
    "venture",
    "venue",
    "verb",
    "verify",
    "version",
    "very",
    "vessel",
    "veteran",
    "viable",
    "vibrant",
    "vicious",
    "victory",
    "video",
    "view",
    "village",
    "vintage",
    "violin",
    "virtual",
    "virus",
    "visa",
    "visit",
    "visual",
    "vital",
    "vivid",
    "vocal",
    "voice",
    "void",
    "volcano",
    "volume",
    "vote",
    "voyage",
    "wage",
    "wagon",
    "wait",
    "walk",
    "wall",
    "walnut",
    "want",
    "warfare",
    "warm",
    "warrior",
    "wash",
    "wasp",
    "waste",
    "water",
    "wave",
    "way",
    "wealth",
    "weapon",
    "wear",
    "weasel",
    "weather",
    "web",
    "wedding",
    "weekend",
    "weird",
    "welcome",
    "west",
    "wet",
    "whale",
    "what",
    "wheat",
    "wheel",
    "when",
    "where",
    "whip",
    "whisper",
    "wide",
    "width",
    "wife",
    "wild",
    "will",
    "win",
    "window",
    "wine",
    "wing",
    "wink",
    "winner",
    "winter",
    "wire",
    "wisdom",
    "wise",
    "wish",
    "witness",
    "wolf",
    "woman",
    "wonder",
    "wood",
    "wool",
    "word",
    "work",
    "world",
    "worry",
    "worth",
    "wrap",
    "wreck",
    "wrestle",
    "wrist",
    "write",
    "wrong",
    "yard",
    "year",
    "yellow",
    "you",
    "young",
    "youth",
    "zebra",
    "zero",
    "zone",
    "zoo",
]

