"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const ecs = require("aws-cdk-lib/aws-ecs");
const assertions_1 = require("aws-cdk-lib/assertions");
const test_helper_1 = require("./test-helper");
const ec2 = require("aws-cdk-lib/aws-ec2");
test('New service/new bucket, public API, new VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const clusterName = "custom-cluster-name";
    const containerName = "custom-container-name";
    const serviceName = "custom-service-name";
    const familyName = "family-name";
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const construct = new lib_1.FargateToKinesisFirehose(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr('172.0.0.0/16') },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingKinesisFirehose: destination.kinesisFirehose
    });
    expect(construct.vpc !== null);
    expect(construct.service !== null);
    expect(construct.container !== null);
    expect(construct.kinesisFirehose !== null);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: familyName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: clusterName
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "firehose:DeleteDeliveryStream",
                        "firehose:PutRecord",
                        "firehose:PutRecordBatch",
                        "firehose:UpdateDestination"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testdestinationKinesisFirehose5F7F1053",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    // Confirm we created a Public/Private VPC
    expect(stack).toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('New service/new bucket, private API, new VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr('172.0.0.0/16') },
        existingKinesisFirehose: destination.kinesisFirehose
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "firehose:DeleteDeliveryStream",
                        "firehose:PutRecord",
                        "firehose:PutRecordBatch",
                        "firehose:UpdateDestination"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testdestinationKinesisFirehose5F7F1053",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
    // Confirm we created an Isolated VPC
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::S3::Bucket', 2);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('Default construct has all expected properties', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const construct = new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose
    });
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.kinesisFirehose).toBeDefined();
});
test('Construct deploys Fargate Service in isolated subnets when publicApi is set to false', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose
    });
    const template = assertions_1.Template.fromStack(stack);
    // The default isolated VPC should have two subnets, 2 route tables, and no nat/internet gateway, or routes
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::EC2::Subnet', 2);
    expect(stack).toCountResources('AWS::EC2::RouteTable', 2);
    expect(stack).toCountResources('AWS::EC2::Route', 0);
    expect(stack).toCountResources('AWS::EC2::NatGateway', 0);
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 0);
    // Get the first isolated subnet
    const subnet1Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/IsolatedSubnet1")
                }
            ]
        }
    });
    const [subnet1] = Object.keys(subnet1Resource);
    // Get the second isolated subnet
    const subnet2Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/IsolatedSubnet2")
                }
            ]
        }
    });
    const [subnet2] = Object.keys(subnet2Resource);
    // Verify the Fargate Service is deployed into the two isolated subnets
    expect(stack).toHaveResourceLike('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: subnet1
                    },
                    {
                        Ref: subnet2
                    }
                ]
            },
        }
    });
});
test('Construct deploys Fargate Service in private subnets when publicApi is set to true', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose
    });
    // The default public/private VPC should have 4 subnets (two public, two private)
    // 4 route tables, 4 routes, 2 NAT Gateways and 1 Internet Gateway
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::EC2::Subnet', 4);
    expect(stack).toCountResources('AWS::EC2::RouteTable', 4);
    expect(stack).toCountResources('AWS::EC2::Route', 4);
    expect(stack).toCountResources('AWS::EC2::NatGateway', 2);
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 1);
    const template = assertions_1.Template.fromStack(stack);
    // Get the first private subnet
    const privateSubnet1Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/PrivateSubnet1")
                }
            ]
        }
    });
    const [privateSubnet1] = Object.keys(privateSubnet1Resource);
    // Get the second private subnet
    const privateSubnet2Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/PrivateSubnet2")
                }
            ]
        }
    });
    const [privateSubnet2] = Object.keys(privateSubnet2Resource);
    // Verify the Fargate Service is deployed into the two private subnets
    expect(stack).toHaveResourceLike('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: privateSubnet1
                    },
                    {
                        Ref: privateSubnet2
                    }
                ]
            },
        }
    });
});
test('Construct uses vpcProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose,
        vpcProps: {
            vpcName: 'my-vpc'
        }
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct uses existingVpc when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const existingVpc = defaults.getTestVpc(stack, false, {
        vpcName: 'my-vpc'
    });
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose,
        existingVpc
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct creates VPC Interface Endpoints for ECR and Kinesis Streams', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.kinesis-firehose'
                ]
            ]
        },
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.ecr.api'
                ]
            ]
        },
    });
});
test('Container has default stream name environment variable', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'FIREHOSE_DELIVERYSTREAM_NAME',
                        Value: destination.kinesisFirehose.deliveryStreamName
                    }
                ],
            }
        ]
    });
});
test('Container stream name environment variable can be overridden', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        firehoseEnvironmentVariableName: 'my-stream-name'
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'my-stream-name',
                        Value: destination.kinesisFirehose.deliveryStreamName
                    }
                ],
            }
        ]
    });
});
test('Construct grants PutRecord permission for the Fargate Service to write to the Kinesis Stream', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "firehose:DeleteDeliveryStream",
                        "firehose:PutRecord",
                        "firehose:PutRecordBatch",
                        "firehose:UpdateDestination"
                    ],
                    Effect: 'Allow',
                    Resource: {
                        "Fn::GetAtt": [
                            "testdestinationKinesisFirehose5F7F1053",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Construct defaults to the latest version of the ECR image', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:latest'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses ecrImageVersion when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        ecrImageVersion: 'my-version'
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:my-version'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses clusterProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: {
            clusterName: 'my-cluster'
        }
    });
    expect(stack).toCountResources('AWS::ECS::Cluster', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
});
test('Construct uses containerDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        containerDefinitionProps: {
            containerName: 'my-container'
        }
    });
    expect(stack).toCountResources('AWS::ECS::TaskDefinition', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
});
test('Construct uses fargateTaskDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateTaskDefinitionProps: {
            family: 'my-family'
        }
    });
    expect(stack).toCountResources('AWS::ECS::TaskDefinition', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: 'my-family'
    });
});
test('Construct uses fargateServiceProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateServiceProps: {
            serviceName: 'my-service',
            desiredCount: 7
        }
    });
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: 'my-service',
        DesiredCount: 7
    });
});
test('Construct uses existingFargateServiceObject when provided', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const existingVpc = defaults.getTestVpc(stack);
    const [existingFargateServiceObject, existingContainerDefinitionObject] = defaults.CreateFargateService(stack, 'test-existing-fargate-service', existingVpc, { clusterName: 'my-cluster' }, defaults.fakeEcrRepoArn, undefined, { family: 'my-family' }, { containerName: 'my-container' }, { serviceName: 'my-service' });
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingVpc,
        existingKinesisFirehose: destination.kinesisFirehose,
        existingFargateServiceObject,
        existingContainerDefinitionObject
    });
    expect(stack).toCountResources('AWS::ECS::Cluster', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
    expect(stack).toCountResources('AWS::ECS::TaskDefinition', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: 'my-family',
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: 'my-service',
    });
});
test('Test fail if existingFirehose does not have a stream name', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const mutatedFirehose = defaults.overrideProps(destination.kinesisFirehose, {});
    mutatedFirehose.deliveryStreamName = undefined;
    const app = () => {
        new lib_1.FargateToKinesisFirehose(stack, 'test-target', {
            existingKinesisFirehose: mutatedFirehose,
            publicApi: false,
            ecrRepositoryArn: defaults.fakeEcrRepoArn,
        });
    };
    expect(app).toThrowError(/existingKinesisFirehose must have a defined deliveryStreamName/);
});
//# sourceMappingURL=data:application/json;base64,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