import functools
import json
import pytest

from platone.packages.eth_utils import (
    event_signature_to_log_topic,
)

CONTRACT_CODE = "0x606060405261022e806100126000396000f360606040523615610074576000357c01000000000000000000000000000000000000000000000000000000009004806316216f391461007657806361bc221a146100995780637cf5dab0146100bc578063a5f3c23b146100e8578063d09de08a1461011d578063dcf537b11461014057610074565b005b610083600480505061016c565b6040518082815260200191505060405180910390f35b6100a6600480505061017f565b6040518082815260200191505060405180910390f35b6100d26004808035906020019091905050610188565b6040518082815260200191505060405180910390f35b61010760048080359060200190919080359060200190919050506101ea565b6040518082815260200191505060405180910390f35b61012a6004805050610201565b6040518082815260200191505060405180910390f35b6101566004808035906020019091905050610217565b6040518082815260200191505060405180910390f35b6000600d9050805080905061017c565b90565b60006000505481565b6000816000600082828250540192505081905550600060005054905080507f3496c3ede4ec3ab3686712aa1c238593ea6a42df83f98a5ec7df9834cfa577c5816040518082815260200191505060405180910390a18090506101e5565b919050565b6000818301905080508090506101fb565b92915050565b600061020d6001610188565b9050610214565b90565b60006007820290508050809050610229565b91905056"  # noqa: E501


CONTRACT_RUNTIME = "0x60606040523615610074576000357c01000000000000000000000000000000000000000000000000000000009004806316216f391461007657806361bc221a146100995780637cf5dab0146100bc578063a5f3c23b146100e8578063d09de08a1461011d578063dcf537b11461014057610074565b005b610083600480505061016c565b6040518082815260200191505060405180910390f35b6100a6600480505061017f565b6040518082815260200191505060405180910390f35b6100d26004808035906020019091905050610188565b6040518082815260200191505060405180910390f35b61010760048080359060200190919080359060200190919050506101ea565b6040518082815260200191505060405180910390f35b61012a6004805050610201565b6040518082815260200191505060405180910390f35b6101566004808035906020019091905050610217565b6040518082815260200191505060405180910390f35b6000600d9050805080905061017c565b90565b60006000505481565b6000816000600082828250540192505081905550600060005054905080507f3496c3ede4ec3ab3686712aa1c238593ea6a42df83f98a5ec7df9834cfa577c5816040518082815260200191505060405180910390a18090506101e5565b919050565b6000818301905080508090506101fb565b92915050565b600061020d6001610188565b9050610214565b90565b60006007820290508050809050610229565b91905056"  # noqa: E501


CONTRACT_ABI = json.loads('[{"constant":false,"inputs":[],"name":"return13","outputs":[{"name":"result","type":"int256"}],"type":"function"},{"constant":true,"inputs":[],"name":"counter","outputs":[{"name":"","type":"uint256"}],"type":"function"},{"constant":false,"inputs":[{"name":"amt","type":"uint256"}],"name":"increment","outputs":[{"name":"result","type":"uint256"}],"type":"function"},{"constant":false,"inputs":[{"name":"a","type":"int256"},{"name":"b","type":"int256"}],"name":"add","outputs":[{"name":"result","type":"int256"}],"type":"function"},{"constant":false,"inputs":[],"name":"increment","outputs":[{"name":"","type":"uint256"}],"type":"function"},{"constant":false,"inputs":[{"name":"a","type":"int256"}],"name":"multiply7","outputs":[{"name":"result","type":"int256"}],"type":"function"},{"anonymous":false,"inputs":[{"indexed":false,"name":"value","type":"uint256"}],"name":"Increased","type":"event"}]')  # noqa: E501


@pytest.fixture(scope="session")
def MATH_CODE():
    return CONTRACT_CODE


@pytest.fixture(scope="session")
def MATH_RUNTIME():
    return CONTRACT_RUNTIME


@pytest.fixture(scope="session")
def MATH_ABI():
    return CONTRACT_ABI


@pytest.fixture()
def MathContract(web3, MATH_ABI, MATH_CODE, MATH_RUNTIME):
    return web3.platone.contract(
        abi=MATH_ABI,
        bytecode=MATH_CODE,
        bytecode_runtime=MATH_RUNTIME,
    )


CONTRACT_SIMPLE_CONSTRUCTOR_CODE = '0x60606040526003600055602c8060156000396000f3606060405260e060020a600035046373d4a13a8114601a575b005b602260005481565b6060908152602090f3'  # noqa: E501
CONTRACT_SIMPLE_CONSTRUCTOR_RUNTIME = '0x606060405260e060020a600035046373d4a13a8114601a575b005b602260005481565b6060908152602090f3'  # noqa: E501
CONTRACT_SIMPLE_CONSTRUCTOR_ABI = json.loads('[{"constant":true,"inputs":[],"name":"data","outputs":[{"name":"","type":"uint256"}],"type":"function"},{"inputs":[],"type":"constructor"}]')  # noqa: E501


@pytest.fixture(scope="session")
def SIMPLE_CONSTRUCTOR_CODE():
    return CONTRACT_SIMPLE_CONSTRUCTOR_CODE


@pytest.fixture(scope="session")
def SIMPLE_CONSTRUCTOR_RUNTIME():
    return CONTRACT_SIMPLE_CONSTRUCTOR_RUNTIME


@pytest.fixture(scope="session")
def SIMPLE_CONSTRUCTOR_ABI():
    return CONTRACT_SIMPLE_CONSTRUCTOR_ABI


@pytest.fixture()
def SimpleConstructorContract(web3,
                              SIMPLE_CONSTRUCTOR_CODE,
                              SIMPLE_CONSTRUCTOR_RUNTIME,
                              SIMPLE_CONSTRUCTOR_ABI):
    return web3.platone.contract(
        abi=SIMPLE_CONSTRUCTOR_ABI,
        bytecode=SIMPLE_CONSTRUCTOR_CODE,
        bytecode_runtime=SIMPLE_CONSTRUCTOR_RUNTIME,
    )


CONTRACT_WITH_CONSTRUCTOR_ARGUMENTS_CODE = "0x60606040818152806066833960a09052516080516000918255600155603e908190602890396000f3606060405260e060020a600035046388ec134681146024578063d4c46c7614602c575b005b603460005481565b603460015481565b6060908152602090f3"  # noqa: E501
CONTRACT_WITH_CONSTRUCTOR_ARGUMENTS_RUNTIME = "0x606060405260e060020a600035046388ec134681146024578063d4c46c7614602c575b005b603460005481565b603460015481565b6060908152602090f3"  # noqa: E501
CONTRACT_WITH_CONSTRUCTOR_ARGUMENTS_ABI = json.loads('[{"constant":true,"inputs":[],"name":"data_a","outputs":[{"name":"","type":"uint256"}],"type":"function"},{"constant":true,"inputs":[],"name":"data_b","outputs":[{"name":"","type":"bytes32"}],"type":"function"},{"inputs":[{"name":"a","type":"uint256"},{"name":"b","type":"bytes32"}],"type":"constructor"}]')  # noqa: E501


@pytest.fixture()
def WITH_CONSTRUCTOR_ARGUMENTS_CODE():
    return CONTRACT_WITH_CONSTRUCTOR_ARGUMENTS_CODE


@pytest.fixture()
def WITH_CONSTRUCTOR_ARGUMENTS_RUNTIME():
    return CONTRACT_WITH_CONSTRUCTOR_ARGUMENTS_RUNTIME


@pytest.fixture()
def WITH_CONSTRUCTOR_ARGUMENTS_ABI():
    return CONTRACT_WITH_CONSTRUCTOR_ARGUMENTS_ABI


@pytest.fixture()
def WithConstructorArgumentsContract(web3,
                                     WITH_CONSTRUCTOR_ARGUMENTS_CODE,
                                     WITH_CONSTRUCTOR_ARGUMENTS_RUNTIME,
                                     WITH_CONSTRUCTOR_ARGUMENTS_ABI):
    return web3.platone.contract(
        abi=WITH_CONSTRUCTOR_ARGUMENTS_ABI,
        bytecode=WITH_CONSTRUCTOR_ARGUMENTS_CODE,
        bytecode_runtime=WITH_CONSTRUCTOR_ARGUMENTS_RUNTIME,
    )


CONTRACT_WITH_CONSTRUCTOR_ADDRESS_CODE = "0x6060604052604051602080607683395060806040525160008054600160a060020a031916821790555060428060346000396000f3606060405260e060020a600035046334664e3a8114601a575b005b603860005473ffffffffffffffffffffffffffffffffffffffff1681565b6060908152602090f3"  # noqa: E501
CONTRACT_WITH_CONSTRUCTOR_ADDRESS_RUNTIME = "0x606060405260e060020a600035046334664e3a8114601a575b005b603860005473ffffffffffffffffffffffffffffffffffffffff1681565b6060908152602090f3"  # noqa: E501
CONTRACT_WITH_CONSTRUCTOR_ADDRESS_ABI = json.loads('[{"constant":true,"inputs":[],"name":"testAddr","outputs":[{"name":"","type":"address"}],"type":"function"},{"inputs":[{"name":"_testAddr","type":"address"}],"type":"constructor"}]')  # noqa: E501


@pytest.fixture()
def WITH_CONSTRUCTOR_ADDRESS_CODE():
    return CONTRACT_WITH_CONSTRUCTOR_ADDRESS_CODE


@pytest.fixture()
def WITH_CONSTRUCTOR_ADDRESS_RUNTIME():
    return CONTRACT_WITH_CONSTRUCTOR_ADDRESS_RUNTIME


@pytest.fixture()
def WITH_CONSTRUCTOR_ADDRESS_ABI():
    return CONTRACT_WITH_CONSTRUCTOR_ADDRESS_ABI


@pytest.fixture()
def WithConstructorAddressArgumentsContract(web3,
                                            WITH_CONSTRUCTOR_ADDRESS_CODE,
                                            WITH_CONSTRUCTOR_ADDRESS_RUNTIME,
                                            WITH_CONSTRUCTOR_ADDRESS_ABI):
    return web3.platone.contract(
        abi=WITH_CONSTRUCTOR_ADDRESS_ABI,
        bytecode=WITH_CONSTRUCTOR_ADDRESS_CODE,
        bytecode_runtime=WITH_CONSTRUCTOR_ADDRESS_RUNTIME,
    )


@pytest.fixture()
def AddressReflectorContract(web3):
    return web3.platone.contract(
        abi=json.loads('[{"constant":true,"inputs":[{"name":"arg","type":"address"}],"name":"reflect","outputs":[{"name":"","type":"address"}],"payable":false,"stateMutability":"pure","type":"function"},{"constant":true,"inputs":[{"name":"arg","type":"address[]"}],"name":"reflect","outputs":[{"name":"","type":"address[]"}],"payable":false,"stateMutability":"pure","type":"function"}]'),  # noqa: 501
        bytecode="6060604052341561000f57600080fd5b6101ca8061001e6000396000f30060606040526000357c0100000000000000000000000000000000000000000000000000000000900463ffffffff1680630b816c1614610048578063c04d11fc146100c157600080fd5b341561005357600080fd5b61007f600480803573ffffffffffffffffffffffffffffffffffffffff16906020019091905050610170565b604051808273ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200191505060405180910390f35b34156100cc57600080fd5b61011960048080359060200190820180359060200190808060200260200160405190810160405280939291908181526020018383602002808284378201915050505050509190505061017a565b6040518080602001828103825283818151815260200191508051906020019060200280838360005b8381101561015c578082015181840152602081019050610141565b505050509050019250505060405180910390f35b6000819050919050565b61018261018a565b819050919050565b6020604051908101604052806000815250905600a165627a7a723058206b15d98a803b91327d94f943e9712291539701b2f7370e10f5873633941484930029",  # noqa: 501
        bytecode_runtime="60606040526000357c0100000000000000000000000000000000000000000000000000000000900463ffffffff1680630b816c1614610048578063c04d11fc146100c157600080fd5b341561005357600080fd5b61007f600480803573ffffffffffffffffffffffffffffffffffffffff16906020019091905050610170565b604051808273ffffffffffffffffffffffffffffffffffffffff1673ffffffffffffffffffffffffffffffffffffffff16815260200191505060405180910390f35b34156100cc57600080fd5b61011960048080359060200190820180359060200190808060200260200160405190810160405280939291908181526020018383602002808284378201915050505050509190505061017a565b6040518080602001828103825283818151815260200191508051906020019060200280838360005b8381101561015c578082015181840152602081019050610141565b505050509050019250505060405180910390f35b6000819050919050565b61018261018a565b819050919050565b6020604051908101604052806000815250905600a165627a7a723058206b15d98a803b91327d94f943e9712291539701b2f7370e10f5873633941484930029",  # noqa: 501
    )


CONTRACT_STRING_CODE = "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"  # noqa: E501

CONTRACT_STRING_RUNTIME = "0x606060405260e060020a600035046320965255811461003c57806330de3cee1461009f5780633fa4f245146100c457806393a0935214610121575b005b6101c7600060608181528154602060026001831615610100026000190190921691909104601f810182900490910260a0908101604052608082815292939190828280156102605780601f1061023557610100808354040283529160200191610260565b6101c7600060609081526101a06040526101006080818152906102d860a03990505b90565b6101c760008054602060026001831615610100026000190190921691909104601f810182900490910260809081016040526060828152929190828280156102975780601f1061026c57610100808354040283529160200191610297565b60206004803580820135601f81018490049093026080908101604052606084815261003a946024939192918401918190838280828437509496505050505050508060006000509080519060200190828054600181600116156101000203166002900490600052602060002090601f016020900481019282601f1061029f57805160ff19168380011785555b506102cf9291505b808211156102d4578381556001016101b4565b60405180806020018281038252838181518152602001915080519060200190808383829060006004602084601f0104600f02600301f150905090810190601f1680156102275780820380516001836020036101000a031916815260200191505b509250505060405180910390f35b820191906000526020600020905b81548152906001019060200180831161024357829003601f168201915b505050505090506100c1565b820191906000526020600020905b81548152906001019060200180831161027a57829003601f168201915b505050505081565b828001600101855582156101ac579182015b828111156101ac5782518260005055916020019190600101906102b1565b505050565b509056000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebfc0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedfe0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfeff"  # noqa: E501

CONTRACT_STRING_ABI = json.loads('[{"constant":false,"inputs":[],"name":"getValue","outputs":[{"name":"","type":"string"}],"type":"function"},{"constant":false,"inputs":[],"name":"constValue","outputs":[{"name":"","type":"string"}],"type":"function"},{"constant":true,"inputs":[],"name":"value","outputs":[{"name":"","type":"string"}],"type":"function"},{"constant":false,"inputs":[{"name":"_value","type":"string"}],"name":"setValue","outputs":[],"type":"function"},{"inputs":[{"name":"_value","type":"string"}],"type":"constructor"}]')  # noqa: E501


@pytest.fixture()
def STRING_CODE():
    return CONTRACT_STRING_CODE


@pytest.fixture()
def STRING_RUNTIME():
    return CONTRACT_STRING_RUNTIME


@pytest.fixture()
def STRING_ABI():
    return CONTRACT_STRING_ABI


@pytest.fixture()
def STRING_CONTRACT(STRING_CODE, STRING_RUNTIME, STRING_ABI):
    return {
        'bytecode': STRING_CODE,
        'bytecode_runtime': STRING_RUNTIME,
        'abi': STRING_ABI,
    }


@pytest.fixture()
def StringContract(web3, STRING_CONTRACT):
    return web3.platone.contract(**STRING_CONTRACT)


CONTRACT_BYTES_CODE = "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"  # noqa: E501

CONTRACT_BYTES_RUNTIME = "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"  # noqa: E501

CONTRACT_BYTES_ABI = json.loads('[{"constant":false,"inputs":[],"name":"getValue","outputs":[{"name":"","type":"bytes"}],"payable":false,"type":"function"},{"constant":false,"inputs":[],"name":"constValue","outputs":[{"name":"","type":"bytes"}],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"value","outputs":[{"name":"","type":"bytes"}],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"_value","type":"bytes"}],"name":"setValue","outputs":[],"payable":false,"type":"function"},{"inputs":[{"name":"_value","type":"bytes"}],"payable":false,"type":"constructor"}]')  # noqa: E501


@pytest.fixture()
def BYTES_CODE():
    return CONTRACT_BYTES_CODE


@pytest.fixture()
def BYTES_RUNTIME():
    return CONTRACT_BYTES_RUNTIME


@pytest.fixture()
def BYTES_ABI():
    return CONTRACT_BYTES_ABI


@pytest.fixture()
def BYTES_CONTRACT(BYTES_CODE, BYTES_RUNTIME, BYTES_ABI):
    return {
        'bytecode': BYTES_CODE,
        'bytecode_runtime': BYTES_RUNTIME,
        'abi': BYTES_ABI,
    }


@pytest.fixture()
def BytesContract(web3, BYTES_CONTRACT):
    return web3.platone.contract(**BYTES_CONTRACT)


CONTRACT_BYTES32_CODE = "60606040527f0123012301230123012301230123012301230123012301230123012301230123600090600019169055341561003957600080fd5b6040516020806101e2833981016040528080519060200190919050505b80600181600019169055505b505b61016f806100736000396000f30060606040526000357c0100000000000000000000000000000000000000000000000000000000900463ffffffff168063209652551461005f57806330de3cee146100905780633fa4f245146100c157806358825b10146100f2575b600080fd5b341561006a57600080fd5b610072610119565b60405180826000191660001916815260200191505060405180910390f35b341561009b57600080fd5b6100a3610124565b60405180826000191660001916815260200191505060405180910390f35b34156100cc57600080fd5b6100d461012e565b60405180826000191660001916815260200191505060405180910390f35b34156100fd57600080fd5b610117600480803560001916906020019091905050610134565b005b600060015490505b90565b6000805490505b90565b60015481565b80600181600019169055505b505600a165627a7a7230582043b15c20378b1603d330561258ccf291d08923a4c25fa8af0d590a010a6322180029"  # noqa: E501

CONTRACT_BYTES32_RUNTIME = "60606040526000357c0100000000000000000000000000000000000000000000000000000000900463ffffffff168063209652551461005f57806330de3cee146100905780633fa4f245146100c157806358825b10146100f2575b600080fd5b341561006a57600080fd5b610072610119565b60405180826000191660001916815260200191505060405180910390f35b341561009b57600080fd5b6100a3610124565b60405180826000191660001916815260200191505060405180910390f35b34156100cc57600080fd5b6100d461012e565b60405180826000191660001916815260200191505060405180910390f35b34156100fd57600080fd5b610117600480803560001916906020019091905050610134565b005b600060015490505b90565b6000805490505b90565b60015481565b80600181600019169055505b505600a165627a7a7230582043b15c20378b1603d330561258ccf291d08923a4c25fa8af0d590a010a6322180029"  # noqa: E501

CONTRACT_BYTES32_ABI = json.loads('[{"constant":false,"inputs":[],"name":"getValue","outputs":[{"name":"","type":"bytes32"}],"payable":false,"type":"function"},{"constant":false,"inputs":[],"name":"constValue","outputs":[{"name":"","type":"bytes32"}],"payable":false,"type":"function"},{"constant":true,"inputs":[],"name":"value","outputs":[{"name":"","type":"bytes32"}],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"_value","type":"bytes32"}],"name":"setValue","outputs":[],"payable":false,"type":"function"},{"inputs":[{"name":"_value","type":"bytes32"}],"payable":false,"type":"constructor"}]')  # noqa: E501


@pytest.fixture()
def BYTES32_CODE():
    return CONTRACT_BYTES32_CODE


@pytest.fixture()
def BYTES32_RUNTIME():
    return CONTRACT_BYTES32_RUNTIME


@pytest.fixture()
def BYTES32_ABI():
    return CONTRACT_BYTES32_ABI


@pytest.fixture()
def BYTES32_CONTRACT(BYTES32_CODE, BYTES32_RUNTIME, BYTES32_ABI):
    return {
        'bytecode': BYTES32_CODE,
        'bytecode_runtime': BYTES32_RUNTIME,
        'abi': BYTES32_ABI,
    }


@pytest.fixture()
def Bytes32Contract(web3, BYTES32_CONTRACT):
    return web3.platone.contract(**BYTES32_CONTRACT)


CONTRACT_EMITTER_CODE = "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"  # noqa: E501

CONTRACT_EMITTER_RUNTIME = "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"  # noqa: E501

CONTRACT_EMITTER_ABI = json.loads('[{"constant":false,"inputs":[{"name":"v","type":"string"}],"name":"logString","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"which","type":"uint8"}],"name":"logNoArgs","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"which","type":"uint8"},{"name":"arg0","type":"uint256"},{"name":"arg1","type":"uint256"},{"name":"arg2","type":"uint256"},{"name":"arg3","type":"uint256"}],"name":"logQuadruple","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"which","type":"uint8"},{"name":"arg0","type":"uint256"},{"name":"arg1","type":"uint256"}],"name":"logDouble","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"which","type":"uint8"},{"name":"arg0","type":"uint256"},{"name":"arg1","type":"uint256"},{"name":"arg2","type":"uint256"}],"name":"logTriple","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"which","type":"uint8"},{"name":"arg0","type":"uint256"}],"name":"logSingle","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"arg0","type":"string"},{"name":"arg1","type":"string"}],"name":"logDynamicArgs","outputs":[],"payable":false,"type":"function"},{"constant":false,"inputs":[{"name":"v","type":"bytes"}],"name":"logBytes","outputs":[],"payable":false,"type":"function"},{"anonymous":true,"inputs":[],"name":"LogAnonymous","type":"event"},{"anonymous":false,"inputs":[],"name":"LogNoArguments","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"arg0","type":"uint256"}],"name":"LogSingleArg","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"arg0","type":"uint256"},{"indexed":false,"name":"arg1","type":"uint256"}],"name":"LogDoubleArg","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"arg0","type":"uint256"},{"indexed":false,"name":"arg1","type":"uint256"},{"indexed":false,"name":"arg2","type":"uint256"}],"name":"LogTripleArg","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"arg0","type":"uint256"},{"indexed":false,"name":"arg1","type":"uint256"},{"indexed":false,"name":"arg2","type":"uint256"},{"indexed":false,"name":"arg3","type":"uint256"}],"name":"LogQuadrupleArg","type":"event"},{"anonymous":true,"inputs":[{"indexed":true,"name":"arg0","type":"uint256"}],"name":"LogSingleAnonymous","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"arg0","type":"uint256"}],"name":"LogSingleWithIndex","type":"event"},{"anonymous":true,"inputs":[{"indexed":false,"name":"arg0","type":"uint256"},{"indexed":true,"name":"arg1","type":"uint256"}],"name":"LogDoubleAnonymous","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"arg0","type":"uint256"},{"indexed":true,"name":"arg1","type":"uint256"}],"name":"LogDoubleWithIndex","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"arg0","type":"uint256"},{"indexed":true,"name":"arg1","type":"uint256"},{"indexed":true,"name":"arg2","type":"uint256"}],"name":"LogTripleWithIndex","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"arg0","type":"uint256"},{"indexed":false,"name":"arg1","type":"uint256"},{"indexed":true,"name":"arg2","type":"uint256"},{"indexed":true,"name":"arg3","type":"uint256"}],"name":"LogQuadrupleWithIndex","type":"event"},{"anonymous":false,"inputs":[{"indexed":true,"name":"arg0","type":"string"},{"indexed":false,"name":"arg1","type":"string"}],"name":"LogDynamicArgs","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"v","type":"bytes"}],"name":"LogBytes","type":"event"},{"anonymous":false,"inputs":[{"indexed":false,"name":"v","type":"string"}],"name":"LogString","type":"event"}]')  # noqa: E501


@pytest.fixture()
def EMITTER_CODE():
    return CONTRACT_EMITTER_CODE


@pytest.fixture()
def EMITTER_RUNTIME():
    return CONTRACT_EMITTER_RUNTIME


@pytest.fixture()
def EMITTER_ABI():
    return CONTRACT_EMITTER_ABI


@pytest.fixture()
def EMITTER(EMITTER_CODE,
            EMITTER_RUNTIME,
            EMITTER_ABI):
    return {
        'bytecode': EMITTER_CODE,
        'bytecode_runtime': EMITTER_RUNTIME,
        'abi': EMITTER_ABI,
    }


@pytest.fixture()
def Emitter(web3_empty, EMITTER):
    web3 = web3_empty
    return web3.platone.contract(**EMITTER)


@pytest.fixture()
def emitter(web3_empty, Emitter, wait_for_transaction, wait_for_block, address_conversion_func):
    web3 = web3_empty

    wait_for_block(web3)
    deploy_txn_hash = Emitter.constructor().transact({'from': web3.platone.coinbase, 'gas': 1000000})
    deploy_receipt = wait_for_transaction(web3, deploy_txn_hash)
    contract_address = address_conversion_func(deploy_receipt['contractAddress'])

    bytecode = web3.platone.getCode(contract_address)
    assert bytecode == Emitter.bytecode_runtime
    emitter_contract = Emitter(address=contract_address)
    assert emitter_contract.address == contract_address
    return emitter_contract


CONTRACT_ARRAYS_SOURCE = """
    contract ArraysContract {

        bytes32[] public bytes32Value;
        bytes32[] public bytes32ConstValue;
        byte[] public byteValue;
        byte[] public byteConstValue;

        function ArraysContract(bytes32[] _bytes32Value, byte[] _byteValue) {
            bytes32Value = _bytes32Value;
            byteValue = _byteValue;
            bytes32ConstValue = [keccak256('A'), keccak256('B')];
            byteConstValue = [bytes1(0), bytes1(1)];
        }

        function setBytes32Value(bytes32[] _bytes32Value) {
            bytes32Value = _bytes32Value;
        }

        function getBytes32Value() returns (bytes32[]) {
            return bytes32Value;
        }

        function getBytes32ConstValue() returns (bytes32[]) {
            return bytes32ConstValue;
        }

        function setByteValue(byte[] _byteValue) {
            byteValue = _byteValue;
        }

        function getByteValue() returns (byte[]) {
            return byteValue;
        }

        function getByteConstValue() returns (byte[]) {
            return byteConstValue;
        }
}
"""


CONTRACT_ARRAYS_CODE = "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"  # noqa: E501

CONTRACT_ARRAYS_RUNTIME = "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"  # noqa: E501

CONTRACT_ARRAYS_ABI = json.loads('[{"constant": false, "inputs": [], "name": "getBytes32ConstValue", "outputs": [{"name": "", "type": "bytes32[]"}], "payable": false, "stateMutability": "nonpayable", "type": "function"}, {"constant": true, "inputs": [{"name": "", "type": "uint256"}], "name": "byteValue", "outputs": [{"name": "", "type": "bytes1"}], "payable": false, "stateMutability": "view", "type": "function"}, {"constant": false, "inputs": [], "name": "getByteConstValue", "outputs": [{"name": "", "type": "bytes1[]"}], "payable": false, "stateMutability": "nonpayable", "type": "function"}, {"constant": false, "inputs": [{"name": "_byteValue", "type": "bytes1[]"}], "name": "setByteValue", "outputs": [], "payable": false, "stateMutability": "nonpayable", "type": "function"}, {"constant": true, "inputs": [{"name": "", "type": "uint256"}], "name": "bytes32ConstValue", "outputs": [{"name": "", "type": "bytes32"}], "payable": false, "stateMutability": "view", "type": "function"}, {"constant": true, "inputs": [{"name": "", "type": "uint256"}], "name": "bytes32Value", "outputs": [{"name": "", "type": "bytes32"}], "payable": false, "stateMutability": "view", "type": "function"}, {"constant": false, "inputs": [], "name": "getBytes32Value", "outputs": [{"name": "", "type": "bytes32[]"}], "payable": false, "stateMutability": "nonpayable", "type": "function"}, {"constant": false, "inputs": [], "name": "getByteValue", "outputs": [{"name": "", "type": "bytes1[]"}], "payable": false, "stateMutability": "nonpayable", "type": "function"}, {"constant": true, "inputs": [{"name": "", "type": "uint256"}], "name": "byteConstValue", "outputs": [{"name": "", "type": "bytes1"}], "payable": false, "stateMutability": "view", "type": "function"}, {"constant": false, "inputs": [{"name": "_bytes32Value", "type": "bytes32[]"}], "name": "setBytes32Value", "outputs": [], "payable": false, "stateMutability": "nonpayable", "type": "function"}, {"inputs": [{"name": "_bytes32Value", "type": "bytes32[]"}, {"name": "_byteValue", "type": "bytes1[]"}], "payable": false, "stateMutability": "nonpayable", "type": "constructor"}]')  # noqa: E501


@pytest.fixture()
def ARRAYS_CODE():
    return CONTRACT_ARRAYS_CODE


@pytest.fixture()
def ARRAYS_RUNTIME():
    return CONTRACT_ARRAYS_RUNTIME


@pytest.fixture()
def ARRAYS_ABI():
    return CONTRACT_ARRAYS_ABI


@pytest.fixture()
def ARRAYS_CONTRACT(ARRAYS_CODE,
                    ARRAYS_RUNTIME,
                    ARRAYS_ABI):
    return {
        'bytecode': ARRAYS_CODE,
        'bytecode_runtime': ARRAYS_RUNTIME,
        'abi': ARRAYS_ABI,
    }


@pytest.fixture()
def ArraysContract(web3, ARRAYS_CONTRACT):
    return web3.platone.contract(**ARRAYS_CONTRACT)


CONTRACT_PAYABLE_TESTER_SOURCE = """
contract PayableTester {
  bool public wasCalled;

  function doNoValueCall() public {
    wasCalled = true;
  }
}
"""

CONTRACT_PAYABLE_TESTER_CODE = "608060405234801561001057600080fd5b5060e88061001f6000396000f3006080604052600436106049576000357c0100000000000000000000000000000000000000000000000000000000900463ffffffff168063c680362214604e578063e4cb8f5c14607a575b600080fd5b348015605957600080fd5b506060608e565b604051808215151515815260200191505060405180910390f35b348015608557600080fd5b50608c60a0565b005b6000809054906101000a900460ff1681565b60016000806101000a81548160ff0219169083151502179055505600a165627a7a723058205362c7376eda918b0dc3a75d0ffab904a241c9b10b68d5268af6ca405242303e0029"  # noqa: E501

CONTRACT_PAYABLE_TESTER_RUNTIME = "6080604052600436106049576000357c0100000000000000000000000000000000000000000000000000000000900463ffffffff168063c680362214604e578063e4cb8f5c14607a575b600080fd5b348015605957600080fd5b506060608e565b604051808215151515815260200191505060405180910390f35b348015608557600080fd5b50608c60a0565b005b6000809054906101000a900460ff1681565b60016000806101000a81548160ff0219169083151502179055505600a165627a7a723058205362c7376eda918b0dc3a75d0ffab904a241c9b10b68d5268af6ca405242303e0029"  # noqa: E501

CONTRACT_PAYABLE_TESTER_ABI = json.loads('[{"constant": true, "inputs": [], "name": "wasCalled", "outputs": [{"name": "", "type": "bool"}], "payable": false, "stateMutability": "view", "type": "function"}, {"constant": false, "inputs": [], "name": "doNoValueCall", "outputs": [], "payable": false, "stateMutability": "nonpayable", "type": "function"}]')  # noqa: E501


@pytest.fixture()
def PAYABLE_TESTER_CODE():
    return CONTRACT_PAYABLE_TESTER_CODE


@pytest.fixture()
def PAYABLE_TESTER_RUNTIME():
    return CONTRACT_PAYABLE_TESTER_RUNTIME


@pytest.fixture()
def PAYABLE_TESTER_ABI():
    return CONTRACT_PAYABLE_TESTER_ABI


@pytest.fixture()
def PAYABLE_TESTER_CONTRACT(PAYABLE_TESTER_CODE,
                            PAYABLE_TESTER_RUNTIME,
                            PAYABLE_TESTER_ABI):
    return {
        'bytecode': PAYABLE_TESTER_CODE,
        'bytecode_runtime': PAYABLE_TESTER_RUNTIME,
        'abi': PAYABLE_TESTER_ABI,
    }


@pytest.fixture()
def PayableTesterContract(web3, PAYABLE_TESTER_CONTRACT):
    return web3.platone.contract(**PAYABLE_TESTER_CONTRACT)


# no matter the function selector, this will return back the 32 bytes of data supplied
CONTRACT_REFLECTION_CODE = (
    "0x610011566020600460003760206000f3005b61000461001103610004600039610004610011036000f3"
)

# reference source used to generate it:
LLL_SOURCE = "['seq', ['return', 0, ['lll', ['seq', ['calldatacopy', 0, 4, 32], ['return', 0, 32], 'stop' ], 0]]])"  # noqa: E501

CONTRACT_FIXED_ABI = [
    {
        "type": "function",
        "constant": False,
        "inputs": [{"type": "fixed8x1"}],
        "name": "reflect",
        "outputs": [{"type": "fixed8x1"}],
    },
    {
        "type": "function",
        "constant": False,
        "inputs": [{"type": "ufixed256x80"}],
        "name": "reflect",
        "outputs": [{"type": "ufixed256x80"}],
    },
    {
        "type": "function",
        "constant": False,
        "inputs": [{"type": "ufixed256x1"}],
        "name": "reflect",
        "outputs": [{"type": "ufixed256x1"}],
    },
    {
        "type": "function",
        "constant": False,
        "inputs": [{"type": "ufixed8x1"}],
        "name": "reflect_short_u",
        "outputs": [{"type": "ufixed8x1"}],
    },
]


@pytest.fixture
def FixedReflectionContract(web3):
    return web3.platone.contract(abi=CONTRACT_FIXED_ABI, bytecode=CONTRACT_REFLECTION_CODE)


CONTRACT_FALLBACK_FUNCTION_SOURCE = """
contract A {
    uint data;
    function A() public payable { data = 0; }
    function getData() returns (uint r) { return data; }
    function() { data = 1; }
}
"""

CONTRACT_FALLBACK_FUNCTION_CODE = "60606040526000808190555060ae806100196000396000f300606060405260043610603f576000357c0100000000000000000000000000000000000000000000000000000000900463ffffffff1680633bc5de30146053575b3415604957600080fd5b6001600081905550005b3415605d57600080fd5b60636079565b6040518082815260200191505060405180910390f35b600080549050905600a165627a7a72305820045439389e4742569ec078687e6a0c81997709778a0097adbe07ccfd9f7b1a330029"  # noqa: E501

CONTRACT_FALLBACK_FUNCTION_RUNTIME = "606060405260043610603f576000357c0100000000000000000000000000000000000000000000000000000000900463ffffffff1680633bc5de30146053575b3415604957600080fd5b6001600081905550005b3415605d57600080fd5b60636079565b6040518082815260200191505060405180910390f35b600080549050905600a165627a7a72305820045439389e4742569ec078687e6a0c81997709778a0097adbe07ccfd9f7b1a330029"  # noqa: E501

CONTRACT_FALLBACK_FUNCTION_ABI = json.loads('[{"constant": false, "inputs": [], "name": "getData", "outputs": [{"name": "r", "type": "uint256"}], "payable": false, "stateMutability": "nonpayable", "type": "function"}, {"inputs": [], "payable": true, "stateMutability": "payable", "type": "constructor"}, {"payable": false, "stateMutability": "nonpayable", "type": "fallback"}]')  # noqa: E501


@pytest.fixture()
def FALLBACK_FUNCTION_CODE():
    return CONTRACT_FALLBACK_FUNCTION_CODE


@pytest.fixture()
def FALLBACK_FUNCTION_RUNTIME():
    return CONTRACT_FALLBACK_FUNCTION_RUNTIME


@pytest.fixture()
def FALLBACK_FUNCTION_ABI():
    return CONTRACT_FALLBACK_FUNCTION_ABI


@pytest.fixture()
def FALLBACK_FUNCTION_CONTRACT(FALLBACK_FUNCTION_CODE,
                               FALLBACK_FUNCTION_RUNTIME,
                               FALLBACK_FUNCTION_ABI):
    return {
        'bytecode': FALLBACK_FUNCTION_CODE,
        'bytecode_runtime': FALLBACK_FUNCTION_RUNTIME,
        'abi': FALLBACK_FUNCTION_ABI,
    }


@pytest.fixture()
def FallballFunctionContract(web3, FALLBACK_FUNCTION_CONTRACT):
    return web3.platone.contract(**FALLBACK_FUNCTION_CONTRACT)


class LogFunctions:
    LogAnonymous = 0
    LogNoArguments = 1
    LogSingleArg = 2
    LogDoubleArg = 3
    LogTripleArg = 4
    LogQuadrupleArg = 5
    LogSingleAnonymous = 6
    LogSingleWithIndex = 7
    LogDoubleAnonymous = 8
    LogDoubleWithIndex = 9
    LogTripleWithIndex = 10
    LogQuadrupleWithIndex = 11


@pytest.fixture()
def emitter_event_ids():
    return LogFunctions


def _encode_to_topic(event_signature):
    return event_signature_to_log_topic(event_signature)


class LogTopics:
    LogAnonymous = _encode_to_topic("LogAnonymous()")
    LogNoArguments = _encode_to_topic("LogNoArguments()")
    LogSingleArg = _encode_to_topic("LogSingleArg(uint256)")
    LogSingleAnonymous = _encode_to_topic("LogSingleAnonymous(uint256)")
    LogSingleWithIndex = _encode_to_topic("LogSingleWithIndex(uint256)")
    LogDoubleArg = _encode_to_topic("LogDoubleArg(uint256,uint256)")
    LogDoubleAnonymous = _encode_to_topic("LogDoubleAnonymous(uint256,uint256)")
    LogDoubleWithIndex = _encode_to_topic("LogDoubleWithIndex(uint256,uint256)")
    LogTripleArg = _encode_to_topic("LogTripleArg(uint256,uint256,uint256)")
    LogTripleWithIndex = _encode_to_topic("LogTripleWithIndex(uint256,uint256,uint256)")
    LogQuadrupleArg = _encode_to_topic("LogQuadrupleArg(uint256,uint256,uint256,uint256)")
    LogQuadrupleWithIndex = _encode_to_topic(
        "LogQuadrupleWithIndex(uint256,uint256,uint256,uint256)",
    )
    LogBytes = _encode_to_topic("LogBytes(bytes)")
    LogString = _encode_to_topic("LogString(string)")
    LogDynamicArgs = _encode_to_topic("LogDynamicArgs(string,string)")


@pytest.fixture()
def emitter_log_topics():
    return LogTopics


@pytest.fixture()
def some_address(address_conversion_func):
    return address_conversion_func('0x5B2063246F2191f18F2675ceDB8b28102e957458')


def invoke_contract(api_style=None,
                    api_call_desig='call',
                    contract=None,
                    contract_function=None,
                    func_args=[],
                    func_kwargs={},
                    tx_params={}):
    allowable_call_desig = ['call', 'transact', 'estimateGas', 'buildTransaction']
    if api_call_desig not in allowable_call_desig:
        raise ValueError("allowable_invoke_method must be one of: %s" % allowable_call_desig)

    if api_style == 'func_first':
        function = contract.functions[contract_function]
        result = getattr(function(*func_args, **func_kwargs), api_call_desig)(tx_params)
    elif api_style == 'func_last':
        api_call_cls = getattr(contract, api_call_desig)
        with pytest.deprecated_call():
            result = getattr(api_call_cls(tx_params), contract_function)(*func_args, **func_kwargs)
    else:
        raise ValueError("api_style must be 'func_first or func_last'")

    return result


@pytest.fixture(params=['func_first', 'func_last'])
def transact(request):
    return functools.partial(invoke_contract, request.param, api_call_desig='transact')


@pytest.fixture(params=['func_first', 'func_last'])
def call(request):
    return functools.partial(invoke_contract, request.param, api_call_desig='call')


@pytest.fixture(params=['func_first', 'func_last'])
def estimateGas(request):
    return functools.partial(invoke_contract, request.param, api_call_desig='estimateGas')


@pytest.fixture(params=['func_first', 'func_last'])
def buildTransaction(request):
    return functools.partial(invoke_contract, request.param, api_call_desig='buildTransaction')
