# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/aggregate.ipynb (unless otherwise specified).


from __future__ import annotations  # To use class name in herself for typing

__all__ = [
    "default_extraction_date",
    "yaml",
    "Perimeter",
    "Reference",
    "Data",
    "LabelKey",
    "Aggregate",
    "AggregateManager",
]

# Cell

# nbdev_comment from __future__ import annotations  # To use class name in herself for typing

import glob
import json
import time
from dataclasses import dataclass, field
from pathlib import Path
from typing import Dict, List, Union

import numpy as np
import pandas as pd
import ruamel.yaml
from dataclasses_json import dataclass_json
from ruamel.yaml.comments import CommentedSeq
from tqdm import tqdm

# from leximpact_socio_fisca_simu_etat.logger import logger

default_extraction_date = time.strftime("%Y-%m-%d")
yaml = ruamel.yaml.YAML()

# Cell


@dataclass_json
@dataclass
class Perimeter:
    """Store a perimeter for a variable.

    Args:
        entity (str): Variable entity. Example : "individual".
        period (str): Date of data. Example: "2019"
        geographic (str): Geographic area. Exemple : "France entière".
    """

    entity: str = ""
    period: str = ""
    geographic: str = ""


@dataclass_json
@dataclass
class Reference:
    title: str
    href: str


@dataclass_json
@dataclass
class Data:
    reference: List[Reference] = field(default_factory=list)
    date: str = ""
    data_structure: str = (
        "list"  # value, list, dict, list_of_dict, distribution, copulas
    )
    extraction_date: str = default_extraction_date
    values: Union[List, Dict] = field(default_factory=Dict)


@dataclass_json
@dataclass
class LabelKey:
    key: str
    label: str


@dataclass_json
@dataclass
class Aggregate:
    description: str = ""
    short_name: str = ""
    ux_template: str = ""
    openfisca_variable: str = ""
    source_variable: str = ""
    unit: str = ""
    extraction_date: str = default_extraction_date
    perimeter: Perimeter = None
    label_of_keys: Dict[str, str] = None
    data: List[Data] = field(default_factory=list)

    def merge_with(self, agg: Aggregate):
        if type(agg) is not Aggregate:
            raise TypeError(f"Param agg is not an Aggregate() but a {type(agg)}")
        keys = self.__dict__.keys()
        for name in keys:
            val = getattr(self, name)
            if type(val) is str and getattr(self, name) == "":
                setattr(self, name, getattr(agg, name))
            elif val is None:
                setattr(self, name, getattr(agg, name))
            elif name == "data":
                if type(agg.data) not in [list, CommentedSeq]:
                    # print(agg.data)
                    raise TypeError(f"agg.data is not a list but a {type(agg.data)}")
                for new_data in agg.data:
                    date = new_data.date
                    ds = new_data.data_structure
                    # We look into new aggregate if same data exist
                    for i, data in enumerate(self.data):
                        if data.date == date and data.data_structure == ds:
                            # print(" meme date et meme datastructure => On remplace")
                            self.data[i] = new_data
                            break
                    else:
                        # Don't exist, add it
                        # print("Don't exist, add it")
                        self.data.append(new_data)


class AggregateManager:
    def __init__(
        self, aggregate: Aggregate = Aggregate(), aggregates_path: str = "../aggregates"
    ):
        self.aggregate = aggregate
        self.aggregates_path = aggregates_path
        yaml = ruamel.yaml.YAML()
        yaml.register_class(Perimeter)
        yaml.register_class(Reference)
        yaml.register_class(Data)
        yaml.register_class(Aggregate)
        self.__yaml = yaml

    def save_to_json(self, filename):
        path = Path(filename)
        path.parent.mkdir(parents=True, exist_ok=True)
        # logger.debug(f"Writing to {path}")
        with open(path, mode="w") as file:
            file.write(self.aggregate.to_json(indent=2))

    def _save_to_yaml(self, filename):
        path = Path(filename)
        path.parent.mkdir(parents=True, exist_ok=True)
        # print(f"DEBUG Writing to {path}")
        with open(path, mode="w") as file:
            ruamel.yaml.YAML().dump(self.aggregate, file)

    def save_to_yaml(self, filename, merge_if_exist: bool = True):
        path = Path(filename)
        if path.is_file() and merge_if_exist:
            # YAML already exist, merging...
            old_agg = AggregateManager()
            old_agg.load_from_yaml(path)
            self.aggregate.merge_with(old_agg.aggregate)
        self._save_to_yaml(filename)

    def load_from_yaml(self, filename):
        with open(filename) as file:
            self.aggregate = ruamel.yaml.YAML().load(file)

    def load_aggregate(self, dataset: str, year: str, variable: str):
        filename = Path(self.aggregates_path) / dataset / year / (variable + ".yaml")
        self.load_from_yaml(filename)

    @staticmethod
    def save_agg_to_yaml_and_json(filepath: Path, aggregate: Aggregate):
        aggm = AggregateManager(aggregate)
        aggm.save_to_yaml(filepath)
        aggm.save_to_json(filepath.with_suffix(".json"))

    @staticmethod
    def save_agg_from_list(
        agg_list: List[Aggregate],
        dataset: str = "POTE",
        year: str = "2019",
        aggregates_path: str = "../aggregates",
    ) -> None:
        for agg in tqdm(agg_list):
            if agg.openfisca_variable == "":
                var_name = agg.source_variable
            else:
                var_name = agg.openfisca_variable

            path = Path(aggregates_path) / dataset / year / (var_name + ".yaml")
            # print(agg,f"{var_name}\n-------------------------\n")
            AggregateManager.save_agg_to_yaml_and_json(path, agg)

    @staticmethod
    def get_key_value_aggregate(
        csv_filepath: str,
        var_name: str,
        metadata: dict,
        year: str,
        ref: Reference,
        perimeter: Perimeter,
    ) -> Aggregate:
        df = pd.read_csv(csv_filepath)
        short_name = description = label_of_keys = ux_template = openfisca_variable = ""
        if metadata:
            short_name = metadata.get("short_name", metadata.get("openfisca_variable"))
            description = metadata.get("description", "")
            ux_template = metadata.get("ux_template", "")
            openfisca_variable = metadata.get("openfisca_variable", "")
            label_of_keys = metadata.get("labels", "")
        values = {}
        _ = df.apply(AggregateManager.row_to_dict, args=[values], axis=1)

        data = Data(reference=[ref], date=year, values=values, data_structure="dict")
        aggregate = Aggregate(
            description=description,
            short_name=short_name,
            ux_template=ux_template,
            openfisca_variable=openfisca_variable,
            source_variable=var_name,
            unit="foyer",
            perimeter=perimeter,
            label_of_keys=label_of_keys,
            data=[data],
        )
        return aggregate

    @staticmethod
    def get_key_value_aggregates(
        dataset_folder: str,
        year: str,
        metadata: dict,
        ref: Reference,
        perimeter: Perimeter,
    ) -> List[Aggregate]:
        aggregates = []
        files = glob.glob(dataset_folder)
        for file in files:
            # print(file)
            var_name = file.split("_")[-1].split(".")[0]
            year = file.split("_")[-2]
            aggregate = AggregateManager.get_key_value_aggregate(
                file, var_name, metadata.get(var_name), year, ref, perimeter
            )
            aggregates.append(aggregate)
        return aggregates

    @staticmethod
    def get_aggregats_from_row(
        row,
        agregats,
        year,
        col_name,
        ref: Reference,
        metadata: dict,
        perimeter: Perimeter,
    ):
        """Parcours les lignes du dataframe et enregistre les agrégats dans le
        dictionnaire.

        ::row:: une ligne de dataframe
        ::year:: year of the data
        """
        values = {}
        for i, cell in enumerate(row):
            if i == 0:
                var_name = cell
            else:
                values[col_name[i]] = cell

        data = Data(reference=[ref], date=year, data_structure="dict", values=values)
        of_name = ""
        if metadata.get(var_name):
            of_name = metadata.get(var_name).get("openfisca_variable", "")

        aggregate = Aggregate(
            source_variable=var_name,
            openfisca_variable=of_name,
            unit="foyer",
            perimeter=perimeter,
            data=[data],
        )
        agregats.append(aggregate)

    @staticmethod
    def get_aggregate_from_df(df, variable_name, agg_type: str = "sum"):
        """Permet de récupérer un agrégat de type determiné dans le dataframe.

        ::df:: Dataframe à lire
        ::variable_name:: Nom de la variable
        """
        return df.loc[df["name"] == variable_name, agg_type].iloc[0]

    @staticmethod
    def get_aggregats_from_json(
        var_name: str,
        calib_filepath: str,
        ref: Reference,
        perimeter: Perimeter,
        metadata: dict,
        year: str = "2019",
        unit: str = "euros",
        data_structure="distribution",
    ) -> Aggregate:
        """Convert calib file to an Aggregate.

        ::var_name:: une ligne de dataframe
        ::calib_filepath:: Path to calib file
        """
        with open(calib_filepath) as myfile:
            json_content = json.loads(myfile.read())
        # keep_upper_bound_secret(json_content)
        data = Data(
            reference=[ref],
            date=year,
            data_structure=data_structure,
            values=json_content,
        )
        of_name = ""
        if metadata.get(var_name):
            of_name = metadata.get(var_name).get("openfisca_variable", "")
        aggregate = Aggregate(
            source_variable=var_name,
            openfisca_variable=of_name,
            unit=unit,
            perimeter=perimeter,
            data=[data],
        )
        return aggregate

    @staticmethod
    def process_all_json(
        path: str,
        ref: Reference,
        perimeter: Perimeter,
        metadata: dict,
        data_structure="distribution",
    ) -> List[Aggregate]:
        aggregates = []
        files = glob.glob(path)
        if len(files) == 0:
            print(f"No file for {path} !")
            return []
        for file in files:
            #    Parcours les fichier de calibration et enregistre les agrégats dans le dictionnaire.
            var_name = file.split("-")[-1].split(".")[0]
            bucket_size = file.split("-")[-3]
            # print(f"Processing {var_name} {bucket_size} from {file}")
            aggregate = AggregateManager.get_aggregats_from_json(
                var_name,
                file,
                ref,
                perimeter,
                metadata,
                data_structure=f"{data_structure}_{bucket_size}",
            )
            # if bucket_size == "10" and var_name == "revkire":
            #     print(aggregate)
            aggregates.append(aggregate)
        return aggregates

    @staticmethod
    def row_to_dict(row, dest_dict: dict):
        """
        Return the first column as key and second columns as row
        WARNING : Use only with 2 columns
        """
        # Force cast of float to avoid problems with ramel.YAML() dump
        key = row[0] if not isinstance(row[0], np.float64) else float(row[0])
        val = row[1] if not isinstance(row[1], np.float64) else float(row[1])
        val = "secret" if np.isnan(val) else val

        dest_dict[key] = val
        return row


# Cell


# Cell
