# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/core.ipynb (unless otherwise specified).

__all__ = ['TSFeatures']

# Cell
import requests
import json
from requests.auth import HTTPBasicAuth

import pandas as pd

# Cell
class TSFeatures:
    """Compute time series features at scale.
    Send an email to ..@.. to request access.

    Parameters
    ----------
    api_id: str
        API identifier.
    api_key: str
        API key.
    """

    def __init__(self, api_id: str, api_key: str) -> 'TSFeatures':
        self.invoke_url = f'https://{api_id}.execute-api.us-east-1.amazonaws.com/main'
        self.api_key = api_key

    def _from_api_to_df(self, response: str) -> pd.DataFrame:
        """Transforms api output to df."""
        df = pd.DataFrame(json.loads(response), index=[0])

        return df

    def calculate_features_from_s3_uri(self, s3_uri: str, freq: int,
                                       kind: str = 'static',
                                       unique_id_column: str = 'unique_id',
                                       ds_column: str = 'ds',
                                       y_column: str = 'y') -> pd.DataFrame:
        """Calculates features from S3 URL.

        Parameters
        ----------
        s3_uri: str
            S3 uri of the dataset.
            The dataset should contain at least three columns:
                - Time series identifier.
                - Time identifier.
                - Target identifier.
        freq: int
            Frequency of the time series.
        kind: str
            Kind of features. Static returns features for each time series.
            temporal returns for each ds and each time series.
        unique_id_column: str
            Column name identifying each time series.
        ds_column: str
            Column name identifying each time stamp.
        y_column: str
            Column name identifying the target variable.

        Notes
        -----
        [1] The dataset should contain time series of the same frequency.
        """
        query = dict(url=s3_uri, freq=freq, kind=kind,
                     unique_id_column=unique_id_column,
                     ds_column=ds_column,
                     y_column=y_column)
        resp = requests.post(f'{self.invoke_url}/tsfeatures',
                             headers={'x-api-key': self.api_key},
                             data=json.dumps(query))

        return self._from_api_to_df(resp.text)

    def get_status(self, job_id: str) -> pd.DataFrame:
        """Gets job status.

        Parameters
        ----------
        job_id: str
            ID job from `calculate_features_from_s3_uri`.
        """
        resp = requests.get(f'{self.invoke_url}/tsfeatures/jobs/',
                            params={'job_id': job_id},
                            headers={'x-api-key': self.api_key})

        return self._from_api_to_df(resp.text)