# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['typingx']

package_data = \
{'': ['*']}

extras_require = \
{':python_version < "3.9"': ['typing_extensions>=3.7.4.3,<4.0.0.0']}

setup_kwargs = {
    'name': 'typingx',
    'version': '0.2.1',
    'description': 'Extend typing package functionalities',
    'long_description': '# typingx\n[![Tests](https://github.com/PrettyWood/typingx/workflows/Tests/badge.svg)](https://github.com/PrettyWood/typingx/actions)\n[![codecov](https://codecov.io/gh/PrettyWood/typingx/branch/main/graph/badge.svg)](https://codecov.io/gh/PrettyWood/typingx)\n[![pypi](https://img.shields.io/pypi/v/typingx.svg)](https://pypi.python.org/pypi/typingx)\n[![versions](https://img.shields.io/pypi/pyversions/typingx.svg)](https://github.com/PrettyWood/typingx)\n[![license](https://img.shields.io/github/license/PrettyWood/typingx.svg)](https://github.com/PrettyWood/typingx/blob/master/LICENSE)\n\n`typing` is great but it changed a lot since 3.6 and it\'s not over!\n\nThis library purpose is to have a consistent behaviour for all those versions to mimic the most recent one\nand go even further with `typing` (and `typing_extensions`).\n\nIt provides:\n- `get_args` and `get_origin` for python `3.6` to `3.9` that mimic `3.10` behaviour\n- `is_literal`, `is_typeddict` helpers\n- most `typing` types but with homogeneous behaviour\n  (e.g. with `3.8`, `typing.TypedDict` won\'t store information to distinguish optional and required keys. This lib will hence choose `typing_extensions` version)\n\nbut also:\n- `isinstancex`: like `isinstance` but with `typing(x)` types\n- extra types:\n  * `Listx` and `Tuplex`: more sophisticated versions of `List` and `Tuple` to add `...` anywhere in the parameters\n- extanded types:\n  * `TypedDict` has a `__extra__` field (value can be changed) to allow type checking on optional fields\n\n## Installation\n\n``` bash\n    pip install typingx\n```\n\n## Usage\n```python\nfrom collections import ChainMap, Counter\n\nfrom typingx import (\n    Any,\n    Dict,\n    List,\n    Listx,\n    Literal,\n    Mapping,\n    Sequence,\n    Set,\n    Tuple,\n    Tuplex,\n    Type,\n    TypedDict,\n    Union,\n    isinstancex,\n)\n\n# Dict\nassert isinstancex({"a": 1, "b": 2}, Dict[str, int]) is True\nassert isinstancex({"a": 1, "b": 2}, Dict[str, str]) is False\nassert isinstancex({"a": 1, "b": 2}, Dict[int, str]) is False\nassert isinstancex({"a": 1, "b": 2}, Dict[str, Any]) is True\n\n# List\nassert isinstancex([1, 2, 3], List[int]) is True\nassert isinstancex([1, 2, "q"], List[int]) is False\nassert isinstancex([1, 2, "q"], List[Union[str, int]]) is True\n\n# Listx\nassert isinstancex([1, 2, 3, 4], Listx[int]) is True\nassert isinstancex([1, 2, "q"], Listx[int, ..., str]) is True\nassert isinstancex([1, 2, "q", "w", "e"], Listx[int, ..., str]) is False\nassert isinstancex([1, 2, "q", "w", "e"], Listx[int, ..., str, ...]) is True\nassert isinstancex([1, 2, "q", "w", b"xyz", "e"], Listx[int, ..., str, ...]) is False\nassert isinstancex([1, 2, "q", "w", b"xyz", "e"], Listx[int, ..., Union[str, bytes], ...]) is True\n\n# Literal\nassert isinstancex("a", Literal["a"]) is True\nassert isinstancex(Literal["a"], Literal["a"]) is True\nassert isinstancex("b", Literal["a"]) is False\nassert isinstancex("b", Literal["a", Literal[Literal["b"]]]) is True\nassert isinstancex(Literal["a", "b"], Literal["b", "a", "c"]) is True\n\n# Mapping\nassert isinstancex(Counter({"red": 4, "blue": 2}), Mapping[str, int]) is True\nassert isinstancex(ChainMap({"art": "van gogh"}, {"music": "bach"}), Mapping[str, str]) is True\n\n# Sequence\nassert isinstancex("abc", Sequence[Any]) is True\nassert isinstancex("abc", Sequence[int]) is False\nassert isinstancex((1, 3, 5), Sequence[int]) is True\n\n# Set\nassert isinstancex({"a", "b"}, Set[str]) is True\nassert isinstancex({"a", "b"}, Set[int]) is False\n\n# Tuple\nassert isinstancex((1, 2), Tuple[int, ...]) is True\nassert isinstancex((1, 2), Tuple[int, int]) is True\nassert isinstancex((1, 2), Tuple[int, int, int]) is False\n\n# Tuplex\nassert isinstancex((3, "a", "b"), Tuplex[int, str, ...]) is True\nassert isinstancex((3, "a", "b", "c"), Tuplex[int, str, ...]) is True\nassert isinstancex((3, "a", "b", "c"), Tuplex[int, str, ..., bool]) is False\nassert isinstancex((3, "a", "b", "c", True), Tuplex[int, str, ..., bool]) is True\nassert isinstancex((3, "a", "b", "c", 3), Tuplex[int, str, ..., bool]) is False\nassert isinstancex((3, "a", "b", "c", True, False), Tuplex[int, str, ..., bool, ...]) is True\n\n# Type\nclass User: ...\nclass BaseUser(User): ...\n\nassert isinstancex(BaseUser, Type[BaseUser]) is True\nassert isinstancex(BaseUser, Type[User]) is True\nassert isinstancex(User, Type[User]) is True\nassert isinstancex(User, Type[BaseUser]) is False\n\n# TypedDict\nFullMovie = TypedDict("FullMovie", {"name": str, "year": int})\n\nclass PartialMovie(TypedDict, total=False):\n    name: str\n    year: int\n\nclass ExtraMovie(TypedDict):\n    name: str\n    year: int\n    __extra__: str\n\nassert isinstancex({"name": "The Matrix", "year": 1999}, FullMovie) is True\nassert isinstancex({"name": "The Matrix", "year": "1999"}, FullMovie) is False\nassert isinstancex({"name": "The Matrix"}, FullMovie) is False\nassert isinstancex({"name": "The Matrix", "year": 1999, "extra": "qwe"}, FullMovie) is False\n\nassert isinstancex({"name": "The Matrix", "year": 1999}, PartialMovie) is True\nassert isinstancex({"name": "The Matrix"}, PartialMovie) is True\nassert isinstancex({"name": "The Matrix", "year": 1999, "extra": "qwe"}, PartialMovie) is False\n\nassert isinstancex({"name": "The Matrix", "year": 1999}, ExtraMovie) is True\nassert isinstancex({"name": "The Matrix", "year": 1999, "q": "w", "e": "r"}, ExtraMovie) is True\nassert isinstancex({"name": "The Matrix", "year": 1999, "q": "w", "e": 1}, ExtraMovie) is False\n```\n',
    'author': 'Eric Jolibois',
    'author_email': 'em.jolibois@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/PrettyWood/typingx',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
