"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RESERVED_TUNNEL_INSIDE_CIDR = exports.VpnConnection = exports.VpnGateway = exports.VpnConnectionType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const net = require("net");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The VPN connection type.
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    /**
     * The IPsec 1 VPN connection type.
     */
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    /**
     * Dummy member
     * TODO: remove once https://github.com/aws/jsii/issues/231 is fixed
     */
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * The VPN Gateway that shall be added to the VPC
 *
 * @resource AWS::EC2::VPNGateway
 */
class VpnGateway extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpnGatewayProps(props);
        // This is 'Default' instead of 'Resource', because using 'Default' will generate
        // a logical ID for a VpnGateway which is exactly the same as the logical ID that used
        // to be created for the CfnVPNGateway (and 'Resource' would not do that).
        const vpnGW = new ec2_generated_1.CfnVPNGateway(this, 'Default', props);
        this.gatewayId = vpnGW.ref;
    }
}
exports.VpnGateway = VpnGateway;
_a = JSII_RTTI_SYMBOL_1;
VpnGateway[_a] = { fqn: "@aws-cdk/aws-ec2.VpnGateway", version: "1.146.0" };
/**
 * Define a VPN Connection
 *
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpnConnectionProps(props);
        if (!props.vpc.vpnGatewayId) {
            props.vpc.enableVpnGateway({
                type: 'ipsec.1',
                amazonSideAsn: props.asn,
            });
        }
        if (!core_1.Token.isUnresolved(props.ip) && !net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && !core_1.Token.isUnresolved(options.preSharedKey) && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    /* eslint-disable max-len */
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    /* eslint-enable max-len */
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        /* eslint-disable-next-line max-len */
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: props.tunnelOptions,
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * Return the given named metric for all VPN connections in the account/region.
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
_b = JSII_RTTI_SYMBOL_1;
VpnConnection[_b] = { fqn: "@aws-cdk/aws-ec2.VpnConnection", version: "1.146.0" };
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,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