"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Instance = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const aspects_1 = require("./aspects");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const ebs_util_1 = require("./private/ebs-util");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * This represents a single EC2 instance
 */
class Instance extends core_1.Resource {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.securityGroups = [];
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_InstanceProps(props);
        if (props.initOptions && !props.init) {
            throw new Error('Setting \'initOptions\' requires that \'init\' is also set');
        }
        if (props.securityGroup) {
            this.securityGroup = props.securityGroup;
        }
        else {
            this.securityGroup = new security_group_1.SecurityGroup(this, 'InstanceSecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound !== false,
            });
        }
        this.connections = new connections_1.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tags.of(this).add(NAME_TAG, props.instanceName || this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_b = props.userData) !== null && _b !== void 0 ? _b : imageConfig.userData;
        const userDataToken = core_1.Lazy.string({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.list({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        let subnet;
        if (props.availabilityZone) {
            const selected = subnets.filter(sn => sn.availabilityZone === props.availabilityZone);
            if (selected.length === 1) {
                subnet = selected[0];
            }
            else {
                core_1.Annotations.of(this).addError(`Need exactly 1 subnet to match AZ '${props.availabilityZone}', found ${selected.length}. Use a different availabilityZone.`);
            }
        }
        else {
            if (subnets.length > 0) {
                subnet = subnets[0];
            }
            else {
                core_1.Annotations.of(this).addError(`Did not find any subnets matching '${JSON.stringify(props.vpcSubnets)}', please use a different selection.`);
            }
        }
        if (!subnet) {
            // We got here and we don't have a subnet because of validation errors.
            // Invent one on the spot so the code below doesn't fail.
            subnet = vpc_1.Subnet.fromSubnetAttributes(this, 'DummySubnet', {
                subnetId: 's-notfound',
                availabilityZone: 'az-notfound',
            });
        }
        this.instance = new ec2_generated_1.CfnInstance(this, 'Resource', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            securityGroupIds: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            subnetId: subnet.subnetId,
            availabilityZone: subnet.availabilityZone,
            sourceDestCheck: props.sourceDestCheck,
            blockDeviceMappings: props.blockDevices !== undefined ? ebs_util_1.instanceBlockDeviceMappings(this, props.blockDevices) : undefined,
            privateIpAddress: props.privateIpAddress,
            propagateTagsToVolumeOnCreation: props.propagateTagsToVolumeOnCreation,
        });
        this.instance.node.addDependency(this.role);
        this.osType = imageConfig.osType;
        this.node.defaultChild = this.instance;
        this.instanceId = this.instance.ref;
        this.instanceAvailabilityZone = this.instance.attrAvailabilityZone;
        this.instancePrivateDnsName = this.instance.attrPrivateDnsName;
        this.instancePrivateIp = this.instance.attrPrivateIp;
        this.instancePublicDnsName = this.instance.attrPublicDnsName;
        this.instancePublicIp = this.instance.attrPublicIp;
        if (props.init) {
            this.applyCloudFormationInit(props.init, props.initOptions);
        }
        this.applyUpdatePolicies(props);
        // Trigger replacement (via new logical ID) on user data change, if specified or cfn-init is being used.
        //
        // This is slightly tricky -- we need to resolve the UserData string (in order to get at actual Asset hashes,
        // instead of the Token stringifications of them ('${Token[1234]}'). However, in the case of CFN Init usage,
        // a UserData is going to contain the logicalID of the resource itself, which means infinite recursion if we
        // try to naively resolve. We need a recursion breaker in this.
        const originalLogicalId = core_1.Stack.of(this).getLogicalId(this.instance);
        let recursing = false;
        this.instance.overrideLogicalId(core_1.Lazy.uncachedString({
            produce: (context) => {
                var _b;
                if (recursing) {
                    return originalLogicalId;
                }
                if (!((_b = props.userDataCausesReplacement) !== null && _b !== void 0 ? _b : props.initOptions)) {
                    return originalLogicalId;
                }
                const md5 = crypto.createHash('md5');
                recursing = true;
                try {
                    md5.update(JSON.stringify(context.resolve(this.userData.render())));
                }
                finally {
                    recursing = false;
                }
                const digest = md5.digest('hex').substr(0, 16);
                return `${originalLogicalId}${digest}`;
            },
        }));
        if (props.requireImdsv2) {
            core_1.Aspects.of(this).add(new aspects_1.InstanceRequireImdsv2Aspect());
        }
    }
    /**
     * Add the security group to the instance.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ISecurityGroup(securityGroup);
        this.securityGroups.push(securityGroup);
    }
    /**
     * Add command to the startup script of the instance.
     * The command must be in the scripting language supported by the instance's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Use a CloudFormation Init configuration at instance startup
     *
     * This does the following:
     *
     * - Attaches the CloudFormation Init metadata to the Instance resource.
     * - Add commands to the instance UserData to run `cfn-init` and `cfn-signal`.
     * - Update the instance's CreationPolicy to wait for the `cfn-signal` commands.
     */
    applyCloudFormationInit(init, options = {}) {
        var _b;
        init.attach(this.instance, {
            platform: this.osType,
            instanceRole: this.role,
            userData: this.userData,
            configSets: options.configSets,
            embedFingerprint: options.embedFingerprint,
            printLog: options.printLog,
            ignoreFailures: options.ignoreFailures,
            includeRole: options.includeRole,
            includeUrl: options.includeUrl,
        });
        this.waitForResourceSignal((_b = options.timeout) !== null && _b !== void 0 ? _b : core_1.Duration.minutes(5));
    }
    /**
     * Wait for a single additional resource signal
     *
     * Add 1 to the current ResourceSignal Count and add the given timeout to the current timeout.
     *
     * Use this to pause the CloudFormation deployment to wait for the instances
     * in the AutoScalingGroup to report successful startup during
     * creation and updates. The UserData script needs to invoke `cfn-signal`
     * with a success or failure code after it is done setting up the instance.
     */
    waitForResourceSignal(timeout) {
        var _b, _c;
        const oldResourceSignal = (_b = this.instance.cfnOptions.creationPolicy) === null || _b === void 0 ? void 0 : _b.resourceSignal;
        this.instance.cfnOptions.creationPolicy = {
            ...this.instance.cfnOptions.creationPolicy,
            resourceSignal: {
                count: ((_c = oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.count) !== null && _c !== void 0 ? _c : 0) + 1,
                timeout: ((oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout) ? core_1.Duration.parse(oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout).plus(timeout) : timeout).toIsoString(),
            },
        };
    }
    /**
     * Apply CloudFormation update policies for the instance
     */
    applyUpdatePolicies(props) {
        if (props.resourceSignalTimeout !== undefined) {
            this.instance.cfnOptions.creationPolicy = {
                ...this.instance.cfnOptions.creationPolicy,
                resourceSignal: {
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toIsoString(),
                },
            };
        }
    }
}
exports.Instance = Instance;
_a = JSII_RTTI_SYMBOL_1;
Instance[_a] = { fqn: "@aws-cdk/aws-ec2.Instance", version: "1.146.0" };
//# sourceMappingURL=data:application/json;base64,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