"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SchedulingStrategy = exports.EcsService = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
class EcsService extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { cluster, serviceName, launchType = aws_ecs_1.LaunchType.FARGATE, platformVersion = '1.4.0', desiredCount = 1, containerPort = 80, prodTargetGroup, taskDefinition, } = props;
        const { vpc } = cluster;
        const securityGroups = props.securityGroups || [
            new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} service`,
                vpc,
            }),
        ];
        const serviceToken = core_1.CustomResourceProvider.getOrCreate(this, 'Custom::BlueGreenService', {
            codeDirectory: path.join(__dirname, 'lambdas', 'ecs-service'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                {
                    Effect: aws_iam_1.Effect.ALLOW,
                    Action: [
                        'ecs:CreateService',
                        'ecs:UpdateService',
                        'ecs:DeleteService',
                        'ecs:DescribeServices',
                    ],
                    Resource: '*',
                },
                {
                    Effect: aws_iam_1.Effect.ALLOW,
                    Action: ['iam:PassRole'],
                    Resource: taskDefinition.executionRole.roleArn,
                },
            ],
        });
        const service = new core_1.CustomResource(this, 'CustomResource', {
            serviceToken,
            resourceType: 'Custom::BlueGreenService',
            properties: {
                Cluster: cluster.clusterName,
                ServiceName: serviceName,
                TaskDefinition: taskDefinition.taskDefinitionArn,
                LaunchType: launchType,
                PlatformVersion: platformVersion,
                DesiredCount: desiredCount,
                Subnets: vpc.privateSubnets.map((sn) => sn.subnetId),
                SecurityGroups: securityGroups.map((sg) => sg.securityGroupId),
                TargetGroupArn: prodTargetGroup.targetGroupArn,
                ContainerPort: containerPort,
                SchedulingStrategy: SchedulingStrategy.REPLICA,
            },
        });
        service.node.addDependency(prodTargetGroup.loadBalancerAttached);
        this.serviceName = service.getAttString('ServiceName');
        this.clusterName = cluster.clusterName;
        this.connections = new aws_ec2_1.Connections({
            securityGroups,
            defaultPort: aws_ec2_1.Port.tcp(containerPort),
        });
    }
}
exports.EcsService = EcsService;
var SchedulingStrategy;
(function (SchedulingStrategy) {
    SchedulingStrategy["REPLICA"] = "REPLICA";
    SchedulingStrategy["DAEMON"] = "DAEMON";
})(SchedulingStrategy = exports.SchedulingStrategy || (exports.SchedulingStrategy = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWNzLXNlcnZpY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvZWNzLXNlcnZpY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLHdDQUt1QjtBQUN2Qiw4Q0FLMEI7QUFFMUIsOENBQXdEO0FBQ3hELDhDQUEwQztBQXFCMUMsTUFBYSxVQUFXLFNBQVEsZ0JBQVM7SUFLdkMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFzQjtRQUM5RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sRUFDSixPQUFPLEVBQ1AsV0FBVyxFQUNYLFVBQVUsR0FBRyxvQkFBVSxDQUFDLE9BQU8sRUFDL0IsZUFBZSxHQUFHLE9BQU8sRUFDekIsWUFBWSxHQUFHLENBQUMsRUFDaEIsYUFBYSxHQUFHLEVBQUUsRUFDbEIsZUFBZSxFQUNmLGNBQWMsR0FDZixHQUFHLEtBQUssQ0FBQztRQUVWLE1BQU0sRUFBRSxHQUFHLEVBQUUsR0FBRyxPQUFPLENBQUM7UUFFeEIsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSTtZQUM3QyxJQUFJLHVCQUFhLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRTtnQkFDdkMsV0FBVyxFQUFFLHNCQUFzQixJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsVUFBVTtnQkFDekQsR0FBRzthQUNKLENBQUM7U0FDSCxDQUFDO1FBRUYsTUFBTSxZQUFZLEdBQUcsNkJBQXNCLENBQUMsV0FBVyxDQUNyRCxJQUFJLEVBQ0osMEJBQTBCLEVBQzFCO1lBQ0UsYUFBYSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxhQUFhLENBQUM7WUFDN0QsT0FBTyxFQUFFLG9DQUE2QixDQUFDLFNBQVM7WUFDaEQsZ0JBQWdCLEVBQUU7Z0JBQ2hCO29CQUNFLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7b0JBQ3BCLE1BQU0sRUFBRTt3QkFDTixtQkFBbUI7d0JBQ25CLG1CQUFtQjt3QkFDbkIsbUJBQW1CO3dCQUNuQixzQkFBc0I7cUJBQ3ZCO29CQUNELFFBQVEsRUFBRSxHQUFHO2lCQUNkO2dCQUNEO29CQUNFLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7b0JBQ3BCLE1BQU0sRUFBRSxDQUFDLGNBQWMsQ0FBQztvQkFDeEIsUUFBUSxFQUFFLGNBQWMsQ0FBQyxhQUFhLENBQUMsT0FBTztpQkFDL0M7YUFDRjtTQUNGLENBQ0YsQ0FBQztRQUVGLE1BQU0sT0FBTyxHQUFHLElBQUkscUJBQWMsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDekQsWUFBWTtZQUNaLFlBQVksRUFBRSwwQkFBMEI7WUFDeEMsVUFBVSxFQUFFO2dCQUNWLE9BQU8sRUFBRSxPQUFPLENBQUMsV0FBVztnQkFDNUIsV0FBVyxFQUFFLFdBQVc7Z0JBQ3hCLGNBQWMsRUFBRSxjQUFjLENBQUMsaUJBQWlCO2dCQUNoRCxVQUFVLEVBQUUsVUFBVTtnQkFDdEIsZUFBZSxFQUFFLGVBQWU7Z0JBQ2hDLFlBQVksRUFBRSxZQUFZO2dCQUMxQixPQUFPLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUM7Z0JBQ3BELGNBQWMsRUFBRSxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDO2dCQUM5RCxjQUFjLEVBQUUsZUFBZSxDQUFDLGNBQWM7Z0JBQzlDLGFBQWEsRUFBRSxhQUFhO2dCQUM1QixrQkFBa0IsRUFBRSxrQkFBa0IsQ0FBQyxPQUFPO2FBQy9DO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsZUFBZSxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFFakUsSUFBSSxDQUFDLFdBQVcsR0FBRyxPQUFPLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3ZELElBQUksQ0FBQyxXQUFXLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztRQUV2QyxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUkscUJBQVcsQ0FBQztZQUNqQyxjQUFjO1lBQ2QsV0FBVyxFQUFFLGNBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDO1NBQ3JDLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQWxGRCxnQ0FrRkM7QUFFRCxJQUFZLGtCQUdYO0FBSEQsV0FBWSxrQkFBa0I7SUFDNUIseUNBQW1CLENBQUE7SUFDbkIsdUNBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQUhXLGtCQUFrQixHQUFsQiwwQkFBa0IsS0FBbEIsMEJBQWtCLFFBRzdCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7XG4gIENvbnN0cnVjdCxcbiAgQ3VzdG9tUmVzb3VyY2UsXG4gIEN1c3RvbVJlc291cmNlUHJvdmlkZXIsXG4gIEN1c3RvbVJlc291cmNlUHJvdmlkZXJSdW50aW1lLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7XG4gIElDb25uZWN0YWJsZSxcbiAgQ29ubmVjdGlvbnMsXG4gIFNlY3VyaXR5R3JvdXAsXG4gIFBvcnQsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0IHsgSVRhcmdldEdyb3VwIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5ndjInO1xuaW1wb3J0IHsgSUNsdXN0ZXIsIExhdW5jaFR5cGUgfSBmcm9tICdAYXdzLWNkay9hd3MtZWNzJztcbmltcG9ydCB7IEVmZmVjdCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuXG5pbXBvcnQgeyBEdW1teVRhc2tEZWZpbml0aW9uIH0gZnJvbSAnLi9kdW1teS10YXNrLWRlZmluaXRpb24nO1xuXG5leHBvcnQgaW50ZXJmYWNlIElFY3NTZXJ2aWNlIHtcbiAgcmVhZG9ubHkgY2x1c3Rlck5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgc2VydmljZU5hbWU6IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBFY3NTZXJ2aWNlUHJvcHMge1xuICByZWFkb25seSBzZWN1cml0eUdyb3Vwcz86IFNlY3VyaXR5R3JvdXBbXTtcbiAgcmVhZG9ubHkgY2x1c3RlcjogSUNsdXN0ZXI7XG4gIHJlYWRvbmx5IHNlcnZpY2VOYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGxhdW5jaFR5cGU/OiBMYXVuY2hUeXBlO1xuICByZWFkb25seSBwbGF0Zm9ybVZlcnNpb24/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IGRlc2lyZWRDb3VudD86IG51bWJlcjtcbiAgcmVhZG9ubHkgY29udGFpbmVyUG9ydD86IG51bWJlcjtcbiAgcmVhZG9ubHkgcHJvZFRhcmdldEdyb3VwOiBJVGFyZ2V0R3JvdXA7XG4gIHJlYWRvbmx5IHRhc2tEZWZpbml0aW9uOiBEdW1teVRhc2tEZWZpbml0aW9uO1xufVxuXG5leHBvcnQgY2xhc3MgRWNzU2VydmljZSBleHRlbmRzIENvbnN0cnVjdCBpbXBsZW1lbnRzIElDb25uZWN0YWJsZSwgSUVjc1NlcnZpY2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgY2x1c3Rlck5hbWU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHNlcnZpY2VOYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogQ29ubmVjdGlvbnM7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEVjc1NlcnZpY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCB7XG4gICAgICBjbHVzdGVyLFxuICAgICAgc2VydmljZU5hbWUsXG4gICAgICBsYXVuY2hUeXBlID0gTGF1bmNoVHlwZS5GQVJHQVRFLFxuICAgICAgcGxhdGZvcm1WZXJzaW9uID0gJzEuNC4wJyxcbiAgICAgIGRlc2lyZWRDb3VudCA9IDEsXG4gICAgICBjb250YWluZXJQb3J0ID0gODAsXG4gICAgICBwcm9kVGFyZ2V0R3JvdXAsXG4gICAgICB0YXNrRGVmaW5pdGlvbixcbiAgICB9ID0gcHJvcHM7XG5cbiAgICBjb25zdCB7IHZwYyB9ID0gY2x1c3RlcjtcblxuICAgIGNvbnN0IHNlY3VyaXR5R3JvdXBzID0gcHJvcHMuc2VjdXJpdHlHcm91cHMgfHwgW1xuICAgICAgbmV3IFNlY3VyaXR5R3JvdXAodGhpcywgJ1NlY3VyaXR5R3JvdXAnLCB7XG4gICAgICAgIGRlc2NyaXB0aW9uOiBgU2VjdXJpdHkgZ3JvdXAgZm9yICR7dGhpcy5ub2RlLmlkfSBzZXJ2aWNlYCxcbiAgICAgICAgdnBjLFxuICAgICAgfSksXG4gICAgXTtcblxuICAgIGNvbnN0IHNlcnZpY2VUb2tlbiA9IEN1c3RvbVJlc291cmNlUHJvdmlkZXIuZ2V0T3JDcmVhdGUoXG4gICAgICB0aGlzLFxuICAgICAgJ0N1c3RvbTo6Qmx1ZUdyZWVuU2VydmljZScsXG4gICAgICB7XG4gICAgICAgIGNvZGVEaXJlY3Rvcnk6IHBhdGguam9pbihfX2Rpcm5hbWUsICdsYW1iZGFzJywgJ2Vjcy1zZXJ2aWNlJyksXG4gICAgICAgIHJ1bnRpbWU6IEN1c3RvbVJlc291cmNlUHJvdmlkZXJSdW50aW1lLk5PREVKU18xMixcbiAgICAgICAgcG9saWN5U3RhdGVtZW50czogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdlY3M6Q3JlYXRlU2VydmljZScsXG4gICAgICAgICAgICAgICdlY3M6VXBkYXRlU2VydmljZScsXG4gICAgICAgICAgICAgICdlY3M6RGVsZXRlU2VydmljZScsXG4gICAgICAgICAgICAgICdlY3M6RGVzY3JpYmVTZXJ2aWNlcycsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgUmVzb3VyY2U6ICcqJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgQWN0aW9uOiBbJ2lhbTpQYXNzUm9sZSddLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHRhc2tEZWZpbml0aW9uLmV4ZWN1dGlvblJvbGUucm9sZUFybixcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgY29uc3Qgc2VydmljZSA9IG5ldyBDdXN0b21SZXNvdXJjZSh0aGlzLCAnQ3VzdG9tUmVzb3VyY2UnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW4sXG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OkJsdWVHcmVlblNlcnZpY2UnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBDbHVzdGVyOiBjbHVzdGVyLmNsdXN0ZXJOYW1lLFxuICAgICAgICBTZXJ2aWNlTmFtZTogc2VydmljZU5hbWUsXG4gICAgICAgIFRhc2tEZWZpbml0aW9uOiB0YXNrRGVmaW5pdGlvbi50YXNrRGVmaW5pdGlvbkFybixcbiAgICAgICAgTGF1bmNoVHlwZTogbGF1bmNoVHlwZSxcbiAgICAgICAgUGxhdGZvcm1WZXJzaW9uOiBwbGF0Zm9ybVZlcnNpb24sXG4gICAgICAgIERlc2lyZWRDb3VudDogZGVzaXJlZENvdW50LFxuICAgICAgICBTdWJuZXRzOiB2cGMucHJpdmF0ZVN1Ym5ldHMubWFwKChzbikgPT4gc24uc3VibmV0SWQpLFxuICAgICAgICBTZWN1cml0eUdyb3Vwczogc2VjdXJpdHlHcm91cHMubWFwKChzZykgPT4gc2cuc2VjdXJpdHlHcm91cElkKSxcbiAgICAgICAgVGFyZ2V0R3JvdXBBcm46IHByb2RUYXJnZXRHcm91cC50YXJnZXRHcm91cEFybixcbiAgICAgICAgQ29udGFpbmVyUG9ydDogY29udGFpbmVyUG9ydCxcbiAgICAgICAgU2NoZWR1bGluZ1N0cmF0ZWd5OiBTY2hlZHVsaW5nU3RyYXRlZ3kuUkVQTElDQSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBzZXJ2aWNlLm5vZGUuYWRkRGVwZW5kZW5jeShwcm9kVGFyZ2V0R3JvdXAubG9hZEJhbGFuY2VyQXR0YWNoZWQpO1xuXG4gICAgdGhpcy5zZXJ2aWNlTmFtZSA9IHNlcnZpY2UuZ2V0QXR0U3RyaW5nKCdTZXJ2aWNlTmFtZScpO1xuICAgIHRoaXMuY2x1c3Rlck5hbWUgPSBjbHVzdGVyLmNsdXN0ZXJOYW1lO1xuXG4gICAgdGhpcy5jb25uZWN0aW9ucyA9IG5ldyBDb25uZWN0aW9ucyh7XG4gICAgICBzZWN1cml0eUdyb3VwcyxcbiAgICAgIGRlZmF1bHRQb3J0OiBQb3J0LnRjcChjb250YWluZXJQb3J0KSxcbiAgICB9KTtcbiAgfVxufVxuXG5leHBvcnQgZW51bSBTY2hlZHVsaW5nU3RyYXRlZ3kge1xuICBSRVBMSUNBID0gJ1JFUExJQ0EnLFxuICBEQUVNT04gPSAnREFFTU9OJyxcbn1cbiJdfQ==