"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RollbackEvent = exports.EcsDeploymentGroup = void 0;
const path = require("path");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_codedeploy_1 = require("@aws-cdk/aws-codedeploy");
const core_1 = require("@aws-cdk/core");
class EcsDeploymentGroup extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const { applicationName, deploymentGroupName, deploymentConfig, ecsServices, targetGroupNames, prodTrafficListener, testTrafficListener, terminationWaitTimeInMinutes = 60, autoRollbackOnEvents, } = props;
        if (terminationWaitTimeInMinutes > 2880) {
            throw new Error('Invalid TerminationWaitTimeInMinutes: The maximum setting is 2880 minutes (2 days).');
        }
        const codeDeployEcsRole = new aws_iam_1.Role(this, 'EcsCodeDeployRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSCodeDeployRoleForECS'),
            ],
        });
        this.application = new aws_codedeploy_1.EcsApplication(this, 'EcsApplication', {
            applicationName,
        });
        const serviceToken = core_1.CustomResourceProvider.getOrCreate(this, 'Custom::EcsDeploymentGroup', {
            codeDirectory: path.join(__dirname, 'lambdas', 'ecs-deployment-group'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                {
                    Effect: aws_iam_1.Effect.ALLOW,
                    Action: [
                        'codeDeploy:CreateDeploymentGroup',
                        'codeDeploy:UpdateDeploymentGroup',
                        'codeDeploy:DeleteDeploymentGroup',
                    ],
                    Resource: '*',
                },
                {
                    Effect: aws_iam_1.Effect.ALLOW,
                    Action: ['iam:PassRole'],
                    Resource: codeDeployEcsRole.roleArn,
                },
            ],
        });
        const ecsDeploymentGroup = new core_1.CustomResource(this, 'CustomResource', {
            serviceToken,
            resourceType: 'Custom::EcsDeploymentGroup',
            properties: {
                ApplicationName: this.application.applicationName,
                DeploymentGroupName: deploymentGroupName,
                ServiceRoleArn: codeDeployEcsRole.roleArn,
                TargetGroupNames: targetGroupNames,
                EcsServices: ecsServices.map((service) => ({
                    ClusterName: service.clusterName,
                    ServiceName: service.serviceName,
                })),
                ProdTrafficListenerArn: prodTrafficListener.listenerArn,
                TestTrafficListenerArn: testTrafficListener.listenerArn,
                TerminationWaitTimeInMinutes: terminationWaitTimeInMinutes,
                AutoRollbackOnEvents: autoRollbackOnEvents,
            },
        });
        this.deploymentGroupName = ecsDeploymentGroup.ref;
        this.deploymentGroupArn = this.arnForDeploymentGroup(this.application.applicationName, this.deploymentGroupName);
        this.deploymentConfig = deploymentConfig || aws_codedeploy_1.EcsDeploymentConfig.ALL_AT_ONCE;
    }
    arnForDeploymentGroup(applicationName, deploymentGroupName) {
        return `arn:${core_1.Aws.PARTITION}:codedeploy:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:deploymentgroup:${applicationName}/${deploymentGroupName}`;
    }
}
exports.EcsDeploymentGroup = EcsDeploymentGroup;
var RollbackEvent;
(function (RollbackEvent) {
    RollbackEvent["DEPLOYMENT_FAILURE"] = "DEPLOYMENT_FAILURE";
    RollbackEvent["DEPLOYMENT_STOP_ON_ALARM"] = "DEPLOYMENT_STOP_ON_ALARM";
    RollbackEvent["DEPLOYMENT_STOP_ON_REQUEST"] = "DEPLOYMENT_STOP_ON_REQUEST";
})(RollbackEvent = exports.RollbackEvent || (exports.RollbackEvent = {}));
//# sourceMappingURL=data:application/json;base64,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