"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BlueGreenService = exports.SchedulingStrategy = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
var SchedulingStrategy;
(function (SchedulingStrategy) {
    SchedulingStrategy["REPLICA"] = "REPLICA";
    SchedulingStrategy["DAEMON"] = "DAEMON";
})(SchedulingStrategy = exports.SchedulingStrategy || (exports.SchedulingStrategy = {}));
class BlueGreenService extends core_1.Construct {
    constructor(parent, id, props) {
        super(parent, id);
        const { cluster, serviceName, launchType = aws_ecs_1.LaunchType.FARGATE, platformVersion = '1.4.0', desiredCount = 1, containerPort = 80, prodTargetGroup, taskDefinition, } = props;
        const { vpc } = cluster;
        const securityGroups = props.securityGroups || [
            new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} service`,
                vpc,
            }),
        ];
        const serviceToken = core_1.CustomResourceProvider.getOrCreate(this, 'Custom::BlueGreenService', {
            codeDirectory: path.join(__dirname, 'lambdas', 'blue-green-service'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                {
                    Effect: aws_iam_1.Effect.ALLOW,
                    Action: [
                        'ecs:CreateService',
                        'ecs:UpdateService',
                        'ecs:DeleteService',
                    ],
                    Resource: '*',
                },
                {
                    Effect: aws_iam_1.Effect.ALLOW,
                    Action: ['iam:PassRole'],
                    Resource: taskDefinition.executionRole.roleArn,
                },
            ],
        });
        const service = new core_1.CustomResource(this, 'CustomResource', {
            serviceToken,
            resourceType: 'Custom::BlueGreenService',
            properties: {
                Cluster: cluster.clusterName,
                ServiceName: serviceName,
                TaskDefinition: taskDefinition.taskDefinitionArn,
                LaunchType: launchType,
                PlatformVersion: platformVersion,
                DesiredCount: desiredCount,
                Subnets: vpc.privateSubnets.map((sn) => sn.subnetId),
                SecurityGroups: securityGroups.map((sg) => sg.securityGroupId),
                TargetGroupArn: prodTargetGroup.targetGroupArn,
                ContainerPort: containerPort,
                SchedulingStrategy: SchedulingStrategy.REPLICA,
            },
        });
        service.node.addDependency(prodTargetGroup.loadBalancerAttached);
        this.serviceName = service.getAttString('ServiceName');
        this.connections = new aws_ec2_1.Connections({
            securityGroups,
            defaultPort: aws_ec2_1.Port.tcp(containerPort),
        });
    }
}
exports.BlueGreenService = BlueGreenService;
//# sourceMappingURL=data:application/json;base64,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