'''
This file contains the implementation of two variants of the InputAware layer.
The architecture is as follows:

Class Hidden:
    Takes the input image and generates 'out_channels' * 'in_channels' kernels to be used by the convolution operator,
    i.e batch_size * 'out_channels' * 'in_channels' kernels each one has 'kernel_size' in width and height.

Class ConvBlock:
    Given an input image (or a feature map) with 'in_channels' channels, 
    it generates kernels for each image using Hidden, and returns a feature map having 
    'out_channels' channels and the same width and height of the input.

'''
from typing import Tuple, Union
import torch
import torch.nn as nn
import torch.nn.functional as F


class Hidden(nn.Module):
    def __init__(self, in_channels, out_channels, mid_channels, drop_pb, kernel_size: int, use_linear: bool = True):
        ''' Initializes the architectue: conv2d_1 - AdaptiveMaxPool2d - Dropout - conv2d_2 - Optional[Linear_Layers]

        Args:
            - in_channels: the number of channels of the input tensor.
            - out_channels: the number of kernels to generate for every tensor in the batch.
            - mid_channels: the number of output channels for conv2d_1.
            - kernel_size: the size of the desired kernel.
            - drop_pb: p for the Dropout layer.
            - use_linear: if True, apply a linear layer to each kernel. 
                        There's a linear layer for each kernel, but they are shared across input tensors.
        '''
        super().__init__()

        # Model definition
        self.conv_block = nn.Sequential(
            nn.Conv2d(in_channels, mid_channels, 3, stride=1, padding=1),
            nn.AdaptiveMaxPool2d(kernel_size),
            nn.Dropout(drop_pb),
            nn.Conv2d(mid_channels, out_channels, 3, stride=1, padding=1),
        )
        
        # 
        self.use_linear = use_linear
        if use_linear:
            self.linears = nn.Conv2d(
                out_channels, out_channels*(kernel_size[0]*kernel_size[1]), kernel_size, padding=0, groups=out_channels)

    def forward(self, z):
        '''
        Args:
            - z: an input tensor whose shape: [batch, in_channels, H, W]

        Returns:
            - A tensor whose shape is [batch, out_channels, kernel_size, kernel_size]
        '''
        # Extract kernels
        z = self.conv_block(z)

        if not self.use_linear:
            return z

        # Apply 'linears[i]' to each kernel in the i^th channel.
        b, c, h, w = z.size()
        z = self.linears(z)
        # Eliminate width & height, since they are always equal to 1.
        z = z.squeeze(-1).squeeze(-1)
        # unpack channel dimension back to: channel, height, width
        z = z.unflatten(-1, (c, h, w))

        return z


class InputAware(nn.Module):
    def __init__(self, in_channels, out_channels, mid_channels, drop_pb,
                 kernel_size: Union[int, Tuple[int, int]] = 3, stride: int = 1, padding: int = 1, use_linear: bool = True):
        ''' Initializes the architectue: Hidden - conv2d (using the kernels generated by Hidden)

        Parameters:
            - in_channels: the number of channels of the input tensor.
            - out_channels: out_channels of the resutling tensor.
            - mid_channels: the number of output channels for the intermediate conv2d layer in 'Hidden'.
            - kernel_size: the size of the desired kernel. It should be an odd integer, tuples are not supported yet.
            - drop_pb: p for the Dropout layer in 'Hidden'.
            - use_linear: if True, apply a linear layer to each kernel before using it. 
                        There's a linear layer for each channel, but they are shared across input tensors.
            - stride, padding: used by the final conv2d layer.
        '''
        super().__init__()

        # Make 'kernel_size' a tuple of two ints
        if isinstance(kernel_size, int):
            kernel_size = (kernel_size, kernel_size)
        if len(kernel_size) == 1:
            kernel_size = (kernel_size[0], kernel_size[0])

        # Check the parity of 'kernel_size'
        assert torch.tensor(kernel_size).prod()%2==1, f'The kernel size must be an odd number {kernel_size}'

        self.in_channels, self.out_channels = in_channels, out_channels
        self.stride, self.padding = stride, padding
        self.hidden = Hidden(in_channels, in_channels*out_channels, mid_channels,
                             drop_pb, kernel_size,
                             use_linear=use_linear)
        self.bias = nn.Parameter(torch.randn(out_channels))

    def forward(self, x):
        '''
        Parameter:
            - x: an input tensor whose shape: [batch, in_channels, H, W]

        Returns:
            - A tensor whose shape is [batch, out_channels, H', W'], 
                H' & W' depending the values of stride & padding
        '''
        batch = x.size(0)
        kernels = self.hidden(x)

        # The bias isn't input-sensitive, it's shared among all inputs.
        bias = self.bias.repeat(batch)

        # Reshape a batch of feature maps to be an image with a looot of channels (batch x in_channels)
        x = x.flatten(0, 1).unsqueeze(0)

        # Reshape the kernels to have the following shape:
        # (batch x out_channels), in_channels, kernel_size, kernel_size
        kernels = kernels.unflatten(
            1, (self.out_channels, self.in_channels)).flatten(0, 1)

        # Apply 2d Conv in an image-wise manner, first in_channels kernels applied to first images,
        # 2nd kernels to 2nd image, etc.
        # Then separate the out_channels of each image, yielding a batch dimension.
        z = F.conv2d(x, kernels, bias=bias, padding=self.padding, stride=self.stride, groups=batch).unflatten(
            1, (batch, self.out_channels)).squeeze(0)
        return z