# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['braincube_connector',
 'braincube_connector.bases',
 'braincube_connector.data',
 'braincube_connector.memory_base',
 'braincube_connector.memory_base.nested_resources']

package_data = \
{'': ['*']}

install_requires = \
['pandas>=1.1.4,<2.0.0', 'requests>=2.23.0,<3.0.0']

setup_kwargs = {
    'name': 'braincube-connector',
    'version': '2.4.1',
    'description': 'python client to the braincube web services',
    'long_description': '# braincube_connector: a python client for Braincube\n\n## Description\n\nThe python package `braincube_connector` provides a tool for datascientists to access their data on Braincube directly from python.\n\n## Installation\n\nInstall with pip:\n\n```bash\npip install braincube_connector\n```\n\n## Configuration and Authentication\n\nSince version `2.2.0`, the authentication uses a personal access token (PAT).\n\nIn order to create a PAT, you need to go in your braincube personal `menu > Account > Access tokens > +/Add`\n\nThe scopes of the token should include `BRAINCUBE` and `SSO_READ`.\n\nThen two options exist to pass the PAT to the braincube_connector:\n 1. Using a configuration dictionary when creating a client:\n```python\nfrom braincube_connector import client\n\nclient.get_instance(config_dict={"api_key":"<my_personal_access_token>", "domain":"mybraincube.com"})\n```\n 2. Using a configuration file:\n ```python\n from braincube_connector import client\n\n client.get_instance("config_file"="myfile.json")\n ```\n *myfile.json*\n ```json\n {"api_key":"<my_personal_access_token>", "domain":"mybraincube.com"}\n ```\n\n### Authentication with an Oauth2 token.\n\nThe *braincube_connector* used to support only this type of authentication. This is not the method we encourage\nthe most since the PAT is available, because the Oauth2 is obtained with the [braincube-token-getter](https://pypi.org/project/braincube-token-getter/) that is not under active development.\nHowever if you still want to use this method, you need to setup the configuration file (or dictionary) as follows:\n`config.json`\n\n```json\n{\n    "client_id": "app id",\n    "client_secret": "app key",\n    "domain": "mybraincube.com",\n    "verify": true,\n    "oauth2_token": "token value"\n}\n```\n\nBy default the connector searches for a PAT and uses the oauth2_token when the PAT is not present in the dictionary.\n\n### Configuration parameters\n\nHere is a list of the settings available in the configuration file:\n\n- `domain`(optional if `sso_base_url` and `braincube_base_url` exist): The domain of the braincube to access.\n- `sso_base_url`(optional if `domain` exists): The base URL of the SSO used to check the validity of your access token.\n- `braincube_base_url`(optional if `domain` exists): The base URL of the Braincube API used to fetch data from.\n- `api_key`(optional if `oauth2_token` exists): a personal access token generated in the braincube account configuration.\n- `oauth2_token`(optional if `api_key` exists): an OAuth2 token obtained with the [braincube-token-getter](https://pypi.org/project/braincube-token-getter/). Used only when `api_key` does not exist.\n- `verify`(optional, default is `True`): If `False`, the requests do not verify the SSL certificate.\n> Setting `verify` to false must be used with care, it\'s a security threat (see [requests documentation](https://requests.readthedocs.io/en/latest/api/#requests.Session.verify)\n\nThe `client_id`,  `client_secret` from the last section are used only by the *braincube_token_getter* when requesting a new OAuth token.\n\n### Note:\nIf the client is not initialized manually or if no configuration is passed to `get_instance`, the package creates a client instance from one of these two files `./config.json`  or `~/.braincube/config.json` (in this priority order) when they exist.\n\n\n\n## Usage\n\n### Client\n\nA client can be inialized manually from a custom configuration file.\n\n```python\nfrom braincube_connector import client\n\nclient.get_instance(config_file="pathto/config.json")\n```\n\n**Note:** If the client is not initialized manually, the package creates a client instance from one of these two files `./config.json`  or `~/.braincube/config.json` (in this priority order) if they exist.\n\n### Features of the connector entities.\n\nThe connector gives access to different entities(described in more details in the following sections) that share multiple methods:\n\n- `<entity>.get_name()`: Returns the name of the entity.\n- `<entity>.get_bcid()`: Returns the bcId identifier of the entity.\n- `<entity>.get_uuid()`: Returns the braincube unique uuid identifier of the entity.\n### Braincube\n\nTo obtain a list of all the available `Braincube` entities with a client:\n```python\nfrom braincube_connector import braincube\n\nbraincube.get_braincube_list()\n```\n\nOr to select a specific `Braincube` entity from its name:\n```python\nbc = braincube.get_braincube("demo")\n```\n\n### MemoryBase\n\nThe list of all the memory bases available within a `Braincube` is obtained with\n\n```python\nmb_list = bc.get_memory_base_list()\n```\n\n**Note:** The number of memory bases in a braincube can be numerous,  hence `get_memory_base_list` allows paginated requests `bc.get_memory_base_list(page=0)`\n\nTo select a unique memory base, go with its bcId:\n\n```python\nmb = bc.get_memory_base(20)\n```\n\n\n### VariableDescriptions\n\nThe variable description are linked to a memory base.\n\n```python\nvar_desc = mb.get_variable(bcid="2000034")\n```\n\nFor multiple variable descriptions:\n\n```python\nmb.get_variable_list(page=0)\n```\n\n**Note:** Similarly to memory bases, providing no argument to `get_variable_list` retrieves all the descriptions available in the memory base.\n\nThe type of variable is obtained with the function `get_type`\n\n```python\nvar_desc.get_type()\n```\n\n### DataGroup\nDataGroup are obtained from a memory base:\n```python\ndatagroup = mb.get_datagroup(bcid="10")\n```\nThe list of the available datagroups can also be obtained with `mb.get_datagroup_list()`.\n\nA datagroup is a container that includes multiple variables. They are accessed with\n```python\ndatagroup.get_variable_ids() # Gets the variable bcIds\ndatagroup.get_variable_list() # Gets the list of VariableDescription objects.\n```\n\n### Event\nAn event is a predifined set of conditions in braincube. It is accessed as follows:\n```python\nevent = mb.get_event(bcid="10")\nevent_list = mb.get_event_list()\n```\n\nThe interest of events is that you can access the conditions they contain in order create new [filters](#data-filters) for a `get_data` function:\n\n```python\nevent.get_conditions()\n```\n\n### JobDescription\n\nThe job desciption contains the settings used to build an analysis and gives a proxy to access these parameters easily. A JobDescription is obtained from a memory base as follows:\n\n```python\njob_desc = mb.get_job(bcid="573")\njob_list = mb.get_job_list(page=0)\n```\n\nThe properties are acced with the following methods:  \n\n- **get_conditions:**  \n  Gets a list of the conditions used to select the job variables.\n  ```python\n  job_desc.get_conditions()\n  job_desc.get_conditions(combine=True) # Merge the conditions into one\n  job_desc.get_conditions(include_events=True) # Includes the conditions from\n                                               # the job\'s events\n  ```\n\n- **get_variable_ids:**  \n  Gets a list of the variables involved in the job, including the target variables and the influence variables.\n  ```python\n  job_desc.get_variable_ids()\n  ```\n\n- **get_events:**  \n  Gets a list of the event objects used by the job.\n  ```python\n  job_desc.get_events()\n  ```\n- **get_categories:**  \n  Gets a list of conditions used to categorise a job\'s data as *good* or *bad*. You may have a *middle* category, it\'s an old categorisation which will not be used anymore.\n  ```python\n  job_desc.get_categories()\n  ```\n\n- **get_data:**  \n  When a job is created on braincube, a separate copy of the data is made. As for now this copy is not available from the webservices. However the `get_data` method collects the job\'s data from the memory base using the same filters as when the job was created. Be aware that these data might be different from the job\'s data if the memory base has been updated since the job creation.  \n\n  Similarly to other object `get_data`, a `filters` parameter is available to add additional [filters](#data-filters) to the job\'s conditions.\n\n  ```python\n  job_desc.get_data()\n  ```\n\n### Job rules\nThe job rule descriptions are obtained with the methods `get_rule` or `get_rule_list` either from a job or a memory base. The only difference being that in the case of a memory base `get_rule_list` gets all the rules existing in the memory base whereas for a job, it gets the rules specific to the job under consideration.\n\n```python\nrule = job.get_rule(bcid="200")\nrule_list = job.get_rule_list()\n```\n\nTo access a `RuleDescription` object\'s metadata, you can calle the `get_metadata` function\n```python\nrule.get_metadata()\n```\n\n\n### Get variable data\n\nA memory base can also request the data for a custom set of variable ids. Adding [filters](#data-filters) restricts the returned data to a desired subset of the data. The method is called as follows:\n```python\ndata = mb.get_data(["2000001", "2000034"], filters=my_filters, label_type="name", dataframe=True)\n```\n\nThe output format is a dictionary or a pandas DataFrame when the `dataframe` parameter is set to `True`. The keys/column labels are the variable bcIds or names depending on whether `label_type` is set to `"bcid"` or `"name"` respectively.\n\n**Note:** By default the dates are not parsed to `datetime` objects in order to speed up the `get_data` function but it is possible to enable the parsing:\n```python\nfrom braincube_connector import parameters\nparameters.set_parameter({"parse_date": True})\n```\n\n### Data filters\nThe `get_data` methods have the option to restrict the data that are collected by using a set of filters. The `filters` parameter must be a list conditions (even for a single condition):\n```python\nobject.get_data(filters=[{"BETWEEN": ["mb20/d2000002",0,10]},{"BETWEEN": ["mb20/d2000003", -1, 1]}])\n```\n\nHere is a selection of the most common types of filters:\n- **Equals to**  \n  Selects data when a variable is equal to\n  ```json\n  {\n    "EQUALS": [ "mb20/d2000002", 2.0]\n  }\n  ```\n- **Between**  \n  Selects the data when a variable belongs to a range.\n  ```json\n  {\n    "BETWEEN": [ "mb20/d2000003", -1, 1]\n  }\n  ```\n- **Lower than**  \n  Selects the data when a variable is lower than a certain value.\n  ```json\n  {\n    "LESS": [ "mb20/d2000003", 10]\n  }\n  ```\n  **Note:** The `LESS_EQUALS` filter also exists.\n\n\n- **Greater than**  \n  Selects the data when a variable is greater than a certain value.\n  ```json\n  {\n    "GREAT": [ "mb20/d2000003", 10]\n  }\n  ```\n  **Note:** The `GREAT_EQUALS` filter also exists.\n\n- **Not:**  \n  The `NOT` condition creates the opposite of an existing condition.\n  ```json\n  {\n    "Not": [{"filter":...}]\n  }\n  ```\n\n- **And gate**  \n  It is possible to combine filters using a *and* gate.\n  ```json\n  {\n    "AND": [{"filter1":...}, {"filter2":...}]\n  }\n  ```\n  **Notes:**  \n    - A `AND` filter can only host two conditions. In order to join more than two filters multiple `AND` conditions should be nested one into another.\n    - When multiple filters are provided in the `get_data`\'s `filters` parameters, they are joined together within the function using `AND` gates.  \n\n- **Or gate:**  \n  Similar to `AND` but uses a `OR` gate.\n  ```json\n  {\n    "OR": [{"filter1":...}, {"filter2":...}]\n  }\n  ```\n## Advanced Usage\n\nThe *braincube_connector* provides a simple interface for the most common features of the *braincube web-services* or *braindata* but it is not extensive.\n\nIf you need to access an endpoint of [*braincube webservices*](https://braincube.io/ws-doc/?urls.primaryName=Braincube%20WS) or [*braindata*](https://braincube.io/ws-doc/?urls.primaryName=Braindata%20WS), the `request_ws` function of the library can help you. The function uses the configuration passed to the client creation to manage the authentication.\n\n```python\nfrom braincube_connector import client\n\nclient.get_instance(config_dict={...})\njson_result = client.request_ws("braincube/demo/braindata/mb20/simple")\n```\n\nMost braincube requests return a json, but for a few of them it might be better to deactivate the parsing by setting the `response_as_json` parameter to `False`. In the latter case, `request_ws` returns the response object.\n\n```python\njson_result = client.request_ws("braincube/demo/braindata/mb20/simple", response_as_json=False)\n```\n\n## Library parameters\nThe library parameters can be set to custom values:\n\n```python\nfrom braincube_connector import parameters\n\n# Change the request pagination size to 10\nparameters.set_parameter({"page_size": 10})\n\n# Parse dates to datetime objects\nparameters.set_parameter({"parse_date": True})\n\n# The Braincube database stores multiple names (`tag`, `standard`, or `local`) for a variable\n# By default `standard` id used, but you can change it as follows:\nparameters.set_parameter(({"VariableDescription_name_key": "tag"}))\n```\n',
    'author': 'Braincube',
    'author_email': 'io@braincube.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://braincube-io.github.io/python-connector/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8.5,<4.0.0',
}


setup(**setup_kwargs)
