import unittest
from copy import deepcopy

import numpy as np
from sklearn.decomposition import PCA
from sklearn.manifold import Isomap

from gbquality.GM import compute_x_leaves, global_judge_x_precomputed, brute_compute_minimum_K, euclidean_distance, \
    compute_paths, binary_search_minimum_K


def generate_data():
    # GENERATE SAMPLED DATA
    # S-CURVE DATASET
    angle = np.pi * (1.5 * np.random.rand(1, 600) - 1)
    height = 5 * np.random.rand(1, 600)
    X1 = np.vstack((np.cos(angle), height, np.sin(angle)))
    angle = np.pi * (1.5 * np.random.rand(1, 100) - 1)
    height = 5 * np.random.rand(1, 100)
    X2 = np.vstack((-np.cos(angle), height, 2 - np.sin(angle)))
    X = np.hstack((X1, X2))
    return X


class TestGM(unittest.TestCase):

    def test_scurve_search_k(self):
        X = generate_data()
        print(X)

        K, pairwise_X, PP = brute_compute_minimum_K(X)
        print('Brute search gives K of {}'.format(K))
        _K, pairwise_X, PP = binary_search_minimum_K(X,max_k=600)
        print('Binary search gives K of {}'.format(_K))
        self.assertEqual(K,_K)

        leaf_indices, leaf_dists, centre_index = compute_x_leaves(X, K, pairwise_X, PP)

        y_isomap = Isomap(n_neighbors=6).fit_transform(deepcopy(X.T)).T
        y_pca = PCA(n_components=2).fit_transform(deepcopy(X.T)).T
        g_isomap = global_judge_x_precomputed(leaf_indices, leaf_dists, centre_index, y_isomap)
        g_pca = global_judge_x_precomputed(leaf_indices, leaf_dists, centre_index, y_pca)
        print('Isomap:' + str(g_isomap))
        print('PCA:' + str(g_pca))
        self.assertTrue(np.isfinite(g_pca))
        self.assertTrue(np.isfinite(g_isomap))

    def test_scurve(self):
        X = generate_data()

        print(X)
        leaf_indices, leaf_dists, centre_index = compute_x_leaves(X, 6)

        y_isomap = Isomap(n_neighbors=6).fit_transform(deepcopy(X.T)).T
        y_pca = PCA(n_components=2).fit_transform(deepcopy(X.T)).T
        g_isomap = global_judge_x_precomputed(leaf_indices, leaf_dists, centre_index, y_isomap)
        g_pca = global_judge_x_precomputed(leaf_indices, leaf_dists, centre_index, y_pca)
        print('Isomap:' + str(g_isomap))
        print('PCA:' + str(g_pca))
        self.assertTrue(np.isfinite(g_pca))
        self.assertTrue(np.isfinite(g_isomap))

    def test_random_data(self):
        X = np.array([
            [4.44, 1.16, 9.00, 1.66, 4.35, 5.72, 2.93, 8.49, 0.13, 8.89, 7.83, 1.02, 5.12, 1.05, 1.25, 1.81, 0.32, 8.88,
             1.76, 1.57, 1.57, 4.95, 2.38, 6.55, 1.29, 4.55, 0.69, 0.96, 6.79, 0.24],
            [2.68, 8.88, 3.44, 2.36, 8.06, 5.11, 7.66, 5.86, 2.17, 5.37, 5.84, 6.34, 6.32, 2.56, 4.06, 4.71, 6.13, 8.36,
             7.93, 1.31, 8.35, 8.27, 4.17, 6.74, 2.61, 1.32, 5.75, 4.15, 0.29, 3.97],
            [3.01, 2.25, 7.26, 3.21, 6.34, 1.39, 5.61, 8.97, 0.39, 4.98, 7.86, 7.47, 1.03, 0.92, 7.40, 0.34, 3.18, 1.61,
             1.64, 8.40, 7.24, 6.90, 2.02, 8.72, 0.87, 4.99, 4.90, 2.31, 8.48, 3.39],
            [4.21, 1.07, 3.42, 8.47, 2.39, 4.13, 5.30, 6.79, 2.10, 7.09, 4.20, 3.05, 3.38, 4.37, 7.68, 0.68, 7.16, 3.08,
             0.62, 5.25, 0.80, 6.87, 1.27, 6.98, 4.51, 5.21, 7.06, 4.28, 5.74, 5.74],
            [2.80, 8.58, 4.20, 8.40, 1.01, 4.92, 1.10, 3.03, 0.05, 6.06, 3.74, 1.45, 0.85, 4.29, 8.64, 1.03, 0.68, 7.61,
             1.38, 7.40, 1.53, 3.25, 7.30, 2.17, 4.05, 8.04, 4.87, 0.55, 0.40, 0.79],
            [6.97, 1.93, 4.77, 3.94, 0.27, 8.41, 0.04, 0.11, 0.17, 7.19, 6.81, 0.22, 2.58, 7.08, 7.06, 4.85, 5.72, 3.66,
             1.56, 3.40, 2.87, 6.82, 3.06, 3.50, 1.88, 4.06, 4.81, 5.60, 3.03, 4.77],
            [8.15, 2.69, 3.37, 0.80, 4.84, 0.56, 7.06, 6.97, 2.35, 8.41, 2.03, 3.10, 2.00, 3.94, 7.04, 1.66, 2.81, 7.35,
             2.97, 3.16, 9.00, 2.86, 6.14, 8.84, 4.82, 7.74, 1.86, 6.72, 1.19, 1.86],
            [6.75, 1.49, 2.22, 0.30, 6.87, 3.30, 8.88, 6.12, 5.99, 1.19, 4.34, 3.20, 0.71, 1.83, 7.10, 4.24, 1.66, 1.00,
             5.78, 0.96, 4.66, 4.77, 1.04, 7.78, 1.09, 2.41, 0.07, 3.25, 6.08, 4.34],
            [1.56, 2.17, 0.60, 7.57, 8.95, 3.17, 6.18, 7.13, 8.54, 3.49, 2.34, 1.51, 3.17, 7.55, 6.96, 3.75, 6.90, 2.32,
             6.64, 2.21, 2.77, 0.90, 1.07, 5.76, 8.95, 8.55, 3.34, 6.32, 6.43, 4.91],
            [2.06, 2.82, 8.43, 4.86, 5.50, 5.66, 7.67, 6.63, 0.69, 0.04, 6.64, 0.35, 2.53, 8.84, 3.12, 5.70, 8.20, 6.88,
             4.27, 0.41, 3.17, 3.22, 6.05, 4.05, 0.60, 0.79, 2.60, 3.29, 0.91, 1.30],
            [5.66, 4.23, 0.60, 4.10, 5.12, 6.03, 2.20, 0.09, 0.52, 5.85, 0.17, 8.69, 8.69, 0.83, 6.74, 1.66, 5.51, 0.47,
             7.12, 3.08, 7.21, 0.75, 0.83, 4.82, 7.16, 7.19, 2.60, 0.50, 3.68, 5.82],
            [4.97, 2.04, 4.34, 6.42, 0.30, 3.06, 9.00, 2.35, 6.43, 2.41, 0.62, 7.19, 4.22, 1.07, 8.29, 8.47, 6.96, 2.53,
             2.35, 5.79, 0.09, 7.67, 4.60, 0.76, 8.86, 6.86, 6.59, 3.61, 6.20, 2.93],
            [4.34, 6.72, 5.86, 8.12, 3.07, 2.25, 7.54, 0.62, 2.66, 5.27, 2.62, 2.17, 8.51, 5.14, 6.79, 7.29, 2.39, 2.89,
             4.14, 7.91, 5.46, 0.96, 7.63, 5.03, 3.13, 2.37, 7.47, 5.31, 7.85, 5.12],
            [8.64, 4.21, 0.49, 0.19, 5.46, 4.67, 2.21, 7.04, 5.47, 3.82, 7.13, 3.52, 1.58, 2.00, 0.38, 4.57, 4.29, 2.17,
             3.49, 7.35, 3.86, 2.24, 6.83, 8.52, 6.79, 2.75, 3.63, 4.69, 6.96, 1.19],
            [6.32, 7.97, 2.59, 2.26, 6.06, 3.91, 7.71, 2.48, 7.47, 0.14, 7.81, 4.99, 3.91, 5.22, 7.20, 0.81, 3.22, 8.47,
             2.85, 0.31, 7.90, 5.19, 0.40, 3.50, 3.69, 2.25, 2.24, 8.71, 1.67, 7.64],
            [1.29, 7.39, 5.56, 1.32, 1.38, 0.33, 6.93, 6.13, 8.76, 6.79, 5.51, 5.84, 4.30, 1.46, 7.71, 4.72, 2.81, 2.48,
             4.98, 7.59, 5.14, 1.86, 4.67, 6.09, 8.82, 0.70, 5.51, 7.28, 8.02, 7.97],
            [4.02, 7.93, 1.47, 8.58, 2.41, 2.51, 0.06, 2.97, 8.71, 6.38, 7.65, 4.55, 2.79, 6.68, 6.97, 0.21, 8.98, 0.30,
             3.08, 7.65, 3.17, 0.68, 3.64, 8.42, 5.31, 4.05, 6.80, 1.61, 3.47, 2.94],
            [1.92, 6.97, 2.56, 7.84, 1.65, 2.83, 7.50, 1.83, 8.65, 6.01, 5.85, 6.83, 8.88, 7.68, 5.20, 6.69, 1.45, 8.88,
             6.74, 5.70, 1.02, 8.76, 0.18, 6.36, 7.39, 3.20, 6.74, 0.87, 8.19, 5.91],
            [3.38, 7.82, 1.76, 4.81, 5.29, 4.65, 4.90, 8.00, 0.04, 3.12, 4.48, 3.15, 6.91, 4.16, 1.65, 6.50, 0.30, 0.83,
             3.99, 8.72, 0.29, 1.74, 6.60, 2.61, 3.04, 2.01, 3.29, 0.93, 5.80, 4.59],
            [7.39, 1.08, 8.40, 2.22, 0.10, 4.56, 2.86, 0.95, 4.74, 3.71, 0.08, 6.54, 4.64, 1.00, 0.58, 4.79, 0.05, 4.61,
             7.36, 8.15, 2.99, 4.12, 5.29, 4.17, 0.17, 6.78, 8.60, 3.51, 5.43, 7.48],
            [7.53, 5.00, 8.42, 6.88, 7.45, 5.86, 1.37, 6.61, 4.92, 7.17, 7.02, 1.61, 5.82, 8.78, 8.18, 0.96, 8.22, 4.41,
             6.02, 7.26, 3.36, 1.64, 2.33, 3.37, 7.14, 7.52, 4.84, 6.01, 6.99, 5.94],
            [4.55, 3.20, 7.16, 7.46, 2.93, 0.21, 7.66, 0.15, 7.33, 5.01, 8.75, 7.41, 4.98, 7.40, 0.55, 6.17, 7.31, 3.52,
             7.61, 8.62, 0.94, 1.91, 7.29, 7.77, 2.00, 1.44, 7.49, 2.59, 3.53, 5.63],
            [8.22, 7.79, 1.76, 2.13, 7.44, 0.50, 4.89, 5.86, 6.72, 0.88, 8.73, 6.71, 5.97, 0.61, 2.91, 2.99, 5.23, 2.99,
             1.43, 4.73, 2.78, 7.15, 7.35, 7.12, 5.49, 4.59, 3.33, 3.62, 1.67, 1.40],
            [7.41, 0.12, 7.09, 5.29, 1.97, 0.94, 6.39, 3.21, 0.77, 8.49, 2.43, 5.51, 4.35, 0.78, 7.67, 2.54, 8.34, 0.83,
             4.00, 4.22, 6.06, 8.47, 2.70, 7.22, 4.63, 8.43, 0.28, 0.07, 2.04, 5.28],
            [8.86, 7.28, 3.64, 5.61, 5.89, 1.78, 2.54, 4.27, 0.27, 1.92, 2.89, 7.19, 4.68, 6.23, 4.38, 8.98, 8.72, 1.44,
             3.36, 1.50, 3.41, 0.36, 8.87, 7.38, 8.63, 2.33, 4.26, 5.79, 0.13, 8.22],
            [3.19, 0.89, 1.31, 5.46, 3.33, 3.04, 1.06, 8.83, 6.22, 1.53, 8.64, 8.30, 0.16, 0.92, 3.06, 5.81, 5.91, 7.50,
             8.89, 1.91, 1.69, 1.51, 8.52, 4.89, 6.01, 3.72, 0.19, 4.65, 6.62, 4.36],
            [1.97, 3.67, 4.13, 8.74, 8.51, 7.43, 0.45, 1.58, 2.35, 1.88, 5.37, 2.35, 8.10, 2.16, 7.06, 4.16, 6.63, 4.69,
             8.85, 3.84, 0.52, 7.88, 1.83, 8.59, 5.12, 6.43, 6.30, 3.02, 5.71, 4.86],
            [2.95, 6.26, 2.48, 3.63, 5.14, 7.91, 6.62, 2.69, 7.14, 2.34, 2.66, 6.62, 4.33, 2.40, 4.06, 3.54, 3.73, 5.31,
             1.84, 1.87, 4.19, 8.96, 0.56, 5.12, 6.72, 1.00, 8.13, 8.35, 5.66, 6.58],
            [2.59, 3.94, 8.88, 8.49, 2.37, 0.19, 0.42, 5.33, 4.93, 1.71, 2.00, 1.69, 3.95, 2.75, 4.80, 8.76, 8.22, 4.23,
             1.99, 3.10, 0.09, 6.48, 4.70, 7.10, 1.88, 6.52, 3.69, 1.76, 0.66, 3.41],
            [8.66, 0.41, 7.69, 4.85, 1.57, 7.70, 6.30, 0.92, 7.34, 6.96, 8.89, 1.13, 6.70, 6.96, 2.22, 4.80, 7.51, 4.28,
             8.12, 4.34, 1.84, 6.31, 5.79, 1.08, 8.17, 2.74, 2.28, 5.33, 0.59, 7.34],
            [8.79, 4.04, 0.56, 4.34, 8.54, 7.28, 4.90, 2.80, 8.38, 4.30, 3.49, 8.52, 6.74, 0.96, 0.88, 4.14, 0.86, 7.84,
             4.61, 4.40, 5.24, 1.48, 3.24, 4.29, 6.65, 0.29, 0.13, 7.98, 6.58, 4.78],
            [5.32, 0.60, 8.26, 8.99, 2.81, 8.09, 6.93, 7.72, 4.17, 7.64, 1.95, 7.69, 4.09, 2.33, 4.47, 2.45, 6.24, 0.34,
             6.98, 3.42, 2.44, 3.52, 5.43, 5.96, 6.87, 4.53, 8.97, 5.56, 3.52, 8.71],
            [7.43, 0.13, 2.50, 5.63, 1.90, 8.40, 8.06, 4.85, 4.28, 1.62, 0.28, 4.93, 4.68, 6.19, 4.09, 8.18, 0.46, 3.68,
             4.96, 7.26, 6.54, 2.49, 5.25, 8.91, 4.95, 6.88, 8.87, 3.49, 6.59, 6.95],
            [7.51, 8.26, 6.23, 4.49, 1.22, 0.36, 5.98, 2.81, 0.16, 3.71, 1.38, 1.76, 1.98, 8.46, 6.75, 3.52, 7.02, 6.43,
             8.98, 7.84, 3.97, 7.86, 8.52, 7.86, 5.50, 0.30, 2.67, 1.16, 3.43, 1.62],
            [0.32, 6.60, 4.86, 0.51, 8.86, 1.92, 4.31, 6.56, 1.39, 0.12, 2.12, 7.34, 1.97, 0.70, 3.85, 8.42, 4.09, 8.77,
             0.35, 1.18, 8.57, 5.33, 8.47, 2.77, 8.74, 5.72, 1.14, 2.12, 3.75, 8.29],
            [7.02, 2.20, 1.85, 4.02, 1.91, 0.75, 5.54, 5.36, 4.67, 5.29, 4.63, 6.50, 8.07, 8.58, 7.12, 2.33, 2.36, 3.94,
             4.04, 8.07, 3.43, 3.00, 0.64, 4.90, 7.72, 6.26, 6.08, 8.39, 5.68, 2.22],
            [1.69, 7.61, 0.65, 5.58, 4.90, 5.78, 0.15, 6.67, 2.39, 7.76, 7.60, 1.82, 1.40, 3.42, 4.88, 2.50, 5.31, 2.72,
             0.05, 8.27, 1.25, 5.04, 1.35, 5.87, 5.52, 6.76, 8.54, 2.25, 1.14, 8.44],
            [4.33, 1.17, 4.47, 1.02, 0.78, 8.16, 3.67, 7.81, 5.54, 7.95, 1.82, 3.44, 7.68, 3.59, 3.27, 4.70, 3.11, 6.42,
             0.28, 4.63, 1.24, 7.10, 4.57, 5.43, 0.61, 8.42, 4.03, 6.44, 8.97, 8.78],
            [0.87, 8.65, 5.05, 4.74, 5.20, 7.07, 5.75, 8.06, 1.11, 2.48, 8.15, 2.21, 6.90, 6.03, 0.44, 2.35, 6.52, 2.15,
             7.96, 5.55, 0.96, 0.65, 3.65, 5.28, 4.30, 3.60, 3.52, 0.45, 6.18, 8.77],
            [3.62, 4.69, 4.41, 6.60, 5.30, 1.53, 8.40, 8.01, 0.24, 7.34, 2.66, 8.57, 3.87, 7.28, 5.33, 6.61, 1.02, 8.74,
             5.55, 2.63, 8.87, 2.27, 7.34, 4.34, 7.78, 6.13, 4.54, 6.88, 2.09, 0.92],
            [8.52, 4.23, 1.99, 3.32, 5.18, 3.21, 2.71, 2.58, 6.69, 6.57, 1.99, 7.24, 1.74, 0.77, 6.95, 2.38, 2.41, 6.42,
             1.74, 4.32, 5.35, 0.06, 3.60, 0.25, 0.13, 8.82, 5.64, 4.96, 2.35, 0.74],
            [4.53, 6.15, 5.18, 4.33, 4.18, 7.08, 7.09, 1.74, 1.63, 2.17, 8.21, 3.54, 3.77, 1.14, 5.52, 7.35, 8.34, 7.30,
             0.21, 8.76, 6.34, 2.43, 8.53, 4.48, 0.69, 1.60, 7.05, 0.31, 5.24, 0.14],
            [2.33, 0.90, 4.07, 8.07, 1.94, 8.04, 6.98, 8.33, 4.88, 0.73, 8.70, 8.23, 8.50, 6.83, 4.76, 0.61, 5.72, 0.57,
             1.20, 8.93, 3.20, 8.44, 6.46, 8.19, 4.27, 3.63, 7.56, 2.44, 7.19, 7.74],
            [8.46, 1.06, 3.86, 0.45, 1.68, 3.81, 5.91, 3.85, 5.14, 0.67, 5.49, 7.32, 0.57, 6.60, 8.12, 7.59, 7.68, 4.15,
             8.50, 4.13, 3.58, 3.19, 8.17, 1.99, 2.41, 2.41, 3.65, 0.71, 6.64, 1.36],
            [2.37, 0.43, 7.08, 1.46, 4.79, 4.06, 7.28, 2.59, 5.60, 3.68, 7.75, 5.04, 0.82, 6.54, 6.00, 2.01, 8.73, 0.29,
             3.65, 3.58, 6.52, 8.07, 2.58, 4.33, 3.21, 8.47, 1.52, 8.26, 3.65, 7.09],
            [7.10, 2.82, 7.28, 2.16, 7.83, 5.57, 5.99, 3.87, 0.29, 4.98, 5.58, 6.61, 7.77, 8.30, 1.22, 5.63, 2.83, 1.82,
             7.55, 1.03, 6.11, 0.96, 1.89, 6.86, 3.64, 0.03, 4.63, 3.09, 7.42, 5.54],
            [0.43, 2.14, 6.46, 4.80, 0.13, 5.57, 8.64, 2.23, 4.36, 1.09, 2.02, 5.54, 4.78, 6.65, 2.30, 5.64, 0.46, 8.97,
             2.71, 6.74, 6.67, 8.76, 3.06, 6.53, 3.29, 3.89, 5.94, 6.29, 2.04, 2.42],
            [2.89, 5.21, 2.27, 3.93, 3.21, 7.69, 8.45, 8.84, 7.99, 3.44, 7.43, 4.43, 0.65, 4.47, 2.21, 7.49, 3.68, 0.41,
             3.27, 5.72, 5.93, 2.22, 3.75, 4.37, 5.29, 4.88, 8.74, 7.89, 3.72, 8.20],
            [7.23, 4.11, 5.27, 8.90, 4.34, 3.67, 3.85, 3.33, 1.54, 5.37, 3.48, 2.45, 3.31, 2.44, 6.52, 7.90, 8.41, 3.57,
             2.38, 2.83, 7.85, 6.10, 3.16, 7.05, 2.83, 2.69, 6.90, 2.39, 2.80, 6.55],
            [3.86, 7.42, 2.06, 6.18, 0.33, 4.98, 6.59, 0.32, 3.22, 6.23, 0.57, 3.43, 8.57, 1.34, 0.53, 6.19, 1.10, 7.13,
             4.69, 8.62, 6.95, 0.40, 6.93, 2.78, 3.73, 1.63, 1.64, 4.12, 6.43, 1.49],
            [0.60, 3.73, 8.79, 8.07, 8.43, 6.67, 0.16, 3.25, 5.85, 4.58, 3.03, 7.35, 0.76, 4.74, 7.81, 4.33, 8.18, 3.95,
             4.27, 0.45, 5.66, 5.14, 0.72, 4.98, 2.84, 4.81, 1.40, 8.39, 3.94, 7.93],
            [5.59, 4.20, 8.45, 1.03, 3.07, 1.74, 4.75, 4.27, 6.31, 4.44, 5.94, 5.15, 2.41, 1.92, 4.37, 1.21, 0.24, 4.29,
             6.03, 1.22, 7.21, 6.30, 7.96, 1.90, 2.85, 2.08, 3.59, 0.55, 4.65, 8.97],
            [1.90, 2.44, 4.16, 4.10, 4.08, 1.63, 6.73, 0.69, 8.63, 7.57, 0.79, 8.03, 7.88, 6.51, 7.40, 0.01, 8.44, 1.65,
             4.20, 2.93, 7.79, 2.09, 0.94, 5.10, 0.15, 4.97, 6.42, 1.44, 7.61, 7.67],
            [3.75, 5.07, 8.30, 0.33, 0.03, 6.52, 1.84, 1.37, 7.98, 1.83, 5.90, 0.56, 0.02, 2.27, 6.88, 4.60, 2.02, 7.86,
             4.93, 0.58, 5.47, 5.50, 8.84, 3.96, 2.35, 8.48, 4.36, 3.31, 8.98, 5.38],
            [5.37, 6.68, 7.48, 6.72, 8.18, 4.56, 6.96, 4.98, 3.35, 3.53, 4.01, 1.28, 6.53, 3.51, 3.58, 5.84, 7.58, 0.04,
             8.75, 1.23, 8.38, 7.57, 8.85, 7.61, 3.76, 5.76, 5.48, 3.82, 6.94, 7.74],
            [2.35, 1.86, 0.51, 6.68, 2.44, 6.87, 5.30, 7.29, 4.33, 8.56, 4.31, 5.86, 0.06, 6.64, 6.29, 4.00, 7.03, 3.70,
             4.99, 8.60, 5.54, 0.01, 0.89, 7.91, 4.75, 0.78, 2.79, 5.32, 7.81, 3.94],
            [6.34, 4.69, 1.41, 3.85, 7.22, 4.54, 6.21, 6.29, 8.51, 2.80, 4.57, 1.97, 8.93, 2.31, 2.21, 7.88, 8.37, 0.06,
             4.53, 4.83, 4.57, 4.18, 5.61, 2.85, 2.80, 5.24, 8.44, 7.09, 7.63, 1.65],
            [0.04, 5.05, 2.22, 5.05, 6.10, 2.58, 1.40, 8.94, 3.60, 6.50, 5.93, 7.12, 5.47, 7.20, 8.15, 4.88, 3.99, 5.45,
             7.85, 6.28, 1.61, 0.75, 1.47, 1.14, 8.50, 8.43, 2.46, 0.67, 8.21, 7.49],
            [0.39, 1.76, 7.73, 4.98, 0.11, 3.99, 0.50, 3.85, 7.27, 2.46, 7.54, 4.76, 4.64, 5.54, 5.29, 0.25, 0.66, 5.94,
             1.08, 4.28, 7.21, 3.73, 2.06, 4.87, 8.26, 4.32, 1.04, 3.00, 7.95, 4.35],
            [2.60, 0.84, 8.57, 5.85, 0.49, 7.85, 1.42, 0.71, 8.88, 6.45, 0.81, 3.93, 4.39, 3.28, 2.08, 6.01, 0.98, 3.56,
             4.44, 7.99, 8.79, 5.80, 7.63, 6.65, 1.67, 7.31, 2.56, 1.99, 4.64, 5.31],
            [2.95, 7.98, 8.71, 1.41, 3.54, 8.99, 3.83, 5.21, 7.82, 8.81, 7.46, 1.89, 7.92, 6.25, 3.44, 3.94, 8.15, 0.17,
             5.25, 6.51, 3.85, 7.07, 1.24, 5.17, 5.55, 2.38, 3.01, 7.62, 7.68, 4.06],
            [7.79, 0.98, 6.26, 5.80, 6.43, 4.35, 3.49, 2.04, 4.23, 4.66, 5.53, 0.98, 4.80, 3.12, 3.38, 7.38, 3.05, 7.07,
             4.40, 4.26, 6.09, 2.53, 8.09, 2.16, 6.05, 3.02, 2.23, 1.33, 1.03, 3.40],
            [2.30, 2.51, 1.64, 1.75, 4.49, 6.31, 4.17, 8.52, 3.47, 0.37, 5.59, 8.55, 6.88, 8.10, 0.21, 0.34, 1.12, 1.89,
             3.80, 7.70, 8.96, 4.66, 1.59, 5.49, 6.31, 0.77, 3.68, 5.90, 3.92, 2.62],
            [4.55, 2.48, 6.24, 2.51, 6.20, 7.13, 6.69, 7.47, 8.85, 7.18, 7.93, 1.53, 7.95, 1.76, 3.43, 6.28, 6.53, 0.10,
             6.65, 0.60, 8.45, 8.43, 3.63, 7.09, 5.79, 0.64, 0.84, 6.40, 2.16, 3.48],
            [3.03, 4.17, 7.31, 7.97, 6.58, 1.61, 5.27, 6.80, 4.91, 5.89, 6.98, 3.44, 8.88, 0.98, 7.77, 3.45, 3.54, 5.25,
             2.08, 3.69, 1.43, 3.51, 4.97, 4.05, 2.47, 4.12, 3.77, 4.59, 5.14, 3.46],
            [8.16, 5.36, 7.83, 4.33, 0.41, 0.79, 7.01, 6.28, 7.64, 0.18, 6.24, 1.50, 2.17, 5.65, 8.93, 3.33, 1.97, 2.26,
             0.74, 2.88, 3.74, 5.13, 7.93, 0.50, 7.63, 7.90, 5.21, 6.27, 5.29, 6.82],
            [3.49, 7.92, 1.59, 2.16, 4.85, 2.33, 7.59, 2.34, 0.54, 2.38, 6.25, 8.25, 5.91, 8.82, 4.62, 1.68, 6.63, 8.20,
             8.60, 4.46, 3.74, 1.47, 0.98, 5.25, 8.41, 0.68, 3.37, 4.42, 1.58, 5.54],
            [6.35, 0.07, 2.39, 3.10, 8.72, 5.82, 7.16, 0.65, 7.29, 0.90, 0.55, 0.23, 1.51, 7.76, 8.18, 4.37, 1.66, 1.18,
             4.47, 7.90, 0.96, 0.17, 6.89, 7.97, 6.82, 5.29, 0.04, 5.62, 5.25, 3.53],
            [0.99, 8.81, 1.97, 1.10, 6.71, 3.67, 3.28, 0.47, 2.57, 3.52, 1.36, 1.94, 8.54, 4.22, 4.06, 0.49, 4.05, 7.70,
             5.26, 3.50, 3.02, 0.96, 6.57, 0.09, 1.70, 0.33, 5.29, 6.39, 5.58, 0.85],
            [3.75, 5.40, 7.84, 7.30, 5.36, 8.95, 1.16, 0.58, 4.40, 5.23, 1.38, 0.47, 5.22, 6.81, 7.70, 8.84, 3.24, 0.26,
             0.57, 2.06, 2.71, 7.54, 7.87, 3.80, 1.24, 6.00, 8.73, 6.15, 4.95, 7.05],
            [3.86, 0.94, 8.71, 8.97, 3.27, 8.27, 3.60, 0.36, 6.03, 4.76, 1.30, 4.14, 5.49, 0.71, 5.29, 7.39, 8.70, 6.39,
             4.15, 7.59, 3.62, 0.15, 2.15, 1.15, 5.04, 8.47, 6.64, 2.57, 0.95, 0.82],
            [8.46, 8.80, 2.70, 7.95, 5.96, 6.11, 8.01, 0.96, 5.79, 8.16, 4.78, 8.30, 6.23, 4.46, 5.13, 6.33, 7.35, 5.53,
             6.99, 3.96, 8.50, 5.23, 3.34, 4.53, 6.21, 1.89, 8.97, 1.17, 6.63, 5.90],
            [6.82, 3.56, 1.68, 8.93, 3.96, 3.33, 1.86, 5.26, 3.69, 8.53, 6.47, 1.78, 8.08, 3.45, 8.33, 1.64, 8.46, 3.93,
             7.27, 2.58, 6.66, 3.41, 0.95, 1.30, 8.15, 8.24, 8.83, 6.31, 1.43, 8.96],
            [2.35, 1.91, 1.85, 4.04, 7.34, 3.68, 3.19, 7.95, 4.56, 1.51, 6.62, 5.48, 5.99, 5.15, 6.18, 6.46, 1.55, 0.71,
             0.45, 6.69, 7.01, 7.45, 7.79, 8.54, 5.54, 7.84, 0.32, 1.57, 2.96, 7.53],
            [1.41, 6.60, 6.40, 7.52, 3.09, 6.26, 7.35, 7.10, 1.32, 5.04, 8.84, 4.61, 1.55, 7.75, 7.37, 3.16, 8.10, 8.00,
             6.40, 5.44, 1.71, 3.00, 7.01, 0.15, 6.69, 0.71, 3.04, 5.11, 0.75, 3.69],
            [4.13, 5.75, 8.47, 8.27, 0.97, 4.19, 3.89, 3.03, 4.96, 3.40, 1.62, 1.16, 1.95, 4.46, 5.13, 4.29, 1.67, 8.40,
             2.54, 4.19, 4.73, 7.29, 5.12, 8.46, 4.49, 1.06, 4.56, 6.54, 8.19, 0.45],
            [6.78, 2.21, 3.39, 7.05, 5.94, 7.80, 5.03, 5.03, 1.03, 8.20, 2.03, 3.53, 5.08, 0.45, 1.69, 5.15, 4.45, 6.88,
             7.23, 4.80, 3.68, 7.90, 1.68, 6.35, 8.46, 1.20, 8.76, 4.96, 1.46, 0.32],
            [4.88, 3.80, 5.02, 5.32, 3.78, 4.70, 6.90, 6.43, 6.91, 0.82, 2.17, 2.75, 5.28, 7.08, 0.53, 6.44, 2.81, 7.07,
             5.07, 0.56, 7.56, 6.58, 2.00, 5.66, 3.54, 2.69, 8.76, 4.03, 1.18, 1.14],
            [2.13, 6.72, 6.77, 7.44, 5.58, 1.57, 4.93, 6.39, 8.05, 2.01, 4.05, 8.04, 7.41, 0.09, 1.09, 3.37, 3.41, 5.63,
             6.93, 0.21, 0.85, 5.98, 7.64, 0.12, 8.54, 7.52, 1.19, 1.17, 8.21, 5.98],
            [7.79, 3.32, 1.03, 0.52, 7.67, 3.36, 1.15, 6.32, 3.18, 5.16, 8.31, 1.66, 8.91, 5.62, 6.80, 4.76, 5.38, 5.10,
             1.89, 4.91, 5.77, 5.00, 4.82, 1.38, 6.70, 1.11, 8.49, 0.76, 4.10, 6.05],
            [5.52, 6.61, 1.25, 8.65, 8.32, 5.12, 5.57, 5.86, 8.95, 6.05, 2.53, 0.91, 0.67, 8.52, 2.12, 7.36, 3.38, 1.24,
             0.60, 2.05, 8.35, 3.91, 2.85, 5.24, 1.60, 6.09, 4.05, 0.36, 8.09, 6.84],
            [5.47, 7.70, 2.51, 4.53, 3.78, 5.77, 4.16, 1.34, 3.96, 3.38, 5.26, 4.09, 5.21, 8.21, 6.59, 7.34, 3.30, 0.19,
             6.79, 8.10, 8.04, 8.88, 3.35, 5.25, 3.25, 6.92, 0.25, 1.44, 2.90, 2.35],
            [6.21, 8.10, 0.06, 8.34, 3.53, 3.25, 2.95, 6.66, 0.14, 0.83, 6.94, 8.15, 4.46, 4.12, 7.28, 4.58, 6.88, 2.47,
             0.52, 7.69, 4.43, 7.59, 5.73, 6.70, 4.63, 1.24, 6.40, 7.41, 6.71, 5.46],
            [7.63, 7.83, 1.26, 5.27, 3.98, 3.95, 1.80, 0.23, 8.25, 6.63, 2.00, 6.53, 0.42, 3.90, 1.98, 7.76, 4.38, 5.85,
             0.43, 6.13, 1.96, 2.30, 2.41, 7.23, 0.93, 3.61, 5.24, 1.61, 1.43, 5.01],
            [2.15, 2.75, 1.37, 0.70, 2.14, 3.96, 1.43, 5.10, 3.67, 3.09, 7.04, 8.25, 5.17, 0.84, 2.71, 0.55, 0.31, 5.47,
             6.09, 8.67, 3.90, 7.64, 8.30, 0.83, 5.32, 3.81, 4.25, 7.04, 8.61, 0.58],
            [6.21, 1.62, 4.11, 8.58, 3.10, 2.44, 3.52, 2.88, 6.60, 5.37, 7.44, 0.54, 4.50, 1.94, 6.19, 5.37, 5.83, 3.45,
             6.22, 6.55, 4.15, 5.78, 3.69, 0.43, 5.74, 3.38, 8.71, 8.25, 6.35, 2.89],
            [8.21, 8.02, 4.91, 2.08, 7.05, 4.90, 4.14, 4.30, 4.90, 8.03, 5.94, 2.11, 2.51, 4.64, 3.96, 4.54, 7.26, 6.82,
             4.27, 1.81, 0.26, 7.90, 6.94, 5.30, 6.46, 7.35, 2.26, 2.96, 8.37, 6.60],
            [1.52, 7.01, 8.61, 6.34, 1.61, 3.23, 6.99, 6.19, 0.94, 4.09, 3.74, 7.04, 0.26, 3.56, 2.72, 0.58, 0.61, 8.50,
             5.06, 1.37, 5.32, 8.03, 4.49, 8.90, 3.67, 1.46, 0.22, 0.87, 0.28, 0.03],
            [7.47, 0.91, 0.81, 3.12, 4.11, 5.93, 4.49, 4.78, 3.94, 5.00, 1.06, 7.81, 8.78, 0.32, 3.20, 8.12, 4.86, 0.21,
             7.92, 5.44, 5.18, 0.36, 1.89, 0.23, 5.92, 3.81, 3.80, 7.94, 5.63, 8.73],
            [3.29, 1.08, 0.81, 0.02, 3.08, 5.95, 2.70, 6.26, 3.07, 4.73, 3.50, 4.27, 1.73, 5.65, 5.71, 2.82, 0.91, 2.78,
             6.85, 1.82, 8.46, 5.89, 3.84, 5.04, 7.88, 1.84, 4.34, 0.68, 8.29, 4.10],
            [5.80, 0.34, 8.08, 2.61, 7.12, 1.25, 2.94, 3.00, 0.87, 1.25, 6.40, 0.95, 2.24, 4.62, 6.41, 1.17, 0.15, 3.45,
             5.56, 3.17, 8.78, 6.43, 1.13, 4.78, 2.75, 8.93, 7.29, 1.68, 8.31, 4.90],
            [2.67, 5.21, 7.91, 2.89, 1.74, 5.29, 6.37, 8.88, 1.69, 4.69, 2.19, 8.53, 2.35, 3.97, 5.07, 1.28, 2.01, 1.04,
             8.21, 3.14, 4.48, 8.22, 6.89, 1.94, 1.63, 3.82, 7.55, 8.23, 8.52, 0.05],
            [8.47, 0.92, 3.33, 4.90, 3.46, 4.02, 6.68, 2.53, 3.36, 7.80, 8.54, 4.16, 4.76, 2.73, 1.36, 7.16, 0.20, 6.95,
             2.74, 5.69, 4.66, 0.79, 3.81, 7.45, 1.21, 2.40, 3.15, 7.93, 3.91, 2.25],
            [7.18, 5.04, 2.82, 7.89, 6.50, 1.16, 4.18, 5.35, 1.57, 5.85, 1.09, 0.19, 0.72, 3.13, 7.39, 6.91, 9.00, 6.43,
             7.66, 7.44, 7.67, 3.19, 4.62, 2.54, 8.46, 4.67, 4.39, 1.30, 4.32, 4.19],
            [7.84, 8.90, 1.21, 3.33, 6.99, 2.03, 2.12, 7.64, 5.12, 3.24, 2.99, 8.97, 5.01, 4.01, 5.55, 8.32, 5.77, 6.48,
             2.85, 1.96, 7.49, 3.40, 6.17, 3.71, 2.27, 7.58, 5.51, 0.70, 5.20, 3.89],
            [0.87, 4.66, 7.19, 3.70, 2.41, 6.81, 4.47, 4.99, 2.51, 5.15, 2.04, 6.29, 6.45, 7.94, 4.33, 6.82, 0.69, 5.72,
             7.19, 4.18, 3.54, 8.79, 1.31, 8.95, 5.31, 6.30, 6.59, 4.80, 0.91, 3.76],
            [8.54, 8.28, 2.52, 3.16, 3.22, 1.36, 8.24, 7.90, 4.49, 5.73, 7.07, 2.55, 8.69, 7.77, 1.54, 3.87, 6.28, 2.89,
             0.45, 1.48, 8.54, 3.63, 7.55, 7.41, 8.70, 5.70, 4.44, 8.65, 2.96, 5.54],
            [7.01, 4.50, 0.87, 7.14, 6.48, 5.62, 4.11, 4.53, 3.83, 8.93, 6.15, 2.21, 5.09, 3.89, 0.20, 8.07, 7.44, 6.92,
             7.37, 8.72, 2.04, 8.66, 1.70, 5.70, 5.30, 0.45, 2.16, 5.87, 0.53, 4.19],
            [8.50, 2.42, 7.73, 3.20, 6.12, 7.17, 6.10, 4.50, 3.96, 7.19, 7.98, 5.38, 8.75, 6.89, 7.48, 4.35, 0.08, 2.42,
             7.61, 8.21, 0.24, 5.57, 0.46, 2.58, 8.33, 7.20, 3.49, 8.42, 2.13, 4.56],
            [1.60, 1.54, 8.33, 5.81, 1.94, 2.38, 5.83, 1.99, 6.83, 3.79, 4.47, 6.31, 4.52, 6.28, 2.52, 1.49, 7.33, 8.86,
             5.47, 1.89, 3.83, 8.07, 2.33, 4.82, 4.88, 7.95, 3.11, 6.87, 5.70, 6.88],
            [4.42, 8.49, 5.00, 3.98, 1.39, 3.13, 1.72, 1.85, 1.74, 4.52, 3.93, 1.36, 4.27, 4.84, 7.92, 5.64, 3.48, 8.59,
             1.88, 1.05, 8.80, 6.12, 2.28, 8.70, 1.09, 0.84, 4.27, 1.92, 4.62, 6.89],
            [6.68, 1.01, 5.94, 1.34, 0.07, 8.55, 4.89, 5.96, 3.69, 4.38, 4.36, 4.00, 3.28, 7.55, 8.61, 1.04, 7.76, 3.89,
             5.96, 2.46, 0.76, 4.99, 5.39, 6.35, 3.24, 2.39, 8.91, 7.96, 6.61, 2.37],
            [2.51, 3.37, 8.61, 7.93, 3.20, 0.33, 8.93, 1.28, 2.89, 1.84, 5.61, 1.73, 1.54, 1.77, 0.39, 6.38, 0.61, 0.20,
             2.65, 6.27, 6.99, 7.82, 1.21, 4.99, 1.62, 2.18, 7.12, 0.09, 7.49, 1.04],
            [2.62, 8.44, 8.84, 7.31, 4.84, 0.62, 7.15, 6.09, 8.61, 3.26, 0.62, 1.43, 2.49, 5.00, 5.32, 2.12, 7.21, 5.21,
             5.63, 4.36, 0.11, 7.65, 1.91, 3.90, 7.01, 0.62, 4.13, 7.12, 1.98, 3.47],
            [4.40, 3.27, 3.54, 1.75, 1.41, 0.38, 7.55, 1.06, 8.31, 6.39, 3.10, 7.98, 6.99, 6.30, 8.24, 0.59, 6.24, 4.62,
             1.37, 4.74, 3.69, 3.31, 8.17, 6.81, 3.60, 4.67, 5.59, 6.55, 4.18, 8.56],
            [6.70, 2.21, 4.43, 3.78, 1.90, 7.75, 1.40, 0.02, 0.21, 8.81, 2.76, 7.30, 5.41, 0.59, 7.63, 0.08, 1.85, 7.97,
             1.33, 4.13, 7.49, 8.06, 7.42, 2.37, 2.68, 7.01, 0.41, 1.46, 3.89, 6.83],
            [0.96, 1.99, 6.96, 4.37, 7.57, 7.63, 2.69, 7.43, 7.77, 6.14, 7.05, 5.54, 0.67, 5.71, 8.37, 4.77, 6.16, 2.10,
             5.51, 8.51, 1.40, 7.04, 1.39, 1.25, 5.07, 5.32, 0.96, 4.28, 4.81, 7.56],
            [8.71, 8.16, 3.88, 7.68, 0.74, 0.85, 1.99, 6.52, 3.35, 6.80, 8.22, 6.37, 6.43, 4.92, 4.45, 1.89, 4.11, 7.23,
             4.41, 0.71, 4.56, 5.71, 1.27, 4.33, 0.88, 1.80, 0.89, 6.78, 2.77, 4.12],
            [7.63, 2.20, 5.82, 4.39, 1.08, 8.00, 2.94, 8.27, 3.45, 2.21, 8.48, 6.31, 5.15, 5.03, 4.02, 1.02, 0.95, 3.37,
             1.77, 2.87, 0.81, 6.09, 5.19, 8.13, 0.63, 0.93, 6.80, 4.28, 0.64, 0.06],
            [3.92, 0.88, 7.84, 7.46, 5.08, 2.12, 8.65, 3.57, 5.87, 1.34, 1.30, 7.57, 2.01, 2.13, 2.11, 7.75, 8.40, 2.52,
             8.99, 2.69, 7.23, 0.34, 3.74, 1.01, 0.08, 2.77, 7.12, 7.02, 5.82, 5.39],
            [3.54, 8.87, 1.50, 3.63, 4.84, 1.36, 4.42, 2.29, 6.56, 7.81, 6.12, 6.67, 3.62, 7.71, 3.38, 5.78, 4.33, 2.85,
             6.84, 5.10, 3.11, 6.01, 5.81, 3.09, 3.79, 2.91, 3.02, 4.84, 7.38, 1.96],
            [6.77, 1.03, 0.69, 3.76, 7.29, 2.66, 5.88, 2.82, 0.60, 6.32, 8.14, 1.64, 2.50, 4.30, 1.33, 3.29, 7.59, 6.67,
             5.23, 0.22, 5.37, 6.33, 0.05, 7.85, 8.50, 5.34, 0.66, 4.02, 5.19, 6.49],
            [3.87, 1.19, 3.30, 8.35, 4.36, 8.05, 2.56, 7.45, 8.22, 2.22, 3.59, 2.43, 7.77, 5.83, 0.01, 2.03, 8.85, 5.84,
             0.39, 5.10, 6.51, 5.91, 4.02, 2.39, 1.46, 8.45, 1.24, 3.38, 4.90, 5.07],
            [3.87, 7.12, 5.85, 7.89, 7.91, 6.81, 5.13, 6.27, 8.30, 4.69, 5.85, 5.63, 8.89, 8.66, 4.87, 5.07, 6.29, 7.32,
             8.97, 1.67, 7.07, 1.20, 0.43, 0.93, 0.74, 8.62, 3.87, 7.54, 6.44, 4.57],
            [7.76, 8.95, 2.13, 2.79, 7.86, 4.18, 6.76, 4.81, 6.01, 3.27, 5.73, 2.22, 3.12, 6.66, 2.09, 3.85, 2.65, 1.67,
             0.69, 1.33, 1.45, 3.04, 6.53, 2.29, 5.32, 5.06, 4.30, 3.72, 2.83, 6.60],
            [3.62, 4.38, 3.32, 0.39, 4.91, 6.12, 8.12, 1.42, 1.80, 8.00, 6.23, 8.63, 6.27, 7.32, 8.90, 1.54, 7.74, 8.48,
             8.53, 5.06, 5.83, 1.58, 4.31, 5.64, 8.94, 5.14, 2.37, 8.78, 0.09, 2.29],
            [0.81, 8.63, 8.36, 3.88, 0.53, 4.53, 2.97, 7.31, 6.34, 4.29, 0.48, 2.50, 0.31, 6.53, 7.30, 2.42, 5.10, 7.99,
             7.45, 4.78, 6.72, 3.96, 0.43, 3.00, 8.38, 0.09, 1.38, 8.33, 2.22, 2.68],
            [1.97, 3.95, 6.60, 8.66, 1.82, 7.17, 0.11, 0.96, 2.04, 2.13, 0.07, 7.94, 7.32, 6.14, 7.61, 5.39, 0.62, 1.09,
             3.96, 0.18, 0.02, 1.21, 1.47, 7.47, 6.40, 5.71, 6.16, 2.69, 6.01, 4.13],
            [7.73, 6.62, 6.23, 4.65, 6.84, 1.05, 2.55, 8.49, 5.92, 3.63, 4.90, 3.12, 2.92, 1.53, 8.91, 5.56, 7.51, 6.60,
             2.49, 2.63, 5.64, 1.38, 0.27, 5.22, 0.72, 8.37, 0.28, 1.71, 7.65, 3.35],
            [6.48, 7.48, 4.47, 6.68, 0.82, 8.90, 3.57, 2.13, 1.26, 1.30, 8.60, 6.78, 1.91, 7.99, 4.88, 2.59, 7.87, 7.03,
             3.36, 3.67, 7.07, 1.87, 3.09, 4.37, 8.22, 1.93, 6.24, 8.94, 1.72, 3.02],
            [8.01, 1.08, 2.56, 2.76, 5.77, 3.15, 2.87, 0.50, 4.09, 3.13, 2.05, 0.60, 4.72, 4.35, 7.86, 5.53, 3.26, 8.16,
             3.09, 6.61, 6.28, 2.08, 5.60, 2.48, 2.66, 2.30, 2.46, 1.66, 6.45, 6.03],
            [4.71, 5.19, 6.69, 6.14, 8.25, 2.72, 4.39, 3.19, 7.00, 0.12, 7.39, 3.62, 6.99, 0.62, 2.61, 3.71, 1.75, 5.50,
             5.75, 8.75, 2.59, 7.02, 3.91, 5.11, 4.65, 2.92, 2.89, 2.51, 0.11, 1.30],
            [2.00, 1.79, 8.00, 5.93, 5.49, 6.85, 6.91, 6.63, 3.68, 8.90, 3.25, 0.37, 1.55, 2.98, 7.53, 7.18, 1.76, 7.05,
             5.07, 7.55, 8.78, 0.41, 3.44, 0.04, 1.02, 7.97, 6.32, 1.63, 4.39, 2.49],
            [7.04, 5.87, 7.21, 3.34, 5.11, 1.26, 4.66, 1.28, 3.20, 2.33, 5.15, 8.48, 7.34, 7.57, 4.25, 7.26, 3.88, 8.54,
             2.25, 4.53, 8.07, 6.09, 6.25, 0.01, 4.76, 4.95, 4.28, 5.18, 0.12, 8.68],
            [4.01, 1.80, 7.05, 2.06, 3.47, 5.00, 4.98, 8.62, 1.55, 6.28, 1.55, 6.05, 4.60, 4.32, 1.30, 4.81, 6.16, 1.00,
             7.94, 2.87, 7.86, 5.21, 2.13, 8.21, 4.35, 5.18, 2.98, 3.71, 1.73, 0.36],
            [8.80, 3.35, 5.87, 7.62, 3.82, 2.86, 8.71, 1.39, 1.75, 1.81, 0.08, 7.51, 6.40, 8.65, 4.37, 3.95, 6.64, 7.41,
             8.67, 1.63, 4.92, 5.17, 8.23, 7.62, 1.78, 8.70, 2.01, 1.21, 1.27, 5.89],
            [8.03, 7.52, 8.20, 1.91, 6.29, 2.23, 7.15, 5.40, 5.87, 8.97, 0.57, 2.64, 4.84, 0.17, 0.23, 5.97, 0.60, 4.99,
             4.89, 3.92, 8.24, 0.36, 3.27, 4.52, 1.21, 1.60, 2.96, 8.97, 4.05, 5.48],
            [2.99, 5.57, 2.54, 1.26, 6.29, 7.34, 3.72, 0.63, 7.59, 4.16, 0.91, 8.66, 8.55, 5.22, 2.57, 5.00, 6.98, 1.84,
             1.72, 8.49, 0.27, 4.28, 5.45, 2.09, 8.29, 1.25, 1.99, 2.49, 4.05, 2.75],
            [1.31, 7.74, 7.80, 1.81, 1.35, 8.75, 4.32, 1.76, 0.42, 7.98, 4.27, 4.92, 1.02, 1.25, 6.43, 3.23, 8.32, 6.52,
             3.74, 7.71, 3.26, 7.97, 4.11, 4.67, 1.29, 5.05, 7.54, 8.35, 0.85, 4.09],
            [8.41, 6.79, 6.07, 1.26, 3.84, 2.27, 6.47, 7.37, 7.76, 6.38, 4.33, 3.86, 0.63, 1.90, 2.68, 5.49, 7.33, 8.89,
             1.22, 0.38, 4.56, 1.45, 2.90, 1.53, 0.68, 2.54, 1.04, 1.99, 4.15, 5.93],
            [8.05, 8.98, 0.96, 0.10, 3.77, 5.71, 4.18, 6.36, 6.27, 2.33, 2.84, 8.85, 0.24, 8.69, 6.65, 1.96, 2.46, 0.42,
             0.39, 5.88, 5.20, 8.28, 5.41, 7.39, 6.82, 2.99, 1.67, 8.23, 8.59, 4.75],
            [3.83, 7.33, 5.07, 2.56, 7.00, 3.43, 6.87, 0.09, 0.61, 7.49, 1.24, 3.92, 7.05, 8.04, 6.42, 4.76, 4.11, 3.90,
             6.83, 2.04, 6.32, 3.61, 1.70, 8.20, 4.69, 6.99, 4.44, 1.71, 1.85, 0.84],
            [2.17, 4.99, 7.36, 6.67, 8.54, 5.17, 2.02, 2.35, 1.13, 6.05, 4.93, 0.92, 8.33, 7.49, 8.22, 5.34, 4.12, 1.26,
             5.93, 1.56, 4.19, 5.33, 7.14, 1.62, 7.82, 0.54, 2.17, 6.46, 4.43, 6.04],
            [2.89, 3.37, 1.33, 0.71, 6.03, 0.46, 8.91, 4.80, 8.38, 6.94, 2.16, 5.35, 0.87, 0.47, 7.32, 2.64, 6.82, 8.48,
             8.66, 5.85, 2.85, 7.11, 4.27, 1.27, 7.59, 8.78, 8.48, 2.00, 7.76, 6.66],
            [1.66, 1.63, 4.69, 0.91, 6.05, 0.74, 8.27, 7.62, 7.27, 4.69, 5.93, 3.02, 6.02, 0.76, 0.79, 0.27, 5.30, 3.08,
             2.38, 5.11, 3.20, 2.75, 6.90, 2.07, 7.52, 3.26, 0.03, 4.76, 5.22, 8.94],
            [1.18, 4.27, 8.53, 1.87, 2.92, 2.69, 3.08, 5.37, 5.35, 3.13, 3.44, 7.72, 4.88, 3.69, 2.02, 5.34, 6.12, 2.10,
             1.61, 0.43, 0.95, 1.98, 0.73, 2.97, 0.78, 2.75, 2.10, 4.14, 4.35, 5.19],
            [4.53, 6.95, 2.02, 3.72, 7.72, 8.78, 7.24, 4.16, 6.17, 5.35, 5.87, 4.16, 1.42, 3.58, 3.39, 7.72, 8.96, 5.10,
             3.92, 4.46, 5.27, 1.49, 7.93, 2.65, 4.51, 8.80, 1.73, 0.38, 8.93, 4.44],
            [4.45, 5.42, 2.06, 5.67, 5.56, 4.66, 8.34, 0.30, 3.22, 0.02, 8.48, 4.27, 1.90, 2.61, 1.16, 8.84, 2.17, 3.77,
             0.87, 5.08, 0.20, 6.31, 2.78, 7.00, 1.05, 1.75, 2.66, 0.21, 6.87, 4.41],
            [4.08, 4.97, 0.68, 0.68, 1.40, 0.68, 2.25, 3.74, 7.09, 4.07, 1.47, 2.55, 7.76, 8.40, 3.45, 0.79, 8.00, 8.32,
             0.41, 8.48, 1.63, 8.38, 4.02, 2.03, 4.04, 2.02, 8.87, 3.26, 8.74, 5.41],
            [4.97, 6.47, 1.94, 3.93, 7.63, 7.91, 8.19, 7.93, 8.17, 8.72, 0.24, 8.41, 3.45, 1.74, 3.97, 5.53, 7.23, 2.22,
             8.31, 0.57, 7.92, 4.17, 2.32, 2.36, 8.27, 4.25, 6.84, 3.12, 1.60, 4.23],
            [1.37, 0.33, 3.07, 7.58, 3.69, 3.37, 1.54, 5.78, 1.56, 8.75, 8.87, 3.97, 1.78, 0.45, 2.94, 5.18, 1.80, 6.23,
             1.40, 3.00, 8.63, 4.75, 0.87, 2.79, 7.78, 7.84, 1.84, 0.79, 6.58, 0.40],
            [7.38, 6.90, 6.50, 2.12, 6.56, 5.53, 2.58, 4.54, 1.71, 6.91, 1.51, 7.46, 8.21, 6.01, 0.29, 3.58, 1.36, 6.00,
             2.08, 8.16, 7.38, 1.40, 4.40, 2.97, 7.56, 2.11, 8.31, 3.07, 0.61, 7.32],
            [2.07, 6.17, 7.44, 5.20, 6.62, 1.84, 2.32, 4.12, 7.76, 2.39, 0.07, 4.47, 5.13, 6.37, 2.43, 6.19, 5.17, 3.23,
             7.40, 1.07, 2.58, 2.80, 4.36, 4.48, 8.60, 6.19, 7.38, 8.33, 8.12, 7.03],
            [7.34, 1.41, 5.67, 4.04, 1.77, 4.45, 4.11, 8.85, 4.04, 6.92, 2.10, 3.69, 2.68, 7.12, 6.48, 1.73, 8.78, 2.39,
             5.94, 2.49, 2.00, 0.21, 3.04, 5.99, 3.67, 7.65, 2.75, 8.00, 3.46, 7.11],
            [8.07, 3.80, 7.26, 3.77, 7.82, 8.61, 6.25, 7.07, 7.57, 0.06, 2.58, 4.12, 1.08, 0.23, 8.65, 8.74, 4.76, 5.71,
             8.89, 3.03, 1.38, 7.29, 8.89, 6.08, 5.22, 5.14, 8.56, 8.10, 2.44, 0.50],
            [5.73, 1.17, 6.69, 8.77, 7.11, 7.90, 3.51, 6.71, 4.86, 7.80, 2.28, 1.46, 7.09, 4.71, 4.52, 0.72, 5.30, 5.15,
             6.21, 3.94, 2.61, 2.37, 5.67, 3.24, 0.08, 7.30, 7.74, 5.94, 1.45, 8.54],
            [3.03, 2.10, 4.25, 1.15, 2.80, 4.68, 1.32, 1.81, 4.66, 3.96, 3.84, 3.37, 0.75, 0.86, 3.66, 2.52, 4.90, 1.48,
             0.27, 3.95, 2.05, 3.85, 2.09, 4.61, 7.32, 4.36, 3.14, 1.31, 7.40, 0.85],
            [8.89, 0.13, 7.77, 3.08, 6.24, 7.06, 4.26, 8.85, 8.89, 0.20, 3.59, 4.40, 5.29, 6.94, 5.05, 0.27, 6.87, 2.76,
             2.00, 3.77, 1.58, 3.51, 1.80, 2.46, 1.40, 1.32, 6.87, 8.81, 8.79, 5.76],
            [1.32, 2.21, 4.36, 2.62, 3.77, 7.57, 6.11, 0.90, 8.60, 5.92, 3.03, 0.15, 5.21, 2.97, 5.10, 4.49, 8.43, 8.93,
             5.43, 4.45, 5.89, 2.16, 1.53, 4.75, 7.11, 5.62, 5.02, 6.21, 4.27, 7.39],
            [2.18, 0.52, 8.96, 5.96, 3.90, 0.71, 3.74, 5.22, 3.45, 5.18, 8.84, 6.78, 1.16, 5.07, 2.13, 1.03, 5.02, 3.32,
             2.35, 3.86, 3.46, 7.60, 7.99, 1.25, 1.93, 1.38, 0.69, 2.46, 7.70, 3.91],
            [0.30, 3.63, 5.65, 5.67, 3.73, 4.86, 3.69, 6.51, 1.98, 4.46, 2.88, 6.39, 5.04, 6.26, 6.42, 5.69, 7.75, 6.62,
             0.42, 2.31, 7.83, 6.92, 2.39, 1.18, 7.18, 5.29, 5.92, 5.93, 6.08, 0.81],
            [3.67, 2.51, 2.90, 2.99, 4.19, 5.24, 7.72, 6.58, 8.48, 7.03, 6.10, 5.46, 3.17, 8.06, 6.32, 2.53, 2.73, 2.37,
             4.52, 5.51, 5.21, 5.54, 8.37, 7.99, 5.92, 2.32, 2.01, 6.21, 3.60, 5.01],
            [4.43, 1.02, 7.75, 8.79, 4.37, 6.39, 7.27, 5.55, 5.41, 4.98, 2.36, 6.30, 5.76, 8.65, 1.00, 5.87, 8.90, 3.54,
             5.37, 8.72, 1.61, 2.62, 4.51, 1.75, 8.93, 8.50, 2.64, 4.45, 5.55, 1.27],
            [0.85, 5.53, 1.19, 6.82, 6.57, 5.86, 3.36, 3.05, 1.39, 8.80, 1.19, 0.72, 0.96, 6.33, 2.31, 8.11, 1.56, 8.02,
             4.97, 8.93, 4.83, 3.78, 3.55, 8.54, 1.51, 8.29, 1.85, 8.31, 2.71, 8.58],
            [1.64, 8.26, 6.27, 2.67, 2.14, 0.98, 0.28, 5.35, 3.41, 1.77, 7.44, 7.71, 1.79, 8.90, 5.42, 5.19, 6.51, 5.98,
             3.89, 3.23, 7.69, 0.98, 3.44, 8.47, 4.83, 7.30, 6.26, 2.43, 1.52, 3.07],
            [2.11, 5.47, 5.36, 1.01, 7.50, 0.01, 0.59, 4.27, 7.66, 2.76, 4.80, 8.52, 6.85, 2.96, 2.30, 1.24, 5.70, 4.29,
             7.45, 6.55, 5.91, 0.79, 5.52, 5.98, 6.18, 8.50, 5.87, 5.54, 6.18, 4.75],
            [7.59, 6.37, 7.47, 1.68, 1.84, 2.38, 1.32, 4.18, 1.02, 3.33, 7.69, 1.59, 6.91, 5.22, 3.80, 1.26, 6.47, 7.31,
             0.38, 8.95, 1.43, 1.28, 4.78, 3.75, 3.57, 0.67, 4.39, 7.61, 6.83, 5.86],
            [2.81, 7.04, 8.27, 4.71, 4.47, 6.68, 4.60, 2.03, 1.23, 0.24, 1.10, 7.52, 2.62, 6.38, 7.65, 0.91, 1.23, 0.72,
             8.46, 3.77, 8.77, 2.53, 6.85, 8.35, 2.58, 7.06, 7.53, 6.89, 4.96, 2.22],
            [8.76, 1.46, 8.19, 0.80, 1.32, 6.81, 0.28, 7.86, 4.19, 6.65, 4.36, 3.48, 8.04, 4.58, 7.78, 8.44, 3.27, 7.12,
             0.46, 2.03, 4.13, 1.86, 1.56, 3.54, 6.26, 0.82, 4.02, 6.79, 1.00, 1.61],
            [8.80, 6.32, 7.97, 6.01, 5.20, 7.55, 2.54, 6.06, 0.23, 3.53, 2.05, 3.30, 2.45, 8.23, 3.00, 6.02, 3.52, 3.01,
             3.43, 5.51, 5.52, 6.01, 2.77, 6.86, 8.22, 1.15, 4.45, 8.78, 4.68, 7.27],
            [1.47, 0.04, 2.96, 3.73, 2.61, 1.05, 2.35, 2.21, 1.07, 3.55, 2.86, 5.13, 6.78, 7.78, 3.24, 6.96, 3.84, 5.27,
             3.27, 1.85, 7.20, 4.64, 4.91, 0.68, 0.85, 6.93, 7.19, 1.66, 2.75, 5.55],
            [1.55, 3.71, 8.47, 5.38, 6.85, 6.60, 0.83, 2.93, 6.81, 1.99, 1.35, 3.10, 0.77, 8.17, 0.71, 0.84, 3.60, 3.76,
             5.24, 0.04, 6.22, 1.10, 1.27, 1.36, 2.85, 1.83, 7.34, 6.49, 4.07, 2.46],
            [6.45, 0.43, 8.40, 5.73, 1.13, 8.99, 7.84, 1.19, 2.03, 2.58, 6.76, 2.31, 8.72, 7.60, 2.30, 5.10, 8.87, 8.93,
             8.53, 2.46, 4.36, 3.65, 5.16, 2.89, 1.58, 1.57, 5.84, 1.41, 3.27, 5.37],
            [8.35, 6.70, 5.15, 4.56, 3.12, 5.93, 7.09, 6.75, 5.38, 8.88, 1.10, 0.73, 1.29, 2.14, 8.68, 0.68, 4.96, 4.77,
             5.64, 3.14, 0.21, 6.06, 1.78, 7.89, 7.10, 3.55, 7.48, 0.21, 5.34, 5.55],
            [3.31, 0.42, 2.56, 8.08, 0.36, 7.89, 7.40, 5.70, 3.24, 6.31, 7.55, 0.78, 1.23, 0.46, 8.08, 5.79, 4.71, 3.81,
             3.36, 2.72, 7.79, 4.15, 3.11, 7.58, 5.98, 6.54, 2.99, 5.09, 3.61, 3.30],
            [4.60, 3.05, 1.69, 1.56, 6.55, 1.15, 5.50, 1.23, 8.68, 5.42, 4.65, 6.13, 4.69, 3.79, 2.55, 0.65, 6.12, 6.34,
             1.33, 0.67, 6.83, 0.03, 1.06, 2.57, 8.13, 8.29, 7.29, 0.88, 2.34, 4.36],
            [1.47, 6.00, 6.54, 5.85, 5.17, 1.22, 3.20, 2.25, 4.30, 6.72, 0.08, 0.80, 1.89, 7.40, 4.10, 8.71, 5.47, 5.31,
             7.77, 4.46, 0.53, 6.37, 5.64, 3.16, 7.66, 2.36, 1.20, 6.11, 4.85, 7.53],
            [3.41, 7.87, 5.67, 0.81, 1.95, 7.78, 6.39, 5.79, 6.17, 4.83, 0.03, 6.68, 5.28, 2.23, 8.93, 6.08, 2.09, 8.98,
             1.11, 7.81, 4.60, 5.71, 8.21, 8.98, 8.09, 2.61, 5.29, 1.09, 8.74, 4.02],
            [7.78, 1.62, 1.97, 7.06, 1.52, 1.67, 4.36, 4.66, 8.26, 8.08, 2.79, 0.91, 1.75, 4.26, 5.19, 8.90, 4.40, 8.61,
             7.16, 8.67, 2.15, 5.48, 5.95, 0.49, 4.70, 8.05, 4.73, 2.34, 0.02, 3.16],
            [3.54, 4.32, 7.65, 0.04, 4.39, 0.51, 5.80, 8.60, 7.11, 3.25, 4.07, 2.23, 7.24, 0.58, 3.35, 7.96, 7.19, 0.91,
             7.47, 2.47, 6.06, 6.12, 2.77, 1.13, 2.85, 5.30, 1.49, 5.41, 5.31, 3.30],
            [2.27, 5.66, 3.92, 3.02, 0.49, 3.04, 3.05, 6.88, 4.33, 8.66, 1.95, 7.89, 4.76, 6.04, 3.04, 5.95, 1.86, 2.28,
             7.03, 5.79, 3.54, 2.43, 8.16, 6.44, 8.20, 6.73, 5.19, 8.01, 6.22, 1.32],
            [0.18, 3.76, 0.06, 2.56, 2.87, 4.27, 0.90, 1.02, 8.78, 1.34, 6.57, 0.24, 1.57, 7.70, 5.15, 7.14, 1.97, 2.72,
             3.88, 8.74, 5.89, 3.72, 6.56, 5.15, 5.92, 2.73, 1.48, 2.54, 8.32, 4.77],
            [1.60, 3.58, 5.19, 0.48, 3.48, 7.14, 5.00, 7.45, 1.16, 1.76, 6.97, 0.17, 2.87, 4.01, 7.82, 1.91, 8.11, 7.08,
             3.08, 8.94, 5.46, 5.80, 5.28, 1.54, 3.03, 8.74, 1.31, 5.10, 8.81, 3.61],
            [7.63, 6.45, 5.31, 0.62, 6.32, 4.64, 4.55, 8.10, 2.18, 3.22, 3.81, 3.56, 2.39, 4.63, 3.76, 6.29, 6.72, 0.44,
             4.65, 6.58, 7.85, 8.71, 3.21, 5.30, 3.54, 0.58, 6.55, 3.69, 1.77, 7.08],
            [6.66, 1.08, 6.75, 8.35, 4.21, 1.22, 1.56, 7.27, 4.75, 6.77, 8.75, 1.01, 4.70, 8.24, 3.71, 3.66, 6.82, 0.06,
             6.76, 3.09, 8.18, 1.32, 8.42, 0.11, 8.97, 1.93, 4.91, 3.71, 4.38, 5.40],
            [2.12, 6.03, 0.67, 2.88, 4.83, 1.69, 0.51, 4.90, 1.00, 0.05, 0.71, 5.81, 8.97, 6.19, 6.62, 6.32, 5.57, 0.30,
             8.69, 3.47, 8.41, 7.92, 5.07, 4.06, 2.63, 8.50, 4.41, 6.16, 4.30, 6.70],
            [5.42, 7.37, 5.78, 3.38, 7.39, 5.98, 8.91, 6.36, 7.68, 2.52, 3.74, 0.37, 7.18, 6.24, 4.85, 6.98, 0.11, 4.31,
             3.36, 3.82, 0.20, 8.00, 5.54, 1.21, 4.20, 0.85, 4.36, 0.80, 3.52, 2.80],
            [5.14, 4.22, 1.82, 6.12, 0.27, 6.52, 4.82, 0.76, 1.15, 7.55, 3.00, 8.47, 2.24, 6.54, 0.80, 3.35, 4.33, 3.11,
             8.36, 6.72, 8.03, 0.96, 0.54, 4.15, 1.57, 1.61, 5.00, 4.64, 6.27, 4.15],
            [2.21, 1.76, 0.66, 3.41, 1.53, 0.67, 4.99, 3.99, 1.48, 3.80, 2.03, 3.47, 4.43, 0.83, 3.92, 7.69, 6.50, 0.87,
             8.15, 1.17, 0.94, 7.95, 1.55, 1.47, 0.00, 2.87, 1.69, 4.78, 5.85, 0.96],
            [0.45, 0.10, 8.40, 1.98, 5.55, 2.54, 6.70, 1.46, 5.69, 6.44, 4.29, 4.35, 4.91, 0.85, 7.60, 2.98, 6.40, 4.05,
             0.18, 5.88, 0.80, 8.83, 7.01, 8.27, 5.62, 8.73, 8.40, 4.49, 4.66, 5.41],
            [6.08, 1.10, 8.76, 0.60, 6.13, 7.35, 1.93, 8.66, 3.24, 2.64, 2.37, 5.50, 0.42, 6.21, 5.33, 3.84, 8.76, 1.50,
             3.78, 5.90, 0.75, 2.41, 2.77, 6.40, 0.56, 1.97, 5.77, 1.17, 2.55, 0.92],
            [3.62, 2.57, 2.70, 6.38, 3.83, 1.77, 8.26, 8.08, 1.15, 6.15, 8.17, 7.54, 0.95, 4.32, 8.97, 4.70, 6.59, 7.54,
             4.15, 0.58, 4.49, 0.81, 3.44, 1.17, 5.65, 3.45, 8.52, 7.28, 7.37, 5.56],
            [6.36, 7.82, 5.94, 1.55, 3.08, 0.38, 2.69, 3.76, 5.62, 3.87, 7.92, 1.54, 8.12, 3.67, 1.58, 8.76, 3.94, 5.22,
             6.81, 4.22, 1.78, 5.73, 7.33, 1.54, 4.96, 2.41, 4.21, 6.27, 8.43, 5.52],
            [6.85, 5.06, 5.22, 4.83, 5.78, 1.03, 7.49, 8.74, 6.24, 5.21, 7.53, 8.30, 6.40, 6.78, 3.59, 6.99, 0.60, 2.29,
             6.66, 0.17, 8.10, 1.46, 1.48, 0.58, 5.10, 4.19, 3.08, 4.15, 1.52, 7.23],
            [2.25, 0.72, 2.46, 3.58, 1.77, 8.20, 4.00, 4.49, 5.09, 1.90, 8.93, 7.80, 7.05, 2.09, 1.21, 3.68, 0.24, 5.82,
             0.35, 5.75, 0.58, 7.71, 7.37, 1.96, 3.24, 0.22, 1.29, 7.35, 7.85, 1.67],
            [0.49, 3.44, 2.26, 0.33, 7.78, 7.75, 7.50, 4.16, 5.04, 0.28, 7.52, 2.21, 5.45, 3.11, 1.81, 3.80, 3.78, 2.37,
             7.13, 6.55, 6.16, 6.43, 3.86, 7.59, 0.13, 7.92, 5.69, 2.26, 0.24, 8.58],
            [5.16, 8.40, 4.43, 3.48, 4.79, 1.23, 5.82, 2.05, 1.87, 8.02, 3.83, 4.43, 0.01, 2.32, 1.97, 8.24, 6.19, 6.82,
             8.52, 4.96, 7.60, 1.93, 6.33, 4.99, 0.77, 4.66, 6.99, 1.00, 1.21, 6.79],
            [6.18, 2.40, 6.49, 3.46, 3.77, 4.78, 7.65, 2.63, 0.45, 1.84, 1.21, 3.80, 7.50, 6.67, 6.71, 1.58, 3.02, 1.00,
             2.12, 6.71, 3.59, 8.25, 1.53, 7.26, 1.23, 1.18, 7.98, 5.58, 1.25, 4.53],
            [2.36, 1.77, 0.62, 0.99, 7.33, 1.50, 6.86, 2.70, 3.89, 0.48, 4.35, 1.97, 8.31, 7.90, 7.61, 5.42, 4.04, 2.88,
             7.28, 4.31, 4.54, 4.59, 2.92, 4.50, 1.66, 1.56, 2.91, 7.68, 6.13, 7.01],
            [1.38, 4.06, 5.71, 0.18, 1.59, 8.57, 6.06, 6.93, 1.06, 1.57, 5.99, 3.80, 6.91, 5.86, 6.03, 7.29, 2.02, 3.75,
             8.13, 1.38, 1.88, 6.90, 0.79, 8.92, 5.81, 3.82, 4.43, 4.91, 1.39, 0.04],
            [7.96, 6.71, 7.49, 7.53, 1.10, 5.53, 0.16, 4.78, 0.41, 4.76, 1.20, 6.60, 0.15, 0.20, 4.22, 3.70, 5.05, 0.92,
             4.67, 4.40, 1.22, 2.25, 6.98, 3.87, 6.17, 5.10, 8.46, 8.00, 7.55, 3.84],
            [2.12, 3.22, 4.18, 1.78, 6.40, 3.26, 8.95, 3.51, 5.21, 7.32, 7.14, 0.92, 4.92, 5.37, 6.82, 1.84, 4.80, 3.74,
             4.70, 2.65, 8.09, 6.91, 4.92, 8.29, 2.45, 5.24, 2.34, 0.31, 1.66, 7.76],
            [6.35, 6.63, 4.89, 2.80, 8.60, 6.91, 5.62, 1.31, 8.11, 6.04, 7.82, 6.55, 0.51, 0.82, 0.96, 7.13, 1.39, 0.99,
             7.74, 2.79, 0.02, 4.36, 1.61, 4.62, 5.88, 3.75, 8.18, 5.28, 5.19, 5.17],
            [2.97, 7.50, 2.34, 3.82, 4.00, 4.05, 6.14, 6.24, 3.03, 6.50, 8.78, 6.41, 5.55, 1.70, 6.02, 7.90, 3.39, 0.84,
             7.61, 2.83, 5.39, 5.95, 8.44, 6.40, 8.13, 8.06, 0.99, 4.38, 7.19, 2.21],
            [8.53, 1.02, 1.01, 6.92, 6.61, 0.88, 0.13, 5.25, 5.80, 5.35, 2.73, 1.11, 0.85, 1.31, 4.50, 7.20, 6.65, 1.27,
             4.62, 5.11, 1.25, 6.03, 1.93, 8.89, 8.96, 5.50, 8.03, 6.61, 6.34, 6.19],
            [0.87, 2.39, 4.31, 7.23, 2.05, 5.89, 7.92, 6.94, 0.33, 5.98, 3.93, 8.65, 4.86, 2.72, 4.67, 7.85, 2.04, 8.71,
             0.16, 2.25, 3.39, 6.36, 1.82, 7.93, 4.69, 2.33, 6.49, 6.47, 1.20, 7.40],
            [0.56, 4.78, 1.66, 6.44, 1.64, 5.46, 4.24, 3.32, 6.10, 4.53, 5.93, 1.21, 2.92, 2.35, 6.62, 6.53, 1.52, 4.06,
             2.70, 1.72, 8.18, 5.61, 3.29, 7.85, 1.83, 8.20, 8.21, 3.68, 2.96, 3.28],
            [6.22, 1.31, 7.87, 5.82, 2.78, 8.47, 6.81, 8.07, 1.89, 2.28, 5.20, 6.39, 2.99, 0.72, 0.59, 0.89, 2.42, 1.49,
             4.84, 2.54, 3.31, 5.40, 6.22, 6.27, 8.31, 1.09, 5.95, 2.37, 6.60, 0.34],
            [5.02, 5.57, 5.02, 7.56, 0.68, 5.53, 1.21, 3.02, 5.41, 3.17, 6.39, 3.78, 0.25, 4.33, 3.40, 7.24, 2.67, 0.31,
             8.51, 0.09, 8.29, 6.61, 3.84, 0.11, 0.08, 0.47, 4.11, 1.96, 2.24, 0.89],
            [5.34, 2.86, 0.45, 6.96, 3.05, 6.65, 1.49, 4.25, 4.48, 5.92, 4.41, 1.42, 2.08, 7.90, 7.42, 4.91, 6.59, 8.42,
             8.82, 1.88, 8.76, 6.87, 8.41, 2.18, 3.15, 0.73, 7.49, 8.89, 1.18, 2.15]])

        print(X)
        Y_isomap = Isomap(n_neighbors=6).fit_transform(deepcopy(X.T)).T
        Y_pca = PCA(n_components=2).fit_transform(deepcopy(X.T)).T

        pairwise_x = euclidean_distance(X.T)
        PP, new_pairwise_x = compute_paths(pairwise_x, 6)
        leaf_indices, leaf_dists, centre_index = compute_x_leaves(X, 6,new_pairwise_x,PP)

        G_isomap = global_judge_x_precomputed(leaf_indices, leaf_dists, centre_index, Y_isomap)
        G_pca = global_judge_x_precomputed(leaf_indices, leaf_dists, centre_index, Y_pca)
        print('Isomap:' + str(G_isomap))
        print('PCA:' + str(G_pca))
        self.assertAlmostEqual(G_isomap, 0.80536, places=5)
        self.assertAlmostEqual(G_pca, 0.40893, places=5)
