import six
from six.moves.urllib.parse import urlencode
import json
import requests
import re
from multiprocessing.pool import ThreadPool
import copy
import gzip

from webshotapi.ApiException import ApiException
from webshotapi.RESTResponse import RESTResponse

class RequestClient:

    def __init__(self, api_key, api_version='v1', api_host ='https://api.webshotapi.com', pools_size=4):
        '''
        Create Request client object

        :param api_key:
            String api key. Get api key from https://webshotapi.com/dashboard/api/
        :param api_version:
            server api version
        :param api_host:
            server api host
        :pools_size:
            set max process for multi requests processing
        '''
        if not api_key:
            raise Exception("No set API KEY")

        self.api_host = api_host
        self.api_version = api_version
        self.api_key = api_key

        self.max_pool = pools_size

        self.is_multi = False
        self.pool = None
        self.multi_requests = []
        self.version = '1.0.1'

    def __del__(self):
        if self.pool is not None:
            self.pool.close()

    def request(self, path, method, params={}, headers={}):
        '''
        Send request to REST server

        :param path:
            REST path on server
        :param method:
            HTTP METHOD [GET,POST,PUT,DELETE]
        :param params:
            dictionary of parameters to send
        :param headers:
            dictionary of headers to send
        :return:
            RESTResponse object
        '''
        assert(isinstance(headers,dict))
        assert(isinstance(params,dict))

        headers['Authorization'] = 'Bearer '+self.api_key
        headers['Content-Type'] = 'application/json'
        headers['User-Agent'] = 'WebshotApi Client Python ' + self.version
        headers['Accept-Encoding'] = 'gzip'

        #production url
        url = self.api_host + '/' + self.api_version + '/' + path

        return self.run_request(method=method, url=url, headers=headers, body=params)

    def run_request(self, method, url, query_params=None, headers=None,
                body=None, post_params=None, _preload_content=True,
                _request_timeout=30):

        #get method
        method = method.upper()
        assert method in ['GET', 'HEAD', 'DELETE', 'POST', 'PUT',
                          'PATCH', 'OPTIONS']

        if post_params and body:
            raise ValueError(
                "body parameter cannot be used with post_params parameter."
            )

        post_params = post_params or {}
        headers = headers or {}

        #timeout
        timeout = None
        if _request_timeout:
            if isinstance(_request_timeout, (int, ) if six.PY3 else (int, long)):  # noqa: E501,F821
                timeout = _request_timeout
            elif (isinstance(_request_timeout, tuple) and
                  len(_request_timeout) == 2):
                timeout = (_request_timeout[0],_request_timeout[1])

        #set default content type
        if 'Content-Type' not in headers:
            headers['Content-Type'] = 'application/json'

        #run request
        try:
            # For `POST`, `PUT`, `PATCH`, `OPTIONS`, `DELETE`
            if method in ['POST', 'PUT', 'PATCH', 'OPTIONS', 'DELETE']:
                if query_params:
                    url += '?' + urlencode(query_params)
                if re.search('json', headers['Content-Type'], re.IGNORECASE):
                    request_body = '{}'
                    if body is not None:
                        request_body = json.dumps(body)
                    r = requests.request(
                        method, url,
                        data=request_body,
                        timeout=timeout,
                        headers=headers)
                elif headers['Content-Type'] == 'application/x-www-form-urlencoded':  # noqa: E501
                    r = requests.request(
                        method, url,
                        params=post_params,
                        encode_multipart=False,
                        timeout=timeout,
                        headers=headers)
                elif headers['Content-Type'] == 'multipart/form-data':
                    # must del headers['Content-Type'], or the correct
                    # Content-Type which generated by urllib3 will be
                    # overwritten.
                    del headers['Content-Type']
                    r = requests.request(
                        method, url,
                        params=post_params,
                        encode_multipart=True,
                        timeout=timeout,
                        headers=headers)
                # Pass a `string` parameter directly in the body to support
                # other content types than Json when `body` argument is
                # provided in serialized form
                elif isinstance(body, str):
                    request_body = body
                    r = requests.request(
                        method, url,
                        data=request_body,
                        timeout=timeout,
                        headers=headers)
                else:
                    # Cannot generate the request from given parameters
                    msg = """Cannot prepare a request message for provided
                             arguments. Please check that your arguments match
                             declared content type."""
                    raise ApiException(status=0, reason=msg)
            # For `GET`, `HEAD`
            else:
                r = requests.request(method, url,
                                              params=query_params,

                                              timeout=timeout,
                                              headers=headers)
        except Exception as e:
            msg = "{0}\n{1}".format(type(e).__name__, str(e))
            raise ApiException(status=0, reason=msg)

        if _preload_content:
            r = RESTResponse(r)

        if r.status == 404:
            raise ApiException('Error 404. Not found')

        if r.status == 401:
            raise ApiException("Error 401. Cant authorize. Check your api token")

        if r.status == 500:
            raise ApiException("Internal error")

        if r.status==400:
            raise ApiException("Error 400. Bad request:",r.data)

        if not 200 <= r.status <= 299:
            raise ApiException(http_resp=r)

        return r

    def info(self):
        ''''
        Get info about account. Return info about free requests etc.
        '''
        return self.request('info', 'GET', {})

    def multi(self):
        '''
        Create concurrency requests mode
        
        :return:
            RequestClient Object
        '''
        self.is_multi = True
        return self

    def add_multi(self, data):
        '''
        Add to concurrency request mode queue new request.

        :param data:
            tuple of request example:
            ('screenshot/json','POST', {url:'https://example.com',width:1920})
            (REST_SERVER_PATH, HTTP_METHOD, PARAMS)
        :return:
            RequestClient Object
        '''
        #save
        self.multi_requests.append([
            {
                'api_key': self.api_key,
                'api_host': self.api_host,
                'api_version': self.api_version,
                'max_pool': self.max_pool
            }, data, len(self.multi_requests)])

        return self

    def exec(
        self,
         on_request_completed=None,
         on_requests_error=None,
         max_requests:int=None
    ):

        '''
        Run all requests added to concurrency mode queue

        :param on_request_completed:
            function fired async when request return data
        :param on_requests_error:
            function fired async when request return error
        :param max_requests:
            How much concurrency requests can send to server in same time.
        :return:
        '''

        info = None
        #if not set max request send request to server to get info about your account
        if not max_requests:
            info = self.info()

        max_requests = 0
        if info is not None and isinstance(info.data,dict) and info.data['max_concurrent_requests'] is not None:
            max_requests = info.data['max_concurrent_requests']

        self.pool = ThreadPool(max_requests)
        results = []

        for x in self.pool.imap_unordered(worker, self.multi_requests):
            # run on completed data
            real_index = x[2][2]
            if x[0] == 1:
                on_request_completed(x[1],x[2][1],real_index)
                results.append((x[1],x[2][1],real_index))
            else:
                on_requests_error(x[1],x[2][1],real_index)
                results.append((x[1], x[2][1],real_index))

        return results


def worker(d):
    ''''
    Worker function for concurrency requests
    '''
    try:
        a = RequestClient(d[0]['api_key'], pools_size=d[0]['max_pool'], api_version=d[0]['api_version'], api_host=d[0]['api_host'])
        func = getattr(a, 'request')
        data = func(*d[1])

        return 1, data, d
    except Exception as e:
        return 0, e, d