import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IVpc, SubnetSelection } from './vpc';
/**
 * @stability stable
 */
export interface IVpnConnection extends IResource {
    /**
     * The id of the VPN connection.
     *
     * @stability stable
     */
    readonly vpnId: string;
    /**
     * The id of the customer gateway.
     *
     * @stability stable
     */
    readonly customerGatewayId: string;
    /**
     * The ip address of the customer gateway.
     *
     * @stability stable
     */
    readonly customerGatewayIp: string;
    /**
     * The ASN of the customer gateway.
     *
     * @stability stable
     */
    readonly customerGatewayAsn: number;
}
/**
 * The virtual private gateway interface.
 *
 * @stability stable
 */
export interface IVpnGateway extends IResource {
    /**
     * The virtual private gateway Id.
     *
     * @stability stable
     */
    readonly gatewayId: string;
}
/**
 * @stability stable
 */
export interface VpnTunnelOption {
    /**
     * The pre-shared key (PSK) to establish initial authentication between the virtual private gateway and customer gateway.
     *
     * Allowed characters are alphanumeric characters
     * and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).
     *
     * @default an Amazon generated pre-shared key
     * @stability stable
     */
    readonly preSharedKey?: string;
    /**
     * The range of inside IP addresses for the tunnel.
     *
     * Any specified CIDR blocks must be
     * unique across all VPN connections that use the same virtual private gateway.
     * A size /30 CIDR block from the 169.254.0.0/16 range.
     *
     * @default an Amazon generated inside IP CIDR
     * @stability stable
     */
    readonly tunnelInsideCidr?: string;
}
/**
 * @stability stable
 */
export interface VpnConnectionOptions {
    /**
     * The ip address of the customer gateway.
     *
     * @stability stable
     */
    readonly ip: string;
    /**
     * The ASN of the customer gateway.
     *
     * @default 65000
     * @stability stable
     */
    readonly asn?: number;
    /**
     * The static routes to be routed from the VPN gateway to the customer gateway.
     *
     * @default Dynamic routing (BGP)
     * @stability stable
     */
    readonly staticRoutes?: string[];
    /**
     * The tunnel options for the VPN connection.
     *
     * At most two elements (one per tunnel).
     * Duplicates not allowed.
     *
     * @default Amazon generated tunnel options
     * @stability stable
     */
    readonly tunnelOptions?: VpnTunnelOption[];
}
/**
 * The VpnGateway Properties.
 *
 * @stability stable
 */
export interface VpnGatewayProps {
    /**
     * Default type ipsec.1.
     *
     * @stability stable
     */
    readonly type: string;
    /**
     * Explicitly specify an Asn or let aws pick an Asn for you.
     *
     * @default 65000
     * @stability stable
     */
    readonly amazonSideAsn?: number;
}
/**
 * Options for the Vpc.enableVpnGateway() method.
 *
 * @stability stable
 */
export interface EnableVpnGatewayOptions extends VpnGatewayProps {
    /**
     * Provide an array of subnets where the route propagation should be added.
     *
     * @default noPropagation
     * @stability stable
     */
    readonly vpnRoutePropagation?: SubnetSelection[];
}
/**
 * @stability stable
 */
export interface VpnConnectionProps extends VpnConnectionOptions {
    /**
     * The VPC to connect to.
     *
     * @stability stable
     */
    readonly vpc: IVpc;
}
/**
 * The VPN connection type.
 *
 * @stability stable
 */
export declare enum VpnConnectionType {
    /**
     * The IPsec 1 VPN connection type.
     *
     * @stability stable
     */
    IPSEC_1 = "ipsec.1",
    /**
     * Dummy member TODO: remove once https://github.com/aws/jsii/issues/231 is fixed.
     *
     * @stability stable
     */
    DUMMY = "dummy"
}
/**
 * The VPN Gateway that shall be added to the VPC.
 *
 * @stability stable
 * @resource AWS::EC2::VPNGateway
 */
export declare class VpnGateway extends Resource implements IVpnGateway {
    /**
     * The virtual private gateway Id.
     *
     * @stability stable
     */
    readonly gatewayId: string;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: VpnGatewayProps);
}
/**
 * Define a VPN Connection.
 *
 * @stability stable
 * @resource AWS::EC2::VPNConnection
 */
export declare class VpnConnection extends Resource implements IVpnConnection {
    /**
     * Return the given named metric for all VPN connections in the account/region.
     *
     * @stability stable
     */
    static metricAll(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metricAllTunnelState(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllTunnelDataIn(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllTunnelDataOut(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The id of the VPN connection.
     *
     * @stability stable
     */
    readonly vpnId: string;
    /**
     * The id of the customer gateway.
     *
     * @stability stable
     */
    readonly customerGatewayId: string;
    /**
     * The ip address of the customer gateway.
     *
     * @stability stable
     */
    readonly customerGatewayIp: string;
    /**
     * The ASN of the customer gateway.
     *
     * @stability stable
     */
    readonly customerGatewayAsn: number;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: VpnConnectionProps);
}
export declare const RESERVED_TUNNEL_INSIDE_CIDR: string[];
