"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.allRouteTableIds = exports.range = exports.ImportSubnetGroup = exports.subnetId = exports.subnetGroupNameFromConstructId = exports.defaultSubnetName = exports.slugify = void 0;
const vpc_1 = require("./vpc");
/**
 * Turn an arbitrary string into one that can be used as a CloudFormation identifier by stripping special characters
 *
 * (At the moment, no efforts are taken to prevent collisions, but we can add that later when it becomes necessary).
 */
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
exports.slugify = slugify;
/**
 * The default names for every subnet type
 */
function defaultSubnetName(type) {
    switch (type) {
        case vpc_1.SubnetType.PUBLIC: return 'Public';
        case vpc_1.SubnetType.PRIVATE: return 'Private';
        case vpc_1.SubnetType.ISOLATED: return 'Isolated';
    }
}
exports.defaultSubnetName = defaultSubnetName;
/**
 * Return a subnet name from its construct ID
 *
 * All subnet names look like NAME <> "Subnet" <> INDEX
 */
function subnetGroupNameFromConstructId(subnet) {
    return subnet.node.id.replace(/Subnet\d+$/, '');
}
exports.subnetGroupNameFromConstructId = subnetGroupNameFromConstructId;
/**
 * Make the subnet construct ID from a name and number
 */
function subnetId(name, i) {
    return `${name}Subnet${i + 1}`;
}
exports.subnetId = subnetId;
class ImportSubnetGroup {
    constructor(subnetIds, names, routeTableIds, type, availabilityZones, idField, nameField, routeTableIdField) {
        this.availabilityZones = availabilityZones;
        this.subnetIds = subnetIds || [];
        this.routeTableIds = routeTableIds || [];
        this.groups = this.subnetIds.length / this.availabilityZones.length;
        if (Math.floor(this.groups) !== this.groups) {
            // eslint-disable-next-line max-len
            throw new Error(`Number of ${idField} (${this.subnetIds.length}) must be a multiple of availability zones (${this.availabilityZones.length}).`);
        }
        if (this.routeTableIds.length !== this.subnetIds.length && routeTableIds != null) {
            // We don't err if no routeTableIds were provided to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            /* eslint-disable max-len */
            throw new Error(`Number of ${routeTableIdField} (${this.routeTableIds.length}) must be equal to the amount of ${idField} (${this.subnetIds.length}).`);
        }
        this.names = this.normalizeNames(names, defaultSubnetName(type), nameField);
    }
    import(scope) {
        return range(this.subnetIds.length).map(i => {
            const k = Math.floor(i / this.availabilityZones.length);
            return vpc_1.Subnet.fromSubnetAttributes(scope, subnetId(this.names[k], i), {
                availabilityZone: this.pickAZ(i),
                subnetId: this.subnetIds[i],
                routeTableId: this.routeTableIds[i],
            });
        });
    }
    /**
     * Return a list with a name for every subnet
     */
    normalizeNames(names, defaultName, fieldName) {
        // If not given, return default
        if (names === undefined || names.length === 0) {
            return [defaultName];
        }
        // If given, must match given subnets
        if (names.length !== this.groups) {
            throw new Error(`${fieldName} must have an entry for every corresponding subnet group, got: ${JSON.stringify(names)}`);
        }
        return names;
    }
    /**
     * Return the i'th AZ
     */
    pickAZ(i) {
        return this.availabilityZones[i % this.availabilityZones.length];
    }
}
exports.ImportSubnetGroup = ImportSubnetGroup;
/**
 * Generate the list of numbers of [0..n)
 */
function range(n) {
    const ret = [];
    for (let i = 0; i < n; i++) {
        ret.push(i);
    }
    return ret;
}
exports.range = range;
/**
 * Return the union of table IDs from all selected subnets
 */
function allRouteTableIds(subnets) {
    const ret = new Set();
    for (const subnet of subnets) {
        if (subnet.routeTable && subnet.routeTable.routeTableId) {
            ret.add(subnet.routeTable.routeTableId);
        }
    }
    return Array.from(ret);
}
exports.allRouteTableIds = allRouteTableIds;
function flatten(xs) {
    return Array.prototype.concat.apply([], xs);
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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