"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const sinon = require("sinon");
const backoff_generator_1 = require("../../lib/backoff-generator");
const dynamodb_1 = require("../../lib/dynamodb");
const x509_certs_1 = require("../../lib/x509-certs");
const acm_handlers_1 = require("../acm-handlers");
describe('AcmCertificateImporter', () => {
    const physicalId = 'physicalId';
    const certArn = 'certArn';
    const oldEnv = process.env;
    let consoleWarnSpy;
    beforeAll(() => {
        jest.spyOn(global.console, 'log').mockImplementation(() => { });
        jest.spyOn(global.console, 'error').mockImplementation(() => { });
        consoleWarnSpy = jest.spyOn(global.console, 'warn').mockImplementation(() => { });
    });
    afterAll(() => {
        jest.restoreAllMocks();
    });
    beforeEach(() => {
        process.env.DATABASE = 'database';
        AWSMock.setSDKInstance(AWS);
    });
    afterEach(() => {
        process.env = oldEnv;
        sinon.restore();
        AWSMock.restore();
    });
    describe('doCreate', () => {
        const doCreateProps = {
            Tags: [],
            X509CertificatePem: {
                Cert: 'cert',
                CertChain: 'certChain',
                Key: 'key',
                Passphrase: 'passphrase',
            },
        };
        beforeEach(() => {
            sinon.stub(x509_certs_1.Certificate, 'decryptKey').returns(Promise.resolve('key'));
            // Mock out the API call in getSecretString
            AWSMock.mock('SecretsManager', 'getSecretValue', sinon.fake.resolves({ SecretString: 'secret' }));
        });
        test('throws when a secret does not have SecretString', async () => {
            // GIVEN
            const getSecretValueFake = sinon.fake.resolves({});
            AWSMock.remock('SecretsManager', 'getSecretValue', getSecretValueFake);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: new MockCompositeStringIndexTable(),
            });
            // WHEN
            await expect(importer.doCreate(physicalId, doCreateProps))
                // THEN
                .rejects.toThrow(/Secret .* did not contain a SecretString as expected/);
            expect(getSecretValueFake.calledOnce).toBe(true);
        });
        test('retries importing certificate', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const getItemStub = sinon.stub(resourceTable, 'getItem').resolves(undefined);
            const putItemStub = sinon.stub(resourceTable, 'putItem').resolves(true);
            const importCertificateStub = sinon.stub()
                .onFirstCall().rejects('Rate exceeded')
                .onSecondCall().rejects('Rate exceeded')
                .onThirdCall().resolves({ CertificateArn: certArn });
            AWSMock.mock('ACM', 'importCertificate', importCertificateStub);
            const backoffStub = sinon.stub(backoff_generator_1.BackoffGenerator.prototype, 'backoff').resolves(true);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doCreate(physicalId, doCreateProps))
                // THEN
                .resolves.toEqual({ CertificateArn: certArn });
            expect(getItemStub.calledOnce).toBe(true);
            expect(putItemStub.calledOnce).toBe(true);
            expect(importCertificateStub.calledThrice).toBe(true);
            expect(backoffStub.callCount).toEqual(2);
        });
        test('throws after max import retries', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const getItemStub = sinon.stub(resourceTable, 'getItem').resolves(undefined);
            const attempts = 10;
            const importCertificateStub = sinon.stub();
            const backoffStub = sinon.stub(backoff_generator_1.BackoffGenerator.prototype, 'backoff');
            for (let i = 0; i < attempts; i++) {
                importCertificateStub.onCall(i).rejects('Rate exceeded');
                backoffStub.onCall(i).resolves(i < attempts - 1);
            }
            AWSMock.mock('ACM', 'importCertificate', importCertificateStub);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doCreate(physicalId, doCreateProps))
                // THEN
                .rejects.toThrow(/Failed to import certificate .* after [0-9]+ attempts\./);
            expect(getItemStub.calledOnce).toBe(true);
            expect(importCertificateStub.callCount).toBe(attempts);
            expect(backoffStub.callCount).toEqual(attempts);
        });
        describe('existing', () => {
            test('throws if item ARN is missing', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves({});
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .rejects.toEqual(new Error("Database Item missing 'ARN' attribute"));
                expect(getItemStub.calledOnce).toBe(true);
            });
            test('throws if certificate not found in ACM', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves({ ARN: certArn });
                const getCertificateFake = sinon.fake.rejects({});
                AWSMock.mock('ACM', 'getCertificate', getCertificateFake);
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .rejects.toThrow(new RegExp(`Database entry ${certArn} could not be found in ACM:`));
                expect(getItemStub.calledOnce).toBe(true);
                expect(getCertificateFake.calledOnce).toBe(true);
            });
            test('imports certificate', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves({ ARN: certArn });
                const getCertificateFake = sinon.fake.resolves({ Certificate: 'cert' });
                AWSMock.mock('ACM', 'getCertificate', getCertificateFake);
                const importCertificateFake = sinon.fake.resolves({});
                AWSMock.mock('ACM', 'importCertificate', importCertificateFake);
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .resolves.toEqual({ CertificateArn: certArn });
                expect(getItemStub.calledOnce).toBe(true);
                expect(getCertificateFake.calledOnce).toBe(true);
                // Verify that we import the existing certificate to support replacing/updating of it (e.g. to rotate certs)
                expect(importCertificateFake.calledOnce).toBe(true);
            });
        });
        describe('new', () => {
            test('throws if CertificateArn not populated', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves(undefined);
                const importCertificateFake = sinon.fake.resolves({});
                AWSMock.mock('ACM', 'importCertificate', importCertificateFake);
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .rejects.toThrow(/CertificateArn was not properly populated after attempt to import .*$/);
                expect(getItemStub.calledOnce).toBe(true);
                expect(importCertificateFake.calledOnce).toBe(true);
            });
            test('imports certificate', async () => {
                // GIVEN
                const resourceTable = new MockCompositeStringIndexTable();
                const getItemStub = sinon.stub(resourceTable, 'getItem').resolves(undefined);
                const putItemStub = sinon.stub(resourceTable, 'putItem').resolves(true);
                const importCertificateFake = sinon.fake.resolves({ CertificateArn: certArn });
                AWSMock.mock('ACM', 'importCertificate', importCertificateFake);
                const importer = new TestAcmCertificateImporter({
                    acm: new AWS.ACM(),
                    dynamoDb: new AWS.DynamoDB(),
                    secretsManager: new AWS.SecretsManager(),
                    resourceTableOverride: resourceTable,
                });
                // WHEN
                await expect(importer.doCreate(physicalId, doCreateProps))
                    // THEN
                    .resolves.toEqual({ CertificateArn: certArn });
                expect(getItemStub.calledOnce).toBe(true);
                expect(putItemStub.calledOnce).toBe(true);
                expect(importCertificateFake.calledOnce).toBe(true);
            });
        });
    });
    describe('doDelete', () => {
        test('throws if describeCertificate is in use after max attempts', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const queryStub = sinon.stub(resourceTable, 'query').resolves({
                key: { ARN: certArn },
            });
            const describeCertificateFake = sinon.fake.resolves({ Certificate: { InUseBy: ['something'] } });
            AWSMock.mock('ACM', 'describeCertificate', describeCertificateFake);
            // This is hardcoded in the code being tested
            const maxAttempts = 10;
            const backoffStub = sinon.stub(backoff_generator_1.BackoffGenerator.prototype, 'backoff').resolves();
            const shouldContinueStub = sinon.stub(backoff_generator_1.BackoffGenerator.prototype, 'shouldContinue')
                .returns(true)
                .onCall(maxAttempts - 1).returns(false);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doDelete(physicalId))
                // THEN
                .rejects.toEqual(new Error(`Response from describeCertificate did not contain an empty InUseBy list after ${maxAttempts} attempts.`));
            expect(queryStub.calledOnce).toBe(true);
            expect(describeCertificateFake.callCount).toEqual(maxAttempts);
            expect(backoffStub.callCount).toEqual(maxAttempts);
            expect(shouldContinueStub.callCount).toEqual(maxAttempts);
        });
        test('throws when deleting certificate from ACM fails', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const queryStub = sinon.stub(resourceTable, 'query').resolves({
                key: { ARN: certArn },
            });
            const describeCertificateFake = sinon.fake.resolves({ Certificate: { InUseBy: [] } });
            AWSMock.mock('ACM', 'describeCertificate', describeCertificateFake);
            const error = new Error('error');
            const deleteCertificateFake = sinon.fake.rejects(error);
            AWSMock.mock('ACM', 'deleteCertificate', deleteCertificateFake);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doDelete(physicalId))
                // THEN
                .rejects.toEqual(error);
            expect(queryStub.calledOnce).toBe(true);
            expect(describeCertificateFake.calledOnce).toBe(true);
            expect(deleteCertificateFake.calledOnce).toBe(true);
        });
        test('warns when deleting certificate from ACM fails with AccessDeniedException', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const queryStub = sinon.stub(resourceTable, 'query').resolves({
                key: { ARN: certArn },
            });
            const describeCertificateFake = sinon.fake.resolves({ Certificate: { InUseBy: [] } });
            AWSMock.mock('ACM', 'describeCertificate', describeCertificateFake);
            const error = new Error('AccessDeniedException');
            const deleteCertificateFake = sinon.fake.rejects(error);
            AWSMock.mock('ACM', 'deleteCertificate', deleteCertificateFake);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doDelete(physicalId))
                // THEN
                .rejects.toEqual(error);
            expect(queryStub.calledOnce).toBe(true);
            expect(describeCertificateFake.calledOnce).toBe(true);
            expect(deleteCertificateFake.calledOnce).toBe(true);
            expect(consoleWarnSpy.mock.calls.length).toBeGreaterThanOrEqual(1);
            expect(consoleWarnSpy.mock.calls.map(args => args[0]).join('\n')).toMatch(new RegExp(`Could not delete Certificate ${certArn}. Please ensure it has been deleted.`));
        });
        test('deletes the certificate', async () => {
            // GIVEN
            const resourceTable = new MockCompositeStringIndexTable();
            const queryStub = sinon.stub(resourceTable, 'query').resolves({ key: { ARN: certArn } });
            const deleteItemStub = sinon.stub(resourceTable, 'deleteItem').resolves(true);
            const describeCertificateFake = sinon.fake.resolves({ Certificate: { InUseBy: [] } });
            AWSMock.mock('ACM', 'describeCertificate', describeCertificateFake);
            const deleteCertificateFake = sinon.fake.resolves({});
            AWSMock.mock('ACM', 'deleteCertificate', deleteCertificateFake);
            const importer = new TestAcmCertificateImporter({
                acm: new AWS.ACM(),
                dynamoDb: new AWS.DynamoDB(),
                secretsManager: new AWS.SecretsManager(),
                resourceTableOverride: resourceTable,
            });
            // WHEN
            await expect(importer.doDelete(physicalId))
                // THEN
                .resolves.not.toThrow();
            expect(queryStub.calledOnce).toBe(true);
            expect(describeCertificateFake.calledOnce).toBe(true);
            expect(deleteCertificateFake.calledOnce).toBe(true);
            expect(deleteItemStub.calledOnce).toBe(true);
        });
    });
});
/**
 * Specialization of AcmCertificateImporter that overrides methods inherited from
 * DynamoBackedResource so that no API calls are made.
 *
 * This allows the testing code above to focus on the testing the AcmCertificateImporter
 * class without having to deal with mocking out API calls from its parent class.
 */
class TestAcmCertificateImporter extends acm_handlers_1.AcmCertificateImporter {
    constructor(props) {
        super(props.acm, props.dynamoDb, props.secretsManager);
        this.resourceTableOverride = props.resourceTableOverride ?? new MockCompositeStringIndexTable();
    }
    async databasePermissionsCheck() {
        // Do nothing
        return;
    }
    async getResourceTable() {
        return this.resourceTableOverride;
    }
}
/**
 * Mock implementation of CompositeStringIndexTable that does not make API calls.
 *
 * This allows the test code above to instantiate a CompositeStringIndexTable object
 * that can be mocked.
 */
class MockCompositeStringIndexTable extends dynamodb_1.CompositeStringIndexTable {
    constructor() {
        super(new AWS.DynamoDB(), '', '', '');
    }
    async deleteTable() { }
    async putItem(_props) {
        return true;
    }
    async getItem(_props) {
        return {};
    }
    async deleteItem(_props) {
        return true;
    }
    async query(_primaryKeyValue, _pageLimit) {
        return {};
    }
}
//# sourceMappingURL=data:application/json;base64,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