"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.importCert = exports.AcmCertificateImporter = void 0;
/* eslint-disable no-console */
const crypto = require("crypto");
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const backoff_generator_1 = require("../lib/backoff-generator");
const custom_resource_1 = require("../lib/custom-resource");
const x509_certs_1 = require("../lib/x509-certs");
const types_1 = require("./types");
const ACM_VERSION = '2015-12-08';
const DYNAMODB_VERSION = '2012-08-10';
const SECRETS_MANAGER_VERSION = '2017-10-17';
class AcmCertificateImporter extends custom_resource_1.DynamoBackedCustomResource {
    constructor(acmClient, dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient);
        this.acmClient = acmClient;
        this.secretsManagerClient = secretsManagerClient;
    }
    /* istanbul ignore next */
    validateInput(data) {
        return types_1.implementsIAcmImportCertProps(data);
    }
    async doCreate(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const cert = await this.getSecretString(resourceProperties.X509CertificatePem.Cert);
        const certChainArn = resourceProperties.X509CertificatePem.CertChain;
        const certChain = certChainArn.length > 0 ? await this.getSecretString(certChainArn) : undefined;
        const key = await this.getSecretString(resourceProperties.X509CertificatePem.Key);
        const passphrase = await this.getSecretString(resourceProperties.X509CertificatePem.Passphrase);
        const decryptedKey = await x509_certs_1.Certificate.decryptKey(key, passphrase);
        const tags = resourceProperties.Tags;
        const certificateArn = await this.importAndStoreCertificate({
            resourceTable,
            key: decryptedKey,
            cert,
            certChain,
            physicalId,
            tags,
        });
        return { CertificateArn: certificateArn };
    }
    async doDelete(physicalId) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const resources = await resourceTable.query(physicalId);
        const maxAttempts = 10;
        for (const [key, resource] of Object.entries(resources)) {
            const arn = resource.ARN;
            let inUseByResources = [];
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base: 1000,
                jitterDivisor: 4,
                maxAttempts,
                maxIntervalMs: 30000,
            });
            do {
                const { Certificate: cert } = await this.acmClient.describeCertificate({
                    CertificateArn: arn,
                }).promise();
                inUseByResources = cert.InUseBy || [];
                if (inUseByResources.length) {
                    console.log(`Sleeping -- Resource ${arn} in use by ${inUseByResources.join(', ')}`);
                    await backoffGenerator.backoff();
                }
                else {
                    break;
                }
            } while (backoffGenerator.shouldContinue());
            if (inUseByResources.length) {
                throw new Error(`Response from describeCertificate did not contain an empty InUseBy list after ${maxAttempts} attempts.`);
            }
            console.log(`Deleting resource for '${key}'`);
            try {
                await this.acmClient.deleteCertificate({ CertificateArn: arn }).promise();
            }
            catch (e) {
                // AccessDeniedException can happen if either:
                //  a) We do not have the required permission to delete the Certificate (unlikely)
                //  b) The Certificate has already been deleted (more likely)
                if (e.message.indexOf('AccessDeniedException')) {
                    console.warn(`Could not delete Certificate ${arn}. Please ensure it has been deleted.`);
                }
                throw e; // Rethrow so the custom resource handler will error-out.
            }
            await resourceTable.deleteItem({
                primaryKeyValue: physicalId,
                sortKeyValue: key,
            });
        }
    }
    async importAndStoreCertificate(args) {
        let certificateArn;
        const sortKey = crypto.createHash('md5').update(args.cert).digest('hex');
        const existingItem = await args.resourceTable.getItem({
            primaryKeyValue: args.physicalId,
            sortKeyValue: sortKey,
        });
        if (existingItem) {
            if (!existingItem.ARN) {
                throw Error("Database Item missing 'ARN' attribute");
            }
            // Verify that the cert is in ACM
            certificateArn = existingItem.ARN;
            try {
                await this.acmClient.getCertificate({ CertificateArn: certificateArn }).promise();
            }
            catch (e) {
                throw Error(`Database entry ${existingItem.ARN} could not be found in ACM: ${JSON.stringify(e)}`);
            }
            // Update the cert by performing an import again, with the new values.
            const importCertRequest = {
                CertificateArn: certificateArn,
                Certificate: args.cert,
                CertificateChain: args.certChain,
                PrivateKey: args.key,
                Tags: args.tags,
            };
            await this.importCertificate(importCertRequest);
        }
        else {
            const importCertRequest = {
                Certificate: args.cert,
                CertificateChain: args.certChain,
                PrivateKey: args.key,
                Tags: args.tags,
            };
            const resp = await this.importCertificate(importCertRequest);
            if (!resp.CertificateArn) {
                throw new Error(`CertificateArn was not properly populated after attempt to import ${args.cert}`);
            }
            certificateArn = resp.CertificateArn;
            await args.resourceTable.putItem({
                primaryKeyValue: args.physicalId,
                sortKeyValue: sortKey,
                attributes: {
                    ARN: certificateArn,
                },
                allow_overwrite: false,
            });
        }
        return certificateArn;
    }
    async importCertificate(importCertRequest) {
        // ACM cert imports are limited to 1 per second (see https://docs.aws.amazon.com/acm/latest/userguide/acm-limits.html#api-rate-limits)
        // We need to backoff & retry in the event that two imports happen in the same second
        const maxAttempts = 10;
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base: 200,
            jitterDivisor: 4,
            maxAttempts,
        });
        let retry = false;
        do {
            try {
                return await this.acmClient.importCertificate(importCertRequest).promise();
            }
            catch (e) {
                console.warn(`Could not import certificate: ${e}`);
                retry = await backoffGenerator.backoff();
                if (retry) {
                    console.log('Retrying...');
                }
            }
        } while (retry);
        throw new Error(`Failed to import certificate ${importCertRequest.CertificateArn ?? ''} after ${maxAttempts} attempts.`);
    }
    async getSecretString(SecretId) {
        console.debug(`Retrieving secret: ${SecretId}`);
        const resp = await this.secretsManagerClient.getSecretValue({ SecretId }).promise();
        if (!resp.SecretString) {
            throw new Error(`Secret ${SecretId} did not contain a SecretString as expected`);
        }
        return resp.SecretString;
    }
}
exports.AcmCertificateImporter = AcmCertificateImporter;
/**
 * The handler used to import an X.509 certificate to ACM from a Secret
 */
/* istanbul ignore next */
async function importCert(event, context) {
    const handler = new AcmCertificateImporter(new aws_sdk_1.ACM({ apiVersion: ACM_VERSION }), new aws_sdk_1.DynamoDB({ apiVersion: DYNAMODB_VERSION }), new aws_sdk_1.SecretsManager({ apiVersion: SECRETS_MANAGER_VERSION }));
    return await handler.handler(event, context);
}
exports.importCert = importCert;
//# sourceMappingURL=data:application/json;base64,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