"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const version_1 = require("../../lib/version-provider/version");
describe('validateVersionString', () => {
    test('correct input', () => {
        expect(version_1.Version.validateVersionString('10.1.9.2')).toBeTruthy();
    });
    test('malformed input', () => {
        expect(version_1.Version.validateVersionString('10.1.9.2.1')).toBeFalsy();
    });
});
describe('parseVersionString', () => {
    test.each([
        [
            '10.1.10.6',
            ['10', '1', '10', '6'],
        ],
        [
            '10.1.9.2',
            ['10', '1', '9', '2'],
        ],
        [
            '10.1.9',
            ['10', '1', '9'],
        ],
        [
            '10.1',
            ['10', '1'],
        ],
        [
            '10',
            ['10'],
        ],
    ])('correct input: %p', (value, resultArray) => {
        const regexResult = version_1.Version.parseFromVersionString(value);
        expect(regexResult).not.toBeNull();
        if (regexResult === null) {
            return;
        }
        expect(regexResult[0]).toEqual(value);
        for (let x = 0; x < resultArray.length; x++) {
            expect(regexResult[x + 1]).toEqual(resultArray[x]);
        }
    });
    test.each([
        '10.1.9.2.1',
        '10.',
        '10.1.',
        '10.-1',
        'a.b.c',
    ])('incorrect version %s parsing', (versionString) => {
        const result = version_1.Version.parseFromVersionString(versionString);
        expect(result).toBeNull();
    });
});
describe('convertToFullVersionString', () => {
    test('correct input', () => {
        expect(version_1.Version.convertToFullVersionString('10', '1', '9', '2')).toEqual('10.1.9.2');
    });
    test('negative value', () => {
        expect(() => version_1.Version.convertToFullVersionString('10', '-1', '9', '2')).toThrowError(/A component of the version was not in the correct format/);
    });
    test('non-numeric value', () => {
        expect(() => version_1.Version.convertToFullVersionString('10', 'test', '9', '2')).toThrowError(/A component of the version was not in the correct format/);
    });
});
//# sourceMappingURL=data:application/json;base64,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