"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
const child_process = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const util_1 = require("util");
const filesystem_1 = require("../../filesystem");
const certificate_1 = require("../certificate");
const distinguished_name_1 = require("../distinguished-name");
const exec = util_1.promisify(child_process.exec);
let tmpDir;
// Enable/disable debugging statements.
const DEBUG = false;
if (!DEBUG) {
    console.debug = () => { };
}
beforeAll(async () => {
    tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
});
afterAll(async () => {
    const unlinks = [];
    const filenames = await fs.promises.readdir(tmpDir);
    for (const file of filenames) {
        unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
    }
    await Promise.all(unlinks);
    await fs.promises.rmdir(tmpDir);
});
test('generate self-signed', async () => {
    // GIVEN
    const name = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(name, passphrase);
    const crtFileName = path.join(tmpDir, 'ss-ca.crt');
    const keyFileName = path.join(tmpDir, 'ss-ca.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const keyOut = await exec(`openssl rsa -noout -text -check -passin env:PW -in ${keyFileName}`, { env: { PW: passphrase } });
    const keyVerification = keyOut.stdout;
    const expiryDate = new Date();
    expiryDate.setDate(expiryDate.getDate() + 3 * 365);
    // THEN
    expect(certificate.cert).toContain('-----BEGIN CERTIFICATE-----');
    expect(certificate.cert).toContain('-----END CERTIFICATE-----');
    expect(certificate.key).toContain('-----BEGIN ENCRYPTED PRIVATE KEY-----');
    expect(certificate.key).toContain('-----END ENCRYPTED PRIVATE KEY-----');
    expect(certificate.passphrase).toBe(passphrase);
    expect(certificate.certChain).toEqual('');
    expect(certVerification).toMatch(/Issuer: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certVerification).toMatch(/Subject: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certVerification).toContain('Version: 3 (0x2)');
    expect(certVerification).toContain('Public-Key: (2048 bit)');
    // ex: Not After : May 22 22:13:24 2023 GMT
    expect(certVerification).toMatch(new RegExp(`Not After.*${expiryDate.getFullYear()} GMT`));
    expect(keyVerification).toContain('RSA key ok');
    expect(keyVerification).toMatch(/Private-Key: \(2048 bit(, \d+ primes)?\)/);
});
test('generate self-signed with expiry', async () => {
    // GIVEN
    const name = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(name, passphrase, 5 * 365);
    const crtFileName = path.join(tmpDir, 'ss-ca.crt');
    const keyFileName = path.join(tmpDir, 'ss-ca.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const expiryDate = new Date();
    expiryDate.setDate(expiryDate.getDate() + 5 * 365);
    // THEN
    // ex: Not After : May 22 22:13:24 2023 GMT
    expect(certVerification).toMatch(new RegExp(`Not After.*${expiryDate.getFullYear()} GMT`));
});
test('generate signed certificate', async () => {
    // GIVEN
    const caName = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const certName = new distinguished_name_1.DistinguishedName({
        CN: 'CertCN',
        O: 'CertO',
        OU: 'CertOU',
    });
    const ca = await certificate_1.Certificate.fromGenerated(caName, 'signing_passphrase');
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(certName, passphrase, undefined, ca);
    const crtFileName = path.join(tmpDir, 'signed.crt');
    const crtChainFileName = path.join(tmpDir, 'chain.crt');
    const keyFileName = path.join(tmpDir, 'signed.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    if (certificate.certChain) {
        await filesystem_1.writeAsciiFile(crtChainFileName, certificate.certChain);
    }
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const certChainOut = await exec(`openssl x509 -noout -text -in ${crtChainFileName}`);
    const certChainVerification = certChainOut.stdout;
    const keyOut = await exec(`openssl rsa -noout -text -check -passin env:PW -in ${keyFileName}`, { env: { PATH: process.env.PATH, PW: passphrase } });
    const keyVerification = keyOut.stdout;
    const expiryDate = new Date();
    expiryDate.setDate(expiryDate.getDate() + 3 * 365);
    // THEN
    expect(certificate.cert).toContain('-----BEGIN CERTIFICATE-----');
    expect(certificate.cert).toContain('-----END CERTIFICATE-----');
    // The cert chain should contain the signing cert
    expect(certificate.certChain).toContain(ca.cert);
    // The cert should not contain any chain
    expect(certificate.cert.indexOf('-----BEGIN CERTIFICATE-----')).toBe(certificate.cert.lastIndexOf('-----BEGIN CERTIFICATE-----'));
    expect(certificate.certChain).toContain('-----BEGIN CERTIFICATE-----');
    expect(certificate.certChain).toContain('-----END CERTIFICATE-----');
    expect(certificate.key).toContain('-----BEGIN ENCRYPTED PRIVATE KEY-----');
    expect(certificate.key).toContain('-----END ENCRYPTED PRIVATE KEY-----');
    expect(certificate.passphrase).toBe(passphrase);
    expect(certVerification).toMatch(/Issuer: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certVerification).toMatch(/Subject: CN\s*=\s*CertCN, O\s*=\s*CertO, OU\s*=\s*CertOU/);
    expect(certVerification).toContain('Public-Key: (2048 bit)');
    // ex: Not After : May 22 22:13:24 2023 GMT
    expect(certVerification).toMatch(new RegExp(`Not After.*${expiryDate.getFullYear()} GMT`));
    expect(certChainVerification).toMatch(/Issuer: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certChainVerification).toMatch(/Subject: CN\s*=\s*TestCN, O\s*=\s*TestO, OU\s*=\s*TestOU/);
    expect(certChainVerification).toContain('Public-Key: (2048 bit)');
    expect(keyVerification).toContain('RSA key ok');
    expect(keyVerification).toMatch(/Private-Key: \(2048 bit(, \d+ primes)?\)/);
});
test('generate signed certificate with expiry', async () => {
    // GIVEN
    const caName = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const certName = new distinguished_name_1.DistinguishedName({
        CN: 'CertCN',
        O: 'CertO',
        OU: 'CertOU',
    });
    const ca = await certificate_1.Certificate.fromGenerated(caName, 'signing_passphrase');
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(certName, passphrase, 5 * 365, ca);
    const crtFileName = path.join(tmpDir, 'signed.crt');
    const crtChainFileName = path.join(tmpDir, 'chain.crt');
    const keyFileName = path.join(tmpDir, 'signed.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    if (certificate.certChain) {
        await filesystem_1.writeAsciiFile(crtChainFileName, certificate.certChain);
    }
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const expiryDate = new Date();
    expiryDate.setDate(expiryDate.getDate() + 5 * 365);
    // THEN
    // ex: Not After : May 22 22:13:24 2023 GMT
    expect(certVerification).toMatch(new RegExp(`Not After.*${expiryDate.getFullYear()} GMT`));
});
test('convert to PKCS #12', async () => {
    const caName = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const certName = new distinguished_name_1.DistinguishedName({
        CN: 'CertCN',
        O: 'CertO',
        OU: 'CertOU',
    });
    const ca = await certificate_1.Certificate.fromGenerated(caName, 'signing_passphrase');
    const passphrase = 'test_passphrase';
    const certificate = await certificate_1.Certificate.fromGenerated(certName, passphrase, undefined, ca);
    const pkcs12Passphrase = 'test_passphrase';
    // WHEN
    const pkcs12Data = await certificate.toPkcs12(pkcs12Passphrase);
    const fileName = path.join(tmpDir, 'cert.p12');
    await filesystem_1.writeBinaryFile(fileName, pkcs12Data);
    let pkcs12Validation;
    // If the PKCS12 passphrase does not match, openssl will return a non-zero exit code and fail the test
    // This was tested for both OpenSSL 1.0.x and 1.1.x.
    pkcs12Validation = await exec(`openssl pkcs12 -in ${fileName} -info -nodes -passin env:PW`, { env: { PATH: process.env.PATH, PW: pkcs12Passphrase } });
    const validationOut = pkcs12Validation.stdout;
    // THEN
    // Must have the certificate's cert
    expect(validationOut).toMatch(/subject=\/?CN\s*=\s*CertCN[/,]\s*O\s*=\s*CertO[/,]\s*OU\s*=\s*CertOU\n{1,2}issuer=\/?CN\s*=\s*TestCN[/,]\s*O\s*=\s*TestO[/,]\s*OU\s*=\s*TestOU\n{1,2}-----BEGIN CERTIFICATE-----/);
    // Must have the CA cert
    expect(validationOut).toMatch(/subject=\/?CN\s*=\s*TestCN[/,]\s*O\s*=\s*TestO[/,]\s*OU\s*=\s*TestOU\n{1,2}issuer=\/?CN\s*=\s*TestCN[/,]\s*O\s*=\s*TestO[/,]\s*OU\s*=\s*TestOU\n{1,2}-----BEGIN CERTIFICATE-----/);
    // Must have the decrypted private key
    expect(validationOut).toContain('-----BEGIN PRIVATE KEY-----');
});
test('decrypt private key', async () => {
    // GIVEN
    const name = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const passphrase = 'test_passphrase';
    const certificate = await certificate_1.Certificate.fromGenerated(name, passphrase);
    // WHEN
    const decryptedKey = await certificate_1.Certificate.decryptKey(certificate.key, passphrase);
    const crtFileName = path.join(tmpDir, 'ca.crt');
    const keyFileName = path.join(tmpDir, 'ca.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const expectedDecryptedKeyOut = await exec(`openssl rsa -in ${keyFileName} -passin env:PW`, { env: { PATH: process.env.PATH, PW: passphrase } });
    const expectedDecryptedKey = expectedDecryptedKeyOut.stdout;
    // THEN
    expect(decryptedKey).toEqual(expectedDecryptedKey);
    // Must have the decrypted private key
    expect(decryptedKey).toContain('-----BEGIN RSA PRIVATE KEY-----');
});
//# sourceMappingURL=data:application/json;base64,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