"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Certificate = void 0;
/* eslint-disable no-console */
const child_process = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const util_1 = require("util");
const filesystem_1 = require("../filesystem");
const exec = util_1.promisify(child_process.exec);
class Certificate {
    constructor(cert, key, passphrase, certChain) {
        this.cert = cert;
        this.key = key;
        this.passphrase = passphrase;
        this.certChain = certChain;
    }
    static async fromGenerated(subject, passphrase, certValidFor, signingCertificate) {
        const tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
        try {
            let cert;
            let key;
            let certChain;
            if (!signingCertificate) {
                [cert, key] = await Certificate.generateSelfSigned(tmpDir, subject, passphrase, certValidFor ?? 1095);
                // certChain cannot be left undefined. CFN expects that attributes will *always* have values.
                certChain = '';
            }
            else {
                [cert, key, certChain] = await Certificate.generateSigned(tmpDir, subject, passphrase, certValidFor ?? 1095, signingCertificate);
            }
            return new Certificate(cert, key, passphrase, certChain);
        }
        finally {
            const unlinks = [];
            const filenames = await fs.promises.readdir(tmpDir);
            for (const file of filenames) {
                unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
            }
            await Promise.all(unlinks);
            await fs.promises.rmdir(tmpDir);
        }
    }
    static async decryptKey(key, passphrase) {
        const tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
        try {
            const encrypedKeyFile = path.join(tmpDir, 'encrypted.key');
            await filesystem_1.writeAsciiFile(encrypedKeyFile, key);
            const decryptedKeyFile = path.join(tmpDir, 'decrypted.key');
            const cmd = 'openssl rsa ' +
                `-in ${encrypedKeyFile} ` +
                '-passin env:CERT_PASSPHRASE ' +
                `-out ${decryptedKeyFile}`;
            console.debug(`Running: ${cmd}`);
            await exec(cmd, { env: { CERT_PASSPHRASE: passphrase, PATH: process.env.PATH } });
            const keyDecrypted = await filesystem_1.readAsciiFile(decryptedKeyFile);
            return keyDecrypted;
        }
        finally {
            const unlinks = [];
            const filenames = await fs.promises.readdir(tmpDir);
            for (const file of filenames) {
                unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
            }
            await Promise.all(unlinks);
            await fs.promises.rmdir(tmpDir);
        }
    }
    static async generateSelfSigned(tmpDir, subject, passphrase, certValidFor) {
        const crtFile = path.join(tmpDir, 'crt');
        const keyFile = path.join(tmpDir, 'key');
        const cmd = 'openssl req -x509 ' +
            '-passout env:CERT_PASSPHRASE ' +
            '-newkey rsa:2048 ' +
            `-days ${certValidFor} ` +
            '-extensions v3_ca ' +
            `-keyout ${keyFile} -out ${crtFile} ` +
            `-subj ${subject.toString()}`;
        console.debug(`Running: ${cmd}`);
        await exec(cmd, { env: { CERT_PASSPHRASE: passphrase, PATH: process.env.PATH } });
        const cert = await filesystem_1.readAsciiFile(crtFile);
        const key = await filesystem_1.readAsciiFile(keyFile);
        return [cert, key];
    }
    static async generateSigned(tmpDir, subject, passphrase, certValidFor, signingCertificate) {
        const signingCertFile = path.join(tmpDir, 'signing.crt');
        const signingKeyFile = path.join(tmpDir, 'signing.key');
        const caChain = signingCertificate.cert + signingCertificate.certChain;
        await filesystem_1.writeAsciiFile(signingCertFile, caChain);
        await filesystem_1.writeAsciiFile(signingKeyFile, signingCertificate.key);
        const csrFile = path.join(tmpDir, 'cert.csr');
        const crtFile = path.join(tmpDir, 'cert.crt');
        const keyFile = path.join(tmpDir, 'cert.key');
        const certSigningRequest = 'openssl req ' +
            '-passout env:CERT_PASSPHRASE ' +
            '-newkey rsa:2048 ' +
            `-days ${certValidFor} ` +
            `-out ${csrFile} -keyout ${keyFile} ` +
            `-subj ${subject.toString()}`;
        const crtCreate = 'openssl x509 -sha256 -req ' +
            '-passin env:SIGNING_PASSPHRASE ' +
            `-days ${certValidFor} ` +
            `-in ${csrFile} ` +
            `-CA ${signingCertFile} -CAkey ${signingKeyFile} -CAcreateserial ` +
            `-out ${crtFile}`;
        console.debug(`Running: ${certSigningRequest}`);
        await exec(certSigningRequest, { env: { CERT_PASSPHRASE: passphrase, PATH: process.env.PATH } });
        console.debug(`Running: ${crtCreate}`);
        await exec(crtCreate, { env: { PATH: process.env.PATH, SIGNING_PASSPHRASE: signingCertificate.passphrase } });
        const cert = await filesystem_1.readAsciiFile(crtFile);
        const key = await filesystem_1.readAsciiFile(keyFile);
        // Return the certificate, private key, and certificate chain. The certificate chain is the signing certificate
        // prepended to its own certificate chain.
        return [cert, key, caChain];
    }
    async toPkcs12(passphrase) {
        const tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
        try {
            const crtFileName = path.join(tmpDir, 'input.crt');
            const keyFileName = path.join(tmpDir, 'input.key');
            const caCert = this.certChain ? this.cert + this.certChain : this.cert;
            await filesystem_1.writeAsciiFile(crtFileName, caCert);
            await filesystem_1.writeAsciiFile(keyFileName, this.key);
            const pkcs12FileName = path.join(tmpDir, 'cert.p12');
            const command = 'openssl pkcs12 -export -nodes -passin env:PASSIN -passout env:PASSOUT ' +
                `-out ${pkcs12FileName} -inkey ${keyFileName} -in ${crtFileName}`;
            await exec(command, { env: {
                    PASSIN: this.passphrase,
                    PASSOUT: passphrase,
                    PATH: process.env.PATH,
                } });
            const pkcs12Data = await filesystem_1.readBinaryFile(pkcs12FileName);
            return pkcs12Data;
        }
        finally {
            const unlinks = [];
            const filenames = await fs.promises.readdir(tmpDir);
            for (const file of filenames) {
                unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
            }
            await Promise.all(unlinks);
            await fs.promises.rmdir(tmpDir);
        }
    }
}
exports.Certificate = Certificate;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2VydGlmaWNhdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZXJ0aWZpY2F0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7OztHQUdHOzs7QUFFSCwrQkFBK0I7QUFFL0IsK0NBQStDO0FBQy9DLHlCQUF5QjtBQUN6Qix5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLCtCQUFpQztBQUVqQyw4Q0FJdUI7QUFHdkIsTUFBTSxJQUFJLEdBQUcsZ0JBQVMsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7QUFTM0MsTUFBYSxXQUFXO0lBMkl0QixZQUNFLElBQVksRUFDWixHQUFXLEVBQ1gsVUFBa0IsRUFDbEIsU0FBaUI7UUFFakIsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUM7UUFDakIsSUFBSSxDQUFDLEdBQUcsR0FBRyxHQUFHLENBQUM7UUFDZixJQUFJLENBQUMsVUFBVSxHQUFHLFVBQVUsQ0FBQztRQUM3QixJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztJQUM3QixDQUFDO0lBcEpNLE1BQU0sQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUMvQixPQUEwQixFQUMxQixVQUFrQixFQUNsQixZQUFxQixFQUNyQixrQkFBZ0M7UUFFaEMsTUFBTSxNQUFNLEdBQUcsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ3pFLElBQUk7WUFDRixJQUFJLElBQVksQ0FBQztZQUNqQixJQUFJLEdBQVcsQ0FBQztZQUNoQixJQUFJLFNBQWlCLENBQUM7WUFDdEIsSUFBSSxDQUFDLGtCQUFrQixFQUFFO2dCQUN2QixDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsR0FBRyxNQUFNLFdBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLEVBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxZQUFZLElBQUksSUFBSSxDQUFDLENBQUM7Z0JBQ3RHLDZGQUE2RjtnQkFDN0YsU0FBUyxHQUFHLEVBQUUsQ0FBQzthQUNoQjtpQkFBTTtnQkFDTCxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsU0FBUyxDQUFDLEdBQUcsTUFBTSxXQUFXLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxPQUFPLEVBQUUsVUFBVSxFQUFFLFlBQVksSUFBSSxJQUFJLEVBQUUsa0JBQWtCLENBQUMsQ0FBQzthQUNsSTtZQUNELE9BQU8sSUFBSSxXQUFXLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxVQUFVLEVBQUUsU0FBUyxDQUFDLENBQUM7U0FDMUQ7Z0JBQVM7WUFDUixNQUFNLE9BQU8sR0FBeUIsRUFBRSxDQUFDO1lBQ3pDLE1BQU0sU0FBUyxHQUFhLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDOUQsS0FBSyxNQUFNLElBQUksSUFBSSxTQUFTLEVBQUU7Z0JBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQzNEO1lBQ0QsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQzNCLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDakM7SUFDSCxDQUFDO0lBRU0sTUFBTSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQzVCLEdBQVcsRUFDWCxVQUFrQjtRQUVsQixNQUFNLE1BQU0sR0FBRyxNQUFNLEVBQUUsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDekUsSUFBSTtZQUNGLE1BQU0sZUFBZSxHQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQ25FLE1BQU0sMkJBQWMsQ0FBQyxlQUFlLEVBQUUsR0FBRyxDQUFDLENBQUM7WUFDM0MsTUFBTSxnQkFBZ0IsR0FBVyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxlQUFlLENBQUMsQ0FBQztZQUNwRSxNQUFNLEdBQUcsR0FDUCxjQUFjO2dCQUNkLE9BQU8sZUFBZSxHQUFHO2dCQUN6Qiw4QkFBOEI7Z0JBQzlCLFFBQVEsZ0JBQWdCLEVBQUUsQ0FBQztZQUU3QixPQUFPLENBQUMsS0FBSyxDQUFDLFlBQVksR0FBRyxFQUFFLENBQUMsQ0FBQztZQUNqQyxNQUFNLElBQUksQ0FBQyxHQUFHLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxlQUFlLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQztZQUVsRixNQUFNLFlBQVksR0FBRyxNQUFNLDBCQUFhLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztZQUUzRCxPQUFPLFlBQVksQ0FBQztTQUNyQjtnQkFBUztZQUNSLE1BQU0sT0FBTyxHQUF5QixFQUFFLENBQUM7WUFDekMsTUFBTSxTQUFTLEdBQWEsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUM5RCxLQUFLLE1BQU0sSUFBSSxJQUFJLFNBQVMsRUFBRTtnQkFDNUIsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7YUFDM0Q7WUFDRCxNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDM0IsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUNqQztJQUNILENBQUM7SUFFTyxNQUFNLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUNyQyxNQUFjLEVBQ2QsT0FBMEIsRUFDMUIsVUFBa0IsRUFDbEIsWUFBb0I7UUFFcEIsTUFBTSxPQUFPLEdBQVcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDakQsTUFBTSxPQUFPLEdBQVcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDakQsTUFBTSxHQUFHLEdBQ1Asb0JBQW9CO1lBQ3BCLCtCQUErQjtZQUMvQixtQkFBbUI7WUFDbkIsU0FBUyxZQUFZLEdBQUc7WUFDeEIsb0JBQW9CO1lBQ3BCLFdBQVcsT0FBTyxTQUFTLE9BQU8sR0FBRztZQUNyQyxTQUFTLE9BQU8sQ0FBQyxRQUFRLEVBQUUsRUFBRSxDQUFDO1FBRWhDLE9BQU8sQ0FBQyxLQUFLLENBQUMsWUFBWSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ2pDLE1BQU0sSUFBSSxDQUFDLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWxGLE1BQU0sSUFBSSxHQUFXLE1BQU0sMEJBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNsRCxNQUFNLEdBQUcsR0FBVyxNQUFNLDBCQUFhLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFakQsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQztJQUNyQixDQUFDO0lBRU8sTUFBTSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQ2pDLE1BQWMsRUFDZCxPQUEwQixFQUMxQixVQUFrQixFQUNsQixZQUFvQixFQUNwQixrQkFBK0I7UUFFL0IsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDekQsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDeEQsTUFBTSxPQUFPLEdBQUcsa0JBQWtCLENBQUMsSUFBSSxHQUFHLGtCQUFrQixDQUFDLFNBQVMsQ0FBQztRQUN2RSxNQUFNLDJCQUFjLENBQUMsZUFBZSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQy9DLE1BQU0sMkJBQWMsQ0FBQyxjQUFjLEVBQUUsa0JBQWtCLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFN0QsTUFBTSxPQUFPLEdBQVcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDdEQsTUFBTSxPQUFPLEdBQVcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDdEQsTUFBTSxPQUFPLEdBQVcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFdEQsTUFBTSxrQkFBa0IsR0FDaEIsY0FBYztZQUNkLCtCQUErQjtZQUMvQixtQkFBbUI7WUFDbkIsU0FBUyxZQUFZLEdBQUc7WUFDeEIsUUFBUSxPQUFPLFlBQVksT0FBTyxHQUFHO1lBQ3JDLFNBQVMsT0FBTyxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUM7UUFDdEMsTUFBTSxTQUFTLEdBQ1AsNEJBQTRCO1lBQzVCLGlDQUFpQztZQUNqQyxTQUFTLFlBQVksR0FBRztZQUN4QixPQUFPLE9BQU8sR0FBRztZQUNqQixPQUFPLGVBQWUsV0FBVyxjQUFjLG1CQUFtQjtZQUNsRSxRQUFRLE9BQU8sRUFBRSxDQUFDO1FBRTFCLE9BQU8sQ0FBQyxLQUFLLENBQUMsWUFBWSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7UUFDaEQsTUFBTSxJQUFJLENBQUMsa0JBQWtCLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxlQUFlLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxFQUFDLENBQUMsQ0FBQztRQUNoRyxPQUFPLENBQUMsS0FBSyxDQUFDLFlBQVksU0FBUyxFQUFFLENBQUMsQ0FBQztRQUN2QyxNQUFNLElBQUksQ0FBQyxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxJQUFJLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUUsa0JBQWtCLENBQUMsVUFBVSxFQUFFLEVBQUMsQ0FBQyxDQUFDO1FBRTdHLE1BQU0sSUFBSSxHQUFXLE1BQU0sMEJBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNsRCxNQUFNLEdBQUcsR0FBVyxNQUFNLDBCQUFhLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFakQsK0dBQStHO1FBQy9HLDBDQUEwQztRQUMxQyxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBbUJNLEtBQUssQ0FBQyxRQUFRLENBQUMsVUFBa0I7UUFDdEMsTUFBTSxNQUFNLEdBQUcsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ3pFLElBQUk7WUFDRixNQUFNLFdBQVcsR0FBVyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxXQUFXLENBQUMsQ0FBQztZQUMzRCxNQUFNLFdBQVcsR0FBVyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxXQUFXLENBQUMsQ0FBQztZQUMzRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7WUFDdkUsTUFBTSwyQkFBYyxDQUFDLFdBQVcsRUFBRSxNQUFNLENBQUMsQ0FBQztZQUMxQyxNQUFNLDJCQUFjLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUU1QyxNQUFNLGNBQWMsR0FBVyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxVQUFVLENBQUMsQ0FBQztZQUM3RCxNQUFNLE9BQU8sR0FBVyx3RUFBd0U7Z0JBQzlGLFFBQVEsY0FBYyxXQUFXLFdBQVcsUUFBUSxXQUFXLEVBQUUsQ0FBQztZQUNwRSxNQUFNLElBQUksQ0FDUixPQUFPLEVBQ1AsRUFBRSxHQUFHLEVBQUU7b0JBQ0wsTUFBTSxFQUFFLElBQUksQ0FBQyxVQUFVO29CQUN2QixPQUFPLEVBQUUsVUFBVTtvQkFDbkIsSUFBSSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSTtpQkFDdkIsRUFBQyxDQUNILENBQUM7WUFFRixNQUFNLFVBQVUsR0FBVyxNQUFNLDJCQUFjLENBQUMsY0FBYyxDQUFDLENBQUM7WUFFaEUsT0FBTyxVQUFVLENBQUM7U0FDbkI7Z0JBQVM7WUFDUixNQUFNLE9BQU8sR0FBeUIsRUFBRSxDQUFDO1lBQ3pDLE1BQU0sU0FBUyxHQUFhLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDOUQsS0FBSyxNQUFNLElBQUksSUFBSSxTQUFTLEVBQUU7Z0JBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQzNEO1lBQ0QsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQzNCLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDakM7SUFDSCxDQUFDO0NBQ0Y7QUF6TEQsa0NBeUxDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuLyogZXNsaW50LWRpc2FibGUgbm8tY29uc29sZSAqL1xuXG5pbXBvcnQgKiBhcyBjaGlsZF9wcm9jZXNzIGZyb20gJ2NoaWxkX3Byb2Nlc3MnO1xuaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgb3MgZnJvbSAnb3MnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IHByb21pc2lmeSB9IGZyb20gJ3V0aWwnO1xuXG5pbXBvcnQge1xuICByZWFkQXNjaWlGaWxlLFxuICByZWFkQmluYXJ5RmlsZSxcbiAgd3JpdGVBc2NpaUZpbGUsXG59IGZyb20gJy4uL2ZpbGVzeXN0ZW0nO1xuaW1wb3J0IHsgRGlzdGluZ3Vpc2hlZE5hbWUgfSBmcm9tICcuL2Rpc3Rpbmd1aXNoZWQtbmFtZSc7XG5cbmNvbnN0IGV4ZWMgPSBwcm9taXNpZnkoY2hpbGRfcHJvY2Vzcy5leGVjKTtcblxuZXhwb3J0IGludGVyZmFjZSBJQ2VydGlmaWNhdGUge1xuICByZWFkb25seSBjZXJ0OiBzdHJpbmc7XG4gIHJlYWRvbmx5IGtleTogc3RyaW5nO1xuICByZWFkb25seSBwYXNzcGhyYXNlOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGNlcnRDaGFpbjogc3RyaW5nO1xufVxuXG5leHBvcnQgY2xhc3MgQ2VydGlmaWNhdGUgaW1wbGVtZW50cyBJQ2VydGlmaWNhdGUge1xuICBwdWJsaWMgc3RhdGljIGFzeW5jIGZyb21HZW5lcmF0ZWQoXG4gICAgc3ViamVjdDogRGlzdGluZ3Vpc2hlZE5hbWUsXG4gICAgcGFzc3BocmFzZTogc3RyaW5nLFxuICAgIGNlcnRWYWxpZEZvcj86IG51bWJlcixcbiAgICBzaWduaW5nQ2VydGlmaWNhdGU/OiBDZXJ0aWZpY2F0ZSxcbiAgKTogUHJvbWlzZTxDZXJ0aWZpY2F0ZT4ge1xuICAgIGNvbnN0IHRtcERpciA9IGF3YWl0IGZzLnByb21pc2VzLm1rZHRlbXAocGF0aC5qb2luKG9zLnRtcGRpcigpLCAndG1wLicpKTtcbiAgICB0cnkge1xuICAgICAgbGV0IGNlcnQ6IHN0cmluZztcbiAgICAgIGxldCBrZXk6IHN0cmluZztcbiAgICAgIGxldCBjZXJ0Q2hhaW46IHN0cmluZztcbiAgICAgIGlmICghc2lnbmluZ0NlcnRpZmljYXRlKSB7XG4gICAgICAgIFtjZXJ0LCBrZXldID0gYXdhaXQgQ2VydGlmaWNhdGUuZ2VuZXJhdGVTZWxmU2lnbmVkKHRtcERpciwgc3ViamVjdCwgcGFzc3BocmFzZSwgY2VydFZhbGlkRm9yID8/IDEwOTUpO1xuICAgICAgICAvLyBjZXJ0Q2hhaW4gY2Fubm90IGJlIGxlZnQgdW5kZWZpbmVkLiBDRk4gZXhwZWN0cyB0aGF0IGF0dHJpYnV0ZXMgd2lsbCAqYWx3YXlzKiBoYXZlIHZhbHVlcy5cbiAgICAgICAgY2VydENoYWluID0gJyc7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBbY2VydCwga2V5LCBjZXJ0Q2hhaW5dID0gYXdhaXQgQ2VydGlmaWNhdGUuZ2VuZXJhdGVTaWduZWQodG1wRGlyLCBzdWJqZWN0LCBwYXNzcGhyYXNlLCBjZXJ0VmFsaWRGb3IgPz8gMTA5NSwgc2lnbmluZ0NlcnRpZmljYXRlKTtcbiAgICAgIH1cbiAgICAgIHJldHVybiBuZXcgQ2VydGlmaWNhdGUoY2VydCwga2V5LCBwYXNzcGhyYXNlLCBjZXJ0Q2hhaW4pO1xuICAgIH0gZmluYWxseSB7XG4gICAgICBjb25zdCB1bmxpbmtzOiBBcnJheTxQcm9taXNlPHZvaWQ+PiA9IFtdO1xuICAgICAgY29uc3QgZmlsZW5hbWVzOiBzdHJpbmdbXSA9IGF3YWl0IGZzLnByb21pc2VzLnJlYWRkaXIodG1wRGlyKTtcbiAgICAgIGZvciAoY29uc3QgZmlsZSBvZiBmaWxlbmFtZXMpIHtcbiAgICAgICAgdW5saW5rcy5wdXNoKGZzLnByb21pc2VzLnVubGluayhwYXRoLmpvaW4odG1wRGlyLCBmaWxlKSkpO1xuICAgICAgfVxuICAgICAgYXdhaXQgUHJvbWlzZS5hbGwodW5saW5rcyk7XG4gICAgICBhd2FpdCBmcy5wcm9taXNlcy5ybWRpcih0bXBEaXIpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgYXN5bmMgZGVjcnlwdEtleShcbiAgICBrZXk6IHN0cmluZyxcbiAgICBwYXNzcGhyYXNlOiBzdHJpbmcsXG4gICk6IFByb21pc2U8c3RyaW5nPiB7XG4gICAgY29uc3QgdG1wRGlyID0gYXdhaXQgZnMucHJvbWlzZXMubWtkdGVtcChwYXRoLmpvaW4ob3MudG1wZGlyKCksICd0bXAuJykpO1xuICAgIHRyeSB7XG4gICAgICBjb25zdCBlbmNyeXBlZEtleUZpbGU6IHN0cmluZyA9IHBhdGguam9pbih0bXBEaXIsICdlbmNyeXB0ZWQua2V5Jyk7XG4gICAgICBhd2FpdCB3cml0ZUFzY2lpRmlsZShlbmNyeXBlZEtleUZpbGUsIGtleSk7XG4gICAgICBjb25zdCBkZWNyeXB0ZWRLZXlGaWxlOiBzdHJpbmcgPSBwYXRoLmpvaW4odG1wRGlyLCAnZGVjcnlwdGVkLmtleScpO1xuICAgICAgY29uc3QgY21kID1cbiAgICAgICAgJ29wZW5zc2wgcnNhICcgK1xuICAgICAgICBgLWluICR7ZW5jcnlwZWRLZXlGaWxlfSBgICtcbiAgICAgICAgJy1wYXNzaW4gZW52OkNFUlRfUEFTU1BIUkFTRSAnICtcbiAgICAgICAgYC1vdXQgJHtkZWNyeXB0ZWRLZXlGaWxlfWA7XG5cbiAgICAgIGNvbnNvbGUuZGVidWcoYFJ1bm5pbmc6ICR7Y21kfWApO1xuICAgICAgYXdhaXQgZXhlYyhjbWQsIHsgZW52OiB7IENFUlRfUEFTU1BIUkFTRTogcGFzc3BocmFzZSwgUEFUSDogcHJvY2Vzcy5lbnYuUEFUSCB9IH0pO1xuXG4gICAgICBjb25zdCBrZXlEZWNyeXB0ZWQgPSBhd2FpdCByZWFkQXNjaWlGaWxlKGRlY3J5cHRlZEtleUZpbGUpO1xuXG4gICAgICByZXR1cm4ga2V5RGVjcnlwdGVkO1xuICAgIH0gZmluYWxseSB7XG4gICAgICBjb25zdCB1bmxpbmtzOiBBcnJheTxQcm9taXNlPHZvaWQ+PiA9IFtdO1xuICAgICAgY29uc3QgZmlsZW5hbWVzOiBzdHJpbmdbXSA9IGF3YWl0IGZzLnByb21pc2VzLnJlYWRkaXIodG1wRGlyKTtcbiAgICAgIGZvciAoY29uc3QgZmlsZSBvZiBmaWxlbmFtZXMpIHtcbiAgICAgICAgdW5saW5rcy5wdXNoKGZzLnByb21pc2VzLnVubGluayhwYXRoLmpvaW4odG1wRGlyLCBmaWxlKSkpO1xuICAgICAgfVxuICAgICAgYXdhaXQgUHJvbWlzZS5hbGwodW5saW5rcyk7XG4gICAgICBhd2FpdCBmcy5wcm9taXNlcy5ybWRpcih0bXBEaXIpO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIGFzeW5jIGdlbmVyYXRlU2VsZlNpZ25lZChcbiAgICB0bXBEaXI6IHN0cmluZyxcbiAgICBzdWJqZWN0OiBEaXN0aW5ndWlzaGVkTmFtZSxcbiAgICBwYXNzcGhyYXNlOiBzdHJpbmcsXG4gICAgY2VydFZhbGlkRm9yOiBudW1iZXIsXG4gICk6IFByb21pc2U8W3N0cmluZywgc3RyaW5nXT4ge1xuICAgIGNvbnN0IGNydEZpbGU6IHN0cmluZyA9IHBhdGguam9pbih0bXBEaXIsICdjcnQnKTtcbiAgICBjb25zdCBrZXlGaWxlOiBzdHJpbmcgPSBwYXRoLmpvaW4odG1wRGlyLCAna2V5Jyk7XG4gICAgY29uc3QgY21kOiBzdHJpbmcgPVxuICAgICAgJ29wZW5zc2wgcmVxIC14NTA5ICcgK1xuICAgICAgJy1wYXNzb3V0IGVudjpDRVJUX1BBU1NQSFJBU0UgJyArXG4gICAgICAnLW5ld2tleSByc2E6MjA0OCAnICtcbiAgICAgIGAtZGF5cyAke2NlcnRWYWxpZEZvcn0gYCArXG4gICAgICAnLWV4dGVuc2lvbnMgdjNfY2EgJyArXG4gICAgICBgLWtleW91dCAke2tleUZpbGV9IC1vdXQgJHtjcnRGaWxlfSBgICtcbiAgICAgIGAtc3ViaiAke3N1YmplY3QudG9TdHJpbmcoKX1gO1xuXG4gICAgY29uc29sZS5kZWJ1ZyhgUnVubmluZzogJHtjbWR9YCk7XG4gICAgYXdhaXQgZXhlYyhjbWQsIHsgZW52OiB7IENFUlRfUEFTU1BIUkFTRTogcGFzc3BocmFzZSwgUEFUSDogcHJvY2Vzcy5lbnYuUEFUSCB9IH0pO1xuXG4gICAgY29uc3QgY2VydDogc3RyaW5nID0gYXdhaXQgcmVhZEFzY2lpRmlsZShjcnRGaWxlKTtcbiAgICBjb25zdCBrZXk6IHN0cmluZyA9IGF3YWl0IHJlYWRBc2NpaUZpbGUoa2V5RmlsZSk7XG5cbiAgICByZXR1cm4gW2NlcnQsIGtleV07XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBhc3luYyBnZW5lcmF0ZVNpZ25lZChcbiAgICB0bXBEaXI6IHN0cmluZyxcbiAgICBzdWJqZWN0OiBEaXN0aW5ndWlzaGVkTmFtZSxcbiAgICBwYXNzcGhyYXNlOiBzdHJpbmcsXG4gICAgY2VydFZhbGlkRm9yOiBudW1iZXIsXG4gICAgc2lnbmluZ0NlcnRpZmljYXRlOiBDZXJ0aWZpY2F0ZSxcbiAgKTogUHJvbWlzZTxbc3RyaW5nLCBzdHJpbmcsIHN0cmluZ10+IHtcbiAgICBjb25zdCBzaWduaW5nQ2VydEZpbGUgPSBwYXRoLmpvaW4odG1wRGlyLCAnc2lnbmluZy5jcnQnKTtcbiAgICBjb25zdCBzaWduaW5nS2V5RmlsZSA9IHBhdGguam9pbih0bXBEaXIsICdzaWduaW5nLmtleScpO1xuICAgIGNvbnN0IGNhQ2hhaW4gPSBzaWduaW5nQ2VydGlmaWNhdGUuY2VydCArIHNpZ25pbmdDZXJ0aWZpY2F0ZS5jZXJ0Q2hhaW47XG4gICAgYXdhaXQgd3JpdGVBc2NpaUZpbGUoc2lnbmluZ0NlcnRGaWxlLCBjYUNoYWluKTtcbiAgICBhd2FpdCB3cml0ZUFzY2lpRmlsZShzaWduaW5nS2V5RmlsZSwgc2lnbmluZ0NlcnRpZmljYXRlLmtleSk7XG5cbiAgICBjb25zdCBjc3JGaWxlOiBzdHJpbmcgPSBwYXRoLmpvaW4odG1wRGlyLCAnY2VydC5jc3InKTtcbiAgICBjb25zdCBjcnRGaWxlOiBzdHJpbmcgPSBwYXRoLmpvaW4odG1wRGlyLCAnY2VydC5jcnQnKTtcbiAgICBjb25zdCBrZXlGaWxlOiBzdHJpbmcgPSBwYXRoLmpvaW4odG1wRGlyLCAnY2VydC5rZXknKTtcblxuICAgIGNvbnN0IGNlcnRTaWduaW5nUmVxdWVzdCA9XG4gICAgICAgICAgICAnb3BlbnNzbCByZXEgJyArXG4gICAgICAgICAgICAnLXBhc3NvdXQgZW52OkNFUlRfUEFTU1BIUkFTRSAnICtcbiAgICAgICAgICAgICctbmV3a2V5IHJzYToyMDQ4ICcgK1xuICAgICAgICAgICAgYC1kYXlzICR7Y2VydFZhbGlkRm9yfSBgICtcbiAgICAgICAgICAgIGAtb3V0ICR7Y3NyRmlsZX0gLWtleW91dCAke2tleUZpbGV9IGAgK1xuICAgICAgICAgICAgYC1zdWJqICR7c3ViamVjdC50b1N0cmluZygpfWA7XG4gICAgY29uc3QgY3J0Q3JlYXRlID1cbiAgICAgICAgICAgICdvcGVuc3NsIHg1MDkgLXNoYTI1NiAtcmVxICcgK1xuICAgICAgICAgICAgJy1wYXNzaW4gZW52OlNJR05JTkdfUEFTU1BIUkFTRSAnICtcbiAgICAgICAgICAgIGAtZGF5cyAke2NlcnRWYWxpZEZvcn0gYCArXG4gICAgICAgICAgICBgLWluICR7Y3NyRmlsZX0gYCArXG4gICAgICAgICAgICBgLUNBICR7c2lnbmluZ0NlcnRGaWxlfSAtQ0FrZXkgJHtzaWduaW5nS2V5RmlsZX0gLUNBY3JlYXRlc2VyaWFsIGAgK1xuICAgICAgICAgICAgYC1vdXQgJHtjcnRGaWxlfWA7XG5cbiAgICBjb25zb2xlLmRlYnVnKGBSdW5uaW5nOiAke2NlcnRTaWduaW5nUmVxdWVzdH1gKTtcbiAgICBhd2FpdCBleGVjKGNlcnRTaWduaW5nUmVxdWVzdCwgeyBlbnY6IHsgQ0VSVF9QQVNTUEhSQVNFOiBwYXNzcGhyYXNlLCBQQVRIOiBwcm9jZXNzLmVudi5QQVRIIH19KTtcbiAgICBjb25zb2xlLmRlYnVnKGBSdW5uaW5nOiAke2NydENyZWF0ZX1gKTtcbiAgICBhd2FpdCBleGVjKGNydENyZWF0ZSwgeyBlbnY6IHsgUEFUSDogcHJvY2Vzcy5lbnYuUEFUSCwgU0lHTklOR19QQVNTUEhSQVNFOiBzaWduaW5nQ2VydGlmaWNhdGUucGFzc3BocmFzZSB9fSk7XG5cbiAgICBjb25zdCBjZXJ0OiBzdHJpbmcgPSBhd2FpdCByZWFkQXNjaWlGaWxlKGNydEZpbGUpO1xuICAgIGNvbnN0IGtleTogc3RyaW5nID0gYXdhaXQgcmVhZEFzY2lpRmlsZShrZXlGaWxlKTtcblxuICAgIC8vIFJldHVybiB0aGUgY2VydGlmaWNhdGUsIHByaXZhdGUga2V5LCBhbmQgY2VydGlmaWNhdGUgY2hhaW4uIFRoZSBjZXJ0aWZpY2F0ZSBjaGFpbiBpcyB0aGUgc2lnbmluZyBjZXJ0aWZpY2F0ZVxuICAgIC8vIHByZXBlbmRlZCB0byBpdHMgb3duIGNlcnRpZmljYXRlIGNoYWluLlxuICAgIHJldHVybiBbY2VydCwga2V5LCBjYUNoYWluXTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBjZXJ0OiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBrZXk6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHBhc3NwaHJhc2U6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGNlcnRDaGFpbjogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIGNlcnQ6IHN0cmluZyxcbiAgICBrZXk6IHN0cmluZyxcbiAgICBwYXNzcGhyYXNlOiBzdHJpbmcsXG4gICAgY2VydENoYWluOiBzdHJpbmcsXG4gICkge1xuICAgIHRoaXMuY2VydCA9IGNlcnQ7XG4gICAgdGhpcy5rZXkgPSBrZXk7XG4gICAgdGhpcy5wYXNzcGhyYXNlID0gcGFzc3BocmFzZTtcbiAgICB0aGlzLmNlcnRDaGFpbiA9IGNlcnRDaGFpbjtcbiAgfVxuXG4gIHB1YmxpYyBhc3luYyB0b1BrY3MxMihwYXNzcGhyYXNlOiBzdHJpbmcpOiBQcm9taXNlPEJ1ZmZlcj4ge1xuICAgIGNvbnN0IHRtcERpciA9IGF3YWl0IGZzLnByb21pc2VzLm1rZHRlbXAocGF0aC5qb2luKG9zLnRtcGRpcigpLCAndG1wLicpKTtcbiAgICB0cnkge1xuICAgICAgY29uc3QgY3J0RmlsZU5hbWU6IHN0cmluZyA9IHBhdGguam9pbih0bXBEaXIsICdpbnB1dC5jcnQnKTtcbiAgICAgIGNvbnN0IGtleUZpbGVOYW1lOiBzdHJpbmcgPSBwYXRoLmpvaW4odG1wRGlyLCAnaW5wdXQua2V5Jyk7XG4gICAgICBjb25zdCBjYUNlcnQgPSB0aGlzLmNlcnRDaGFpbiA/IHRoaXMuY2VydCArIHRoaXMuY2VydENoYWluIDogdGhpcy5jZXJ0O1xuICAgICAgYXdhaXQgd3JpdGVBc2NpaUZpbGUoY3J0RmlsZU5hbWUsIGNhQ2VydCk7XG4gICAgICBhd2FpdCB3cml0ZUFzY2lpRmlsZShrZXlGaWxlTmFtZSwgdGhpcy5rZXkpO1xuXG4gICAgICBjb25zdCBwa2NzMTJGaWxlTmFtZTogc3RyaW5nID0gcGF0aC5qb2luKHRtcERpciwgJ2NlcnQucDEyJyk7XG4gICAgICBjb25zdCBjb21tYW5kOiBzdHJpbmcgPSAnb3BlbnNzbCBwa2NzMTIgLWV4cG9ydCAtbm9kZXMgLXBhc3NpbiBlbnY6UEFTU0lOIC1wYXNzb3V0IGVudjpQQVNTT1VUICcgK1xuICAgICAgICBgLW91dCAke3BrY3MxMkZpbGVOYW1lfSAtaW5rZXkgJHtrZXlGaWxlTmFtZX0gLWluICR7Y3J0RmlsZU5hbWV9YDtcbiAgICAgIGF3YWl0IGV4ZWMoXG4gICAgICAgIGNvbW1hbmQsXG4gICAgICAgIHsgZW52OiB7XG4gICAgICAgICAgUEFTU0lOOiB0aGlzLnBhc3NwaHJhc2UsXG4gICAgICAgICAgUEFTU09VVDogcGFzc3BocmFzZSxcbiAgICAgICAgICBQQVRIOiBwcm9jZXNzLmVudi5QQVRILFxuICAgICAgICB9fSxcbiAgICAgICk7XG5cbiAgICAgIGNvbnN0IHBrY3MxMkRhdGE6IEJ1ZmZlciA9IGF3YWl0IHJlYWRCaW5hcnlGaWxlKHBrY3MxMkZpbGVOYW1lKTtcblxuICAgICAgcmV0dXJuIHBrY3MxMkRhdGE7XG4gICAgfSBmaW5hbGx5IHtcbiAgICAgIGNvbnN0IHVubGlua3M6IEFycmF5PFByb21pc2U8dm9pZD4+ID0gW107XG4gICAgICBjb25zdCBmaWxlbmFtZXM6IHN0cmluZ1tdID0gYXdhaXQgZnMucHJvbWlzZXMucmVhZGRpcih0bXBEaXIpO1xuICAgICAgZm9yIChjb25zdCBmaWxlIG9mIGZpbGVuYW1lcykge1xuICAgICAgICB1bmxpbmtzLnB1c2goZnMucHJvbWlzZXMudW5saW5rKHBhdGguam9pbih0bXBEaXIsIGZpbGUpKSk7XG4gICAgICB9XG4gICAgICBhd2FpdCBQcm9taXNlLmFsbCh1bmxpbmtzKTtcbiAgICAgIGF3YWl0IGZzLnByb21pc2VzLnJtZGlyKHRtcERpcik7XG4gICAgfVxuICB9XG59XG4iXX0=