"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Secret = exports.sanitizeSecretName = void 0;
const validation_1 = require("./validation");
function sanitizeSecretName(name) {
    // The secret name must be ASCII letters, digits, or the following characters : /_+=.@-
    return name.replace(/[^A-Za-z0-9/_+=.@-]/g, '');
}
exports.sanitizeSecretName = sanitizeSecretName;
class Secret {
    constructor(arn, client) {
        this.client = client;
        this.arn = arn;
    }
    static fromArn(arn, client) {
        if (!validation_1.isArn(arn)) {
            throw Error(`Not a Secret ARN: ${arn}`);
        }
        return new Secret(arn, client);
    }
    /**
     * Creates a Secret with the given name and, optionally, containing the given 'data'. Tags the
     * secret with the given 'tags' if provided.
     * @param args
     * @throws Error if the request fails.
     * @returns A new Secret object for the newly created secret.
     */
    static async create(args) {
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/SecretsManager.html#createSecret-property
        const request = {
            Name: args.name,
            Description: args.description,
            KmsKeyId: args.encryptionKey?.arn,
            Tags: args.tags,
            SecretString: (typeof args.data === 'string') ? args.data : undefined,
            SecretBinary: Buffer.isBuffer(args.data) ? args.data : undefined,
        };
        try {
            const response = await args.client.createSecret(request).promise();
            console.debug(`CreateSecret response: ${JSON.stringify(response)}`);
            if (response.ARN) {
                return Secret.fromArn(response.ARN, args.client);
            }
            return undefined;
        }
        catch (e) {
            throw new Error(`CreateSecret '${args.name}' failed in region '${args.client.config.region}': ` +
                `${e.code} -- ${e.message}`);
        }
    }
    /**
     * Deletes this Secret in AWS SecretsManager
     * @param force If true, then force the delete with no recovery.
     * @throws Error if the request fails.
     */
    async delete(force) {
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/SecretsManager.html#deleteSecret-property
        if (!this.arn) {
            throw Error('Secret has already been deleted');
        }
        const request = {
            SecretId: this.arn,
            ForceDeleteWithoutRecovery: force,
        };
        try {
            console.debug(`Deleting Secret: ${this.arn}`);
            const response = await this.client.deleteSecret(request).promise();
            console.debug(`DeleteSecret response: ${JSON.stringify(response)}`);
            this.arn = undefined;
        }
        catch (e) {
            throw new Error(`DeleteSecret '${this.arn}' failed in region '${this.client.config.region}':` +
                `${e.code} -- ${e.message}`);
        }
    }
    /**
     * Store the given 'data' in the Secret. Text is stored in the 'SecretString'
     * of the Secret, whereas bytes are stored base64-encoded in the 'SecretBinary'
     * of the Secret.
     * @param data
     * @throws Error if the request fails.
     */
    async putValue(data) {
        if (!this.arn) {
            throw Error('Secret has been deleted');
        }
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/SecretsManager.html#putSecretValue-property
        const request = {
            SecretId: this.arn,
            SecretString: (typeof data === 'string') ? data : undefined,
            SecretBinary: Buffer.isBuffer(data) ? data : undefined,
        };
        try {
            const response = await this.client.putSecretValue(request).promise();
            console.debug(`PutSecret response: ${JSON.stringify(response)}`);
        }
        catch (e) {
            throw new Error(`PutSecret '${this.arn}' failed in region '${this.client.config.region}':` +
                `${e.code} -- ${e.message}`);
        }
    }
    /**
     * Get the latest version of the data stored in the secret.
     */
    async getValue() {
        if (!this.arn) {
            throw Error('Secret has been deleted');
        }
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/SecretsManager.html#getSecretValue-property
        const request = {
            SecretId: this.arn,
        };
        try {
            const response = await this.client.getSecretValue(request).promise();
            if (response.SecretBinary) {
                // SecretBinary can be: Buffer|Uint8Array|Blob|string
                const data = response.SecretBinary;
                if (Buffer.isBuffer(data)) {
                    return data;
                }
                else if (typeof data === 'string') {
                    return Buffer.from(data, 'binary');
                }
                else if (ArrayBuffer.isView(data)) {
                    return Buffer.from(data);
                }
                else {
                    throw new Error('Unknown type for SecretBinary data');
                }
            }
            return response.SecretString;
        }
        catch (e) {
            throw new Error(`GetSecret '${this.arn}' failed in region '${this.client.config.region}':` +
                `${e.code} -- ${e.message}`);
        }
    }
}
exports.Secret = Secret;
Secret.API_VERSION = '2017-10-17';
//# sourceMappingURL=data:application/json;base64,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