"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotEventPluginClient = exports.CollectionType = void 0;
/**
 * A type of collection to get/recive from Deadline.
 */
var CollectionType;
(function (CollectionType) {
    CollectionType["Pool"] = "pool";
    CollectionType["Group"] = "group";
})(CollectionType = exports.CollectionType || (exports.CollectionType = {}));
/**
 * Provides a simple interface to send requests to the Render Queue API related to the Deadline Spot Event Plugin.
 */
class SpotEventPluginClient {
    constructor(client) {
        this.deadlineClient = client;
    }
    async saveServerData(config) {
        console.log('Saving server data configuration:');
        console.log(config);
        try {
            // Get the concurrency token required to save server data
            const concurrencyToken = await this.concurrencyToken();
            await this.deadlineClient.PostRequest('/rcs/v1/putServerData', {
                ServerData: [
                    {
                        ID: SpotEventPluginClient.EVENT_PLUGIN_ID,
                        ServerDataDictionary: {
                            Config: config,
                        },
                        ConcurrencyToken: concurrencyToken,
                    },
                ],
            }, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            return true;
        }
        catch (e) {
            console.error(`Failed to save server data. Reason: ${e}`);
            return false;
        }
    }
    async configureSpotEventPlugin(configs) {
        console.log('Saving plugin configuration:');
        console.log(configs);
        try {
            await this.deadlineClient.PostRequest('/db/plugins/event/config/save', {
                ID: 'spot',
                DebugLogging: false,
                DlInit: configs,
                Icon: null,
                Limits: [],
                Meta: [],
                Name: 'Spot',
                PluginEnabled: 1,
            }, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            return true;
        }
        catch (e) {
            console.error(`Failed to save plugin configuration. Reason: ${e}`);
            return false;
        }
    }
    async addGroups(newGroups) {
        if (newGroups && newGroups.length) {
            const deadlineGroups = await this.getCollection(CollectionType.Group);
            if (deadlineGroups) {
                const newDeadlineGroups = deadlineGroups.Pools
                    .concat(newGroups
                    .filter(group => !deadlineGroups.Pools.includes(group)));
                return await this.saveCollection({
                    Pools: newDeadlineGroups,
                    ObsoletePools: deadlineGroups.ObsoletePools,
                }, CollectionType.Group);
            }
            return false;
        }
        return true;
    }
    async addPools(newPools) {
        if (newPools && newPools.length) {
            const deadlinePools = await this.getCollection(CollectionType.Pool);
            if (deadlinePools) {
                const newDeadlinePools = deadlinePools.Pools
                    .concat(newPools
                    .filter(pool => !deadlinePools.Pools.includes(pool)));
                return await this.saveCollection({
                    Pools: newDeadlinePools,
                    ObsoletePools: deadlinePools.ObsoletePools,
                }, CollectionType.Pool);
            }
            return false;
        }
        return true;
    }
    async getCollection(type) {
        console.log(`Getting ${type} collection:`);
        try {
            const response = await this.deadlineClient.GetRequest(`/db/settings/collections/${type}s?invalidateCache=true`, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            const deadlinePools = response.data;
            if (!deadlinePools.Pools || !Array.isArray(deadlinePools.Pools)) {
                console.error(`Failed to receive a ${type} collection. Invalid response: ${JSON.stringify(response.data)}.`);
                return undefined;
            }
            return deadlinePools;
        }
        catch (e) {
            console.error(`Failed to get ${type} collection. Reason: ${e.message}`);
            return undefined;
        }
    }
    async saveCollection(pools, type) {
        console.log(`Saving ${type} collection:`);
        console.log(pools);
        try {
            await this.deadlineClient.PostRequest(`/db/settings/collections/${type}s/save`, {
                Pools: pools.Pools,
                ObsoletePools: pools.ObsoletePools,
            }, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            return true;
        }
        catch (e) {
            console.error(`Failed to save ${type} collection. Reason: ${e.message}`);
            return false;
        }
    }
    async describeServerData() {
        return await this.deadlineClient.PostRequest('/rcs/v1/describeServerData', {
            ServerDataIds: [
                SpotEventPluginClient.EVENT_PLUGIN_ID,
            ],
        }, {
            headers: {
                'Content-Type': 'application/json; charset=utf-8',
            },
        });
    }
    /**
     * Requests a concurrencyToken required to save spot fleet request configuration.
     * If data already exists under the ID, an existing ConcurrencyToken has to be used.
     * First obtain the token and then save the data with the same ConcurrencyToken.
     * If there is no data under the ID, then real token is not required,
     * but the ConcurrencyToken property still has to be set.
     * NOTE:
     * saveServerData() will have a ConcurrencyToken in its response but we do not use it,
     * instead we always call this function to get a latest token.
     */
    async concurrencyToken() {
        const response = await this.describeServerData();
        const describedData = response.data;
        if (!describedData.ServerData || !Array.isArray(describedData.ServerData)) {
            throw new Error(`Failed to receive a ConcurrencyToken. Invalid response: ${describedData}.`);
        }
        const found = describedData.ServerData.find(element => element.ID === SpotEventPluginClient.EVENT_PLUGIN_ID);
        return found?.ConcurrencyToken ?? '';
    }
}
exports.SpotEventPluginClient = SpotEventPluginClient;
SpotEventPluginClient.EVENT_PLUGIN_ID = 'event.plugin.spot';
//# sourceMappingURL=data:application/json;base64,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