"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const sinon = require("sinon");
const backoff_generator_1 = require("../backoff-generator");
describe('BackoffGenerator', () => {
    const base = 100;
    beforeEach(() => {
        // Jest fake timers were upgraded from v26 to v27 and lots of our tests break.
        // Moving from the legacy timers to the modern ones breaks most of these tests, so for now
        // we're forcing the use of the old ones.
        jest.useFakeTimers('legacy');
        jest.spyOn(global, 'setTimeout');
    });
    afterAll(() => {
        jest.useRealTimers();
    });
    test('throws when jitterDivisor is invalid', () => {
        // GIVEN
        const jitterDivisor = 0;
        // WHEN
        expect(() => new backoff_generator_1.BackoffGenerator({
            jitterDivisor,
        }))
            // THEN
            .toThrow(`jitterDivisor must be greater than or equal to 1, got: ${jitterDivisor}`);
    });
    test('stops when max attempts is reached', async () => {
        // GIVEN
        const maxAttempts = 2;
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base,
            maxAttempts,
        });
        // WHEN
        let retvals = [];
        for (let i = 0; i < maxAttempts; i++) {
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base * Math.pow(2, i));
            const retval = await promise;
            retvals.push(retval);
        }
        // THEN
        retvals.slice(0, -1).forEach(retval => expect(retval).toBe(true));
        expect(retvals[retvals.length - 1]).toBe(false);
        expect(backoffGenerator.shouldContinue()).toBe(false);
    });
    test('stops when max cumulative backoff time is reached', async () => {
        // GIVEN
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base,
            maxCumulativeBackoffTimeMs: base,
        });
        // WHEN
        const promise = backoffGenerator.backoff();
        jest.advanceTimersByTime(base);
        const retval = await promise;
        // THEN
        expect(retval).toBe(false);
        expect(backoffGenerator.shouldContinue()).toBe(false);
    });
    test('respects max interval between backoffs', async () => {
        // GIVEN
        const maxIntervalMs = base / 2;
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base,
            maxIntervalMs,
        });
        // WHEN
        const promise = backoffGenerator.backoff();
        jest.advanceTimersByTime(maxIntervalMs);
        await promise;
        // THEN
        expect(maxIntervalMs).toBeLessThan(base);
        expect(setTimeout).toHaveBeenCalledTimes(1);
        expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), maxIntervalMs);
    });
    test('forces backoff', async () => {
        // GIVEN
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base,
            maxAttempts: 0,
        });
        if (backoffGenerator.shouldContinue() !== false) {
            throw new Error('BackoffGenerator.shouldContinue did not return false when it was expected to. Please fix this unit test.');
        }
        // WHEN
        const promise = backoffGenerator.backoff(true);
        jest.advanceTimersByTime(base);
        await promise;
        // THEN
        expect(setTimeout).toHaveBeenCalledTimes(1);
        expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), base);
    });
    describe('.restart()', () => {
        test('resets the number of attempts', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                maxAttempts: 1,
            });
            // This reaches maxAttempts, shouldContinue() will return false
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base);
            await promise;
            if (backoffGenerator.shouldContinue() !== false) {
                throw new Error('BackoffGenerator.shouldContinue did not return false when it was expected to. Please fix this unit test.');
            }
            // WHEN
            backoffGenerator.restart();
            // THEN
            expect(backoffGenerator.shouldContinue()).toBe(true);
        });
        test('resets the cumulative backoff time', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                maxCumulativeBackoffTimeMs: base,
            });
            // This reaches maxCumulativeBackoffTimeMs, shouldContinue() will return false
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base);
            await promise;
            if (backoffGenerator.shouldContinue() !== false) {
                throw new Error('BackoffGenerator.shouldContinue did not return false when it was expected to. Please fix this unit test.');
            }
            // WHEN
            backoffGenerator.restart();
            // THEN
            expect(backoffGenerator.shouldContinue()).toBe(true);
        });
    });
    describe('backs off and continues', () => {
        test('when there are remaining attempts', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                maxAttempts: 2,
            });
            // WHEN
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base);
            const retval = await promise;
            // THEN
            // We have one more attempt left, it should continue
            expect(retval).toBe(true);
            expect(backoffGenerator.shouldContinue()).toBe(true);
        });
        test('when there is remaining cumulative backoff time', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                maxCumulativeBackoffTimeMs: base + 1,
            });
            // WHEN
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base);
            const retval = await promise;
            // THEN
            // We haven't reached max cumulative backoff time yet, it should continue
            expect(retval).toBe(true);
            expect(backoffGenerator.shouldContinue()).toBe(true);
        });
    });
    describe.each([
        0,
        0.25,
        0.5,
        0.75,
        1,
    ])('jitter (factor %d)', (factor) => {
        let randomStub;
        beforeAll(() => {
            randomStub = sinon.stub(Math, 'random').returns(factor);
        });
        afterAll(() => {
            randomStub.restore();
        });
        test('applies full jitter', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                jitterDivisor: 1,
            });
            const interval = base * factor;
            // WHEN
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(interval);
            await promise;
            // THEN
            expect(setTimeout).toHaveBeenCalledTimes(1);
            expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), interval);
        });
        test('correctly calculates jitter with divisor', async () => {
            // GIVEN
            const jitterDivisor = 4;
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                jitterDivisor,
            });
            const interval = (base - base / jitterDivisor) + Math.floor(base / jitterDivisor * factor);
            // WHEN
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(interval);
            await promise;
            // THEN
            expect(setTimeout).toHaveBeenCalledTimes(1);
            expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), interval);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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