"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackoffGenerator = void 0;
/**
 * Class to handle sleeping with exponential backoff.
 *
 * Reference: https://aws.amazon.com/blogs/architecture/exponential-backoff-and-jitter/
 */
class BackoffGenerator {
    constructor(props) {
        this.cumulativeBackoffTimeMs = 0;
        this.attempt = 0;
        this.maxCumulativeBackoffTimeMs = props?.maxCumulativeBackoffTimeMs;
        this.maxAttempts = props?.maxAttempts;
        this.base = props?.base ?? 200;
        this.maxIntervalMs = props?.maxIntervalMs ?? Number.MAX_SAFE_INTEGER;
        this.jitterDivisor = props?.jitterDivisor;
        if (this.jitterDivisor !== undefined && this.jitterDivisor < 1) {
            throw new Error(`jitterDivisor must be greater than or equal to 1, got: ${this.jitterDivisor}`);
        }
        // Initialize internal counters
        this.restart();
    }
    /**
     * Calculates the number of milliseconds to sleep based on the attempt count.
     * @param b The base value for the calculation.
     * @param attempt The attempt count.
     * @param maxIntervalMs The maximum interval between backoffs, in milliseconds.
     * @returns The number of milliseconds to sleep.
     */
    static calculateSleepMs(b, attempt, maxIntervalMs) {
        return Math.min(b * Math.pow(2, attempt), maxIntervalMs, Number.MAX_SAFE_INTEGER);
    }
    /**
     * Restarts the internal counters used by this class.
     */
    restart() {
        this.cumulativeBackoffTimeMs = 0;
        this.attempt = 0;
    }
    /**
     * Sleeps for an exponentially increasing time depending on how many times this class has backed off.
     * If `jitterDivisor` was provided, jitter will be applied to the backoff time.
     *
     * If any of the conditions to stop backing off are met, this method will not sleep and return false.
     * Otherwise, it sleeps and returns true.
     * @param force Force sleeping, regardless of the conditions that indicate when to stop backing off.
     */
    async backoff(force) {
        let interval = BackoffGenerator.calculateSleepMs(this.base, this.attempt, this.maxIntervalMs);
        if (this.jitterDivisor !== undefined) {
            interval = (interval - interval / this.jitterDivisor) + (Math.floor(interval / this.jitterDivisor * Math.random()));
        }
        if (force || this.shouldContinue()) {
            await sleep(interval);
            this.cumulativeBackoffTimeMs += interval;
            this.attempt++;
        }
        return this.shouldContinue();
    }
    /**
     * Returns true if either the maximum number of attempts or maximum cumulative backoff time has been reached.
     * If neither are specified, this will always return true.
     */
    shouldContinue() {
        return (this.maxAttempts === undefined || this.attempt < this.maxAttempts) &&
            (this.maxCumulativeBackoffTimeMs === undefined || this.cumulativeBackoffTimeMs < this.maxCumulativeBackoffTimeMs);
    }
}
exports.BackoffGenerator = BackoffGenerator;
async function sleep(ms) {
    return new Promise(res => setTimeout(res, ms));
}
//# sourceMappingURL=data:application/json;base64,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