"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
function sleep(timeout) {
    return new Promise((resolve) => {
        setTimeout(() => { resolve(); }, timeout);
    });
}
/**
 * Make sure that a task completes successfully. This does increase the length of time the Lambda runs by
 * quite a bit, but Lambdas are cheap and we want to know if our logs are exporting properly.
 */
async function confirmTaskCompletion(taskId) {
    const cloudwatchlogs = new aws_sdk_1.CloudWatchLogs({ apiVersion: '2014-03-28' });
    let errorCount = 0;
    let complete = false;
    while (!complete) {
        try {
            const response = await cloudwatchlogs.describeExportTasks({ taskId }).promise();
            if (response.$response.error) {
                throw new Error(`Task ${taskId} failed with message: ${response.$response.error.message}`);
            }
            if (response.exportTasks?.length !== 1) {
                throw new Error(`Received ${response.exportTasks?.length} export tasks from DescribeExportTasks for task ${taskId}.`);
            }
            const task = response.exportTasks[0];
            if (!task.status || !task.status.code) {
                throw new Error(`Task ${taskId} did not return a status code.`);
            }
            const taskStatus = task.status.code;
            if (taskStatus === 'RUNNING' || taskStatus.indexOf('PENDING') !== -1) {
                await sleep(500);
            }
            else if (taskStatus === 'FAILED' || taskStatus === 'CANCELLED') {
                throw new Error(`Task ${taskId} failed with status code: ${taskStatus}`);
            }
            else {
                console.log(`${taskId}: Task has completed successfully!`);
                complete = true;
            }
        }
        catch (e) {
            // Retry 3 times before giving up
            if (errorCount < 3) {
                console.error(`${taskId}: Encountered failure #${errorCount} with message: ${e.message}`);
                errorCount++;
            }
            else {
                throw e;
            }
        }
    }
}
function getDatePath(dateObj) {
    const year = dateObj.getFullYear();
    const month = dateObj.getMonth() + 1;
    const date = dateObj.getDate();
    return `${year}/${month}/${date}`;
}
async function exportToS3Task(bucketName, exportFrequencyInHours, logGroupName, retentionInHours) {
    const cloudwatchlogs = new aws_sdk_1.CloudWatchLogs({ apiVersion: '2014-03-28' });
    // End time is now minus the retention period in CloudWatch plus one hour. This creates an extra hour buffer to
    // make sure no logs expire before they get exported.
    const endTime = new Date();
    endTime.setHours(endTime.getHours() - retentionInHours + 1);
    // Start time is the end time minus the frequency that the Lambda is run, with an extra minute taken off to account
    // for any drift in Lambda execution times between runs.
    const startTime = new Date();
    startTime.setHours(endTime.getHours() - exportFrequencyInHours);
    startTime.setMinutes(startTime.getMinutes() - 1);
    const destinationPrefix = `${logGroupName}/${getDatePath(new Date())}`;
    const params = {
        destination: bucketName,
        destinationPrefix,
        from: startTime.getTime(),
        logGroupName,
        to: endTime.getTime(),
    };
    const response = await cloudwatchlogs.createExportTask(params).promise();
    if (response.$response.error) {
        throw new Error(response.$response.error.message);
    }
    if (response.taskId) {
        console.log(`${response.taskId}: Successfully created export task for ${logGroupName}.`);
        console.log(`Exporting into ${bucketName} from ${startTime} to ${endTime}.`);
        await confirmTaskCompletion(response.taskId);
    }
    else {
        throw new Error(`For logGroup ${logGroupName}, No error thrown for CreateExportTask, but no task ID was received.`);
    }
}
/**
 * Lambda for exporting logs from CloudWatch to S3.
 *
 * @param event.BucketName The name of the S3 bucket to export the logs into
 * @param event.ExportFrequencyInHours How often this Lambda runs
 * @param event.LogGroupName The name of the LogGroup to export the logs from
 * @param event.RetentionInHours The retention period for logs in CloudWatch
 */
async function handler(event) {
    await exportToS3Task(event.BucketName, event.ExportFrequencyInHours, event.LogGroupName, event.RetentionInHours);
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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