"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const AWS = require("aws-sdk");
const deadline_1 = require("../../../../deadline");
const handler_1 = require("../handler");
jest.mock('../../lib/secrets-manager/read-certificate');
const secretPartialArn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert';
describe('SEPConfiguratorResource', () => {
    const deadlineGroup = 'group_name';
    const deadlinePool = 'pool_name';
    let app;
    let stack;
    let validSepConfiguration;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'Stack');
        const launchTemplate = new aws_ec2_1.LaunchTemplate(stack, 'LaunchTemplate');
        validSepConfiguration = {
            spotPluginConfigurations: {
                AWSInstanceStatus: deadline_1.SpotEventPluginDisplayInstanceStatus.DISABLED,
                DeleteInterruptedSlaves: true,
                DeleteTerminatedSlaves: true,
                IdleShutdown: 20,
                Logging: deadline_1.SpotEventPluginLoggingLevel.STANDARD,
                PreJobTaskMode: deadline_1.SpotEventPluginPreJobTaskMode.CONSERVATIVE,
                Region: 'us-west-2',
                ResourceTracker: true,
                StaggerInstances: 50,
                State: deadline_1.SpotEventPluginState.GLOBAL_ENABLED,
                StrictHardCap: true,
            },
            connection: {
                hostname: 'internal-hostname.com',
                protocol: 'HTTPS',
                port: '4433',
                caCertificateArn: secretPartialArn,
            },
            spotFleetRequestConfigurations: {
                [deadlineGroup]: {
                    AllocationStrategy: deadline_1.SpotFleetAllocationStrategy.CAPACITY_OPTIMIZED,
                    IamFleetRole: 'roleArn',
                    // Explicitly provide empty array for testing comparisons since we inject this for compatibility with SEP
                    LaunchSpecifications: [],
                    LaunchTemplateConfigs: [{
                            LaunchTemplateSpecification: {
                                LaunchTemplateId: launchTemplate.launchTemplateId,
                                LaunchTemplateName: launchTemplate.launchTemplateName,
                                Version: launchTemplate.versionNumber,
                            },
                            Overrides: [],
                        }],
                    ReplaceUnhealthyInstances: true,
                    TargetCapacity: 1,
                    TerminateInstancesWithExpiration: true,
                    Type: deadline_1.SpotFleetRequestType.MAINTAIN,
                    ValidUntil: core_1.Expiration.atDate(new Date(2022, 11, 17)).date.toISOString(),
                    TagSpecifications: [{
                            ResourceType: deadline_1.SpotFleetResourceType.SPOT_FLEET_REQUEST,
                            Tags: [
                                {
                                    Key: 'name',
                                    Value: 'test',
                                },
                            ],
                        }],
                },
            },
            deadlineGroups: [deadlineGroup],
            deadlinePools: [deadlinePool],
        };
    });
    describe('doCreate', () => {
        let handler;
        let mockSpotEventPluginClient;
        beforeEach(() => {
            mockSpotEventPluginClient = {
                saveServerData: jest.fn((_a) => Promise.resolve(true)),
                configureSpotEventPlugin: jest.fn((_a) => Promise.resolve(true)),
                addGroups: jest.fn((_a) => Promise.resolve(true)),
                addPools: jest.fn((_a) => Promise.resolve(true)),
            };
            handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockReturnValue(Promise.resolve('BEGIN CERTIFICATE'));
            async function returnSpotEventPluginClient(_v1) {
                return mockSpotEventPluginClient;
            }
            // eslint-disable-next-line dot-notation
            handler['spotEventPluginClient'] = jest.fn((a) => returnSpotEventPluginClient(a));
        });
        afterEach(() => {
            jest.clearAllMocks();
        });
        test('does not save server data when no configurations are provided', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            // WHEN
            const result = await handler.doCreate('physicalId', {
                connection: validSepConfiguration.connection,
            });
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(0);
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(0);
        });
        test('save spot fleet request configs', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const result = await handler.doCreate('physicalId', {
                ...validSepConfiguration,
                spotPluginConfigurations: undefined,
            });
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(1);
            expect(mockSaveServerData).toBeCalledWith(JSON.stringify(validSepConfiguration.spotFleetRequestConfigurations));
        });
        test('save spot event plugin configs', async () => {
            // GIVEN
            const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            const configs = [];
            for (const [key, value] of Object.entries(validSepConfiguration.spotPluginConfigurations)) {
                configs.push({
                    Key: key,
                    Value: value,
                });
            }
            const securitySettings = [{
                    Key: 'UseLocalCredentials',
                    Value: true,
                },
                {
                    Key: 'NamedProfile',
                    Value: '',
                }];
            // WHEN
            const result = await handler.doCreate('physicalId', {
                ...validSepConfiguration,
                spotFleetRequestConfigurations: undefined,
            });
            // THEN
            expect(result).toBeUndefined();
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(1);
            expect(mockConfigureSpotEventPlugin.mock.calls[0][0]).toEqual([...configs, ...securitySettings]);
        });
        test('save server data', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const result = await handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(1);
            expect(mockSaveServerData.mock.calls[0][0]).toEqual(JSON.stringify(validSepConfiguration.spotFleetRequestConfigurations));
        });
        test('configure spot event plugin', async () => {
            // GIVEN
            const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            const configs = [];
            for (const [key, value] of Object.entries(validSepConfiguration.spotPluginConfigurations)) {
                configs.push({
                    Key: key,
                    Value: value,
                });
            }
            const securitySettings = [{
                    Key: 'UseLocalCredentials',
                    Value: true,
                },
                {
                    Key: 'NamedProfile',
                    Value: '',
                }];
            // WHEN
            await handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(1);
            expect(mockConfigureSpotEventPlugin.mock.calls[0][0]).toEqual([...configs, ...securitySettings]);
        });
        test('create groups', async () => {
            // GIVEN
            const mockAddGroups = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.addGroups = mockAddGroups;
            // WHEN
            await handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            expect(mockAddGroups.mock.calls.length).toBe(1);
            expect(mockAddGroups).toHaveBeenCalledWith([deadlineGroup]);
        });
        test('create pools', async () => {
            // GIVEN
            const mockAddPools = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.addPools = mockAddPools;
            // WHEN
            await handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            expect(mockAddPools.mock.calls.length).toBe(1);
            expect(mockAddPools).toHaveBeenCalledWith([deadlinePool]);
        });
        test('throw when cannot add groups', async () => {
            // GIVEN
            mockSpotEventPluginClient.addGroups = jest.fn((_a) => Promise.resolve(false));
            // WHEN
            const promise = handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(`Failed to add Deadline group(s) ${validSepConfiguration.deadlineGroups}`);
        });
        test('throw when cannot add pools', async () => {
            // GIVEN
            mockSpotEventPluginClient.addPools = jest.fn((_a) => Promise.resolve(false));
            // WHEN
            const promise = handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(`Failed to add Deadline pool(s) ${validSepConfiguration.deadlinePools}`);
        });
        test('throw when cannot save spot fleet request configs', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((_a) => Promise.resolve(false));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const promise = handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(/Failed to save spot fleet request with configuration/);
        });
        test('throw when cannot save spot event plugin configs', async () => {
            // GIVEN
            const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(false));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            // WHEN
            const promise = handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(/Failed to save Spot Event Plugin Configurations/);
        });
    });
    test('doDelete does not do anything', async () => {
        // GIVEN
        const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
        // WHEN
        const promise = await handler.doDelete('physicalId', {
            connection: validSepConfiguration.connection,
        });
        // THEN
        await expect(promise).toBeUndefined();
    });
    describe('.validateInput()', () => {
        describe('should return true', () => {
            test('with valid input', async () => {
                // GIVEN
                const input = validSepConfiguration;
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
            test('without spotPluginConfigurations', async () => {
                // GIVEN
                const input = {
                    ...validSepConfiguration,
                    spotPluginConfigurations: undefined,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
            test('without spotFleetRequestConfigurations', async () => {
                // GIVEN
                const input = {
                    ...validSepConfiguration,
                    spotFleetRequestConfigurations: undefined,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
            test('with only connection', async () => {
                // GIVEN
                const input = {
                    connection: validSepConfiguration.connection,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
        });
        // Invalid connection
        const noProtocolConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            port: '4433',
        };
        const noHostnameConnection = {
            protocol: 'HTTPS',
            port: '4433',
        };
        const noPortConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            caCertificateArn: secretPartialArn,
        };
        const invalidHostnameConnection = {
            hostname: 10,
            protocol: 'HTTPS',
            port: '4433',
        };
        const invalidProtocolConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'TCP',
            port: '4433',
        };
        const invalidProtocolTypeConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: ['HTTPS'],
            port: '4433',
        };
        const invalidPortTypeConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: 4433,
        };
        const invalidPortRange1Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '-1',
        };
        const invalidPortRange2Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '65536',
        };
        const invalidPortRange3Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: Number.NaN.toString(),
        };
        const invalidCaCertConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '4433',
            caCertificateArn: 'notArn',
        };
        describe('should return false if', () => {
            test.each([
                noProtocolConnection,
                noHostnameConnection,
                noPortConnection,
                invalidCaCertConnection,
                invalidHostnameConnection,
                invalidProtocolConnection,
                invalidProtocolTypeConnection,
                invalidPortTypeConnection,
                invalidPortRange1Connection,
                invalidPortRange2Connection,
                invalidPortRange3Connection,
                undefined,
                [],
            ])('invalid connection', (invalidConnection) => {
                // GIVEN
                const input = {
                    spotPluginConfigurations: validSepConfiguration.spotPluginConfigurations,
                    connection: invalidConnection,
                    spotFleetRequestConfigurations: validSepConfiguration.spotFleetRequestConfigurations,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeFalsy();
            });
            test.each([
                undefined,
                [],
                '',
            ])('{input=%s}', (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeFalsy();
            });
        });
    });
    describe('.isSecretArnOrUndefined()', () => {
        describe('should return true if', () => {
            test.each([
                secretPartialArn,
                undefined,
            ])('{input=%s}', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                // eslint-disable-next-line dot-notation
                const returnValue = handler['isSecretArnOrUndefined'](input);
                expect(returnValue).toBeTruthy();
            });
        });
        describe('should return false if', () => {
            test.each([
                'any string',
                10,
                [],
            ])('{input=%s}', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                // eslint-disable-next-line dot-notation
                const returnValue = handler['isSecretArnOrUndefined'](input);
                expect(returnValue).toBeFalsy();
            });
        });
    });
    describe('.spotEventPluginClient()', () => {
        test('creates a valid object with http', async () => {
            // GIVEN
            const validHTTPConnection = {
                hostname: 'internal-hostname.com',
                protocol: 'HTTP',
                port: '8080',
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            // eslint-disable-next-line dot-notation
            const result = await handler['spotEventPluginClient'](validHTTPConnection);
            expect(result).toBeDefined();
        });
        test('creates a valid object with https', async () => {
            // GIVEN
            const validHTTPSConnection = {
                hostname: 'internal-hostname.com',
                protocol: 'HTTP',
                port: '8080',
                caCertificateArn: secretPartialArn,
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockReturnValue(Promise.resolve('BEGIN CERTIFICATE'));
            // eslint-disable-next-line dot-notation
            const result = await handler['spotEventPluginClient'](validHTTPSConnection);
            expect(result).toBeDefined();
        });
    });
    describe('.toKeyValueArray()', () => {
        test('converts to array of key value pairs', () => {
            // GIVEN
            const pluginConfig = {
                AWSInstanceStatus: 'Disabled',
            };
            const expectedResult = {
                Key: 'AWSInstanceStatus',
                Value: 'Disabled',
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            // eslint-disable-next-line dot-notation
            const returnValue = handler['toKeyValueArray'](pluginConfig);
            // THEN
            expect(returnValue).toContainEqual(expectedResult);
        });
        test('throws with undefined values', () => {
            // GIVEN
            const pluginConfig = {
                AWSInstanceStatus: undefined,
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            function toKeyValueArray() {
                // eslint-disable-next-line dot-notation
                handler['toKeyValueArray'](pluginConfig);
            }
            // THEN
            expect(toKeyValueArray).toThrowError();
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaGFuZGxlci50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaGFuZGxlci50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7O0dBR0c7O0FBRUgsOENBRTBCO0FBQzFCLHdDQUl1QjtBQUN2QiwrQkFBK0I7QUFDL0IsbURBUThCO0FBQzlCLHdDQUFxRDtBQU9yRCxJQUFJLENBQUMsSUFBSSxDQUFDLDRDQUE0QyxDQUFDLENBQUM7QUFFeEQsTUFBTSxnQkFBZ0IsR0FBVyxvRUFBb0UsQ0FBQztBQUV0RyxRQUFRLENBQUMseUJBQXlCLEVBQUUsR0FBRyxFQUFFO0lBQ3ZDLE1BQU0sYUFBYSxHQUFHLFlBQVksQ0FBQztJQUNuQyxNQUFNLFlBQVksR0FBSSxXQUFXLENBQUM7SUFFbEMsSUFBSSxHQUFRLENBQUM7SUFDYixJQUFJLEtBQVksQ0FBQztJQUNqQixJQUFJLHFCQUFtRCxDQUFDO0lBRXhELFVBQVUsQ0FBQyxHQUFHLEVBQUU7UUFDZCxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztRQUNoQixLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ2hDLE1BQU0sY0FBYyxHQUFHLElBQUksd0JBQWMsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUVuRSxxQkFBcUIsR0FBRztZQUN0Qix3QkFBd0IsRUFBRTtnQkFDeEIsaUJBQWlCLEVBQUUsK0NBQW9DLENBQUMsUUFBUTtnQkFDaEUsdUJBQXVCLEVBQUUsSUFBSTtnQkFDN0Isc0JBQXNCLEVBQUUsSUFBSTtnQkFDNUIsWUFBWSxFQUFFLEVBQUU7Z0JBQ2hCLE9BQU8sRUFBRSxzQ0FBMkIsQ0FBQyxRQUFRO2dCQUM3QyxjQUFjLEVBQUUsd0NBQTZCLENBQUMsWUFBWTtnQkFDMUQsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLGVBQWUsRUFBRSxJQUFJO2dCQUNyQixnQkFBZ0IsRUFBRSxFQUFFO2dCQUNwQixLQUFLLEVBQUUsK0JBQW9CLENBQUMsY0FBYztnQkFDMUMsYUFBYSxFQUFFLElBQUk7YUFDcEI7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsUUFBUSxFQUFFLHVCQUF1QjtnQkFDakMsUUFBUSxFQUFFLE9BQU87Z0JBQ2pCLElBQUksRUFBRSxNQUFNO2dCQUNaLGdCQUFnQixFQUFFLGdCQUFnQjthQUNuQztZQUNELDhCQUE4QixFQUFFO2dCQUM5QixDQUFDLGFBQWEsQ0FBQyxFQUFFO29CQUNmLGtCQUFrQixFQUFFLHNDQUEyQixDQUFDLGtCQUFrQjtvQkFDbEUsWUFBWSxFQUFFLFNBQVM7b0JBQ3ZCLHlHQUF5RztvQkFDekcsb0JBQW9CLEVBQUUsRUFBRTtvQkFDeEIscUJBQXFCLEVBQUUsQ0FBQzs0QkFDdEIsMkJBQTJCLEVBQUU7Z0NBQzNCLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxnQkFBZ0I7Z0NBQ2pELGtCQUFrQixFQUFFLGNBQWMsQ0FBQyxrQkFBa0I7Z0NBQ3JELE9BQU8sRUFBRSxjQUFjLENBQUMsYUFBYTs2QkFDdEM7NEJBQ0QsU0FBUyxFQUFFLEVBQUU7eUJBQ2QsQ0FBQztvQkFDRix5QkFBeUIsRUFBRSxJQUFJO29CQUMvQixjQUFjLEVBQUUsQ0FBQztvQkFDakIsZ0NBQWdDLEVBQUUsSUFBSTtvQkFDdEMsSUFBSSxFQUFFLCtCQUFvQixDQUFDLFFBQVE7b0JBQ25DLFVBQVUsRUFBRSxpQkFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRTtvQkFDeEUsaUJBQWlCLEVBQUUsQ0FBQzs0QkFDbEIsWUFBWSxFQUFFLGdDQUFxQixDQUFDLGtCQUFrQjs0QkFDdEQsSUFBSSxFQUFFO2dDQUNKO29DQUNFLEdBQUcsRUFBRSxNQUFNO29DQUNYLEtBQUssRUFBRSxNQUFNO2lDQUNkOzZCQUNGO3lCQUNGLENBQUM7aUJBQ0g7YUFDRjtZQUNELGNBQWMsRUFBRSxDQUFDLGFBQWEsQ0FBQztZQUMvQixhQUFhLEVBQUUsQ0FBQyxZQUFZLENBQUM7U0FDOUIsQ0FBQztJQUNKLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLFVBQVUsRUFBRSxHQUFHLEVBQUU7UUFDeEIsSUFBSSxPQUFnQyxDQUFDO1FBQ3JDLElBQUkseUJBS0gsQ0FBQztRQUVGLFVBQVUsQ0FBQyxHQUFHLEVBQUU7WUFDZCx5QkFBeUIsR0FBRztnQkFDMUIsY0FBYyxFQUFFLElBQUksQ0FBQyxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUU7Z0JBQ3hELHdCQUF3QixFQUFFLElBQUksQ0FBQyxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUU7Z0JBQ2xFLFNBQVMsRUFBRSxJQUFJLENBQUMsRUFBRSxDQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFFO2dCQUNuRCxRQUFRLEVBQUUsSUFBSSxDQUFDLEVBQUUsQ0FBRSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBRTthQUNuRCxDQUFDO1lBRUYsT0FBTyxHQUFHLElBQUksaUNBQXVCLENBQUMsSUFBSSxHQUFHLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztZQUVoRSxJQUFJLENBQUMsV0FBVyxDQUFDLDRDQUE0QyxDQUFDLENBQUMsbUJBQW1CLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDO1lBRXpJLEtBQUssVUFBVSwyQkFBMkIsQ0FBQyxHQUFRO2dCQUNqRCxPQUFPLHlCQUF5QixDQUFDO1lBQ25DLENBQUM7WUFDRCx3Q0FBd0M7WUFDeEMsT0FBTyxDQUFDLHVCQUF1QixDQUFDLEdBQUcsSUFBSSxDQUFDLEVBQUUsQ0FBRSxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQyxDQUFDLENBQUUsQ0FBQztRQUN0RixDQUFDLENBQUMsQ0FBQztRQUVILFNBQVMsQ0FBQyxHQUFHLEVBQUU7WUFDYixJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7UUFDdkIsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsK0RBQStELEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDL0UsUUFBUTtZQUNSLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxDQUFDLEVBQUUsQ0FBRSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBRSxDQUFDO1lBQ3BFLHlCQUF5QixDQUFDLGNBQWMsR0FBRyxrQkFBa0IsQ0FBQztZQUM5RCxNQUFNLDRCQUE0QixHQUFHLElBQUksQ0FBQyxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUUsQ0FBQztZQUM5RSx5QkFBeUIsQ0FBQyx3QkFBd0IsR0FBRyw0QkFBNEIsQ0FBQztZQUVsRixPQUFPO1lBQ1AsTUFBTSxNQUFNLEdBQUcsTUFBTSxPQUFPLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRTtnQkFDbEQsVUFBVSxFQUFFLHFCQUFxQixDQUFDLFVBQVU7YUFDN0MsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUMvQixNQUFNLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDckQsTUFBTSxDQUFDLDRCQUE0QixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2pFLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEtBQUssSUFBSSxFQUFFO1lBQ2pELFFBQVE7WUFDUixNQUFNLGtCQUFrQixHQUFHLElBQUksQ0FBQyxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUUsQ0FBQztZQUNwRSx5QkFBeUIsQ0FBQyxjQUFjLEdBQUcsa0JBQWtCLENBQUM7WUFFOUQsT0FBTztZQUNQLE1BQU0sTUFBTSxHQUFHLE1BQU0sT0FBTyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7Z0JBQ2xELEdBQUcscUJBQXFCO2dCQUN4Qix3QkFBd0IsRUFBRSxTQUFTO2FBQ3BDLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDL0IsTUFBTSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3JELE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLHFCQUFxQixDQUFDLDhCQUE4QixDQUFDLENBQUMsQ0FBQztRQUNsSCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxLQUFLLElBQUksRUFBRTtZQUNoRCxRQUFRO1lBQ1IsTUFBTSw0QkFBNEIsR0FBRyxJQUFJLENBQUMsRUFBRSxDQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFFLENBQUM7WUFDOUUseUJBQXlCLENBQUMsd0JBQXdCLEdBQUcsNEJBQTRCLENBQUM7WUFFbEYsTUFBTSxPQUFPLEdBQWtDLEVBQUUsQ0FBQztZQUNsRCxLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQyx3QkFBK0IsQ0FBQyxFQUFFO2dCQUNoRyxPQUFPLENBQUMsSUFBSSxDQUFDO29CQUNYLEdBQUcsRUFBRSxHQUFHO29CQUNSLEtBQUssRUFBRSxLQUFLO2lCQUNiLENBQUMsQ0FBQzthQUNKO1lBRUQsTUFBTSxnQkFBZ0IsR0FBRyxDQUFDO29CQUN4QixHQUFHLEVBQUUscUJBQXFCO29CQUMxQixLQUFLLEVBQUUsSUFBSTtpQkFDWjtnQkFDRDtvQkFDRSxHQUFHLEVBQUUsY0FBYztvQkFDbkIsS0FBSyxFQUFFLEVBQUU7aUJBQ1YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sTUFBTSxHQUFHLE1BQU0sT0FBTyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7Z0JBQ2xELEdBQUcscUJBQXFCO2dCQUN4Qiw4QkFBOEIsRUFBRSxTQUFTO2FBQzFDLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDL0IsTUFBTSxDQUFDLDRCQUE0QixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQy9ELE1BQU0sQ0FBQyw0QkFBNEIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBRyxPQUFPLEVBQUUsR0FBRyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7UUFDbkcsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsa0JBQWtCLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDbEMsUUFBUTtZQUNSLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxDQUFDLEVBQUUsQ0FBRSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBRSxDQUFDO1lBQ3BFLHlCQUF5QixDQUFDLGNBQWMsR0FBRyxrQkFBa0IsQ0FBQztZQUU5RCxPQUFPO1lBQ1AsTUFBTSxNQUFNLEdBQUcsTUFBTSxPQUFPLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1lBRTNFLE9BQU87WUFDUCxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDL0IsTUFBTSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3JELE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMscUJBQXFCLENBQUMsOEJBQThCLENBQUMsQ0FBQyxDQUFDO1FBQzVILENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLDZCQUE2QixFQUFFLEtBQUssSUFBSSxFQUFFO1lBQzdDLFFBQVE7WUFDUixNQUFNLDRCQUE0QixHQUFHLElBQUksQ0FBQyxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUUsQ0FBQztZQUM5RSx5QkFBeUIsQ0FBQyx3QkFBd0IsR0FBRyw0QkFBNEIsQ0FBQztZQUVsRixNQUFNLE9BQU8sR0FBa0MsRUFBRSxDQUFDO1lBQ2xELEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLHdCQUErQixDQUFDLEVBQUU7Z0JBQ2hHLE9BQU8sQ0FBQyxJQUFJLENBQUM7b0JBQ1gsR0FBRyxFQUFFLEdBQUc7b0JBQ1IsS0FBSyxFQUFFLEtBQUs7aUJBQ2IsQ0FBQyxDQUFDO2FBQ0o7WUFFRCxNQUFNLGdCQUFnQixHQUFHLENBQUM7b0JBQ3hCLEdBQUcsRUFBRSxxQkFBcUI7b0JBQzFCLEtBQUssRUFBRSxJQUFJO2lCQUNaO2dCQUNEO29CQUNFLEdBQUcsRUFBRSxjQUFjO29CQUNuQixLQUFLLEVBQUUsRUFBRTtpQkFDVixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxPQUFPLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1lBRTVELE9BQU87WUFDUCxNQUFNLENBQUMsNEJBQTRCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDL0QsTUFBTSxDQUFDLDRCQUE0QixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLE9BQU8sRUFBRSxHQUFHLGdCQUFnQixDQUFDLENBQUMsQ0FBQztRQUNuRyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxlQUFlLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDL0IsUUFBUTtZQUNSLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUUsQ0FBQztZQUMvRCx5QkFBeUIsQ0FBQyxTQUFTLEdBQUcsYUFBYSxDQUFDO1lBRXBELE9BQU87WUFDUCxNQUFNLE9BQU8sQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFLHFCQUFxQixDQUFDLENBQUM7WUFFNUQsT0FBTztZQUNQLE1BQU0sQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEQsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDLG9CQUFvQixDQUFDLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztRQUM5RCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxjQUFjLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDOUIsUUFBUTtZQUNSLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUUsQ0FBQztZQUM5RCx5QkFBeUIsQ0FBQyxRQUFRLEdBQUcsWUFBWSxDQUFDO1lBRWxELE9BQU87WUFDUCxNQUFNLE9BQU8sQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFLHFCQUFxQixDQUFDLENBQUM7WUFFNUQsT0FBTztZQUNQLE1BQU0sQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDL0MsTUFBTSxDQUFDLFlBQVksQ0FBQyxDQUFDLG9CQUFvQixDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztRQUM1RCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxLQUFLLElBQUksRUFBRTtZQUM5QyxRQUFRO1lBQ1IseUJBQXlCLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUUsQ0FBQztZQUVoRixPQUFPO1lBQ1AsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUUscUJBQXFCLENBQUMsQ0FBQztZQUV0RSxPQUFPO1lBQ1AsTUFBTSxNQUFNLENBQUMsT0FBTyxDQUFDO2lCQUNsQixPQUFPO2lCQUNQLFlBQVksQ0FBQyxtQ0FBbUMscUJBQXFCLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUM3RixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyw2QkFBNkIsRUFBRSxLQUFLLElBQUksRUFBRTtZQUM3QyxRQUFRO1lBQ1IseUJBQXlCLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxFQUFFLENBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUUsQ0FBQztZQUUvRSxPQUFPO1lBQ1AsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUUscUJBQXFCLENBQUMsQ0FBQztZQUV0RSxPQUFPO1lBQ1AsTUFBTSxNQUFNLENBQUMsT0FBTyxDQUFDO2lCQUNsQixPQUFPO2lCQUNQLFlBQVksQ0FBQyxrQ0FBa0MscUJBQXFCLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQztRQUMzRixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxtREFBbUQsRUFBRSxLQUFLLElBQUksRUFBRTtZQUNuRSxRQUFRO1lBQ1IsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsRUFBRSxDQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFFLENBQUM7WUFDckUseUJBQXlCLENBQUMsY0FBYyxHQUFHLGtCQUFrQixDQUFDO1lBRTlELE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1lBRXRFLE9BQU87WUFDUCxNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUM7aUJBQ2xCLE9BQU87aUJBQ1AsWUFBWSxDQUFDLHNEQUFzRCxDQUFDLENBQUM7UUFDMUUsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsa0RBQWtELEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDbEUsUUFBUTtZQUNSLE1BQU0sNEJBQTRCLEdBQUcsSUFBSSxDQUFDLEVBQUUsQ0FBRSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBRSxDQUFDO1lBQy9FLHlCQUF5QixDQUFDLHdCQUF3QixHQUFHLDRCQUE0QixDQUFDO1lBRWxGLE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1lBRXRFLE9BQU87WUFDUCxNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUM7aUJBQ2xCLE9BQU87aUJBQ1AsWUFBWSxDQUFDLGlEQUFpRCxDQUFDLENBQUM7UUFDckUsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywrQkFBK0IsRUFBRSxLQUFLLElBQUksRUFBRTtRQUMvQyxRQUFRO1FBQ1IsTUFBTSxPQUFPLEdBQUcsSUFBSSxpQ0FBdUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO1FBRXRFLE9BQU87UUFDUCxNQUFNLE9BQU8sR0FBRyxNQUFNLE9BQU8sQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFO1lBQ25ELFVBQVUsRUFBRSxxQkFBcUIsQ0FBQyxVQUFVO1NBQzdDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztJQUN4QyxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLEVBQUU7UUFDaEMsUUFBUSxDQUFDLG9CQUFvQixFQUFFLEdBQUcsRUFBRTtZQUNsQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsS0FBSyxJQUFJLEVBQUU7Z0JBQ2xDLFFBQVE7Z0JBQ1IsTUFBTSxLQUFLLEdBQUcscUJBQXFCLENBQUM7Z0JBRXBDLE9BQU87Z0JBQ1AsTUFBTSxPQUFPLEdBQUcsSUFBSSxpQ0FBdUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO2dCQUN0RSxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUVqRCxPQUFPO2dCQUNQLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNuQyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxrQ0FBa0MsRUFBRSxLQUFLLElBQUksRUFBRTtnQkFDbEQsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBaUM7b0JBQzFDLEdBQUcscUJBQXFCO29CQUN4Qix3QkFBd0IsRUFBRSxTQUFTO2lCQUNwQyxDQUFDO2dCQUVGLE9BQU87Z0JBQ1AsTUFBTSxPQUFPLEdBQUcsSUFBSSxpQ0FBdUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO2dCQUN0RSxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUVqRCxPQUFPO2dCQUNQLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNuQyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyx3Q0FBd0MsRUFBRSxLQUFLLElBQUksRUFBRTtnQkFDeEQsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBaUM7b0JBQzFDLEdBQUcscUJBQXFCO29CQUN4Qiw4QkFBOEIsRUFBRSxTQUFTO2lCQUMxQyxDQUFDO2dCQUVGLE9BQU87Z0JBQ1AsTUFBTSxPQUFPLEdBQUcsSUFBSSxpQ0FBdUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO2dCQUN0RSxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUVqRCxPQUFPO2dCQUNQLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNuQyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxzQkFBc0IsRUFBRSxLQUFLLElBQUksRUFBRTtnQkFDdEMsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBaUM7b0JBQzFDLFVBQVUsRUFBRSxxQkFBcUIsQ0FBQyxVQUFVO2lCQUM3QyxDQUFDO2dCQUVGLE9BQU87Z0JBQ1AsTUFBTSxPQUFPLEdBQUcsSUFBSSxpQ0FBdUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO2dCQUN0RSxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUVqRCxPQUFPO2dCQUNQLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNuQyxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgscUJBQXFCO1FBQ3JCLE1BQU0sb0JBQW9CLEdBQUc7WUFDM0IsUUFBUSxFQUFFLCtDQUErQztZQUN6RCxJQUFJLEVBQUUsTUFBTTtTQUNiLENBQUM7UUFDRixNQUFNLG9CQUFvQixHQUFHO1lBQzNCLFFBQVEsRUFBRSxPQUFPO1lBQ2pCLElBQUksRUFBRSxNQUFNO1NBQ2IsQ0FBQztRQUNGLE1BQU0sZ0JBQWdCLEdBQUc7WUFDdkIsUUFBUSxFQUFFLCtDQUErQztZQUN6RCxRQUFRLEVBQUUsT0FBTztZQUNqQixnQkFBZ0IsRUFBRSxnQkFBZ0I7U0FDbkMsQ0FBQztRQUNGLE1BQU0seUJBQXlCLEdBQUc7WUFDaEMsUUFBUSxFQUFFLEVBQUU7WUFDWixRQUFRLEVBQUUsT0FBTztZQUNqQixJQUFJLEVBQUUsTUFBTTtTQUNiLENBQUM7UUFDRixNQUFNLHlCQUF5QixHQUFHO1lBQ2hDLFFBQVEsRUFBRSwrQ0FBK0M7WUFDekQsUUFBUSxFQUFFLEtBQUs7WUFDZixJQUFJLEVBQUUsTUFBTTtTQUNiLENBQUM7UUFDRixNQUFNLDZCQUE2QixHQUFHO1lBQ3BDLFFBQVEsRUFBRSwrQ0FBK0M7WUFDekQsUUFBUSxFQUFFLENBQUMsT0FBTyxDQUFDO1lBQ25CLElBQUksRUFBRSxNQUFNO1NBQ2IsQ0FBQztRQUNGLE1BQU0seUJBQXlCLEdBQUc7WUFDaEMsUUFBUSxFQUFFLCtDQUErQztZQUN6RCxRQUFRLEVBQUUsT0FBTztZQUNqQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUM7UUFDRixNQUFNLDJCQUEyQixHQUFHO1lBQ2xDLFFBQVEsRUFBRSwrQ0FBK0M7WUFDekQsUUFBUSxFQUFFLE9BQU87WUFDakIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDO1FBQ0YsTUFBTSwyQkFBMkIsR0FBRztZQUNsQyxRQUFRLEVBQUUsK0NBQStDO1lBQ3pELFFBQVEsRUFBRSxPQUFPO1lBQ2pCLElBQUksRUFBRSxPQUFPO1NBQ2QsQ0FBQztRQUNGLE1BQU0sMkJBQTJCLEdBQUc7WUFDbEMsUUFBUSxFQUFFLCtDQUErQztZQUN6RCxRQUFRLEVBQUUsT0FBTztZQUNqQixJQUFJLEVBQUUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUU7U0FDNUIsQ0FBQztRQUNGLE1BQU0sdUJBQXVCLEdBQUc7WUFDOUIsUUFBUSxFQUFFLCtDQUErQztZQUN6RCxRQUFRLEVBQUUsT0FBTztZQUNqQixJQUFJLEVBQUUsTUFBTTtZQUNaLGdCQUFnQixFQUFFLFFBQVE7U0FDM0IsQ0FBQztRQUVGLFFBQVEsQ0FBQyx3QkFBd0IsRUFBRSxHQUFHLEVBQUU7WUFDdEMsSUFBSSxDQUFDLElBQUksQ0FBTTtnQkFDYixvQkFBb0I7Z0JBQ3BCLG9CQUFvQjtnQkFDcEIsZ0JBQWdCO2dCQUNoQix1QkFBdUI7Z0JBQ3ZCLHlCQUF5QjtnQkFDekIseUJBQXlCO2dCQUN6Qiw2QkFBNkI7Z0JBQzdCLHlCQUF5QjtnQkFDekIsMkJBQTJCO2dCQUMzQiwyQkFBMkI7Z0JBQzNCLDJCQUEyQjtnQkFDM0IsU0FBUztnQkFDVCxFQUFFO2FBQ0gsQ0FBQyxDQUFDLG9CQUFvQixFQUFFLENBQUMsaUJBQXNCLEVBQUUsRUFBRTtnQkFDbEQsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBRztvQkFDWix3QkFBd0IsRUFBRSxxQkFBcUIsQ0FBQyx3QkFBd0I7b0JBQ3hFLFVBQVUsRUFBRSxpQkFBaUI7b0JBQzdCLDhCQUE4QixFQUFFLHFCQUFxQixDQUFDLDhCQUE4QjtpQkFDckYsQ0FBQztnQkFFRixPQUFPO2dCQUNQLE1BQU0sT0FBTyxHQUFHLElBQUksaUNBQXVCLENBQUMsSUFBSSxHQUFHLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztnQkFDdEUsTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFFakQsT0FBTztnQkFDUCxNQUFNLENBQUMsV0FBVyxDQUFDLENBQUMsU0FBUyxFQUFFLENBQUM7WUFDbEMsQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxDQUFNO2dCQUNiLFNBQVM7Z0JBQ1QsRUFBRTtnQkFDRixFQUFFO2FBQ0gsQ0FBQyxDQUFDLFlBQVksRUFBRSxDQUFDLEtBQUssRUFBRSxFQUFFO2dCQUN6QixPQUFPO2dCQUNQLE1BQU0sT0FBTyxHQUFHLElBQUksaUNBQXVCLENBQUMsSUFBSSxHQUFHLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztnQkFDdEUsTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFFakQsT0FBTztnQkFDUCxNQUFNLENBQUMsV0FBVyxDQUFDLENBQUMsU0FBUyxFQUFFLENBQUM7WUFDbEMsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLDJCQUEyQixFQUFFLEdBQUcsRUFBRTtRQUN6QyxRQUFRLENBQUMsdUJBQXVCLEVBQUUsR0FBRyxFQUFFO1lBQ3JDLElBQUksQ0FBQyxJQUFJLENBQXFCO2dCQUM1QixnQkFBZ0I7Z0JBQ2hCLFNBQVM7YUFDVixDQUFDLENBQUMsWUFBWSxFQUFFLEtBQUssRUFBRSxLQUF5QixFQUFFLEVBQUU7Z0JBQ25ELE9BQU87Z0JBQ1AsTUFBTSxPQUFPLEdBQUcsSUFBSSxpQ0FBdUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO2dCQUV0RSx3Q0FBd0M7Z0JBQ3hDLE1BQU0sV0FBVyxHQUFHLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUM3RCxNQUFNLENBQUMsV0FBVyxDQUFDLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDbkMsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyx3QkFBd0IsRUFBRSxHQUFHLEVBQUU7WUFDdEMsSUFBSSxDQUFDLElBQUksQ0FBTTtnQkFDYixZQUFZO2dCQUNaLEVBQUU7Z0JBQ0YsRUFBRTthQUNILENBQUMsQ0FBQyxZQUFZLEVBQUUsS0FBSyxFQUFFLEtBQVUsRUFBRSxFQUFFO2dCQUNwQyxPQUFPO2dCQUNQLE1BQU0sT0FBTyxHQUFHLElBQUksaUNBQXVCLENBQUMsSUFBSSxHQUFHLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztnQkFFdEUsd0NBQXdDO2dCQUN4QyxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsd0JBQXdCLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDN0QsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLFNBQVMsRUFBRSxDQUFDO1lBQ2xDLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQywwQkFBMEIsRUFBRSxHQUFHLEVBQUU7UUFDeEMsSUFBSSxDQUFDLGtDQUFrQyxFQUFFLEtBQUssSUFBSSxFQUFFO1lBQ2xELFFBQVE7WUFDUixNQUFNLG1CQUFtQixHQUFzQjtnQkFDN0MsUUFBUSxFQUFFLHVCQUF1QjtnQkFDakMsUUFBUSxFQUFFLE1BQU07Z0JBQ2hCLElBQUksRUFBRSxNQUFNO2FBQ2IsQ0FBQztZQUVGLE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxJQUFJLGlDQUF1QixDQUFDLElBQUksR0FBRyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUM7WUFDdEUsd0NBQXdDO1lBQ3hDLE1BQU0sTUFBTSxHQUFHLE1BQU0sT0FBTyxDQUFDLHVCQUF1QixDQUFDLENBQUMsbUJBQW1CLENBQUMsQ0FBQztZQUUzRSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7UUFDL0IsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsbUNBQW1DLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDbkQsUUFBUTtZQUNSLE1BQU0sb0JBQW9CLEdBQXNCO2dCQUM5QyxRQUFRLEVBQUUsdUJBQXVCO2dCQUNqQyxRQUFRLEVBQUUsTUFBTTtnQkFDaEIsSUFBSSxFQUFFLE1BQU07Z0JBQ1osZ0JBQWdCLEVBQUUsZ0JBQWdCO2FBQ25DLENBQUM7WUFFRixPQUFPO1lBQ1AsTUFBTSxPQUFPLEdBQUcsSUFBSSxpQ0FBdUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO1lBRXRFLElBQUksQ0FBQyxXQUFXLENBQUMsNENBQTRDLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUM7WUFFekksd0NBQXdDO1lBQ3hDLE1BQU0sTUFBTSxHQUFHLE1BQU0sT0FBTyxDQUFDLHVCQUF1QixDQUFDLENBQUMsb0JBQW9CLENBQUMsQ0FBQztZQUU1RSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7UUFDL0IsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxvQkFBb0IsRUFBRSxHQUFHLEVBQUU7UUFDbEMsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtZQUNoRCxRQUFRO1lBQ1IsTUFBTSxZQUFZLEdBQUc7Z0JBQ25CLGlCQUFpQixFQUFFLFVBQVU7YUFDbkIsQ0FBQztZQUNiLE1BQU0sY0FBYyxHQUFHO2dCQUNyQixHQUFHLEVBQUUsbUJBQW1CO2dCQUN4QixLQUFLLEVBQUUsVUFBVTthQUNsQixDQUFDO1lBRUYsT0FBTztZQUNQLE1BQU0sT0FBTyxHQUFHLElBQUksaUNBQXVCLENBQUMsSUFBSSxHQUFHLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztZQUN0RSx3Q0FBd0M7WUFDeEMsTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLGlCQUFpQixDQUFDLENBQUMsWUFBOEIsQ0FBQyxDQUFDO1lBRS9FLE9BQU87WUFDUCxNQUFNLENBQUMsV0FBVyxDQUFDLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3JELENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLDhCQUE4QixFQUFFLEdBQUcsRUFBRTtZQUN4QyxRQUFRO1lBQ1IsTUFBTSxZQUFZLEdBQUc7Z0JBQ25CLGlCQUFpQixFQUFFLFNBQVM7YUFDbEIsQ0FBQztZQUViLE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxJQUFJLGlDQUF1QixDQUFDLElBQUksR0FBRyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUM7WUFDdEUsU0FBUyxlQUFlO2dCQUN0Qix3Q0FBd0M7Z0JBQ3hDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLFlBQThCLENBQUMsQ0FBQztZQUM3RCxDQUFDO1lBRUQsT0FBTztZQUNQLE1BQU0sQ0FBQyxlQUFlLENBQUMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUN6QyxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQge1xuICBMYXVuY2hUZW1wbGF0ZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQge1xuICBBcHAsXG4gIEV4cGlyYXRpb24sXG4gIFN0YWNrLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIEFXUyBmcm9tICdhd3Mtc2RrJztcbmltcG9ydCB7XG4gIFNwb3RFdmVudFBsdWdpbkRpc3BsYXlJbnN0YW5jZVN0YXR1cyxcbiAgU3BvdEV2ZW50UGx1Z2luTG9nZ2luZ0xldmVsLFxuICBTcG90RXZlbnRQbHVnaW5QcmVKb2JUYXNrTW9kZSxcbiAgU3BvdEV2ZW50UGx1Z2luU3RhdGUsXG4gIFNwb3RGbGVldEFsbG9jYXRpb25TdHJhdGVneSxcbiAgU3BvdEZsZWV0UmVxdWVzdFR5cGUsXG4gIFNwb3RGbGVldFJlc291cmNlVHlwZSxcbn0gZnJvbSAnLi4vLi4vLi4vLi4vZGVhZGxpbmUnO1xuaW1wb3J0IHsgU0VQQ29uZmlndXJhdG9yUmVzb3VyY2UgfSBmcm9tICcuLi9oYW5kbGVyJztcbmltcG9ydCB7XG4gIENvbm5lY3Rpb25PcHRpb25zLFxuICBTRVBDb25maWd1cmF0b3JSZXNvdXJjZVByb3BzLFxuICBQbHVnaW5TZXR0aW5ncyxcbn0gZnJvbSAnLi4vdHlwZXMnO1xuXG5qZXN0Lm1vY2soJy4uLy4uL2xpYi9zZWNyZXRzLW1hbmFnZXIvcmVhZC1jZXJ0aWZpY2F0ZScpO1xuXG5jb25zdCBzZWNyZXRQYXJ0aWFsQXJuOiBzdHJpbmcgPSAnYXJuOmF3czpzZWNyZXRzbWFuYWdlcjp1cy13ZXN0LTE6MTIzNDU2Nzg5MDpzZWNyZXQ6U2VjcmV0UGF0aC9DZXJ0JztcblxuZGVzY3JpYmUoJ1NFUENvbmZpZ3VyYXRvclJlc291cmNlJywgKCkgPT4ge1xuICBjb25zdCBkZWFkbGluZUdyb3VwID0gJ2dyb3VwX25hbWUnO1xuICBjb25zdCBkZWFkbGluZVBvb2wgPSAgJ3Bvb2xfbmFtZSc7XG5cbiAgbGV0IGFwcDogQXBwO1xuICBsZXQgc3RhY2s6IFN0YWNrO1xuICBsZXQgdmFsaWRTZXBDb25maWd1cmF0aW9uOiBTRVBDb25maWd1cmF0b3JSZXNvdXJjZVByb3BzO1xuXG4gIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgIGFwcCA9IG5ldyBBcHAoKTtcbiAgICBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdTdGFjaycpO1xuICAgIGNvbnN0IGxhdW5jaFRlbXBsYXRlID0gbmV3IExhdW5jaFRlbXBsYXRlKHN0YWNrLCAnTGF1bmNoVGVtcGxhdGUnKTtcblxuICAgIHZhbGlkU2VwQ29uZmlndXJhdGlvbiA9IHtcbiAgICAgIHNwb3RQbHVnaW5Db25maWd1cmF0aW9uczoge1xuICAgICAgICBBV1NJbnN0YW5jZVN0YXR1czogU3BvdEV2ZW50UGx1Z2luRGlzcGxheUluc3RhbmNlU3RhdHVzLkRJU0FCTEVELFxuICAgICAgICBEZWxldGVJbnRlcnJ1cHRlZFNsYXZlczogdHJ1ZSxcbiAgICAgICAgRGVsZXRlVGVybWluYXRlZFNsYXZlczogdHJ1ZSxcbiAgICAgICAgSWRsZVNodXRkb3duOiAyMCxcbiAgICAgICAgTG9nZ2luZzogU3BvdEV2ZW50UGx1Z2luTG9nZ2luZ0xldmVsLlNUQU5EQVJELFxuICAgICAgICBQcmVKb2JUYXNrTW9kZTogU3BvdEV2ZW50UGx1Z2luUHJlSm9iVGFza01vZGUuQ09OU0VSVkFUSVZFLFxuICAgICAgICBSZWdpb246ICd1cy13ZXN0LTInLFxuICAgICAgICBSZXNvdXJjZVRyYWNrZXI6IHRydWUsXG4gICAgICAgIFN0YWdnZXJJbnN0YW5jZXM6IDUwLFxuICAgICAgICBTdGF0ZTogU3BvdEV2ZW50UGx1Z2luU3RhdGUuR0xPQkFMX0VOQUJMRUQsXG4gICAgICAgIFN0cmljdEhhcmRDYXA6IHRydWUsXG4gICAgICB9LFxuICAgICAgY29ubmVjdGlvbjoge1xuICAgICAgICBob3N0bmFtZTogJ2ludGVybmFsLWhvc3RuYW1lLmNvbScsXG4gICAgICAgIHByb3RvY29sOiAnSFRUUFMnLFxuICAgICAgICBwb3J0OiAnNDQzMycsXG4gICAgICAgIGNhQ2VydGlmaWNhdGVBcm46IHNlY3JldFBhcnRpYWxBcm4sXG4gICAgICB9LFxuICAgICAgc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25zOiB7XG4gICAgICAgIFtkZWFkbGluZUdyb3VwXToge1xuICAgICAgICAgIEFsbG9jYXRpb25TdHJhdGVneTogU3BvdEZsZWV0QWxsb2NhdGlvblN0cmF0ZWd5LkNBUEFDSVRZX09QVElNSVpFRCxcbiAgICAgICAgICBJYW1GbGVldFJvbGU6ICdyb2xlQXJuJyxcbiAgICAgICAgICAvLyBFeHBsaWNpdGx5IHByb3ZpZGUgZW1wdHkgYXJyYXkgZm9yIHRlc3RpbmcgY29tcGFyaXNvbnMgc2luY2Ugd2UgaW5qZWN0IHRoaXMgZm9yIGNvbXBhdGliaWxpdHkgd2l0aCBTRVBcbiAgICAgICAgICBMYXVuY2hTcGVjaWZpY2F0aW9uczogW10sXG4gICAgICAgICAgTGF1bmNoVGVtcGxhdGVDb25maWdzOiBbe1xuICAgICAgICAgICAgTGF1bmNoVGVtcGxhdGVTcGVjaWZpY2F0aW9uOiB7XG4gICAgICAgICAgICAgIExhdW5jaFRlbXBsYXRlSWQ6IGxhdW5jaFRlbXBsYXRlLmxhdW5jaFRlbXBsYXRlSWQsXG4gICAgICAgICAgICAgIExhdW5jaFRlbXBsYXRlTmFtZTogbGF1bmNoVGVtcGxhdGUubGF1bmNoVGVtcGxhdGVOYW1lLFxuICAgICAgICAgICAgICBWZXJzaW9uOiBsYXVuY2hUZW1wbGF0ZS52ZXJzaW9uTnVtYmVyLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIE92ZXJyaWRlczogW10sXG4gICAgICAgICAgfV0sXG4gICAgICAgICAgUmVwbGFjZVVuaGVhbHRoeUluc3RhbmNlczogdHJ1ZSxcbiAgICAgICAgICBUYXJnZXRDYXBhY2l0eTogMSxcbiAgICAgICAgICBUZXJtaW5hdGVJbnN0YW5jZXNXaXRoRXhwaXJhdGlvbjogdHJ1ZSxcbiAgICAgICAgICBUeXBlOiBTcG90RmxlZXRSZXF1ZXN0VHlwZS5NQUlOVEFJTixcbiAgICAgICAgICBWYWxpZFVudGlsOiBFeHBpcmF0aW9uLmF0RGF0ZShuZXcgRGF0ZSgyMDIyLCAxMSwgMTcpKS5kYXRlLnRvSVNPU3RyaW5nKCksXG4gICAgICAgICAgVGFnU3BlY2lmaWNhdGlvbnM6IFt7XG4gICAgICAgICAgICBSZXNvdXJjZVR5cGU6IFNwb3RGbGVldFJlc291cmNlVHlwZS5TUE9UX0ZMRUVUX1JFUVVFU1QsXG4gICAgICAgICAgICBUYWdzOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBLZXk6ICduYW1lJyxcbiAgICAgICAgICAgICAgICBWYWx1ZTogJ3Rlc3QnLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9XSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBkZWFkbGluZUdyb3VwczogW2RlYWRsaW5lR3JvdXBdLFxuICAgICAgZGVhZGxpbmVQb29sczogW2RlYWRsaW5lUG9vbF0sXG4gICAgfTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ2RvQ3JlYXRlJywgKCkgPT4ge1xuICAgIGxldCBoYW5kbGVyOiBTRVBDb25maWd1cmF0b3JSZXNvdXJjZTtcbiAgICBsZXQgbW9ja1Nwb3RFdmVudFBsdWdpbkNsaWVudDoge1xuICAgICAgc2F2ZVNlcnZlckRhdGE6IGplc3QuTW9jazxhbnksIGFueT47XG4gICAgICBjb25maWd1cmVTcG90RXZlbnRQbHVnaW46IGplc3QuTW9jazxhbnksIGFueT47XG4gICAgICBhZGRHcm91cHM6IGplc3QuTW9jazxhbnksIGFueT47XG4gICAgICBhZGRQb29sczogamVzdC5Nb2NrPGFueSwgYW55PjtcbiAgICB9O1xuXG4gICAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgICBtb2NrU3BvdEV2ZW50UGx1Z2luQ2xpZW50ID0ge1xuICAgICAgICBzYXZlU2VydmVyRGF0YTogamVzdC5mbiggKF9hKSA9PiBQcm9taXNlLnJlc29sdmUodHJ1ZSkgKSxcbiAgICAgICAgY29uZmlndXJlU3BvdEV2ZW50UGx1Z2luOiBqZXN0LmZuKCAoX2EpID0+IFByb21pc2UucmVzb2x2ZSh0cnVlKSApLFxuICAgICAgICBhZGRHcm91cHM6IGplc3QuZm4oIChfYSkgPT4gUHJvbWlzZS5yZXNvbHZlKHRydWUpICksXG4gICAgICAgIGFkZFBvb2xzOiBqZXN0LmZuKCAoX2EpID0+IFByb21pc2UucmVzb2x2ZSh0cnVlKSApLFxuICAgICAgfTtcblxuICAgICAgaGFuZGxlciA9IG5ldyBTRVBDb25maWd1cmF0b3JSZXNvdXJjZShuZXcgQVdTLlNlY3JldHNNYW5hZ2VyKCkpO1xuXG4gICAgICBqZXN0LnJlcXVpcmVNb2NrKCcuLi8uLi9saWIvc2VjcmV0cy1tYW5hZ2VyL3JlYWQtY2VydGlmaWNhdGUnKS5yZWFkQ2VydGlmaWNhdGVEYXRhLm1vY2tSZXR1cm5WYWx1ZShQcm9taXNlLnJlc29sdmUoJ0JFR0lOIENFUlRJRklDQVRFJykpO1xuXG4gICAgICBhc3luYyBmdW5jdGlvbiByZXR1cm5TcG90RXZlbnRQbHVnaW5DbGllbnQoX3YxOiBhbnkpOiBQcm9taXNlPGFueT4ge1xuICAgICAgICByZXR1cm4gbW9ja1Nwb3RFdmVudFBsdWdpbkNsaWVudDtcbiAgICAgIH1cbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBkb3Qtbm90YXRpb25cbiAgICAgIGhhbmRsZXJbJ3Nwb3RFdmVudFBsdWdpbkNsaWVudCddID0gamVzdC5mbiggKGEpID0+IHJldHVyblNwb3RFdmVudFBsdWdpbkNsaWVudChhKSApO1xuICAgIH0pO1xuXG4gICAgYWZ0ZXJFYWNoKCgpID0+IHtcbiAgICAgIGplc3QuY2xlYXJBbGxNb2NrcygpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnZG9lcyBub3Qgc2F2ZSBzZXJ2ZXIgZGF0YSB3aGVuIG5vIGNvbmZpZ3VyYXRpb25zIGFyZSBwcm92aWRlZCcsIGFzeW5jICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBtb2NrU2F2ZVNlcnZlckRhdGEgPSBqZXN0LmZuKCAoX2EpID0+IFByb21pc2UucmVzb2x2ZSh0cnVlKSApO1xuICAgICAgbW9ja1Nwb3RFdmVudFBsdWdpbkNsaWVudC5zYXZlU2VydmVyRGF0YSA9IG1vY2tTYXZlU2VydmVyRGF0YTtcbiAgICAgIGNvbnN0IG1vY2tDb25maWd1cmVTcG90RXZlbnRQbHVnaW4gPSBqZXN0LmZuKCAoX2EpID0+IFByb21pc2UucmVzb2x2ZSh0cnVlKSApO1xuICAgICAgbW9ja1Nwb3RFdmVudFBsdWdpbkNsaWVudC5jb25maWd1cmVTcG90RXZlbnRQbHVnaW4gPSBtb2NrQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCByZXN1bHQgPSBhd2FpdCBoYW5kbGVyLmRvQ3JlYXRlKCdwaHlzaWNhbElkJywge1xuICAgICAgICBjb25uZWN0aW9uOiB2YWxpZFNlcENvbmZpZ3VyYXRpb24uY29ubmVjdGlvbixcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QocmVzdWx0KS50b0JlVW5kZWZpbmVkKCk7XG4gICAgICBleHBlY3QobW9ja1NhdmVTZXJ2ZXJEYXRhLm1vY2suY2FsbHMubGVuZ3RoKS50b0JlKDApO1xuICAgICAgZXhwZWN0KG1vY2tDb25maWd1cmVTcG90RXZlbnRQbHVnaW4ubW9jay5jYWxscy5sZW5ndGgpLnRvQmUoMCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdzYXZlIHNwb3QgZmxlZXQgcmVxdWVzdCBjb25maWdzJywgYXN5bmMgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG1vY2tTYXZlU2VydmVyRGF0YSA9IGplc3QuZm4oIChfYSkgPT4gUHJvbWlzZS5yZXNvbHZlKHRydWUpICk7XG4gICAgICBtb2NrU3BvdEV2ZW50UGx1Z2luQ2xpZW50LnNhdmVTZXJ2ZXJEYXRhID0gbW9ja1NhdmVTZXJ2ZXJEYXRhO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCByZXN1bHQgPSBhd2FpdCBoYW5kbGVyLmRvQ3JlYXRlKCdwaHlzaWNhbElkJywge1xuICAgICAgICAuLi52YWxpZFNlcENvbmZpZ3VyYXRpb24sXG4gICAgICAgIHNwb3RQbHVnaW5Db25maWd1cmF0aW9uczogdW5kZWZpbmVkLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChyZXN1bHQpLnRvQmVVbmRlZmluZWQoKTtcbiAgICAgIGV4cGVjdChtb2NrU2F2ZVNlcnZlckRhdGEubW9jay5jYWxscy5sZW5ndGgpLnRvQmUoMSk7XG4gICAgICBleHBlY3QobW9ja1NhdmVTZXJ2ZXJEYXRhKS50b0JlQ2FsbGVkV2l0aChKU09OLnN0cmluZ2lmeSh2YWxpZFNlcENvbmZpZ3VyYXRpb24uc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25zKSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdzYXZlIHNwb3QgZXZlbnQgcGx1Z2luIGNvbmZpZ3MnLCBhc3luYyAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgbW9ja0NvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbiA9IGplc3QuZm4oIChfYSkgPT4gUHJvbWlzZS5yZXNvbHZlKHRydWUpICk7XG4gICAgICBtb2NrU3BvdEV2ZW50UGx1Z2luQ2xpZW50LmNvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbiA9IG1vY2tDb25maWd1cmVTcG90RXZlbnRQbHVnaW47XG5cbiAgICAgIGNvbnN0IGNvbmZpZ3M6IHsgS2V5OiBzdHJpbmcsIFZhbHVlOiBhbnkgfVtdID0gW107XG4gICAgICBmb3IgKGNvbnN0IFtrZXksIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyh2YWxpZFNlcENvbmZpZ3VyYXRpb24uc3BvdFBsdWdpbkNvbmZpZ3VyYXRpb25zIGFzIGFueSkpIHtcbiAgICAgICAgY29uZmlncy5wdXNoKHtcbiAgICAgICAgICBLZXk6IGtleSxcbiAgICAgICAgICBWYWx1ZTogdmFsdWUsXG4gICAgICAgIH0pO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBzZWN1cml0eVNldHRpbmdzID0gW3tcbiAgICAgICAgS2V5OiAnVXNlTG9jYWxDcmVkZW50aWFscycsXG4gICAgICAgIFZhbHVlOiB0cnVlLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgS2V5OiAnTmFtZWRQcm9maWxlJyxcbiAgICAgICAgVmFsdWU6ICcnLFxuICAgICAgfV07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHJlc3VsdCA9IGF3YWl0IGhhbmRsZXIuZG9DcmVhdGUoJ3BoeXNpY2FsSWQnLCB7XG4gICAgICAgIC4uLnZhbGlkU2VwQ29uZmlndXJhdGlvbixcbiAgICAgICAgc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25zOiB1bmRlZmluZWQsXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHJlc3VsdCkudG9CZVVuZGVmaW5lZCgpO1xuICAgICAgZXhwZWN0KG1vY2tDb25maWd1cmVTcG90RXZlbnRQbHVnaW4ubW9jay5jYWxscy5sZW5ndGgpLnRvQmUoMSk7XG4gICAgICBleHBlY3QobW9ja0NvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbi5tb2NrLmNhbGxzWzBdWzBdKS50b0VxdWFsKFsuLi5jb25maWdzLCAuLi5zZWN1cml0eVNldHRpbmdzXSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdzYXZlIHNlcnZlciBkYXRhJywgYXN5bmMgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG1vY2tTYXZlU2VydmVyRGF0YSA9IGplc3QuZm4oIChfYSkgPT4gUHJvbWlzZS5yZXNvbHZlKHRydWUpICk7XG4gICAgICBtb2NrU3BvdEV2ZW50UGx1Z2luQ2xpZW50LnNhdmVTZXJ2ZXJEYXRhID0gbW9ja1NhdmVTZXJ2ZXJEYXRhO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCByZXN1bHQgPSBhd2FpdCBoYW5kbGVyLmRvQ3JlYXRlKCdwaHlzaWNhbElkJywgdmFsaWRTZXBDb25maWd1cmF0aW9uKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHJlc3VsdCkudG9CZVVuZGVmaW5lZCgpO1xuICAgICAgZXhwZWN0KG1vY2tTYXZlU2VydmVyRGF0YS5tb2NrLmNhbGxzLmxlbmd0aCkudG9CZSgxKTtcbiAgICAgIGV4cGVjdChtb2NrU2F2ZVNlcnZlckRhdGEubW9jay5jYWxsc1swXVswXSkudG9FcXVhbChKU09OLnN0cmluZ2lmeSh2YWxpZFNlcENvbmZpZ3VyYXRpb24uc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25zKSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdjb25maWd1cmUgc3BvdCBldmVudCBwbHVnaW4nLCBhc3luYyAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgbW9ja0NvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbiA9IGplc3QuZm4oIChfYSkgPT4gUHJvbWlzZS5yZXNvbHZlKHRydWUpICk7XG4gICAgICBtb2NrU3BvdEV2ZW50UGx1Z2luQ2xpZW50LmNvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbiA9IG1vY2tDb25maWd1cmVTcG90RXZlbnRQbHVnaW47XG5cbiAgICAgIGNvbnN0IGNvbmZpZ3M6IHsgS2V5OiBzdHJpbmcsIFZhbHVlOiBhbnkgfVtdID0gW107XG4gICAgICBmb3IgKGNvbnN0IFtrZXksIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyh2YWxpZFNlcENvbmZpZ3VyYXRpb24uc3BvdFBsdWdpbkNvbmZpZ3VyYXRpb25zIGFzIGFueSkpIHtcbiAgICAgICAgY29uZmlncy5wdXNoKHtcbiAgICAgICAgICBLZXk6IGtleSxcbiAgICAgICAgICBWYWx1ZTogdmFsdWUsXG4gICAgICAgIH0pO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBzZWN1cml0eVNldHRpbmdzID0gW3tcbiAgICAgICAgS2V5OiAnVXNlTG9jYWxDcmVkZW50aWFscycsXG4gICAgICAgIFZhbHVlOiB0cnVlLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgS2V5OiAnTmFtZWRQcm9maWxlJyxcbiAgICAgICAgVmFsdWU6ICcnLFxuICAgICAgfV07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGF3YWl0IGhhbmRsZXIuZG9DcmVhdGUoJ3BoeXNpY2FsSWQnLCB2YWxpZFNlcENvbmZpZ3VyYXRpb24pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QobW9ja0NvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbi5tb2NrLmNhbGxzLmxlbmd0aCkudG9CZSgxKTtcbiAgICAgIGV4cGVjdChtb2NrQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luLm1vY2suY2FsbHNbMF1bMF0pLnRvRXF1YWwoWy4uLmNvbmZpZ3MsIC4uLnNlY3VyaXR5U2V0dGluZ3NdKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2NyZWF0ZSBncm91cHMnLCBhc3luYyAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgbW9ja0FkZEdyb3VwcyA9IGplc3QuZm4oIChfYSkgPT4gUHJvbWlzZS5yZXNvbHZlKHRydWUpICk7XG4gICAgICBtb2NrU3BvdEV2ZW50UGx1Z2luQ2xpZW50LmFkZEdyb3VwcyA9IG1vY2tBZGRHcm91cHM7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGF3YWl0IGhhbmRsZXIuZG9DcmVhdGUoJ3BoeXNpY2FsSWQnLCB2YWxpZFNlcENvbmZpZ3VyYXRpb24pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QobW9ja0FkZEdyb3Vwcy5tb2NrLmNhbGxzLmxlbmd0aCkudG9CZSgxKTtcbiAgICAgIGV4cGVjdChtb2NrQWRkR3JvdXBzKS50b0hhdmVCZWVuQ2FsbGVkV2l0aChbZGVhZGxpbmVHcm91cF0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnY3JlYXRlIHBvb2xzJywgYXN5bmMgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG1vY2tBZGRQb29scyA9IGplc3QuZm4oIChfYSkgPT4gUHJvbWlzZS5yZXNvbHZlKHRydWUpICk7XG4gICAgICBtb2NrU3BvdEV2ZW50UGx1Z2luQ2xpZW50LmFkZFBvb2xzID0gbW9ja0FkZFBvb2xzO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBhd2FpdCBoYW5kbGVyLmRvQ3JlYXRlKCdwaHlzaWNhbElkJywgdmFsaWRTZXBDb25maWd1cmF0aW9uKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KG1vY2tBZGRQb29scy5tb2NrLmNhbGxzLmxlbmd0aCkudG9CZSgxKTtcbiAgICAgIGV4cGVjdChtb2NrQWRkUG9vbHMpLnRvSGF2ZUJlZW5DYWxsZWRXaXRoKFtkZWFkbGluZVBvb2xdKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3Rocm93IHdoZW4gY2Fubm90IGFkZCBncm91cHMnLCBhc3luYyAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgbW9ja1Nwb3RFdmVudFBsdWdpbkNsaWVudC5hZGRHcm91cHMgPSBqZXN0LmZuKCAoX2EpID0+IFByb21pc2UucmVzb2x2ZShmYWxzZSkgKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgcHJvbWlzZSA9IGhhbmRsZXIuZG9DcmVhdGUoJ3BoeXNpY2FsSWQnLCB2YWxpZFNlcENvbmZpZ3VyYXRpb24pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBhd2FpdCBleHBlY3QocHJvbWlzZSlcbiAgICAgICAgLnJlamVjdHNcbiAgICAgICAgLnRvVGhyb3dFcnJvcihgRmFpbGVkIHRvIGFkZCBEZWFkbGluZSBncm91cChzKSAke3ZhbGlkU2VwQ29uZmlndXJhdGlvbi5kZWFkbGluZUdyb3Vwc31gKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3Rocm93IHdoZW4gY2Fubm90IGFkZCBwb29scycsIGFzeW5jICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBtb2NrU3BvdEV2ZW50UGx1Z2luQ2xpZW50LmFkZFBvb2xzID0gamVzdC5mbiggKF9hKSA9PiBQcm9taXNlLnJlc29sdmUoZmFsc2UpICk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHByb21pc2UgPSBoYW5kbGVyLmRvQ3JlYXRlKCdwaHlzaWNhbElkJywgdmFsaWRTZXBDb25maWd1cmF0aW9uKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgYXdhaXQgZXhwZWN0KHByb21pc2UpXG4gICAgICAgIC5yZWplY3RzXG4gICAgICAgIC50b1Rocm93RXJyb3IoYEZhaWxlZCB0byBhZGQgRGVhZGxpbmUgcG9vbChzKSAke3ZhbGlkU2VwQ29uZmlndXJhdGlvbi5kZWFkbGluZVBvb2xzfWApO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndGhyb3cgd2hlbiBjYW5ub3Qgc2F2ZSBzcG90IGZsZWV0IHJlcXVlc3QgY29uZmlncycsIGFzeW5jICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBtb2NrU2F2ZVNlcnZlckRhdGEgPSBqZXN0LmZuKCAoX2EpID0+IFByb21pc2UucmVzb2x2ZShmYWxzZSkgKTtcbiAgICAgIG1vY2tTcG90RXZlbnRQbHVnaW5DbGllbnQuc2F2ZVNlcnZlckRhdGEgPSBtb2NrU2F2ZVNlcnZlckRhdGE7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHByb21pc2UgPSBoYW5kbGVyLmRvQ3JlYXRlKCdwaHlzaWNhbElkJywgdmFsaWRTZXBDb25maWd1cmF0aW9uKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgYXdhaXQgZXhwZWN0KHByb21pc2UpXG4gICAgICAgIC5yZWplY3RzXG4gICAgICAgIC50b1Rocm93RXJyb3IoL0ZhaWxlZCB0byBzYXZlIHNwb3QgZmxlZXQgcmVxdWVzdCB3aXRoIGNvbmZpZ3VyYXRpb24vKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3Rocm93IHdoZW4gY2Fubm90IHNhdmUgc3BvdCBldmVudCBwbHVnaW4gY29uZmlncycsIGFzeW5jICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBtb2NrQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luID0gamVzdC5mbiggKF9hKSA9PiBQcm9taXNlLnJlc29sdmUoZmFsc2UpICk7XG4gICAgICBtb2NrU3BvdEV2ZW50UGx1Z2luQ2xpZW50LmNvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbiA9IG1vY2tDb25maWd1cmVTcG90RXZlbnRQbHVnaW47XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHByb21pc2UgPSBoYW5kbGVyLmRvQ3JlYXRlKCdwaHlzaWNhbElkJywgdmFsaWRTZXBDb25maWd1cmF0aW9uKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgYXdhaXQgZXhwZWN0KHByb21pc2UpXG4gICAgICAgIC5yZWplY3RzXG4gICAgICAgIC50b1Rocm93RXJyb3IoL0ZhaWxlZCB0byBzYXZlIFNwb3QgRXZlbnQgUGx1Z2luIENvbmZpZ3VyYXRpb25zLyk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RvRGVsZXRlIGRvZXMgbm90IGRvIGFueXRoaW5nJywgYXN5bmMgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgaGFuZGxlciA9IG5ldyBTRVBDb25maWd1cmF0b3JSZXNvdXJjZShuZXcgQVdTLlNlY3JldHNNYW5hZ2VyKCkpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHByb21pc2UgPSBhd2FpdCBoYW5kbGVyLmRvRGVsZXRlKCdwaHlzaWNhbElkJywge1xuICAgICAgY29ubmVjdGlvbjogdmFsaWRTZXBDb25maWd1cmF0aW9uLmNvbm5lY3Rpb24sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgYXdhaXQgZXhwZWN0KHByb21pc2UpLnRvQmVVbmRlZmluZWQoKTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJy52YWxpZGF0ZUlucHV0KCknLCAoKSA9PiB7XG4gICAgZGVzY3JpYmUoJ3Nob3VsZCByZXR1cm4gdHJ1ZScsICgpID0+IHtcbiAgICAgIHRlc3QoJ3dpdGggdmFsaWQgaW5wdXQnLCBhc3luYyAoKSA9PiB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IGlucHV0ID0gdmFsaWRTZXBDb25maWd1cmF0aW9uO1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29uc3QgaGFuZGxlciA9IG5ldyBTRVBDb25maWd1cmF0b3JSZXNvdXJjZShuZXcgQVdTLlNlY3JldHNNYW5hZ2VyKCkpO1xuICAgICAgICBjb25zdCByZXR1cm5WYWx1ZSA9IGhhbmRsZXIudmFsaWRhdGVJbnB1dChpbnB1dCk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QocmV0dXJuVmFsdWUpLnRvQmVUcnV0aHkoKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCd3aXRob3V0IHNwb3RQbHVnaW5Db25maWd1cmF0aW9ucycsIGFzeW5jICgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgaW5wdXQ6IFNFUENvbmZpZ3VyYXRvclJlc291cmNlUHJvcHMgPSB7XG4gICAgICAgICAgLi4udmFsaWRTZXBDb25maWd1cmF0aW9uLFxuICAgICAgICAgIHNwb3RQbHVnaW5Db25maWd1cmF0aW9uczogdW5kZWZpbmVkLFxuICAgICAgICB9O1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29uc3QgaGFuZGxlciA9IG5ldyBTRVBDb25maWd1cmF0b3JSZXNvdXJjZShuZXcgQVdTLlNlY3JldHNNYW5hZ2VyKCkpO1xuICAgICAgICBjb25zdCByZXR1cm5WYWx1ZSA9IGhhbmRsZXIudmFsaWRhdGVJbnB1dChpbnB1dCk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QocmV0dXJuVmFsdWUpLnRvQmVUcnV0aHkoKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCd3aXRob3V0IHNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9ucycsIGFzeW5jICgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgaW5wdXQ6IFNFUENvbmZpZ3VyYXRvclJlc291cmNlUHJvcHMgPSB7XG4gICAgICAgICAgLi4udmFsaWRTZXBDb25maWd1cmF0aW9uLFxuICAgICAgICAgIHNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9uczogdW5kZWZpbmVkLFxuICAgICAgICB9O1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29uc3QgaGFuZGxlciA9IG5ldyBTRVBDb25maWd1cmF0b3JSZXNvdXJjZShuZXcgQVdTLlNlY3JldHNNYW5hZ2VyKCkpO1xuICAgICAgICBjb25zdCByZXR1cm5WYWx1ZSA9IGhhbmRsZXIudmFsaWRhdGVJbnB1dChpbnB1dCk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QocmV0dXJuVmFsdWUpLnRvQmVUcnV0aHkoKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCd3aXRoIG9ubHkgY29ubmVjdGlvbicsIGFzeW5jICgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgaW5wdXQ6IFNFUENvbmZpZ3VyYXRvclJlc291cmNlUHJvcHMgPSB7XG4gICAgICAgICAgY29ubmVjdGlvbjogdmFsaWRTZXBDb25maWd1cmF0aW9uLmNvbm5lY3Rpb24sXG4gICAgICAgIH07XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBoYW5kbGVyID0gbmV3IFNFUENvbmZpZ3VyYXRvclJlc291cmNlKG5ldyBBV1MuU2VjcmV0c01hbmFnZXIoKSk7XG4gICAgICAgIGNvbnN0IHJldHVyblZhbHVlID0gaGFuZGxlci52YWxpZGF0ZUlucHV0KGlucHV0KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChyZXR1cm5WYWx1ZSkudG9CZVRydXRoeSgpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICAvLyBJbnZhbGlkIGNvbm5lY3Rpb25cbiAgICBjb25zdCBub1Byb3RvY29sQ29ubmVjdGlvbiA9IHtcbiAgICAgIGhvc3RuYW1lOiAnaW50ZXJuYWwtaG9zdG5hbWUudXMtZWFzdC0xLmVsYi5hbWF6b25hd3MuY29tJyxcbiAgICAgIHBvcnQ6ICc0NDMzJyxcbiAgICB9O1xuICAgIGNvbnN0IG5vSG9zdG5hbWVDb25uZWN0aW9uID0ge1xuICAgICAgcHJvdG9jb2w6ICdIVFRQUycsXG4gICAgICBwb3J0OiAnNDQzMycsXG4gICAgfTtcbiAgICBjb25zdCBub1BvcnRDb25uZWN0aW9uID0ge1xuICAgICAgaG9zdG5hbWU6ICdpbnRlcm5hbC1ob3N0bmFtZS51cy1lYXN0LTEuZWxiLmFtYXpvbmF3cy5jb20nLFxuICAgICAgcHJvdG9jb2w6ICdIVFRQUycsXG4gICAgICBjYUNlcnRpZmljYXRlQXJuOiBzZWNyZXRQYXJ0aWFsQXJuLFxuICAgIH07XG4gICAgY29uc3QgaW52YWxpZEhvc3RuYW1lQ29ubmVjdGlvbiA9IHtcbiAgICAgIGhvc3RuYW1lOiAxMCxcbiAgICAgIHByb3RvY29sOiAnSFRUUFMnLFxuICAgICAgcG9ydDogJzQ0MzMnLFxuICAgIH07XG4gICAgY29uc3QgaW52YWxpZFByb3RvY29sQ29ubmVjdGlvbiA9IHtcbiAgICAgIGhvc3RuYW1lOiAnaW50ZXJuYWwtaG9zdG5hbWUudXMtZWFzdC0xLmVsYi5hbWF6b25hd3MuY29tJyxcbiAgICAgIHByb3RvY29sOiAnVENQJyxcbiAgICAgIHBvcnQ6ICc0NDMzJyxcbiAgICB9O1xuICAgIGNvbnN0IGludmFsaWRQcm90b2NvbFR5cGVDb25uZWN0aW9uID0ge1xuICAgICAgaG9zdG5hbWU6ICdpbnRlcm5hbC1ob3N0bmFtZS51cy1lYXN0LTEuZWxiLmFtYXpvbmF3cy5jb20nLFxuICAgICAgcHJvdG9jb2w6IFsnSFRUUFMnXSxcbiAgICAgIHBvcnQ6ICc0NDMzJyxcbiAgICB9O1xuICAgIGNvbnN0IGludmFsaWRQb3J0VHlwZUNvbm5lY3Rpb24gPSB7XG4gICAgICBob3N0bmFtZTogJ2ludGVybmFsLWhvc3RuYW1lLnVzLWVhc3QtMS5lbGIuYW1hem9uYXdzLmNvbScsXG4gICAgICBwcm90b2NvbDogJ0hUVFBTJyxcbiAgICAgIHBvcnQ6IDQ0MzMsXG4gICAgfTtcbiAgICBjb25zdCBpbnZhbGlkUG9ydFJhbmdlMUNvbm5lY3Rpb24gPSB7XG4gICAgICBob3N0bmFtZTogJ2ludGVybmFsLWhvc3RuYW1lLnVzLWVhc3QtMS5lbGIuYW1hem9uYXdzLmNvbScsXG4gICAgICBwcm90b2NvbDogJ0hUVFBTJyxcbiAgICAgIHBvcnQ6ICctMScsXG4gICAgfTtcbiAgICBjb25zdCBpbnZhbGlkUG9ydFJhbmdlMkNvbm5lY3Rpb24gPSB7XG4gICAgICBob3N0bmFtZTogJ2ludGVybmFsLWhvc3RuYW1lLnVzLWVhc3QtMS5lbGIuYW1hem9uYXdzLmNvbScsXG4gICAgICBwcm90b2NvbDogJ0hUVFBTJyxcbiAgICAgIHBvcnQ6ICc2NTUzNicsXG4gICAgfTtcbiAgICBjb25zdCBpbnZhbGlkUG9ydFJhbmdlM0Nvbm5lY3Rpb24gPSB7XG4gICAgICBob3N0bmFtZTogJ2ludGVybmFsLWhvc3RuYW1lLnVzLWVhc3QtMS5lbGIuYW1hem9uYXdzLmNvbScsXG4gICAgICBwcm90b2NvbDogJ0hUVFBTJyxcbiAgICAgIHBvcnQ6IE51bWJlci5OYU4udG9TdHJpbmcoKSxcbiAgICB9O1xuICAgIGNvbnN0IGludmFsaWRDYUNlcnRDb25uZWN0aW9uID0ge1xuICAgICAgaG9zdG5hbWU6ICdpbnRlcm5hbC1ob3N0bmFtZS51cy1lYXN0LTEuZWxiLmFtYXpvbmF3cy5jb20nLFxuICAgICAgcHJvdG9jb2w6ICdIVFRQUycsXG4gICAgICBwb3J0OiAnNDQzMycsXG4gICAgICBjYUNlcnRpZmljYXRlQXJuOiAnbm90QXJuJyxcbiAgICB9O1xuXG4gICAgZGVzY3JpYmUoJ3Nob3VsZCByZXR1cm4gZmFsc2UgaWYnLCAoKSA9PiB7XG4gICAgICB0ZXN0LmVhY2g8YW55PihbXG4gICAgICAgIG5vUHJvdG9jb2xDb25uZWN0aW9uLFxuICAgICAgICBub0hvc3RuYW1lQ29ubmVjdGlvbixcbiAgICAgICAgbm9Qb3J0Q29ubmVjdGlvbixcbiAgICAgICAgaW52YWxpZENhQ2VydENvbm5lY3Rpb24sXG4gICAgICAgIGludmFsaWRIb3N0bmFtZUNvbm5lY3Rpb24sXG4gICAgICAgIGludmFsaWRQcm90b2NvbENvbm5lY3Rpb24sXG4gICAgICAgIGludmFsaWRQcm90b2NvbFR5cGVDb25uZWN0aW9uLFxuICAgICAgICBpbnZhbGlkUG9ydFR5cGVDb25uZWN0aW9uLFxuICAgICAgICBpbnZhbGlkUG9ydFJhbmdlMUNvbm5lY3Rpb24sXG4gICAgICAgIGludmFsaWRQb3J0UmFuZ2UyQ29ubmVjdGlvbixcbiAgICAgICAgaW52YWxpZFBvcnRSYW5nZTNDb25uZWN0aW9uLFxuICAgICAgICB1bmRlZmluZWQsXG4gICAgICAgIFtdLFxuICAgICAgXSkoJ2ludmFsaWQgY29ubmVjdGlvbicsIChpbnZhbGlkQ29ubmVjdGlvbjogYW55KSA9PiB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IGlucHV0ID0ge1xuICAgICAgICAgIHNwb3RQbHVnaW5Db25maWd1cmF0aW9uczogdmFsaWRTZXBDb25maWd1cmF0aW9uLnNwb3RQbHVnaW5Db25maWd1cmF0aW9ucyxcbiAgICAgICAgICBjb25uZWN0aW9uOiBpbnZhbGlkQ29ubmVjdGlvbixcbiAgICAgICAgICBzcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbnM6IHZhbGlkU2VwQ29uZmlndXJhdGlvbi5zcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbnMsXG4gICAgICAgIH07XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBoYW5kbGVyID0gbmV3IFNFUENvbmZpZ3VyYXRvclJlc291cmNlKG5ldyBBV1MuU2VjcmV0c01hbmFnZXIoKSk7XG4gICAgICAgIGNvbnN0IHJldHVyblZhbHVlID0gaGFuZGxlci52YWxpZGF0ZUlucHV0KGlucHV0KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChyZXR1cm5WYWx1ZSkudG9CZUZhbHN5KCk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdC5lYWNoPGFueT4oW1xuICAgICAgICB1bmRlZmluZWQsXG4gICAgICAgIFtdLFxuICAgICAgICAnJyxcbiAgICAgIF0pKCd7aW5wdXQ9JXN9JywgKGlucHV0KSA9PiB7XG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29uc3QgaGFuZGxlciA9IG5ldyBTRVBDb25maWd1cmF0b3JSZXNvdXJjZShuZXcgQVdTLlNlY3JldHNNYW5hZ2VyKCkpO1xuICAgICAgICBjb25zdCByZXR1cm5WYWx1ZSA9IGhhbmRsZXIudmFsaWRhdGVJbnB1dChpbnB1dCk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QocmV0dXJuVmFsdWUpLnRvQmVGYWxzeSgpO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCcuaXNTZWNyZXRBcm5PclVuZGVmaW5lZCgpJywgKCkgPT4ge1xuICAgIGRlc2NyaWJlKCdzaG91bGQgcmV0dXJuIHRydWUgaWYnLCAoKSA9PiB7XG4gICAgICB0ZXN0LmVhY2g8c3RyaW5nIHwgdW5kZWZpbmVkPihbXG4gICAgICAgIHNlY3JldFBhcnRpYWxBcm4sXG4gICAgICAgIHVuZGVmaW5lZCxcbiAgICAgIF0pKCd7aW5wdXQ9JXN9JywgYXN5bmMgKGlucHV0OiBzdHJpbmcgfCB1bmRlZmluZWQpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBoYW5kbGVyID0gbmV3IFNFUENvbmZpZ3VyYXRvclJlc291cmNlKG5ldyBBV1MuU2VjcmV0c01hbmFnZXIoKSk7XG5cbiAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGRvdC1ub3RhdGlvblxuICAgICAgICBjb25zdCByZXR1cm5WYWx1ZSA9IGhhbmRsZXJbJ2lzU2VjcmV0QXJuT3JVbmRlZmluZWQnXShpbnB1dCk7XG4gICAgICAgIGV4cGVjdChyZXR1cm5WYWx1ZSkudG9CZVRydXRoeSgpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnc2hvdWxkIHJldHVybiBmYWxzZSBpZicsICgpID0+IHtcbiAgICAgIHRlc3QuZWFjaDxhbnk+KFtcbiAgICAgICAgJ2FueSBzdHJpbmcnLFxuICAgICAgICAxMCxcbiAgICAgICAgW10sXG4gICAgICBdKSgne2lucHV0PSVzfScsIGFzeW5jIChpbnB1dDogYW55KSA9PiB7XG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29uc3QgaGFuZGxlciA9IG5ldyBTRVBDb25maWd1cmF0b3JSZXNvdXJjZShuZXcgQVdTLlNlY3JldHNNYW5hZ2VyKCkpO1xuXG4gICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBkb3Qtbm90YXRpb25cbiAgICAgICAgY29uc3QgcmV0dXJuVmFsdWUgPSBoYW5kbGVyWydpc1NlY3JldEFybk9yVW5kZWZpbmVkJ10oaW5wdXQpO1xuICAgICAgICBleHBlY3QocmV0dXJuVmFsdWUpLnRvQmVGYWxzeSgpO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCcuc3BvdEV2ZW50UGx1Z2luQ2xpZW50KCknLCAoKSA9PiB7XG4gICAgdGVzdCgnY3JlYXRlcyBhIHZhbGlkIG9iamVjdCB3aXRoIGh0dHAnLCBhc3luYyAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgdmFsaWRIVFRQQ29ubmVjdGlvbjogQ29ubmVjdGlvbk9wdGlvbnMgPSB7XG4gICAgICAgIGhvc3RuYW1lOiAnaW50ZXJuYWwtaG9zdG5hbWUuY29tJyxcbiAgICAgICAgcHJvdG9jb2w6ICdIVFRQJyxcbiAgICAgICAgcG9ydDogJzgwODAnLFxuICAgICAgfTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgaGFuZGxlciA9IG5ldyBTRVBDb25maWd1cmF0b3JSZXNvdXJjZShuZXcgQVdTLlNlY3JldHNNYW5hZ2VyKCkpO1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGRvdC1ub3RhdGlvblxuICAgICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgaGFuZGxlclsnc3BvdEV2ZW50UGx1Z2luQ2xpZW50J10odmFsaWRIVFRQQ29ubmVjdGlvbik7XG5cbiAgICAgIGV4cGVjdChyZXN1bHQpLnRvQmVEZWZpbmVkKCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdjcmVhdGVzIGEgdmFsaWQgb2JqZWN0IHdpdGggaHR0cHMnLCBhc3luYyAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgdmFsaWRIVFRQU0Nvbm5lY3Rpb246IENvbm5lY3Rpb25PcHRpb25zID0ge1xuICAgICAgICBob3N0bmFtZTogJ2ludGVybmFsLWhvc3RuYW1lLmNvbScsXG4gICAgICAgIHByb3RvY29sOiAnSFRUUCcsXG4gICAgICAgIHBvcnQ6ICc4MDgwJyxcbiAgICAgICAgY2FDZXJ0aWZpY2F0ZUFybjogc2VjcmV0UGFydGlhbEFybixcbiAgICAgIH07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGhhbmRsZXIgPSBuZXcgU0VQQ29uZmlndXJhdG9yUmVzb3VyY2UobmV3IEFXUy5TZWNyZXRzTWFuYWdlcigpKTtcblxuICAgICAgamVzdC5yZXF1aXJlTW9jaygnLi4vLi4vbGliL3NlY3JldHMtbWFuYWdlci9yZWFkLWNlcnRpZmljYXRlJykucmVhZENlcnRpZmljYXRlRGF0YS5tb2NrUmV0dXJuVmFsdWUoUHJvbWlzZS5yZXNvbHZlKCdCRUdJTiBDRVJUSUZJQ0FURScpKTtcblxuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGRvdC1ub3RhdGlvblxuICAgICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgaGFuZGxlclsnc3BvdEV2ZW50UGx1Z2luQ2xpZW50J10odmFsaWRIVFRQU0Nvbm5lY3Rpb24pO1xuXG4gICAgICBleHBlY3QocmVzdWx0KS50b0JlRGVmaW5lZCgpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnLnRvS2V5VmFsdWVBcnJheSgpJywgKCkgPT4ge1xuICAgIHRlc3QoJ2NvbnZlcnRzIHRvIGFycmF5IG9mIGtleSB2YWx1ZSBwYWlycycsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBwbHVnaW5Db25maWcgPSB7XG4gICAgICAgIEFXU0luc3RhbmNlU3RhdHVzOiAnRGlzYWJsZWQnLFxuICAgICAgfSBhcyB1bmtub3duO1xuICAgICAgY29uc3QgZXhwZWN0ZWRSZXN1bHQgPSB7XG4gICAgICAgIEtleTogJ0FXU0luc3RhbmNlU3RhdHVzJyxcbiAgICAgICAgVmFsdWU6ICdEaXNhYmxlZCcsXG4gICAgICB9O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBoYW5kbGVyID0gbmV3IFNFUENvbmZpZ3VyYXRvclJlc291cmNlKG5ldyBBV1MuU2VjcmV0c01hbmFnZXIoKSk7XG4gICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgZG90LW5vdGF0aW9uXG4gICAgICBjb25zdCByZXR1cm5WYWx1ZSA9IGhhbmRsZXJbJ3RvS2V5VmFsdWVBcnJheSddKHBsdWdpbkNvbmZpZyBhcyBQbHVnaW5TZXR0aW5ncyk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChyZXR1cm5WYWx1ZSkudG9Db250YWluRXF1YWwoZXhwZWN0ZWRSZXN1bHQpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndGhyb3dzIHdpdGggdW5kZWZpbmVkIHZhbHVlcycsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBwbHVnaW5Db25maWcgPSB7XG4gICAgICAgIEFXU0luc3RhbmNlU3RhdHVzOiB1bmRlZmluZWQsXG4gICAgICB9IGFzIHVua25vd247XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGhhbmRsZXIgPSBuZXcgU0VQQ29uZmlndXJhdG9yUmVzb3VyY2UobmV3IEFXUy5TZWNyZXRzTWFuYWdlcigpKTtcbiAgICAgIGZ1bmN0aW9uIHRvS2V5VmFsdWVBcnJheSgpIHtcbiAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGRvdC1ub3RhdGlvblxuICAgICAgICBoYW5kbGVyWyd0b0tleVZhbHVlQXJyYXknXShwbHVnaW5Db25maWcgYXMgUGx1Z2luU2V0dGluZ3MpO1xuICAgICAgfVxuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QodG9LZXlWYWx1ZUFycmF5KS50b1Rocm93RXJyb3IoKTtcbiAgICB9KTtcbiAgfSk7XG59KTtcbiJdfQ==