"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const conversion_1 = require("../conversion");
const propertyName = 'propertyName';
describe('convertSpotEventPluginSettings()', () => {
    test('does not convert properties with correct types', () => {
        // GIVEN
        const defaultPluginConfig = {
            AWSInstanceStatus: 'Disabled',
            DeleteInterruptedSlaves: false,
            DeleteTerminatedSlaves: false,
            IdleShutdown: 10,
            Logging: 'Standard',
            PreJobTaskMode: 'Conservative',
            Region: 'eu-west-1',
            ResourceTracker: true,
            StaggerInstances: 50,
            State: 'Disabled',
            StrictHardCap: false,
        };
        const defaultConvertedPluginConfig = {
            AWSInstanceStatus: 'Disabled',
            DeleteInterruptedSlaves: false,
            DeleteTerminatedSlaves: false,
            IdleShutdown: 10,
            Logging: 'Standard',
            PreJobTaskMode: 'Conservative',
            Region: 'eu-west-1',
            ResourceTracker: true,
            StaggerInstances: 50,
            State: 'Disabled',
            StrictHardCap: false,
        };
        // WHEN
        const returnValue = conversion_1.convertSpotEventPluginSettings(defaultPluginConfig);
        // THEN
        expect(returnValue).toEqual(defaultConvertedPluginConfig);
    });
    test('converts properties of type string', () => {
        // GIVEN
        const defaultPluginConfig = {
            AWSInstanceStatus: 'Disabled',
            DeleteInterruptedSlaves: 'false',
            DeleteTerminatedSlaves: 'false',
            IdleShutdown: '10',
            Logging: 'Standard',
            PreJobTaskMode: 'Conservative',
            Region: 'eu-west-1',
            ResourceTracker: 'true',
            StaggerInstances: '50',
            State: 'Disabled',
            StrictHardCap: 'false',
        };
        const defaultConvertedPluginConfig = {
            AWSInstanceStatus: 'Disabled',
            DeleteInterruptedSlaves: false,
            DeleteTerminatedSlaves: false,
            IdleShutdown: 10,
            Logging: 'Standard',
            PreJobTaskMode: 'Conservative',
            Region: 'eu-west-1',
            ResourceTracker: true,
            StaggerInstances: 50,
            State: 'Disabled',
            StrictHardCap: false,
        };
        // WHEN
        // Need this trick so TS allows to pass config with string properties.
        const config = defaultPluginConfig;
        const returnValue = conversion_1.convertSpotEventPluginSettings(config);
        // THEN
        expect(returnValue).toEqual(defaultConvertedPluginConfig);
    });
});
describe('convertToInt()', () => {
    test.each([
        ['10', 10],
        [10, 10],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.convertToInt(input, 'propertyName');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10.6,
        [],
        {},
        'string',
        undefined,
    ])('throws an error with %p', input => {
        // WHEN
        function callingConvertToInt() {
            conversion_1.convertToInt(input, propertyName);
        }
        // THEN
        expect(callingConvertToInt).toThrowError(`The value of ${propertyName} should be an integer. Received: ${input}`);
    });
});
describe('convertToBoolean()', () => {
    test.each([
        [true, true],
        ['true', true],
        [false, false],
        ['false', false],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.convertToBoolean(input, 'property');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10.6,
        [],
        {},
        'string',
        undefined,
    ])('throws an error with %p', input => {
        // WHEN
        function callingConvertToBoolean() {
            conversion_1.convertToBoolean(input, propertyName);
        }
        // THEN
        expect(callingConvertToBoolean).toThrowError(`The value of ${propertyName} should be a boolean. Received: ${input}`);
    });
});
describe('validateString()', () => {
    test.each([
        ['string', 'string'],
        ['10', '10'],
        ['true', 'true'],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.validateString(input, 'propertyName');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10,
        [],
        {},
        undefined,
    ])('throws an error with %p', input => {
        // WHEN
        function callingValidateString() {
            conversion_1.validateString(input, propertyName);
        }
        // THEN
        expect(callingValidateString).toThrowError(`The value of ${propertyName} should be a string. Received: ${input} of type ${typeof (input)}`);
    });
});
describe('validateStringOptional()', () => {
    test.each([
        ['string', 'string'],
        ['10', '10'],
        ['true', 'true'],
        [undefined, undefined],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.validateStringOptional(input, 'propertyName');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10,
        [],
        {},
    ])('throws an error with %p', input => {
        // WHEN
        function callingValidateStringOptional() {
            conversion_1.validateStringOptional(input, propertyName);
        }
        // THEN
        expect(callingValidateStringOptional).toThrowError(`The value of ${propertyName} should be a string. Received: ${input} of type ${typeof (input)}`);
    });
});
describe('validateArray', () => {
    test.each([
        undefined,
        {},
        [],
    ])('throws with invalid input %p', (invalidInput) => {
        // WHEN
        function callingValidateArray() {
            conversion_1.validateArray(invalidInput, propertyName);
        }
        // THEN
        expect(callingValidateArray).toThrowError(`${propertyName} should be an array with at least one element.`);
    });
    test('passes with not empty array', () => {
        // GIVEN
        const nonEmptyArray = ['value'];
        // WHEN
        function callingValidateArray() {
            conversion_1.validateArray(nonEmptyArray, 'propertyName');
        }
        // THEN
        expect(callingValidateArray).not.toThrowError();
    });
});
describe('isValidTagSpecification', () => {
    // Valid tag specifications
    const validTagSpecification = {
        ResourceType: 'type',
        Tags: [{
                Key: 'key',
                Value: 'value',
            }],
    };
    // Invalid tag specifications
    const noResourceType = {};
    const resourceTypeNotSting = {
        ResourceType: 10,
    };
    const noTags = {
        ResourceType: 'type',
    };
    const tagsNotArray = {
        ResourceType: 'type',
        Tags: '',
    };
    const tagElementUndefined = {
        ResourceType: 'type',
        Tags: [undefined],
    };
    const tagElementWrongType = {
        ResourceType: 'type',
        Tags: [''],
    };
    const tagElementNoKey = {
        ResourceType: 'type',
        Tags: [{}],
    };
    const tagElementKeyNotString = {
        ResourceType: 'type',
        Tags: [{
                Key: 10,
            }],
    };
    const tagElementNoValue = {
        ResourceType: 'type',
        Tags: [{
                Key: 'key',
            }],
    };
    const tagElementValueNotString = {
        ResourceType: 'type',
        Tags: [{
                Key: 'key',
                Value: 10,
            }],
    };
    test.each([
        undefined,
        [],
        '',
        noResourceType,
        resourceTypeNotSting,
        noTags,
        tagsNotArray,
        tagElementUndefined,
        tagElementWrongType,
        tagElementNoKey,
        tagElementKeyNotString,
        tagElementNoValue,
        tagElementValueNotString,
    ])('returns false with invalid input %p', (invalidInput) => {
        // WHEN
        const result = conversion_1.isValidTagSpecification(invalidInput);
        // THEN
        expect(result).toBeFalsy();
    });
    test('returns true with a valid input', () => {
        // WHEN
        const result = conversion_1.isValidTagSpecification(validTagSpecification);
        // THEN
        expect(result).toBeTruthy();
    });
});
describe('validateProperty', () => {
    test('throws with invalid input', () => {
        // WHEN
        function returnFalse(_input) {
            return false;
        }
        function callingValidateProperty() {
            conversion_1.validateProperty(returnFalse, 'anyValue', propertyName);
        }
        // THEN
        expect(callingValidateProperty).toThrowError(`${propertyName} type is not valid.`);
    });
    test('passes with a valid input', () => {
        // WHEN
        function returnTrue(_input) {
            return true;
        }
        function callingValidateProperty() {
            conversion_1.validateProperty(returnTrue, 'anyValue', 'propertyName');
        }
        // THEN
        expect(callingValidateProperty).not.toThrowError();
    });
});
describe('validateLaunchTemplateSpecification', () => {
    test('accepts launch template specification with id', () => {
        // GIVEN
        const spec = {
            Version: '1',
            LaunchTemplateId: 'id',
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateSpecification(spec, propertyName))
            // THEN
            .not.toThrow();
    });
    test('accepts launch template specification with name', () => {
        // GIVEN
        const spec = {
            Version: '1',
            LaunchTemplateName: 'name',
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateSpecification(spec, propertyName))
            // THEN
            .not.toThrow();
    });
    test('throws if both id and name are specified', () => {
        // GIVEN
        const id = 'id';
        const name = 'name';
        const spec = {
            Version: '1',
            LaunchTemplateId: id,
            LaunchTemplateName: name,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateSpecification(spec, propertyName))
            // THEN
            .toThrowError(`Exactly one of ${propertyName}.LaunchTemplateId or ${propertyName}.LaunchTemplateName must be specified, but got: ${id} and ${name} respectively`);
    });
    test('throws if neither id or name are specified', () => {
        // GIVEN
        const spec = {
            Version: '1',
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateSpecification(spec, propertyName))
            // THEN
            .toThrowError(`Exactly one of ${propertyName}.LaunchTemplateId or ${propertyName}.LaunchTemplateName must be specified, but got: ${undefined} and ${undefined} respectively`);
    });
    test('throws if id is invalid', () => {
        // GIVEN
        const invalidValue = 123;
        const spec = {
            Version: '1',
            // @ts-ignore
            LaunchTemplateId: invalidValue,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateSpecification(spec, propertyName))
            // THEN
            .toThrowError(new RegExp(`The value of ${propertyName}.LaunchTemplateId should be a string. Received: ${invalidValue} of type ${typeof (invalidValue)}`));
    });
    test('throws if name is invalid', () => {
        // GIVEN
        const invalidValue = 123;
        const spec = {
            Version: '1',
            // @ts-ignore
            LaunchTemplateName: invalidValue,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateSpecification(spec, propertyName))
            // THEN
            .toThrowError(new RegExp(`The value of ${propertyName}.LaunchTemplateName should be a string. Received: ${invalidValue} of type ${typeof (invalidValue)}`));
    });
    test('throws if version is invalid', () => {
        // GIVEN
        const invalidValue = 123;
        const spec = {
            LaunchTemplateId: '',
            // @ts-ignore
            Version: invalidValue,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateSpecification(spec, propertyName))
            // THEN
            .toThrowError(`The value of ${propertyName}.Version should be a string. Received: ${invalidValue} of type ${typeof (invalidValue)}`);
    });
});
describe('validateLaunchTemplateOverrides', () => {
    test('accepts valid overrides', () => {
        // GIVEN
        const overrides = {
            AvailabilityZone: 'AvailabilityZone',
            InstanceType: 'InstanceType',
            SpotPrice: 'SpotPrice',
            SubnetId: 'SubnetId',
            WeightedCapacity: 123,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateOverrides(overrides, propertyName))
            // THEN
            .not.toThrow();
    });
    test('throws if AvailabilityZone is invalid', () => {
        // GIVEN
        const invalidValue = 123;
        const overrides = {
            // @ts-ignore
            AvailabilityZone: invalidValue,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateOverrides(overrides, propertyName))
            // THEN
            .toThrowError(new RegExp(`The value of ${propertyName}.AvailabilityZone should be a string. Received: ${invalidValue} of type ${typeof (invalidValue)}`));
    });
    test('throws if InstanceType is invalid', () => {
        // GIVEN
        const invalidValue = 123;
        const overrides = {
            // @ts-ignore
            InstanceType: invalidValue,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateOverrides(overrides, propertyName))
            // THEN
            .toThrowError(new RegExp(`The value of ${propertyName}.InstanceType should be a string. Received: ${invalidValue} of type ${typeof (invalidValue)}`));
    });
    test('throws if SpotPrice is invalid', () => {
        // GIVEN
        const invalidValue = 123;
        const overrides = {
            // @ts-ignore
            SpotPrice: invalidValue,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateOverrides(overrides, propertyName))
            // THEN
            .toThrowError(new RegExp(`The value of ${propertyName}.SpotPrice should be a string. Received: ${invalidValue} of type ${typeof (invalidValue)}`));
    });
    test('throws if SubnetId is invalid', () => {
        // GIVEN
        const invalidValue = 123;
        const overrides = {
            // @ts-ignore
            SubnetId: invalidValue,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateOverrides(overrides, propertyName))
            // THEN
            .toThrowError(new RegExp(`The value of ${propertyName}.SubnetId should be a string. Received: ${invalidValue} of type ${typeof (invalidValue)}`));
    });
    test('throws if WeightedCapacity is invalid', () => {
        // GIVEN
        const invalidValue = 'WeightedCapacity';
        const overrides = {
            // @ts-ignore
            WeightedCapacity: invalidValue,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateOverrides(overrides, propertyName))
            // THEN
            .toThrowError(`${propertyName}.WeightedCapacity type is not valid.`);
    });
});
describe('validateLaunchTemplateConfigs', () => {
    const LaunchTemplateSpec = {
        Version: '1',
        LaunchTemplateId: 'id',
    };
    const Overrides = [];
    test('accepts valid LaunchTemplateConfig', () => {
        // GIVEN
        const config = {
            LaunchTemplateSpecification: LaunchTemplateSpec,
            Overrides,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateConfigs([config], propertyName))
            // THEN
            .not.toThrow();
    });
    test('throws when not given an array of LaunchTemplateConfigs', () => {
        // WHEN
        expect(() => {
            // @ts-ignore
            conversion_1.validateLaunchTemplateConfigs({}, propertyName);
        })
            // THEN
            .toThrowError(`${propertyName} should be an array with at least one element.`);
    });
    test('throws when LaunchTemplateSpecification is the wrong type', () => {
        // GIVEN
        const invalidValue = 123;
        const config = {
            // @ts-ignore
            LaunchTemplateSpecification: invalidValue,
            Overrides,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateConfigs([config], propertyName))
            // THEN
            .toThrowError(`${propertyName}[0].LaunchTemplateSpecification type is not valid.`);
    });
    test('throws when Version is invalid', () => {
        // GIVEN
        const invalidValue = 123;
        const config = {
            LaunchTemplateSpecification: {
                LaunchTemplateId: '',
                // @ts-ignore
                Version: invalidValue,
            },
            Overrides,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateConfigs([config], propertyName))
            // THEN
            .toThrowError(`The value of ${propertyName}[0].LaunchTemplateSpecification.Version should be a string. Received: ${invalidValue} of type ${typeof (invalidValue)}`);
    });
    test('throws when Overrides is not an array', () => {
        // GIVEN
        const config = {
            LaunchTemplateSpecification: LaunchTemplateSpec,
            // @ts-ignore
            Overrides: 123,
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateConfigs([config], propertyName))
            // THEN
            .toThrowError(`${propertyName}[0].Overrides type is not valid.`);
    });
    test('throws when a LaunchTemplateOverride is invalid', () => {
        // GIVEN
        const invalidValue = 123;
        const config = {
            LaunchTemplateSpecification: LaunchTemplateSpec,
            Overrides: [{
                    // @ts-ignore
                    AvailabilityZone: invalidValue,
                }],
        };
        // WHEN
        expect(() => conversion_1.validateLaunchTemplateConfigs([config], propertyName))
            // THEN
            .toThrowError(`The value of ${propertyName}[0].Overrides[0].AvailabilityZone should be a string. Received: ${invalidValue} of type ${typeof (invalidValue)}`);
    });
});
//# sourceMappingURL=data:application/json;base64,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