"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../core/lib");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_2 = require("../lib");
const asset_constants_1 = require("./asset-constants");
let app;
let stack;
let wfstack;
let vpc;
let renderQueue;
let rcsImage;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'infraStack', {
        env: {
            region: 'us-east-1',
        },
    });
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
    const version = new lib_2.VersionQuery(stack, 'VersionQuery');
    renderQueue = new lib_2.RenderQueue(stack, 'RQ', {
        vpc,
        images: { remoteConnectionServer: rcsImage },
        repository: new lib_2.Repository(stack, 'Repository', {
            vpc,
            version,
            secretsManagementSettings: { enabled: false },
        }),
        trafficEncryption: { externalTLS: { enabled: false } },
        version,
    });
    wfstack = new core_1.Stack(app, 'workerFleetStack', {
        env: {
            region: 'us-east-1',
        },
    });
});
test('default worker fleet is created correctly', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.large',
        spotPrice: assert_1.ABSENT,
    }));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                'GroupId',
            ],
        },
        GroupId: {
            'Fn::ImportValue': 'infraStack:ExportsOutputFnGetAttRQLBSecurityGroupAC643AEDGroupId8F9F7830',
        },
    }));
    assert_1.expect(wfstack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/workerFleet',
    }));
    expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadataEntry[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    expect(fleet.node.metadataEntry[1].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadataEntry[1].data).toContain('being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy');
});
test('security group is added to fleet after its creation', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    fleet.addSecurityGroup(aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
        allowAllOutbound: false,
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            {
                'Fn::GetAtt': [
                    stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                    'GroupId',
                ],
            },
            'sg-123456789',
        ],
    }));
});
test('WorkerFleet uses given security group', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
            allowAllOutbound: false,
        }),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            'sg-123456789',
        ],
    }));
});
describe('allowing log listener port', () => {
    test('from CIDR', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        fleet.allowListenerPortFrom(aws_ec2_1.Peer.ipv4('127.0.0.1/24').connections);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
            SecurityGroupIngress: [
                {
                    CidrIp: '127.0.0.1/24',
                    Description: 'Worker remote command listening port',
                    FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
                    IpProtocol: 'tcp',
                    ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
                },
            ],
        }));
    });
    test('to CIDR', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        fleet.allowListenerPortTo(aws_ec2_1.Peer.ipv4('127.0.0.1/24').connections);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
            SecurityGroupIngress: [
                {
                    CidrIp: '127.0.0.1/24',
                    Description: 'Worker remote command listening port',
                    FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
                    IpProtocol: 'tcp',
                    ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
                },
            ],
        }));
    });
    test('from SecurityGroup', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
        fleet.allowListenerPortFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        }));
    });
    test('to SecurityGroup', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
        fleet.allowListenerPortTo(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        }));
    });
    test('from other stack', () => {
        const otherStack = new core_1.Stack(app, 'otherStack', {
            env: { region: 'us-east-1' },
        });
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
        fleet.allowListenerPortFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        }));
    });
    test('to other stack', () => {
        const otherStack = new core_1.Stack(app, 'otherStack', {
            env: { region: 'us-east-1' },
        });
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
        fleet.allowListenerPortTo(securityGroup);
        // THEN
        assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        }));
    });
});
test('default worker fleet is created correctly with linux image', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration'));
});
test('default worker fleet is created correctly with spot config', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        spotPrice: 2.5,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SpotPrice: '2.5',
    }));
});
test('default worker fleet is not created with incorrect spot config', () => {
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT + 1,
        });
    }).toThrowError(/Invalid value: 256 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT / 2,
        });
    }).toThrowError(/Invalid value: 0.0005 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
});
test('default worker fleet is created correctly custom Instance type', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.medium',
    }));
});
test.each([
    'test-prefix/',
    '',
])('default worker fleet is created correctly with custom LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'workerFleet';
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, id, {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: testPrefix + id,
    }));
});
test('default worker fleet is created correctly custom subnet values', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        healthCheckConfig: {
            port: 6161,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: [{
                Ref: 'VPC1AzPublicSubnet1Subnet9649CC17',
            }],
    }));
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': [
            '',
            [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "\')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'set -e\n' +
                    'chmod +x \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n\'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\' -i us-east-1 ',
                { Ref: 'workerFleetStringParameterDB3717DA' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                '\' \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                "'\nmkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                `' '' '' '' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION}' ${lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`,
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
            ],
        ],
    });
});
test('default worker fleet is created correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        groups: ['A', 'B'],
        pools: ['C', 'D'],
        region: 'E',
    });
    // THEN
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': ['', [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' -i us-east-1 ",
                { Ref: 'workerFleetStringParameterDB3717DA' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                {
                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Bucket,
                },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                "'\nmkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                `' 'a,b' 'c,d' 'E' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION}' ${lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`,
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                },
                            ],
                        },
                    ],
                },
            ]],
    });
});
test('worker fleet does validation correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // group name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'none'],
        });
    }).toThrowError();
    // group name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet1', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'no ne'],
        });
    }).toThrowError(/Invalid value: no ne for property 'groups'/);
    // pool name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // pool name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet3', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // region as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet4', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none',
        });
    }).toThrowError(/Invalid value: none for property 'region'/);
    // region as 'all'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet5', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'all',
        });
    }).toThrowError(/Invalid value: all for property 'region'/);
    // region as 'unrecognized'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet6', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'unrecognized',
        });
    }).toThrowError(/Invalid value: unrecognized for property 'region'/);
    // region with invalid characters
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet7', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none@123',
        });
    }).toThrowError(/Invalid value: none@123 for property 'region'/);
    // region with reserved name as substring
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet8', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none123',
        });
    }).not.toThrowError();
    // region with case-insensitive name
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet9', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'None',
        });
    }).toThrowError(/Invalid value: None for property 'region'/);
});
describe('Block Device Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Warning if no BlockDevices provided', () => {
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadataEntry[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    });
    test('No Warnings if Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: true,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadataEntry).toHaveLength(0);
    });
    test('Warnings if non-Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: false,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadataEntry[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('Warnings for BlockDevices without encryption specified', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadataEntry[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('No warnings for Ephemeral blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                    VirtualName: 'ephemeral0',
                },
            ],
        }));
        expect(fleet.node.metadataEntry).toHaveLength(0);
    });
    test('No warnings for Suppressed blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                },
            ],
        }));
        expect(fleet.node.metadataEntry).toHaveLength(0);
    });
});
describe('HealthMonitor Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Monitor is configured for Windows', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('&\'C:/temp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'63415\' \'10.1.9.2\'')));
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
    });
    test('Monitor is configured for Linux', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'63415\' \'10.1.9.2\'')));
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
    });
    test('UserData is added', () => {
        // WHEN
        class UserDataProvider extends lib_2.InstanceUserDataProvider {
            preCloudWatchAgent(host) {
                host.userData.addCommands('echo preCloudWatchAgent');
            }
            preRenderQueueConfiguration(host) {
                host.userData.addCommands('echo preRenderQueueConfiguration');
            }
            preWorkerConfiguration(host) {
                host.userData.addCommands('echo preWorkerConfiguration');
            }
            postWorkerLaunch(host) {
                host.userData.addCommands('echo postWorkerLaunch');
            }
        }
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            userDataProvider: new UserDataProvider(wfstack, 'UserDataProvider'),
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        expect(userData).toContain('echo preCloudWatchAgent');
        expect(userData).toContain('echo preRenderQueueConfiguration');
        expect(userData).toContain('echo preWorkerConfiguration');
        expect(userData).toContain('echo postWorkerLaunch');
    });
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'WorkerInstanceFleet',
        createConstruct: () => {
            // GIVEN
            const healthMonitorStack = new core_1.Stack(app, 'HealthMonitorStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            const healthMonitor = new lib_1.HealthMonitor(healthMonitorStack, 'healthMonitor', {
                vpc,
            });
            const deviceName = '/dev/xvda';
            // WHEN
            new lib_2.WorkerInstanceFleet(wfstack, 'WorkerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                healthMonitor,
                blockDevices: [{
                        deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                    }],
            });
            return wfstack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
test('worker fleet signals when non-zero minCapacity', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        minCapacity: 1,
    });
    // WHEN
    const userData = fleet.fleet.userData.render();
    // THEN
    expect(userData).toContain('cfn-signal');
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
            },
        },
    }, assert_1.ResourcePart.CompleteDefinition));
    // [0] = warning about block devices. [1] = warning about no health monitor
    expect(fleet.node.metadataEntry).toHaveLength(2);
});
test('worker fleet does not signal when zero minCapacity', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        minCapacity: 0,
    });
    // WHEN
    const userData = fleet.fleet.userData.render();
    // THEN
    // There should be no cfn-signal call in the UserData.
    expect(userData).not.toContain('cfn-signal');
    // Make sure we don't have a CreationPolicy
    assert_1.expect(wfstack).notTo(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: assert_1.objectLike({}),
    }, assert_1.ResourcePart.CompleteDefinition));
    // There should be a warning in the construct's metadata about deploying with no capacity.
    expect(fleet.node.metadataEntry).toHaveLength(3);
    // [0] = warning about block devices. [2] = warning about no health monitor
    expect(fleet.node.metadataEntry[1].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadataEntry[1].data).toMatch(/Deploying with 0 minimum capacity./);
});
describe('secrets management enabled', () => {
    let props;
    // GIVEN
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_2.VersionQuery(stack, 'VersionQuery');
        renderQueue = new lib_2.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_2.Repository(stack, 'Repository', {
                vpc,
                version,
            }),
            version,
        });
        wfstack = new core_1.Stack(app, 'workerFleetStack');
        props = {
            renderQueue,
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({}),
        };
    });
    test('vpc subnets not specified => warns about dedicated subnets', () => {
        // WHEN
        const workerInstanceFleet = new lib_2.WorkerInstanceFleet(wfstack, 'WorkerInstanceFleet', props);
        // THEN
        expect(workerInstanceFleet.node.metadataEntry).toContainEqual(expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components',
        }));
    });
    test('vpc subnets specified => does not emit dedicated subnets warning', () => {
        // WHEN
        const workerInstanceFleet = new lib_2.WorkerInstanceFleet(wfstack, 'WorkerInstanceFleet', {
            ...props,
            vpcSubnets: {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
        });
        // THEN
        expect(workerInstanceFleet.node.metadataEntry).not.toContainEqual(expect.objectContaining({
            type: 'aws:cdk:warning',
            data: expect.stringMatching(/dedicated subnet/i),
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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