"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const wait_for_stable_service_1 = require("../lib/wait-for-stable-service");
describe('WaitForStableService', () => {
    let app;
    let stack;
    let isolatedStack;
    let cluster;
    let taskDefinition;
    let service;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'Stack');
        isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        cluster = new aws_ecs_1.Cluster(stack, 'Cluster');
        cluster.addCapacity('ASG', {
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            minCapacity: 1,
            maxCapacity: 1,
        });
        taskDefinition = new aws_ecs_1.Ec2TaskDefinition(stack, 'RCSTask');
        taskDefinition.addContainer('Test', {
            image: aws_ecs_1.ContainerImage.fromAsset(__dirname),
            memoryLimitMiB: 7500,
        });
        service = new aws_ecs_1.Ec2Service(stack, 'Service', {
            cluster,
            taskDefinition,
        });
    });
    test('creates a custom resource', () => {
        // WHEN
        new wait_for_stable_service_1.WaitForStableService(isolatedStack, 'WaitForStableService', {
            service,
        });
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_WaitForStableService', {
            cluster: isolatedStack.resolve(cluster.clusterArn),
            services: [isolatedStack.resolve(service.serviceArn)],
        }));
    });
    test('creates lambda correctly', () => {
        // WHEN
        new wait_for_stable_service_1.WaitForStableService(isolatedStack, 'WaitForStableService', {
            service,
        });
        assert_1.expect(isolatedStack).to(assert_1.countResourcesLike('AWS::Lambda::Function', 1, {
            Handler: 'wait-for-stable-service.wait',
            Environment: {
                Variables: {
                    DEBUG: 'false',
                },
            },
            Runtime: 'nodejs16.x',
            Timeout: 900,
        }));
    });
    test('adds policies to the lambda role', () => {
        // WHEN
        new wait_for_stable_service_1.WaitForStableService(isolatedStack, 'WaitForStableService', {
            service,
        });
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
            ManagedPolicyArns: assert_1.arrayWith(isolatedStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole').managedPolicyArn)),
            Policies: [{
                    PolicyDocument: assert_1.objectLike({
                        Statement: [{
                                Action: 'ecs:DescribeServices',
                                Effect: 'Allow',
                                Resource: assert_1.arrayWith(isolatedStack.resolve(cluster.clusterArn), isolatedStack.resolve(service.serviceArn)),
                            }],
                    }),
                }],
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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