"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('VersionQuery constructor full version', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery', { version: '10.1.9.2' });
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
        forceRun: assert_1.ABSENT,
        versionString: '10.1.9.2',
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: 'lambda.amazonaws.com',
                    },
                },
            ],
        },
        ManagedPolicyArns: [
            {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole',
                    ],
                ],
            },
        ],
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'version-provider.handler',
        Role: {
            'Fn::GetAtt': [
                assert_1.stringLike('SingletonLambda*ServiceRole*'),
                'Arn',
            ],
        },
        Runtime: 'nodejs16.x',
    }));
});
test('VersionQuery constructor no versionString', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery');
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
        forceRun: assert_1.stringLike('*'),
        versionString: assert_1.ABSENT,
    }));
});
test.each([
    ['10.1.9'],
    ['10.1'],
    ['10'],
])('VersionQuery constructor partial version: %s', (version) => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery', { version });
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
        versionString: version,
        forceRun: assert_1.stringLike('*'),
    }));
});
describe('VersionQuery.linuxInstallers', () => {
    let customResource;
    let versionQuery;
    let stack;
    beforeAll(() => {
        // GIVEN
        const app = new core_1.App();
        stack = new core_1.Stack(app, 'Stack');
        versionQuery = new lib_1.VersionQuery(stack, 'VersionQuery');
        customResource = versionQuery.node.findChild('DeadlineResource');
    });
    describe('.repository', () => {
        let repoInstaller;
        beforeAll(() => {
            // WHEN
            repoInstaller = versionQuery.linuxInstallers.repository;
        });
        test('S3 bucket from Custom::RFDK_DEADLINE_INSTALLERS "S3Bucket" attribute', () => {
            // THEN
            expect(stack.resolve(repoInstaller.s3Bucket.bucketName))
                .toEqual(stack.resolve(customResource.getAtt('S3Bucket')));
        });
        test('S3 object key from Custom::RFDK_DEADLINE_INSTALLERS "LinuxRepositoryInstaller" attribute', () => {
            // THEN
            expect(stack.resolve(repoInstaller.objectKey))
                .toEqual(stack.resolve(customResource.getAtt('LinuxRepositoryInstaller')));
        });
    });
    describe('.client', () => {
        let clientInstaller;
        beforeAll(() => {
            // WHEN
            clientInstaller = versionQuery.linuxInstallers.client;
        });
        test('S3 bucket from Custom::RFDK_DEADLINE_INSTALLERS "S3Bucket" attribute', () => {
            // THEN
            expect(stack.resolve(clientInstaller.s3Bucket.bucketName))
                .toEqual(stack.resolve(customResource.getAtt('S3Bucket')));
        });
        test('S3 object key from Custom::RFDK_DEADLINE_INSTALLERS "LinuxClientInstaller" attribute', () => {
            // THEN
            expect(stack.resolve(clientInstaller.objectKey))
                .toEqual(stack.resolve(customResource.getAtt('LinuxClientInstaller')));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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