"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('ThinkboxDockerRecipes', () => {
    let app;
    let stack;
    let stage;
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    const MAJOR_VERSION = 10;
    const MINOR_VERSION = 1;
    const RELEASE_VERSION = 9;
    const PATCH_VERSION = 2;
    const RELEASE_VERSION_STRING = `${MAJOR_VERSION}.${MINOR_VERSION}.${RELEASE_VERSION}`;
    const FULL_VERSION_STRING = `${RELEASE_VERSION_STRING}.${PATCH_VERSION}`;
    const RCS_RECIPE_NAME = 'rcs';
    const RCS_RECIPE = {
        description: 'rcs',
        title: 'rcs',
        buildArgs: {
            DL_VERSION: FULL_VERSION_STRING,
            a: 'a',
            b: 'b',
        },
        target: undefined,
    };
    const LICENSE_FORWARDER_RECIPE_NAME = 'license-forwarder';
    const LICENSE_FORWARDER_RECIPE = {
        title: 'license-forwarder',
        description: 'license-forwarder',
        buildArgs: {
            DL_VERSION: FULL_VERSION_STRING,
            c: 'c',
            d: 'd',
        },
        target: 'lf',
    };
    beforeEach(() => {
        app = new core_1.App();
        class TestStage extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        stage = new TestStage({
            path: STAGE_PATH,
            manifest: {
                schema: 1,
                version: FULL_VERSION_STRING,
                recipes: {
                    [RCS_RECIPE_NAME]: RCS_RECIPE,
                    [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
                },
            },
        });
        stack = new core_1.Stack(app, 'Stack');
    });
    /**
     * Tests that the remote connection server and license forwarder Docker recipes
     * create the correct {@link DockerImageAsset} instances using the build arguments,
     * and target from the supplied manifest.
     *
     * This is done by comparing the `assetHash` property of the expected vs actual
     * DockerImageAsset instances.
     */
    test.each([
        [
            'remoteConnnectionServer',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: RCS_RECIPE.buildArgs,
                    target: RCS_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.remoteConnectionServer;
            },
        ],
        [
            'licenseForwarder',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: LICENSE_FORWARDER_RECIPE.buildArgs,
                    target: LICENSE_FORWARDER_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.licenseForwarder;
            },
        ],
    ])('has proper %p container image', (_imageName, getExpected, getActual) => {
        // GIVEN
        const expectedImage = getExpected();
        // WHEN
        const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
            stage,
        });
        const actualImage = getActual(recipes);
        // THEN
        expect(actualImage.assetHash).toEqual(expectedImage.assetHash);
    });
    describe('.version', () => {
        test('creates a VersionQuery when referenced', () => {
            // GIVEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            // WHEN
            recipes.version;
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
                forceRun: assert_1.stringLike('*'),
                versionString: RELEASE_VERSION_STRING,
            }));
        });
        test('does not create a VersionQuery when not referenced', () => {
            // GIVEN
            new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_DEADLINE_INSTALLERS'));
        });
        test('.linuxInstallers.client creates an Asset using the client installer', () => {
            // GIVEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            // WHEN
            const clientInstaller = recipes.version.linuxInstallers.client;
            // THEN
            const asset = recipes.node.findChild('ClientInstallerAsset');
            expect(clientInstaller.s3Bucket).toEqual(asset.bucket);
            expect(clientInstaller.objectKey).toEqual(asset.s3ObjectKey);
        });
        test('.linuxInstallers.client successive accesses return the same bucket/key', () => {
            // GIVEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            // WHEN
            const firstClientInstaller = recipes.version.linuxInstallers.client;
            const secondClientInstaller = recipes.version.linuxInstallers.client;
            // THEN
            expect(firstClientInstaller.objectKey).toBe(secondClientInstaller.objectKey);
            expect(firstClientInstaller.s3Bucket).toBe(secondClientInstaller.s3Bucket);
        });
        describe('.isLessThan()', () => {
            let recipes;
            beforeEach(() => {
                // GIVEN
                recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                    stage,
                });
            });
            test.each([
                [{ majorOffset: -1 }, false],
                [{ minorOffset: -1 }, false],
                [{ releaseOffset: -1 }, false],
                [{}, false],
                [{ majorOffset: 1 }, true],
                [{ minorOffset: 1 }, true],
                [{ releaseOffset: 1 }, true],
            ])('%s = %s', ({ majorOffset, minorOffset, releaseOffset }, expectedResult) => {
                // GIVEN
                majorOffset = majorOffset ?? 0;
                minorOffset = minorOffset ?? 0;
                releaseOffset = releaseOffset ?? 0;
                const other = new lib_1.Version([
                    MAJOR_VERSION + majorOffset,
                    MINOR_VERSION + minorOffset,
                    RELEASE_VERSION + releaseOffset,
                    0,
                ]);
                // WHEN
                const result = recipes.version.isLessThan(other);
                // THEN
                expect(result).toEqual(expectedResult);
            });
        });
        test('.linuxfullVersionString matches the stage manifest version', () => {
            // GIVEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            // WHEN
            const linuxFullVersionString = recipes.version.linuxFullVersionString();
            // THEN
            expect(linuxFullVersionString).toEqual(FULL_VERSION_STRING);
        });
    });
    test.each([
        ['rcs', {
                [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
            }],
        ['license-forwarder', {
                [RCS_RECIPE_NAME]: RCS_RECIPE,
            }],
    ])('manifest missing required recipe %s throws error', (_recipeName, recipes) => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        class StageWithPublicConstructor extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        const manifest = {
            recipes,
            schema: 1,
            version: '1.2.3.4',
        };
        // WHEN
        expect(() => {
            new lib_1.ThinkboxDockerRecipes(isolatedStack, 'Recipes', {
                stage: new StageWithPublicConstructor({
                    manifest,
                    path: '/some/path',
                }),
            });
        })
            // THEN
            .toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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