"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const deployment_instance_1 = require("../../core/lib/deployment-instance");
const lib_1 = require("../lib");
const secrets_management_1 = require("../lib/secrets-management");
class MockUserData extends aws_ec2_1.UserData {
    constructor() {
        super();
        this.addCommands = jest.fn();
        this.addCommands = jest.fn();
        this.addOnExitCommands = jest.fn();
        this.render = jest.fn();
        this.addExecuteFileCommand = jest.fn();
        this.addS3DownloadCommand = jest.fn();
        this.addSignalOnExitCommand = jest.fn();
    }
}
class MockDeploymentInstance extends deployment_instance_1.DeploymentInstance {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mockUserData = new MockUserData();
    }
    get userData() {
        return this.mockUserData;
    }
}
function writeSynthedTemplate(stack, outputFile) {
    const template = assert_1.SynthUtils.synthesize(stack).template;
    fs.writeFileSync(outputFile, JSON.stringify(template, null, 2), { encoding: 'utf8' });
}
const DEADLINE_CLIENT_SUBNET_NAME = 'DeadlineClient';
const RENDER_QUEUE_ALB_SUBNET_NAME = 'RenderQueueALB';
describe('SecretsManagementIdentityRegistration', () => {
    let app;
    let dependencyStack;
    let deploymentInstanceStack;
    let stack;
    let vpc;
    let version;
    let repository;
    let deploymentInstance;
    let deploymentInstanceRole;
    let renderQueueSubnets;
    let target;
    // @ts-ignore
    function writeSynthedTemplates() {
        writeSynthedTemplate(deploymentInstanceStack, 'deployment-instance-stack.json');
        writeSynthedTemplate(stack, 'secrets-management-stack.json');
    }
    beforeEach(() => {
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack');
        deploymentInstanceStack = new core_1.Stack(app, 'DeploymentInstanceStack');
        stack = new core_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc', {
            subnetConfiguration: [
                {
                    name: RENDER_QUEUE_ALB_SUBNET_NAME,
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
                    cidrMask: 28,
                },
                {
                    name: 'Public',
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 28,
                },
                {
                    name: DEADLINE_CLIENT_SUBNET_NAME,
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 28,
                },
            ],
        });
        version = new lib_1.VersionQuery(dependencyStack, 'Version');
        deploymentInstance = new MockDeploymentInstance(deploymentInstanceStack, 'DeploymentInstance', {
            vpc,
        });
        renderQueueSubnets = vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME });
    });
    describe('when Repository uses secrets management', () => {
        beforeEach(() => {
            // GIVEN
            repository = new lib_1.Repository(dependencyStack, 'Repository', {
                version,
                vpc,
                secretsManagementSettings: {
                    enabled: true,
                },
            });
            jest.spyOn(repository, 'configureClientInstance');
            // Get a reference to the DeploymentInstance's IAM role L1 resource
            deploymentInstanceRole = (deploymentInstance
                .node.findChild('ASG')
                .node.findChild('InstanceRole')
                .node.defaultChild);
        });
        function createTarget() {
            target = new secrets_management_1.SecretsManagementIdentityRegistration(stack, 'IdentityRegistration', {
                deploymentInstance,
                renderQueueSubnets,
                repository,
                version,
            });
        }
        describe('Deadline Client installer', () => {
            test('grant S3 read to client installer', () => {
                // WHEN
                createTarget();
                // THEN
                assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: assert_1.arrayWith(...deploymentInstanceStack.resolve([
                                version.linuxInstallers.client.s3Bucket.bucketArn,
                                version.linuxInstallers.client.s3Bucket.arnForObjects(version.linuxInstallers.client.objectKey),
                            ])),
                        }),
                    },
                    Roles: [
                        deploymentInstanceStack.resolve(deploymentInstanceRole.ref),
                    ],
                }));
            });
            test('downloads and executes Client installer', () => {
                // GIVEN
                const clientInstallerLocalFilename = 'clientInstallerLocalFilename';
                const userData = deploymentInstance.userData;
                userData.addS3DownloadCommand.mockReturnValueOnce(clientInstallerLocalFilename);
                // WHEN
                createTarget();
                // THEN
                expect(userData.addS3DownloadCommand).toHaveBeenCalledWith({
                    bucket: version.linuxInstallers.client.s3Bucket,
                    bucketKey: version.linuxInstallers.client.objectKey,
                });
                expect(userData.addCommands).toHaveBeenCalledWith(`chmod +x "${clientInstallerLocalFilename}"`);
                expect(userData.addCommands).toHaveBeenCalledWith([
                    // This is required b/c USER and HOME environment variables are not defined when running
                    // user-data
                    'sudo', '--login',
                    // Run the Deadline Client installer
                    `"${clientInstallerLocalFilename}"`,
                    '--mode', 'unattended',
                    '--connectiontype', 'Remote',
                    '--proxyrootdir', '127.0.0.1:8080',
                    '--noguimode', 'true',
                    '--slavestartup', 'false',
                    '--launcherdaemon', 'false',
                    '--restartstalled', 'true',
                    '--autoupdateoverride', 'False',
                ].join(' '));
            });
        });
        test('grants DeploymentInstance role permissions to describe subnets', () => {
            // WHEN
            createTarget();
            // THEN
            assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 'ec2:DescribeSubnets',
                        Effect: 'Allow',
                        Resource: '*',
                    }),
                },
                Roles: [stack.resolve(deploymentInstanceRole.ref)],
            }));
        });
        test('configures direct repository connection', () => {
            // WHEN
            createTarget();
            // THEN
            expect(repository.configureClientInstance).toHaveBeenCalledWith({
                host: deploymentInstance,
                mountPoint: expect.any(String),
            });
        });
        test('grants DeploymentInstance read access to the Deadline Secrets Management admin credentials secret', () => {
            // WHEN
            createTarget();
            // THEN
            assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: deploymentInstanceStack.resolve(repository.secretsManagementSettings.credentials.secretArn),
                    }),
                },
                Roles: [
                    deploymentInstanceStack.resolve(deploymentInstanceRole.ref),
                ],
            }));
        });
        describe('Identity registration settings script', () => {
            function getIdentityRegistrationSettingsScript() {
                return target.node.findChild('ConfigureIdentityRegistrationSettingScript');
            }
            test('DeploymentInstance granted S3 read access', () => {
                // WHEN
                createTarget();
                const identityRegistrationSettingsScript = getIdentityRegistrationSettingsScript();
                // THEN
                assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: deploymentInstanceStack.resolve([
                                identityRegistrationSettingsScript.bucket.bucketArn,
                                identityRegistrationSettingsScript.bucket.arnForObjects('*'),
                            ]),
                        }),
                    },
                    Roles: [deploymentInstanceStack.resolve(deploymentInstanceRole.ref)],
                }));
            });
            test('DeploymentInstance downloads script', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                deploymentInstance.userData.addS3DownloadCommand.mockReturnValueOnce('deadlineClientLocalPath');
                deploymentInstance.userData.addS3DownloadCommand.mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath);
                // WHEN
                createTarget();
                const identityRegistrationSettingsScript = getIdentityRegistrationSettingsScript();
                // THEN
                expect(deploymentInstance.userData.addS3DownloadCommand).toHaveBeenCalledWith({
                    bucket: identityRegistrationSettingsScript.bucket,
                    bucketKey: identityRegistrationSettingsScript.s3ObjectKey,
                    localFile: expect.stringMatching(/^\/home\/ec2-user\//),
                });
            });
            test('DeploymentInstance sets ownership and executable permissions for ec2-user', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                (deploymentInstance.userData.addS3DownloadCommand
                    .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                    .mockReturnValueOnce('efsMountScriptLocalPath')
                    .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                    .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                // WHEN
                createTarget();
                // THEN
                expect(deploymentInstance.userData.addCommands).toHaveBeenCalledWith(`chmod +x ${identityRegistrationSettingsScriptLocalPath}`, `chown ec2-user:ec2-user ${identityRegistrationSettingsScriptLocalPath}`);
            });
        });
        describe('.addSubnetIdentityRegistrationSetting(...)', () => {
            describe.each([
                [lib_1.SecretsManagementRole.SERVER],
                [lib_1.SecretsManagementRole.CLIENT],
            ])('identityRole=%s', (identityRole) => {
                describe.each([
                    [lib_1.SecretsManagementRegistrationStatus.PENDING],
                    [lib_1.SecretsManagementRegistrationStatus.REGISTERED],
                    [lib_1.SecretsManagementRegistrationStatus.REVOKED],
                ])('registrationStatus=%s', (registrationStatus) => {
                    test('executes identity registration settings configuration script with proper arguments', () => {
                        // GIVEN
                        const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                        (deploymentInstance.userData.addS3DownloadCommand
                            .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                            .mockReturnValueOnce('efsMountScriptLocalPath')
                            .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                            .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                        const clientStack = new core_1.Stack(app, 'ClientStack');
                        createTarget();
                        // WHEN
                        target.addSubnetIdentityRegistrationSetting({
                            dependent: new core_1.Construct(clientStack, 'DeadlineClient'),
                            registrationStatus,
                            role: identityRole,
                            vpc,
                            vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                        });
                        // THEN
                        const resolvedCalls = deploymentInstance.userData.addCommands.mock.calls.map(call => {
                            return deploymentInstanceStack.resolve(call);
                        });
                        const expectedCall = [{
                                'Fn::Join': [
                                    '',
                                    [
                                        // Command is run as "ec2-user" which has the database credentials stored
                                        `sudo --login -u ec2-user ${identityRegistrationSettingsScriptLocalPath} `,
                                        stack.resolve(core_1.Fn.join(' ', [
                                            '--region',
                                            stack.region,
                                            // The Deadline Secrets Management admin credentials secret ARN is passed
                                            '--credentials',
                                            `"${repository.secretsManagementSettings.credentials.secretArn}"`,
                                            // The Render Queue's ALB subnets are passed as --connection-subnet args
                                            ...(vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME })
                                                .subnetIds.map(subnetID => `--connection-subnet "${subnetID}"`)),
                                            // The Deadline Client's subnets, desired role, and registration status are passed as --source-subnet args
                                            ...(vpc.selectSubnets({ subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME })
                                                .subnetIds.map(subnetID => {
                                                return `--source-subnet "${subnetID},${identityRole},${registrationStatus}"`;
                                            })),
                                        ])),
                                    ],
                                ],
                            }];
                        expect(resolvedCalls).toContainEqual(expectedCall);
                    });
                });
            });
            test('throws execption when using Administrator role', () => {
                // GIVEN
                createTarget();
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: new core_1.Construct(stack, 'Dependent'),
                        registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                        role: lib_1.SecretsManagementRole.ADMINISTRATOR,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError('The Administrator role cannot be set using a Deadline identity registration setting');
            });
            test('throws when two rules for same source subnet with different roles', () => {
                // GIVEN
                const client1 = new core_1.Construct(stack, 'client1');
                const client2 = new core_1.Construct(stack, 'client2');
                const existingRole = lib_1.SecretsManagementRole.SERVER;
                const newRole = lib_1.SecretsManagementRole.CLIENT;
                createTarget();
                target.addSubnetIdentityRegistrationSetting({
                    dependent: client1,
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: existingRole,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                });
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: client2,
                        registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                        role: newRole,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError(`Subnet is already registered with role "${existingRole}" but another caller requested "${newRole}"`);
            });
            test('throws when two rules for same source subnet with different registration statuses', () => {
                // GIVEN
                const client1 = new core_1.Construct(stack, 'client1');
                const client2 = new core_1.Construct(stack, 'client2');
                const role = lib_1.SecretsManagementRole.CLIENT;
                const existingStatus = lib_1.SecretsManagementRegistrationStatus.REGISTERED;
                const newStatus = lib_1.SecretsManagementRegistrationStatus.PENDING;
                createTarget();
                target.addSubnetIdentityRegistrationSetting({
                    dependent: client1,
                    registrationStatus: existingStatus,
                    role,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                });
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: client2,
                        registrationStatus: newStatus,
                        role,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError(`Subnet is already registered with registrationStatus "${existingStatus}" but another caller requested "${newStatus}"`);
            });
            test('de-duplicates subnets', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                (deploymentInstance.userData.addS3DownloadCommand
                    .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                    .mockReturnValueOnce('efsMountScriptLocalPath')
                    .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                    .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                const clientStack = new core_1.Stack(app, 'ClientStack');
                const client1 = new core_1.Construct(clientStack, 'client1');
                const client2 = new core_1.Construct(clientStack, 'client2');
                createTarget();
                const baseProps = {
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: lib_1.SecretsManagementRole.CLIENT,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                };
                target.addSubnetIdentityRegistrationSetting({
                    ...baseProps,
                    dependent: client1,
                });
                // WHEN
                target.addSubnetIdentityRegistrationSetting({
                    ...baseProps,
                    dependent: client2,
                });
                // THEN
                const resolvedCalls = deploymentInstance.userData.addCommands.mock.calls.map(call => {
                    return deploymentInstanceStack.resolve(call);
                });
                const expectedCall = [{
                        'Fn::Join': [
                            '',
                            [
                                // Command is run as "ec2-user" which has the database credentials stored
                                `sudo --login -u ec2-user ${identityRegistrationSettingsScriptLocalPath} `,
                                stack.resolve(core_1.Fn.join(' ', [
                                    '--region',
                                    stack.region,
                                    // The Deadline Secrets Management admin credentials secret ARN is passed
                                    '--credentials',
                                    `"${repository.secretsManagementSettings.credentials.secretArn}"`,
                                    // The Render Queue's ALB subnets are passed as --connection-subnet args
                                    ...(vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME })
                                        .subnetIds.map(subnetID => `--connection-subnet "${subnetID}"`)),
                                    // The Deadline Client's subnets, desired role, and registration status are passed as --source-subnet args
                                    ...(vpc.selectSubnets({ subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME })
                                        .subnetIds.map(subnetID => {
                                        return `--source-subnet "${subnetID},${baseProps.role},${baseProps.registrationStatus}"`;
                                    })),
                                ])),
                            ],
                        ],
                    }];
                expect(resolvedCalls).toContainEqual(expectedCall);
            });
            test('warns about dedicated subnets when render queue ALB and source subnets match', () => {
                // GIVEN
                createTarget();
                const dependent = new core_1.Construct(stack, 'Dependent');
                const registrationStatus = lib_1.SecretsManagementRegistrationStatus.REGISTERED;
                const role = lib_1.SecretsManagementRole.CLIENT;
                const vpcSubnets = {
                    subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME,
                };
                // WHEN
                target.addSubnetIdentityRegistrationSetting({
                    dependent,
                    registrationStatus,
                    role,
                    vpc,
                    vpcSubnets,
                });
                expect(dependent.node.metadataEntry).toContainEqual(expect.objectContaining({
                    type: 'aws:cdk:warning',
                    data: `Deadline Secrets Management is enabled on the Repository and VPC subnets of the Render Queue match the subnets of ${dependent.node.path}. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components`,
                }));
            });
        });
        test('Repository with no admin credentials throws an error', () => {
            // GIVEN
            class RepositoryNoCredentials extends lib_1.Repository {
                constructor(scope, id, props) {
                    super(scope, id, props);
                    this.secretsManagementSettings = {
                        enabled: true,
                        credentials: undefined,
                    };
                }
            }
            repository = new RepositoryNoCredentials(dependencyStack, 'RepoNoCreds', {
                version,
                vpc,
            });
            // WHEN
            const when = createTarget;
            // THEN
            expect(when).toThrowError('Repository does not contain secrets management credentials');
        });
    });
    test('when Repository disables secrets management throws an exception', () => {
        // GIVEN
        repository = new lib_1.Repository(stack, 'Repository', {
            version,
            vpc,
            secretsManagementSettings: {
                enabled: false,
            },
        });
        // WHEN
        function when() {
            new secrets_management_1.SecretsManagementIdentityRegistration(stack, 'IdentityRegistrationSettings', {
                deploymentInstance,
                renderQueueSubnets: vpc.selectSubnets({
                    subnetGroupName: 'RenderQueueALB',
                }),
                repository,
                version,
            });
        }
        // THEN
        expect(when).toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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