"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const sinon = require("sinon");
const lib_1 = require("../../core/lib");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_2 = require("../lib");
describe('DocumentDB', () => {
    let stack;
    let vpc;
    let database;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
            engineVersion: '3.6.0',
        });
        if (!database.secret) {
            throw new Error('secret cannot be null');
        }
    });
    test('Grants access to Document DB Secret', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        connection.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: { Ref: 'DbClusterSecretAttachment4201A1ED' },
                    }],
            },
        }));
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=(');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            ToPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
        }));
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'DbClusterInstance155835CE5',
                'InstanceInstanceRoleE9785DE5',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('add child dependency to attributes', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('add child dependency throws when cluster implementation changed', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        docdb.node.defaultChild = database; // Trick addChildDependency() into thinking this is a real construct.
        // THEN
        expect(() => {
            connection.addChildDependency(instance);
        }).toThrowError(/The internal implementation of the AWS CDK's DocumentDB cluster construct may have changed./);
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
    test('adds warning annotation when a security group cannot be added due to unsupported IDatabaseCluster implementation', () => {
        // GIVEN
        class FakeDatabaseCluster extends core_1.Resource {
            constructor(scope, id) {
                super(scope, id);
                this.clusterIdentifier = '';
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new aws_docdb_1.Endpoint('address', 123);
                this.clusterReadEndpoint = new aws_docdb_1.Endpoint('readAddress', 123);
                this.instanceEndpoints = [];
                this.securityGroupId = '';
                this.connections = new aws_ec2_1.Connections();
                this.stack = core_1.Stack.of(scope);
                this.env = { account: this.stack.account, region: this.stack.region };
            }
            asSecretAttachmentTarget() {
                throw new Error('Method not implemented.');
            }
        }
        const fakeDatabase = new FakeDatabaseCluster(stack, 'FakeDatabase');
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', { vpc });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: fakeDatabase, login: database.secret });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(fakeDatabase.node.metadataEntry).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:warning',
                data: expect.stringMatching(new RegExp(`Failed to add the following security groups to ${fakeDatabase.node.id}: .*\\. ` +
                    'The \\"database\\" property passed to this class is not an instance of AWS CDK\'s DocumentDB cluster construct.')),
            }),
        ]));
    });
    // This test can be removed once the following CDK PR is merged:
    // https://github.com/aws/aws-cdk/pull/13290
    test('adds warning annotation when a security group cannot be added due to implementation changes in DatabaseCluster', () => {
        // GIVEN
        if (!database.node.tryRemoveChild('Resource')) {
            throw new Error('The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed. The addSecurityGroup method needs to be updated.');
        }
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', { vpc });
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(database.node.metadataEntry).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:warning',
                data: expect.stringMatching(new RegExp(`Failed to add the following security groups to ${database.node.id}: .*\\. ` +
                    'The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed.')),
            }),
        ]));
    });
    test('Document DB connection is pointed to correct construct', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(connection.databaseConstruct).toEqual(database);
    });
});
describe('DocumentDB Version Checks', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    });
    test('Compatible version', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
            engineVersion: '3.6.0',
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(database.node.metadataEntry.length).toBe(0);
    });
    test('When from attributes', () => {
        // GIVEN
        const sg = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        const secret = new aws_secretsmanager_1.Secret(stack, 'Secret');
        const database = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'DbCluster', {
            clusterEndpointAddress: '1.2.3.4',
            clusterIdentifier: 'foo',
            instanceEndpointAddresses: ['1.2.3.5'],
            instanceIdentifiers: ['i0'],
            port: 27001,
            readerEndpointAddress: '1.2.3.6',
            securityGroup: sg,
        });
        // WHEN
        const databaseConnection = lib_2.DatabaseConnection.forDocDB({ database, login: secret });
        // THEN
        expect(database.node.metadataEntry.length).toBe(0);
        expect(databaseConnection.databaseConstruct).toBeUndefined();
    });
    test('No engineVersion given', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(database.node.metadataEntry).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:error',
                data: 'engineVersion must be 3.6.0 to be compatible with Deadline',
            }),
        ]));
    });
    test('engineVersion not 3.6.0', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
            engineVersion: '4.0.0',
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(database.node.metadataEntry).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:error',
                data: 'engineVersion must be 3.6.0 to be compatible with Deadline',
            }),
        ]));
    });
});
describe('MongoDB', () => {
    let stack;
    let vpc;
    let database;
    let clientCert;
    let dnsZone;
    let serverCert;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        const hostname = 'mongo';
        const zoneName = 'deadline.internal';
        dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        const caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
        clientCert = new lib_1.X509CertificatePem(stack, 'ClientCert', {
            subject: {
                cn: 'dbuser',
                ou: 'TestClient',
            },
            signingCertificate: caCert,
        });
        database = new lib_1.MongoDbInstance(stack, 'MongoDb', {
            vpc,
            mongoDb: {
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                hostname,
                dnsZone,
                serverCertificate: serverCert,
            },
        });
    });
    test('Grants access to certificate Secrets', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        connection.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'ClientCert',
                                'Cert',
                            ],
                        },
                    },
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            Ref: 'ClientCertPassphrase8A71E1E1',
                        },
                    },
                ],
            },
        }));
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/' + token + token + '\' ' + token + ' /opt/Thinkbox/certs/mongo_client.pfx')));
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true');
        expect(userData).toContain('["--dbhost"]="mongo.deadline.internal"');
        expect(userData).toContain('["--dbport"]=27017');
        expect(userData).toContain('["--dbclientcert"]=');
        expect(userData).toContain('["--dbcertpass"]=$CERT_PASSWORD');
    });
    test('addConnectionDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addConnectionDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/' + token + token + '\' ' + token + ' /opt/Thinkbox/certs/mongo_client.pfx')));
        expect(userData).toContain('configure_deadline_database(){\n');
        expect(userData).toContain('\nexport -f configure_deadline_database');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nexport DB_CERT_FILE=');
        expect(userData).toContain('\nexport DB_CERT_PASSWORD=');
    });
    test('defines required container environment variables', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // THEN
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_URI');
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_PASSWORD_URI');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 27017,
            ToPort: 27017,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'MongoDbServerAsgInstanceSecurityGroupCE623335',
                    'GroupId',
                ],
            },
        }));
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
                'MongoDbServerAsgASG47B3D94E',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
    test('adds security group', () => {
        // GIVEN
        const dbSpy = sinon.spy(database, 'addSecurityGroup');
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', {
            vpc,
        });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(dbSpy.calledOnce).toBeTruthy();
    });
    test('Mongo DB connection is pointed to correct construct', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // THEN
        expect(connection.databaseConstruct).toEqual(database.mongoDataVolume);
    });
    test('Mongo DB imported from attributes', () => {
        // GIVEN
        const volume = aws_ec2_1.Volume.fromVolumeAttributes(stack, 'Volume', {
            availabilityZone: 'dummy zone',
            volumeId: 'vol-05abe246af',
        });
        const mongoDB = new lib_1.MongoDbInstance(stack, 'ImportedMongoDb', {
            vpc,
            mongoDb: {
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                hostname: 'mongo',
                dnsZone,
                serverCertificate: serverCert,
                mongoDataVolume: { volume },
            },
        });
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database: mongoDB, clientCertificate: clientCert });
        // THEN
        expect(connection.databaseConstruct).toBeUndefined();
    });
});
//# sourceMappingURL=data:application/json;base64,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