"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const secrets_management_ref_1 = require("./secrets-management-ref");
const version_1 = require("./version");
const worker_configuration_1 = require("./worker-configuration");
/**
 *  A new or imported Deadline Worker Fleet.
 */
class WorkerInstanceFleetBase extends core_1.Construct {
}
/**
 * This construct reperesents a fleet of Deadline Workers.
 *
 * The construct consists of an Auto Scaling Group (ASG) of instances using a provided AMI which has Deadline and any number
 * of render applications installed.  Whenever an instance in the ASG start it will connect Deadline to the desired render queue.
 *
 * When the worker fleet is deployed if it has been provided a HealthMonitor the Worker fleet will register itself against the Monitor
 * to ensure that the fleet remains healthy.
 *
 * ![architecture diagram](/diagrams/deadline/WorkerInstanceFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An EC2 Auto Scaling Group to maintain the number of instances.
 * - An Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *   in the fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *   that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *   the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *   patching strategy to keep this software current with the latest security patches. Please see
 *   https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 */
class WorkerInstanceFleet extends WorkerInstanceFleetBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps(props);
        const minCapacity = props.minCapacity ?? 1;
        const signals = minCapacity > 0 ? aws_autoscaling_1.Signals.waitForMinCapacity({
            timeout: WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT,
        }) : undefined;
        if (signals === undefined) {
            core_1.Annotations.of(this).addWarning('Deploying with 0 minimum capacity. If there is an error in the EC2 UserData for this fleet, then your stack deployment will not fail. Watch for errors in your CloudWatch logs.');
        }
        const vpcSubnets = props.vpcSubnets ? props.vpcSubnets : {
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        };
        // Launching the fleet with deadline workers.
        this.fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'Default', {
            vpc: props.vpc,
            instanceType: (props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE)),
            machineImage: props.workerMachineImage,
            keyName: props.keyName,
            vpcSubnets,
            securityGroup: props.securityGroup,
            minCapacity,
            maxCapacity: props.maxCapacity,
            desiredCapacity: props.desiredCapacity,
            signals,
            healthCheck: aws_autoscaling_1.HealthCheck.elb({
                grace: WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL,
            }),
            role: props.role,
            spotPrice: props.spotPrice?.toString(),
            blockDevices: props.blockDevices,
        });
        this.targetCapacity = parseInt(this.fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = this.fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensionsMap: {
                AutoScalingGroupName: this.fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [this.fleet.autoScalingGroupArn],
                })],
        });
        this.fleet.node.defaultChild.metricsCollection = [{
                granularity: '1Minute',
                metrics: ['GroupDesiredCapacity'],
            }];
        this.grantPrincipal = this.fleet.grantPrincipal;
        this.connections = this.fleet.connections;
        // Configure the health monitoring if provided.
        // Note: This must be done *BEFORE* configuring the worker. We rely on the worker configuration
        // script restarting the launcher.
        this.configureHealthMonitor(props);
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this.fleet,
            cloudWatchLogSettings: {
                logGroupPrefix: WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: props,
            userDataProvider: props.userDataProvider,
        });
        this.listeningPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + WorkerInstanceFleet.MAX_WORKERS_PER_HOST);
        if (props.renderQueue.repository.secretsManagementSettings.enabled) {
            if (!props.vpcSubnets) {
                core_1.Annotations.of(this).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
            }
            props.renderQueue.configureSecretsManagementAutoRegistration({
                vpc: props.vpc,
                vpcSubnets,
                role: secrets_management_ref_1.SecretsManagementRole.CLIENT,
                registrationStatus: secrets_management_ref_1.SecretsManagementRegistrationStatus.REGISTERED,
                dependent: this.fleet,
            });
        }
        // Updating the user data with successful cfn-signal commands.
        if (signals) {
            this.fleet.userData.addSignalOnExitCommand(this.fleet);
        }
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Add the security group to all workers
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.fleet.addSecurityGroup(securityGroup);
    }
    /**
     * @inheritdoc
     */
    allowListenerPortFrom(other) {
        this.connections.allowFrom(other.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    /**
     * @inheritdoc
     */
    allowListenerPortTo(other) {
        other.connections.allowTo(this.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    validateProps(props) {
        this.validateSpotPrice(props.spotPrice);
        this.validateArrayGroupsPoolsSyntax(props.groups, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'groups');
        this.validateArrayGroupsPoolsSyntax(props.pools, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'pools');
        this.validateRegion(props.region, /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i);
        this.validateBlockDevices(props.blockDevices);
    }
    validateSpotPrice(spotPrice) {
        if (spotPrice && !(spotPrice >= WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT && spotPrice <= WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT)) {
            throw new Error(`Invalid value: ${spotPrice} for property 'spotPrice'. Valid values can be any decimal between ${WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT} and ${WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT}.`);
        }
    }
    validateRegion(region, regex) {
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property 'region'. Valid characters are A-Z, a-z, 0-9, - and _. 'All', 'none' and 'unrecognized' are reserved names that cannot be used.`);
        }
    }
    validateArrayGroupsPoolsSyntax(array, regex, property) {
        if (array) {
            array.forEach(value => {
                if (!regex.test(value)) {
                    throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
                }
            });
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            core_1.Annotations.of(this).addWarning(`The worker-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    core_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the worker-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
    configureHealthMonitor(props) {
        if (props.healthMonitor) {
            const healthCheckPort = props.healthCheckConfig?.port ?? core_2.HealthMonitor.DEFAULT_HEALTH_CHECK_PORT;
            const configureHealthMonitorScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'WorkerConfigurationScript', {
                osType: this.fleet.osType,
                baseName: 'configureWorkerHealthCheck',
                rootDir: path.join(__dirname, '..', 'scripts/'),
            });
            configureHealthMonitorScriptAsset.executeOn({
                host: this.fleet,
                args: [
                    `'${healthCheckPort}'`,
                    `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
                ],
            });
            props.healthMonitor.registerFleet(this, props.healthCheckConfig || {
                port: healthCheckPort,
            });
        }
        else {
            core_1.Annotations.of(this).addWarning(`The worker-fleet ${this.node.id} is being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy.`);
        }
    }
}
exports.WorkerInstanceFleet = WorkerInstanceFleet;
_a = JSII_RTTI_SYMBOL_1;
WorkerInstanceFleet[_a] = { fqn: "aws-rfdk.deadline.WorkerInstanceFleet", version: "0.42.0" };
/**
 * The min limit for spot price.
 */
WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT = 0.001;
/**
 * The max limit for spot price.
 */
WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT = 255;
/**
 * This determines worker's health based on any hardware or software issues of EC2 instance.
 * Resource Tracker does deep ping every 5 minutes. These checks should be more frequent so
 * that any EC2 level issues are identified ASAP. Hence setting it to 1 minute.
 */
WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(1);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
WorkerInstanceFleet.MAX_WORKERS_PER_HOST = 8;
/**
 * Setting the default signal timeout to 15 min. This is the max time, a single instance is expected
 * to take for launch and execute the user-data for deadline worker configuration. As we are setting
 * failure signals in the user-data, any failure will terminate deployment immediately.
 */
WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT = core_1.Duration.minutes(15);
//# sourceMappingURL=data:application/json;base64,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