"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.WaitForStableService = void 0;
const crypto_1 = require("crypto");
const path = require("path");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
/**
 * Depend on this construct to wait until the ECS Service becomes stable.
 * See https://docs.aws.amazon.com/cli/latest/reference/ecs/wait/services-stable.html.
 */
class WaitForStableService extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaRole = new aws_iam_1.Role(this, 'ECSWaitLambdaRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
            inlinePolicies: {
                describeServices: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'ecs:DescribeServices',
                            ],
                            effect: aws_iam_1.Effect.ALLOW,
                            resources: [props.service.cluster.clusterArn, props.service.serviceArn],
                        }),
                    ],
                }),
            },
        });
        const waitingFunction = new aws_lambda_1.Function(this, 'ECSWait', {
            role: lambdaRole,
            description: `Used by a WaitForStableService ${this.node.addr} to wait until ECS service becomes stable`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {}),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'wait-for-stable-service.wait',
            timeout: core_1.Duration.minutes(15),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const properties = {
            cluster: props.service.cluster.clusterArn,
            services: [props.service.serviceArn],
            forceRun: this.forceRun(),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: waitingFunction.functionArn,
            resourceType: 'Custom::RFDK_WaitForStableService',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(lambdaRole);
        resource.node.addDependency(props.service);
        this.node.defaultChild = resource;
    }
    forceRun() {
        return crypto_1.randomBytes(32).toString('base64').slice(0, 32);
    }
}
exports.WaitForStableService = WaitForStableService;
//# sourceMappingURL=data:application/json;base64,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