"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path_1 = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends core_1.Construct {
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 */
class VersionQuery extends VersionQueryBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.expression = props?.version;
        const match = (props?.version ?? '').match(VersionQuery.EXPRESSION_REGEX);
        if (match === null) {
            throw new Error(`Invalid version expression "${props.version}`);
        }
        this.pinnedVersionComponents = (match
            // First capture group is the entire matched string, so slice it off
            .slice(1)
            // Capture groups that are not matched return as undefined, so we filter them out
            .filter(component => component)
            // Parse strings to numbers
            .map(component => Number(component)));
        const lambdaCode = aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '2e19e243-16ee-4d1a-a3c9-18d35eddd446',
            description: 'Used by the Version construct to get installer locations for a specific Deadline version.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'version-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const deadlineProperties = {
            versionString: props?.version,
            // If we don't have a full static version string, create a random string that will force the Lambda to always
            // run on redeploys, effectively checking for version updates.
            forceRun: this.forceRun(props?.version),
        };
        this.deadlineResource = new core_1.CustomResource(this, 'DeadlineResource', {
            serviceToken: lambdaFunc.functionArn,
            properties: deadlineProperties,
            resourceType: 'Custom::RFDK_DEADLINE_INSTALLERS',
        });
        this.majorVersion = this.versionComponent({
            expressionIndex: 0,
            customResourceAttribute: 'MajorVersion',
        });
        this.minorVersion = this.versionComponent({
            expressionIndex: 1,
            customResourceAttribute: 'MinorVersion',
        });
        this.releaseVersion = this.versionComponent({
            expressionIndex: 2,
            customResourceAttribute: 'ReleaseVersion',
        });
        const installerBucket = aws_s3_1.Bucket.fromBucketName(scope, 'InstallerBucket', this.deadlineResource.getAttString('S3Bucket'));
        this.linuxInstallers = {
            patchVersion: core_1.Token.asNumber(this.deadlineResource.getAtt('LinuxPatchVersion')),
            repository: {
                objectKey: this.deadlineResource.getAttString('LinuxRepositoryInstaller'),
                s3Bucket: installerBucket,
            },
            client: {
                objectKey: this.deadlineResource.getAttString('LinuxClientInstaller'),
                s3Bucket: installerBucket,
            },
        };
    }
    versionComponent(args) {
        const { expressionIndex, customResourceAttribute } = args;
        return this.pinnedVersionComponents.length > expressionIndex
            ? this.pinnedVersionComponents[expressionIndex]
            : core_1.Token.asNumber(this.deadlineResource.getAtt(customResourceAttribute));
    }
    linuxFullVersionString() {
        const major = core_1.Token.isUnresolved(this.majorVersion) ? core_1.Token.asString(this.majorVersion) : this.majorVersion.toString();
        const minor = core_1.Token.isUnresolved(this.minorVersion) ? core_1.Token.asString(this.minorVersion) : this.minorVersion.toString();
        const release = core_1.Token.isUnresolved(this.releaseVersion) ? core_1.Token.asString(this.releaseVersion) : this.releaseVersion.toString();
        const patch = core_1.Token.isUnresolved(this.linuxInstallers.patchVersion)
            ? core_1.Token.asString(this.linuxInstallers.patchVersion)
            : this.linuxInstallers.patchVersion.toString();
        return `${major}.${minor}.${release}.${patch}`;
    }
    isLessThan(other) {
        if (other.patchVersion !== 0) {
            throw new Error('Cannot compare a VersionQuery to a fully-qualified version with a non-zero patch number');
        }
        // We compare each component from highest order to lowest
        const componentGetters = [
            v => v.majorVersion,
            v => v.minorVersion,
            v => v.releaseVersion,
        ];
        for (const componentGetter of componentGetters) {
            const thisComponent = componentGetter(this);
            const otherComponent = componentGetter(other);
            if (core_1.Token.isUnresolved(thisComponent)) {
                // Unresolved components are unpinned. These will resolve to the latest and are not less than any provided
                // version
                return false;
            }
            else {
                const componentDiff = thisComponent - otherComponent;
                if (componentDiff !== 0) {
                    // If the components are different, return whether this component is smaller than the other component
                    return componentDiff < 0;
                }
            }
        }
        // If we've exited the loop naturally, it means all version components are pinned and equal. This means the version
        // is not less than the other, they are the same
        return false;
    }
    /**
     * Check if we have a full version in the supplied version string. If we don't, we want to make sure the Lambda
     * that fetches the full version number and the installers for it is always run. This allows for Deadline updates
     * to be discovered.
     */
    forceRun(version) {
        return !this.isFullVersion(version) ? crypto_1.randomBytes(32).toString('base64').slice(0, 32) : undefined;
    }
    /**
     * Checks if the supplied version contains the major, minor, release, and patch version numbers,
     * and returns true only if all 4 are supplied.
     */
    isFullVersion(version) {
        const components = version?.split('.').map(x => parseInt(x));
        if (!components || components?.length != 4) {
            return false;
        }
        return true;
    }
    get versionString() {
        return this.expression ?? '(latest)';
    }
}
exports.VersionQuery = VersionQuery;
_a = JSII_RTTI_SYMBOL_1;
VersionQuery[_a] = { fqn: "aws-rfdk.deadline.VersionQuery", version: "0.42.0" };
/**
 * Regular expression for valid version query expressions
 */
VersionQuery.EXPRESSION_REGEX = /^(?:(\d+)(?:\.(\d+)(?:\.(\d+)(?:\.(\d+))?)?)?)?$/;
//# sourceMappingURL=data:application/json;base64,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