/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Construct } from '@aws-cdk/core';
import { Version } from './version';
import { IVersion, PlatformInstallers } from './version-ref';
/**
 * Properties for the Deadline Version
 */
export interface VersionQueryProps {
    /**
     * String containing the complete or partial deadline version.
     *
     * @default - the latest available version of deadline installer.
     */
    readonly version?: string;
}
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
declare abstract class VersionQueryBase extends Construct implements IVersion {
    /**
     * @inheritdoc
     */
    readonly abstract majorVersion: number;
    /**
     * @inheritdoc
     */
    readonly abstract minorVersion: number;
    /**
     * @inheritdoc
     */
    readonly abstract releaseVersion: number;
    /**
     * @inheritdoc
     */
    readonly abstract linuxInstallers: PlatformInstallers;
    /**
     * @inheritdoc
     */
    readonly abstract versionString: string;
    /**
     * @inheritdoc
     */
    abstract linuxFullVersionString(): string;
    /**
     * @inheritdoc
     */
    abstract isLessThan(other: Version): boolean;
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 */
export declare class VersionQuery extends VersionQueryBase {
    /**
     * Regular expression for valid version query expressions
     */
    private static readonly EXPRESSION_REGEX;
    /**
     * The expression used as input to the `VersionQuery`
     */
    readonly expression?: string;
    /**
     * @inheritdoc
     */
    readonly majorVersion: number;
    /**
     * @inheritdoc
     */
    readonly minorVersion: number;
    /**
     * @inheritdoc
     */
    readonly releaseVersion: number;
    /**
     * @inheritdoc
     */
    readonly linuxInstallers: PlatformInstallers;
    /**
     * Custom resource that provides the resolved Deadline version components and installer URIs
     */
    private readonly deadlineResource;
    /**
     * The pinned numeric version components extracted from the VersionQuery expression.
     */
    private readonly pinnedVersionComponents;
    constructor(scope: Construct, id: string, props?: VersionQueryProps);
    private versionComponent;
    linuxFullVersionString(): string;
    isLessThan(other: Version): boolean;
    /**
     * Check if we have a full version in the supplied version string. If we don't, we want to make sure the Lambda
     * that fetches the full version number and the installers for it is always run. This allows for Deadline updates
     * to be discovered.
     */
    private forceRun;
    /**
     * Checks if the supplied version contains the major, minor, release, and patch version numbers,
     * and returns true only if all 4 are supplied.
     */
    private isFullVersion;
    get versionString(): string;
}
export {};
