"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsageBasedLicensing = exports.UsageBasedLicense = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
/**
 * Instances of this class represent a usage-based license for a particular product.
 * It encapsulates all of the information specific to a product that the UsageBasedLicensing
 * construct requires to interoperate with that product.
 */
class UsageBasedLicense {
    constructor(props) {
        this.licenseName = props.licenseName;
        this.ports = props.ports;
        this.limit = props.limit;
    }
    /**
     * Method for 3dsMax license limit.
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static for3dsMax(limit) {
        return new UsageBasedLicense({
            licenseName: 'max',
            ports: [aws_ec2_1.Port.tcp(27002)],
            limit,
        });
    }
    /**
     * Method for Arnold license limit
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forArnold(limit) {
        return new UsageBasedLicense({
            licenseName: 'arnold',
            ports: [aws_ec2_1.Port.tcp(5056), aws_ec2_1.Port.tcp(7056)],
            limit,
        });
    }
    /**
     * Method for Cinema 4D license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forCinema4D(limit) {
        return new UsageBasedLicense({
            licenseName: 'cinema4d',
            ports: [aws_ec2_1.Port.tcp(5057), aws_ec2_1.Port.tcp(7057)],
            limit,
        });
    }
    /**
     * Method for Clarisse license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forClarisse(limit) {
        return new UsageBasedLicense({
            licenseName: 'clarisse',
            ports: [aws_ec2_1.Port.tcp(40500)],
            limit,
        });
    }
    /**
     * Method for Houdini license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forHoudini(limit) {
        return new UsageBasedLicense({
            licenseName: 'houdini',
            ports: [aws_ec2_1.Port.tcp(1715)],
            limit,
        });
    }
    /**
     * Method for Katana license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKatana(limit) {
        return new UsageBasedLicense({
            licenseName: 'katana',
            ports: [aws_ec2_1.Port.tcp(4151), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for KeyShot license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKeyShot(limit) {
        return new UsageBasedLicense({
            licenseName: 'keyshot',
            ports: [aws_ec2_1.Port.tcp(27003), aws_ec2_1.Port.tcp(2703)],
            limit,
        });
    }
    /**
     * Method for krakatoa license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKrakatoa(limit) {
        return new UsageBasedLicense({
            licenseName: 'krakatoa',
            ports: [aws_ec2_1.Port.tcp(27000), aws_ec2_1.Port.tcp(2700)],
            limit,
        });
    }
    /**
     * Method for Mantra license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMantra(limit) {
        return new UsageBasedLicense({
            licenseName: 'mantra',
            ports: [aws_ec2_1.Port.tcp(1716)],
            limit,
        });
    }
    /**
     * Method for maxwell license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMaxwell(limit) {
        return new UsageBasedLicense({
            licenseName: 'maxwell',
            ports: [aws_ec2_1.Port.tcp(5555), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for Maya license limit
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMaya(limit) {
        return new UsageBasedLicense({
            licenseName: 'maya',
            ports: [aws_ec2_1.Port.tcp(27002), aws_ec2_1.Port.tcp(2702)],
            limit,
        });
    }
    /**
     * Method for Nuke license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forNuke(limit) {
        return new UsageBasedLicense({
            licenseName: 'nuke',
            ports: [aws_ec2_1.Port.tcp(4101), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for RealFlow license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forRealFlow(limit) {
        return new UsageBasedLicense({
            licenseName: 'realflow',
            ports: [aws_ec2_1.Port.tcp(5055), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for RedShift license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forRedShift(limit) {
        return new UsageBasedLicense({
            licenseName: 'redshift',
            ports: [aws_ec2_1.Port.tcp(5054), aws_ec2_1.Port.tcp(7054)],
            limit,
        });
    }
    /**
     * Method for V-Ray license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forVray(limit) {
        return new UsageBasedLicense({
            licenseName: 'vray',
            ports: [aws_ec2_1.Port.tcp(30306)],
            limit,
        });
    }
    /**
     * Method for Yeti license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forYeti(limit) {
        return new UsageBasedLicense({
            licenseName: 'yeti',
            ports: [aws_ec2_1.Port.tcp(5053), aws_ec2_1.Port.tcp(7053)],
            limit,
        });
    }
}
exports.UsageBasedLicense = UsageBasedLicense;
_a = JSII_RTTI_SYMBOL_1;
UsageBasedLicense[_a] = { fqn: "aws-rfdk.deadline.UsageBasedLicense", version: "0.42.0" };
/**
 * Constant used to signify unlimited overage.
 */
UsageBasedLicense.UNLIMITED = 2147483647;
/**
 * This construct is an implementation of the Deadline component that is required for Usage-based Licensing (UBL)
 * (see: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/licensing-usage-based.html )
 * in a render farm.
 *
 * Internally this is implemented as one or more instances of the Deadline License Forwarder application set up
 * to communicate to the render queue and Thinkbox’s licensing system, and to allow ingress connections
 * from the worker nodes so that they can acquire licenses as needed.
 *
 * The Deadline License Forwarder is set up to run within an AWS ECS task.
 *
 * Access to the running License Forwarder is gated by a security group that, by default, only allows ingress from the
 * Render Queue (in order to register Workers for license forwarding).
 *
 * When a Deadline Worker requires access to licensing via `UsageBasedLicensing.grantPortAccess(...)`, then the RFDK
 * constructs will grant that worker’s security group ingress on TCP port 17004 as well as other ports as required by
 * the specific licenses being used.
 *
 * Note: This construct does not currently implement the Deadline License Forwarder's Web Forwarding functionality.
 * This construct is not usable in any China region.
 *
 * ![architecture diagram](/diagrams/deadline/UsageBasedLicensing.svg)
 *
 * Resources Deployed
 * ------------------------
 * - The Auto Scaling Group (ASG) added to the Amazon Elastic Container Service cluster that is hosting the Deadline
 *   License Forwarder for UBL. This creates one C5 Large instance by default.
 * - Amazon Elastic Block Store (EBS) device(s) associated with the EC2 instance(s) in the ASG. The default volume size is 30 GiB.
 * - An Amazon CloudWatch log group that contains the logs from the Deadline License Forwarder application.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The Deadline License Forwarder is designed to be secured by restricting network access to it. For security, only the Deadline
 *   Workers that require access to Usage-based Licenses should be granted network access to the instances deployed by this construct.
 *   Futhermore, you should restrict that access to only the product(s) that those workers require when deploying this construct.
 */
class UsageBasedLicensing extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const usageBasedLicenses = new Array();
        props.licenses.forEach(license => {
            usageBasedLicenses.push(`${license.licenseName}:${license.limit ? license.limit : UsageBasedLicense.UNLIMITED}`);
        });
        if (usageBasedLicenses.length < 1) {
            throw new Error('Should be specified at least one license with defined limit.');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', { vpc: props.vpc });
        if (!props.vpcSubnets && props.renderQueue.repository.secretsManagementSettings.enabled) {
            core_1.Annotations.of(this).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
        }
        const vpcSubnets = props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT };
        this.asg = this.cluster.addCapacity('ASG', {
            vpcSubnets,
            instanceType: props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            minCapacity: props.desiredCount ?? 1,
            maxCapacity: props.desiredCount ?? 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: props.securityGroup,
        });
        const taskDefinition = new aws_ecs_1.TaskDefinition(this, 'TaskDefinition', {
            compatibility: aws_ecs_1.Compatibility.EC2,
            networkMode: aws_ecs_1.NetworkMode.HOST,
        });
        this.grantPrincipal = taskDefinition.taskRole;
        const containerEnv = {
            UBL_CERTIFICATES_URI: '',
            UBL_LIMITS: usageBasedLicenses.join(';'),
            ...props.renderQueue.configureClientECS({
                hosts: [this.asg],
                grantee: this,
            }),
        };
        containerEnv.UBL_CERTIFICATES_URI = props.certificateSecret.secretArn;
        props.certificateSecret.grantRead(taskDefinition.taskRole);
        const prefix = props.logGroupProps?.logGroupPrefix ?? UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...props.logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, defaultedLogGroupProps);
        logGroup.grantWrite(this.asg);
        const container = taskDefinition.addContainer('LicenseForwarderContainer', {
            image: props.images.licenseForwarder,
            environment: containerEnv,
            memoryReservationMiB: 1024,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup,
                streamPrefix: 'LicenseForwarder',
            }),
        });
        // Increase ulimits
        container.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        this.service = new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition,
            desiredCount: props.desiredCount ?? 1,
            placementConstraints: [aws_ecs_1.PlacementConstraint.distinctInstances()],
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
        });
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.service.node.addDependency(this.asg);
        this.node.defaultChild = this.service;
        if (props.renderQueue.repository.secretsManagementSettings.enabled) {
            props.renderQueue.configureSecretsManagementAutoRegistration({
                dependent: this.service.node.defaultChild,
                registrationStatus: _1.SecretsManagementRegistrationStatus.REGISTERED,
                role: _1.SecretsManagementRole.CLIENT,
                vpc: props.vpc,
                vpcSubnets,
            });
        }
        // Grant the render queue the ability to connect to the license forwarder to register workers
        this.asg.connections.allowFrom(props.renderQueue.backendConnections, aws_ec2_1.Port.tcp(UsageBasedLicensing.LF_PORT));
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * This method grant access of worker fleet to ports that required
     *
     * @param workerFleet - worker fleet
     * @param licenses - UBL licenses
     */
    grantPortAccess(workerFleet, licenses) {
        licenses.forEach(license => {
            license.ports.forEach(port => {
                workerFleet.connections.allowTo(this, port);
            });
        });
        workerFleet.connections.allowTo(this, aws_ec2_1.Port.tcp(UsageBasedLicensing.LF_PORT));
    }
    /**
     * The connections object that allows you to control network egress/ingress to the License Forwarder.
     */
    get connections() {
        return this.service.connections;
    }
}
exports.UsageBasedLicensing = UsageBasedLicensing;
_b = JSII_RTTI_SYMBOL_1;
UsageBasedLicensing[_b] = { fqn: "aws-rfdk.deadline.UsageBasedLicensing", version: "0.42.0" };
/**
 * The port that the License Forwarder listens on
 */
UsageBasedLicensing.LF_PORT = 17004;
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
//# sourceMappingURL=data:application/json;base64,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