"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxDockerImages = exports.AwsThinkboxEulaAcceptance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path = require("path");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
/**
 * Choices for signifying the user's stance on the terms of the AWS Thinkbox End-User License Agreement (EULA).
 * See: https://www.awsthinkbox.com/end-user-license-agreement
 */
var AwsThinkboxEulaAcceptance;
(function (AwsThinkboxEulaAcceptance) {
    /**
     * The user signifies their explicit rejection of the tems of the AWS Thinkbox EULA.
     *
     * See: https://www.awsthinkbox.com/end-user-license-agreement
     */
    AwsThinkboxEulaAcceptance[AwsThinkboxEulaAcceptance["USER_REJECTS_AWS_THINKBOX_EULA"] = 0] = "USER_REJECTS_AWS_THINKBOX_EULA";
    /**
     * The user signifies their explicit acceptance of the terms of the AWS Thinkbox EULA.
     *
     * See: https://www.awsthinkbox.com/end-user-license-agreement
     */
    AwsThinkboxEulaAcceptance[AwsThinkboxEulaAcceptance["USER_ACCEPTS_AWS_THINKBOX_EULA"] = 1] = "USER_ACCEPTS_AWS_THINKBOX_EULA";
})(AwsThinkboxEulaAcceptance = exports.AwsThinkboxEulaAcceptance || (exports.AwsThinkboxEulaAcceptance = {}));
/**
 * An API for interacting with publicly available Deadline container images published by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * Successful usage of the published Deadline container images with this class requires:
 *
 * 1) Explicit acceptance of the terms of the AWS Thinkbox End User License Agreement, under which Deadline is
 *    distributed; and
 * 2) The lambda on which the custom resource looks up the Thinkbox container images is able to make HTTPS
 *    requests to the official AWS Thinbox download site: https://downloads.thinkboxsoftware.com
 *
 * Resources Deployed
 * ------------------------
 * - A Lambda function containing a script to look up the AWS Thinkbox container image registry
 *
 * Security Considerations
 * ------------------------
 * - CDK deploys the code for this lambda as an S3 object in the CDK bootstrap bucket. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly
 *   recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS
 *   CloudTrail on your account to assist in post-incident analysis of compromised production environments.
 *
 * For example, to construct a RenderQueue using the images:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { AwsThinkboxEulaAcceptance, RenderQueue, Repository, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *   version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *   version,
 *   // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *   // of the AWS Thinkbox End User License Agreement
 *   userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *   vpc,
 *   version,
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *   images: images.forRenderQueue(),
 *   // ...
 * });
 * ```
 */
class ThinkboxDockerImages extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.userAwsThinkboxEulaAcceptance = props.userAwsThinkboxEulaAcceptance;
        this.version = props?.version;
        const lambdaCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '08553416-1fc9-4be9-a818-609a31ae1b5b',
            description: 'Used by the ThinkboxDockerImages construct to look up the ECR repositories where AWS Thinkbox publishes Deadline container images.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'ecr-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const ecrProvider = new core_1.CustomResource(this, 'ThinkboxEcrProvider', {
            serviceToken: lambdaFunc.functionArn,
            properties: {
                // create a random string that will force the Lambda to "update" on each deployment. Changes to its output will
                // be propagated to any CloudFormation resource providers that reference the output ARN
                ForceRun: this.forceRun(),
            },
            resourceType: 'Custom::RFDK_EcrProvider',
        });
        this.node.defaultChild = ecrProvider;
        this.ecrBaseURI = ecrProvider.getAtt('EcrURIPrefix').toString();
        this.remoteConnectionServer = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER);
        this.licenseForwarder = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER);
    }
    onValidate() {
        const errors = [];
        // Users must accept the AWS Thinkbox EULA to use the container images
        if (this.userAwsThinkboxEulaAcceptance !== AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA) {
            errors.push(ThinkboxDockerImages.AWS_THINKBOX_EULA_MESSAGE);
        }
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof _1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                errors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return errors;
    }
    ecrImageForRecipe(recipe) {
        let registryName = `${this.ecrBaseURI}${recipe}`;
        if (this.versionString) {
            registryName += `:${this.versionString}`;
        }
        return aws_ecs_1.ContainerImage.fromRegistry(registryName);
    }
    /**
     * Returns container images for use with the {@link RenderQueue} construct
     */
    forRenderQueue() {
        return this;
    }
    /**
     * Returns container images for use with the {@link UsageBasedLicensing} construct
     */
    forUsageBasedLicensing() {
        return this;
    }
    /**
     * A string representation of the Deadline version to retrieve images for.
     *
     * This can be undefined - in which case the latest available version of Deadline is used.
     */
    get versionString() {
        function numAsString(num) {
            return core_1.Token.isUnresolved(num) ? core_1.Token.asString(num) : num.toString();
        }
        const version = this.version;
        if (version) {
            const major = numAsString(version.majorVersion);
            const minor = numAsString(version.minorVersion);
            const release = numAsString(version.releaseVersion);
            return `${major}.${minor}.${release}`;
        }
        return undefined;
    }
    forceRun() {
        return crypto_1.randomBytes(32).toString('base64').slice(0, 32);
    }
}
exports.ThinkboxDockerImages = ThinkboxDockerImages;
_a = JSII_RTTI_SYMBOL_1;
ThinkboxDockerImages[_a] = { fqn: "aws-rfdk.deadline.ThinkboxDockerImages", version: "0.42.0" };
/**
 * The AWS Thinkbox licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the AWS Thinkbox EULA.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * witout approval.
 */
ThinkboxDockerImages.AWS_THINKBOX_EULA_MESSAGE = `
The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.

Deadline is provided by AWS Thinkbox under the AWS Thinkbox End User License
Agreement (EULA). By installing Deadline, you are agreeing to the terms of this
license. Follow the link below to read the terms of the AWS Thinkbox EULA.

https://www.awsthinkbox.com/end-user-license-agreement

By using the ThinkboxDockerImages to install Deadline you agree to the terms of
the AWS Thinkbox EULA.

Please set the userAwsThinkboxEulaAcceptance property to
USER_ACCEPTS_AWS_THINKBOX_EULA to signify your acceptance of the terms of the
AWS Thinkbox EULA.
`;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGhpbmtib3gtZG9ja2VyLWltYWdlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRoaW5rYm94LWRvY2tlci1pbWFnZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0dBR0c7QUFFSCxtQ0FBcUM7QUFDckMsNkJBQTZCO0FBRTdCLDhDQUcwQjtBQUMxQixvREFJNkI7QUFDN0IsZ0RBQWtEO0FBQ2xELHdDQU11QjtBQUV2Qix3QkFNVztBQUVYOzs7R0FHRztBQUNILElBQVkseUJBY1g7QUFkRCxXQUFZLHlCQUF5QjtJQUNuQzs7OztPQUlHO0lBQ0gsNkhBQWtDLENBQUE7SUFFbEM7Ozs7T0FJRztJQUNILDZIQUFrQyxDQUFBO0FBQ3BDLENBQUMsRUFkVyx5QkFBeUIsR0FBekIsaUNBQXlCLEtBQXpCLGlDQUF5QixRQWNwQztBQXFCRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FxREc7QUFDSCxNQUFhLG9CQUFxQixTQUFRLGdCQUFTO0lBMERqRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWdDO1FBQ3hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLDZCQUE2QixHQUFHLEtBQUssQ0FBQyw2QkFBNkIsQ0FBQztRQUN6RSxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssRUFBRSxPQUFPLENBQUM7UUFFOUIsTUFBTSxVQUFVLEdBQUcsaUJBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQztRQUV6RixNQUFNLFVBQVUsR0FBRyxJQUFJLDhCQUFpQixDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFBRTtZQUN4RSxJQUFJLEVBQUUsc0NBQXNDO1lBQzVDLFdBQVcsRUFBRSxvSUFBb0k7WUFDakosSUFBSSxFQUFFLFVBQVU7WUFDaEIsT0FBTyxFQUFFLG9CQUFPLENBQUMsV0FBVztZQUM1QixPQUFPLEVBQUUsc0JBQXNCO1lBQy9CLE9BQU8sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixZQUFZLEVBQUUsd0JBQWEsQ0FBQyxRQUFRO1NBQ3JDLENBQUMsQ0FBQztRQUVILE1BQU0sV0FBVyxHQUFHLElBQUkscUJBQWMsQ0FBQyxJQUFJLEVBQUUscUJBQXFCLEVBQUU7WUFDbEUsWUFBWSxFQUFFLFVBQVUsQ0FBQyxXQUFXO1lBQ3BDLFVBQVUsRUFBRTtnQkFDViwrR0FBK0c7Z0JBQy9HLHVGQUF1RjtnQkFDdkYsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRLEVBQUU7YUFDMUI7WUFDRCxZQUFZLEVBQUUsMEJBQTBCO1NBQ3pDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxHQUFHLFdBQVcsQ0FBQztRQUVyQyxJQUFJLENBQUMsVUFBVSxHQUFHLFdBQVcsQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7UUFFaEUsSUFBSSxDQUFDLHNCQUFzQixHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyx1Q0FBb0MsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ3BILElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsdUNBQW9DLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUN6RyxDQUFDO0lBRVMsVUFBVTtRQUNsQixNQUFNLE1BQU0sR0FBYSxFQUFFLENBQUM7UUFFNUIsc0VBQXNFO1FBQ3RFLElBQUksSUFBSSxDQUFDLDZCQUE2QixLQUFLLHlCQUF5QixDQUFDLDhCQUE4QixFQUFFO1lBQ25HLE1BQU0sQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMseUJBQXlCLENBQUMsQ0FBQztTQUM3RDtRQUVELGlIQUFpSDtRQUNqSCxpREFBaUQ7UUFDakQsSUFBSSxJQUFJLENBQUMsT0FBTyxZQUFZLGVBQVksRUFBRTtZQUN4QyxNQUFNLFlBQVksR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUM1QyxNQUFNLFNBQVMsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ2pDLElBQUksWUFBWSxJQUFJLFNBQVMsRUFBRTtnQkFDN0IsTUFBTSxDQUFDLElBQUksQ0FBQywyREFBMkQsQ0FBQyxDQUFDO2FBQzFFO1NBQ0Y7UUFDRCxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDO0lBRU8saUJBQWlCLENBQUMsTUFBNEM7UUFDcEUsSUFBSSxZQUFZLEdBQUcsR0FBRyxJQUFJLENBQUMsVUFBVSxHQUFHLE1BQU0sRUFBRSxDQUFDO1FBQ2pELElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUN0QixZQUFZLElBQUksSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7U0FDMUM7UUFDRCxPQUFPLHdCQUFjLENBQUMsWUFBWSxDQUNoQyxZQUFZLENBQ2IsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILElBQVksYUFBYTtRQUN2QixTQUFTLFdBQVcsQ0FBQyxHQUFXO1lBQzlCLE9BQU8sWUFBSyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsWUFBSyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ3hFLENBQUM7UUFFRCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksT0FBTyxFQUFFO1lBQ1gsTUFBTSxLQUFLLEdBQUcsV0FBVyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUNoRCxNQUFNLEtBQUssR0FBRyxXQUFXLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDO1lBQ2hELE1BQU0sT0FBTyxHQUFHLFdBQVcsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUM7WUFFcEQsT0FBTyxHQUFHLEtBQUssSUFBSSxLQUFLLElBQUksT0FBTyxFQUFFLENBQUM7U0FDdkM7UUFFRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRU8sUUFBUTtRQUNkLE9BQU8sb0JBQVcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUN6RCxDQUFDOztBQWxLSCxvREFtS0M7OztBQWxLQzs7Ozs7O0dBTUc7QUFDcUIsOENBQXlCLEdBQVc7Ozs7Ozs7Ozs7Ozs7OztDQWU3RCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHsgcmFuZG9tQnl0ZXMgfSBmcm9tICdjcnlwdG8nO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcblxuaW1wb3J0IHtcbiAgQ29udGFpbmVySW1hZ2UsXG4gIFJlcG9zaXRvcnlJbWFnZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjcyc7XG5pbXBvcnQge1xuICBDb2RlLFxuICBTaW5nbGV0b25GdW5jdGlvbixcbiAgUnVudGltZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBSZXRlbnRpb25EYXlzIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0IHtcbiAgQ29uc3RydWN0LFxuICBDdXN0b21SZXNvdXJjZSxcbiAgRHVyYXRpb24sXG4gIFN0YWNrLFxuICBUb2tlbixcbn0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmltcG9ydCB7XG4gIElWZXJzaW9uLFxuICBSZW5kZXJRdWV1ZUltYWdlcyxcbiAgVGhpbmtib3hNYW5hZ2VkRGVhZGxpbmVEb2NrZXJSZWNpcGVzLFxuICBVc2FnZUJhc2VkTGljZW5zaW5nSW1hZ2VzLFxuICBWZXJzaW9uUXVlcnksXG59IGZyb20gJy4nO1xuXG4vKipcbiAqIENob2ljZXMgZm9yIHNpZ25pZnlpbmcgdGhlIHVzZXIncyBzdGFuY2Ugb24gdGhlIHRlcm1zIG9mIHRoZSBBV1MgVGhpbmtib3ggRW5kLVVzZXIgTGljZW5zZSBBZ3JlZW1lbnQgKEVVTEEpLlxuICogU2VlOiBodHRwczovL3d3dy5hd3N0aGlua2JveC5jb20vZW5kLXVzZXItbGljZW5zZS1hZ3JlZW1lbnRcbiAqL1xuZXhwb3J0IGVudW0gQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZSB7XG4gIC8qKlxuICAgKiBUaGUgdXNlciBzaWduaWZpZXMgdGhlaXIgZXhwbGljaXQgcmVqZWN0aW9uIG9mIHRoZSB0ZW1zIG9mIHRoZSBBV1MgVGhpbmtib3ggRVVMQS5cbiAgICpcbiAgICogU2VlOiBodHRwczovL3d3dy5hd3N0aGlua2JveC5jb20vZW5kLXVzZXItbGljZW5zZS1hZ3JlZW1lbnRcbiAgICovXG4gIFVTRVJfUkVKRUNUU19BV1NfVEhJTktCT1hfRVVMQSA9IDAsXG5cbiAgLyoqXG4gICAqIFRoZSB1c2VyIHNpZ25pZmllcyB0aGVpciBleHBsaWNpdCBhY2NlcHRhbmNlIG9mIHRoZSB0ZXJtcyBvZiB0aGUgQVdTIFRoaW5rYm94IEVVTEEuXG4gICAqXG4gICAqIFNlZTogaHR0cHM6Ly93d3cuYXdzdGhpbmtib3guY29tL2VuZC11c2VyLWxpY2Vuc2UtYWdyZWVtZW50XG4gICAqL1xuICBVU0VSX0FDQ0VQVFNfQVdTX1RISU5LQk9YX0VVTEEgPSAxLFxufVxuXG4vKipcbiAqIEludGVyZmFjZSB0byBzcGVjaWZ5IHRoZSBwcm9wZXJ0aWVzIHdoZW4gaW5zdGFudGlhdGluZyBhIHtAbGluayBUaGlua2JveERvY2tlckltYWdlc30gaW5zdG5hY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVGhpbmtib3hEb2NrZXJJbWFnZXNQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgRGVhZGxpbmUgdmVyc2lvbiB0byBvYnRhaW4gaW1hZ2VzIGZvci5cbiAgICogQGRlZmF1bHQgbGF0ZXN0XG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9uPzogSVZlcnNpb247XG5cbiAgLyoqXG4gICAqIERlYWRsaW5lIGlzIGxpY2Vuc2VkIHVuZGVyIHRoZSB0ZXJtcyBvZiB0aGUgQVdTIFRoaW5rYm94IEVuZC1Vc2VyIExpY2Vuc2UgQWdyZWVtZW50IChzZWU6IGh0dHBzOi8vd3d3LmF3c3RoaW5rYm94LmNvbS9lbmQtdXNlci1saWNlbnNlLWFncmVlbWVudCkuXG4gICAqIFVzZXJzIG9mIFRoaW5rYm94RG9ja2VySW1hZ2VzIG11c3QgZXhwbGljaXRseSBzaWduaWZ5IHRoZWlyIGFjY2VwdGFuY2Ugb2YgdGhlIHRlcm1zIG9mIHRoZSBBV1MgVGhpbmtib3ggRVVMQSB0aHJvdWdoIHRoaXNcbiAgICogcHJvcGVydHkgYmVmb3JlIHRoZSB7QGxpbmsgVGhpbmtib3hEb2NrZXJJbWFnZXN9IHdpbGwgYmUgYWxsb3dlZCB0byBkZXBsb3kgRGVhZGxpbmUuXG4gICAqL1xuICAvLyBEZXZlbG9wZXIgbm90ZTogSXQgaXMgYSBsZWdhbCByZXF1aXJlbWVudCB0aGF0IHRoZSBkZWZhdWx0IGJlIFVTRVJfUkVKRUNUU19BV1NfVEhJTktCT1hfRVVMQS5cbiAgcmVhZG9ubHkgdXNlckF3c1RoaW5rYm94RXVsYUFjY2VwdGFuY2U6IEF3c1RoaW5rYm94RXVsYUFjY2VwdGFuY2U7XG59XG5cbi8qKlxuICogQW4gQVBJIGZvciBpbnRlcmFjdGluZyB3aXRoIHB1YmxpY2x5IGF2YWlsYWJsZSBEZWFkbGluZSBjb250YWluZXIgaW1hZ2VzIHB1Ymxpc2hlZCBieSBBV1MgVGhpbmtib3guXG4gKlxuICogVGhpcyBwcm92aWRlcyBjb250YWluZXIgaW1hZ2VzIGFzIHJlcXVpcmVkIGJ5IFJGREsncyBEZWFkbGluZSBjb25zdHJ1Y3RzIHN1Y2ggYXNcbiAqXG4gKiAqIHtAbGluayBAYXdzLXJmZGsvZGVhZGxpbmUjUmVuZGVyUXVldWV9XG4gKiAqIHtAbGluayBAYXdzLXJmZGsvZGVhZGxpbmUjVXNhZ2VCYXNlZExpY2Vuc2luZ31cbiAqXG4gKiBTdWNjZXNzZnVsIHVzYWdlIG9mIHRoZSBwdWJsaXNoZWQgRGVhZGxpbmUgY29udGFpbmVyIGltYWdlcyB3aXRoIHRoaXMgY2xhc3MgcmVxdWlyZXM6XG4gKlxuICogMSkgRXhwbGljaXQgYWNjZXB0YW5jZSBvZiB0aGUgdGVybXMgb2YgdGhlIEFXUyBUaGlua2JveCBFbmQgVXNlciBMaWNlbnNlIEFncmVlbWVudCwgdW5kZXIgd2hpY2ggRGVhZGxpbmUgaXNcbiAqICAgIGRpc3RyaWJ1dGVkOyBhbmRcbiAqIDIpIFRoZSBsYW1iZGEgb24gd2hpY2ggdGhlIGN1c3RvbSByZXNvdXJjZSBsb29rcyB1cCB0aGUgVGhpbmtib3ggY29udGFpbmVyIGltYWdlcyBpcyBhYmxlIHRvIG1ha2UgSFRUUFNcbiAqICAgIHJlcXVlc3RzIHRvIHRoZSBvZmZpY2lhbCBBV1MgVGhpbmJveCBkb3dubG9hZCBzaXRlOiBodHRwczovL2Rvd25sb2Fkcy50aGlua2JveHNvZnR3YXJlLmNvbVxuICpcbiAqIFJlc291cmNlcyBEZXBsb3llZFxuICogLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gKiAtIEEgTGFtYmRhIGZ1bmN0aW9uIGNvbnRhaW5pbmcgYSBzY3JpcHQgdG8gbG9vayB1cCB0aGUgQVdTIFRoaW5rYm94IGNvbnRhaW5lciBpbWFnZSByZWdpc3RyeVxuICpcbiAqIFNlY3VyaXR5IENvbnNpZGVyYXRpb25zXG4gKiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAqIC0gQ0RLIGRlcGxveXMgdGhlIGNvZGUgZm9yIHRoaXMgbGFtYmRhIGFzIGFuIFMzIG9iamVjdCBpbiB0aGUgQ0RLIGJvb3RzdHJhcCBidWNrZXQuIFlvdSBtdXN0IGxpbWl0IHdyaXRlIGFjY2VzcyB0b1xuICogICB5b3VyIENESyBib290c3RyYXAgYnVja2V0IHRvIHByZXZlbnQgYW4gYXR0YWNrZXIgZnJvbSBtb2RpZnlpbmcgdGhlIGFjdGlvbnMgcGVyZm9ybWVkIGJ5IHRoZXNlIHNjcmlwdHMuIFdlIHN0cm9uZ2x5XG4gKiAgIHJlY29tbWVuZCB0aGF0IHlvdSBlaXRoZXIgZW5hYmxlIEFtYXpvbiBTMyBzZXJ2ZXIgYWNjZXNzIGxvZ2dpbmcgb24geW91ciBDREsgYm9vdHN0cmFwIGJ1Y2tldCwgb3IgZW5hYmxlIEFXU1xuICogICBDbG91ZFRyYWlsIG9uIHlvdXIgYWNjb3VudCB0byBhc3Npc3QgaW4gcG9zdC1pbmNpZGVudCBhbmFseXNpcyBvZiBjb21wcm9taXNlZCBwcm9kdWN0aW9uIGVudmlyb25tZW50cy5cbiAqXG4gKiBGb3IgZXhhbXBsZSwgdG8gY29uc3RydWN0IGEgUmVuZGVyUXVldWUgdXNpbmcgdGhlIGltYWdlczpcbiAqXG4gKiBgYGB0c1xuICogaW1wb3J0IHsgQXBwLCBTdGFjaywgVnBjIH0gZnJvbSAnQGF3cy1yZmRrL2NvcmUnO1xuICogaW1wb3J0IHsgQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZSwgUmVuZGVyUXVldWUsIFJlcG9zaXRvcnksIFRoaW5rYm94RG9ja2VySW1hZ2VzLCBWZXJzaW9uUXVlcnkgfSBmcm9tICdAYXdzLXJmZGsvZGVhZGxpbmUnO1xuICogY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICogY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2snKTtcbiAqIGNvbnN0IHZwYyA9IG5ldyBWcGMoc3RhY2ssICdWcGMnKTtcbiAqIGNvbnN0IHZlcnNpb24gPSBuZXcgVmVyc2lvblF1ZXJ5KHN0YWNrLCAnVmVyc2lvbicsIHtcbiAqICAgdmVyc2lvbjogJzEwLjEuMTInLFxuICogfSk7XG4gKiBjb25zdCBpbWFnZXMgPSBuZXcgVGhpbmtib3hEb2NrZXJJbWFnZXMoc3RhY2ssICdJbWFnZScsIHtcbiAqICAgdmVyc2lvbixcbiAqICAgLy8gQ2hhbmdlIHRoaXMgdG8gQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZS5VU0VSX0FDQ0VQVFNfQVdTX1RISU5LQk9YX0VVTEEgdG8gYWNjZXB0IHRoZSB0ZXJtc1xuICogICAvLyBvZiB0aGUgQVdTIFRoaW5rYm94IEVuZCBVc2VyIExpY2Vuc2UgQWdyZWVtZW50XG4gKiAgIHVzZXJBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlOiBBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlLlVTRVJfUkVKRUNUU19BV1NfVEhJTktCT1hfRVVMQSxcbiAqIH0pO1xuICogY29uc3QgcmVwb3NpdG9yeSA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAnUmVwb3NpdG9yeScsIHtcbiAqICAgdnBjLFxuICogICB2ZXJzaW9uLFxuICogfSk7XG4gKlxuICogY29uc3QgcmVuZGVyUXVldWUgPSBuZXcgUmVuZGVyUXVldWUoc3RhY2ssICdSZW5kZXJRdWV1ZScsIHtcbiAqICAgaW1hZ2VzOiBpbWFnZXMuZm9yUmVuZGVyUXVldWUoKSxcbiAqICAgLy8gLi4uXG4gKiB9KTtcbiAqIGBgYFxuICovXG5leHBvcnQgY2xhc3MgVGhpbmtib3hEb2NrZXJJbWFnZXMgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAvKipcbiAgICogVGhlIEFXUyBUaGlua2JveCBsaWNlbnNpbmcgbWVzc2FnZSB0aGF0IGlzIHByZXNlbnRlZCB0byB0aGUgdXNlciBpZiB0aGV5IGNyZWF0ZSBhbiBpbnN0YW5jZSBvZlxuICAgKiB0aGlzIGNsYXNzIHdpdGhvdXQgZXhwbGljaXRseSBhY2NlcHRpbmcgdGhlIEFXUyBUaGlua2JveCBFVUxBLlxuICAgKlxuICAgKiBOb3RlIHRvIGRldmVsb3BlcnM6IFRoZSB0ZXh0IG9mIHRoaXMgc3RyaW5nIGlzIGEgbGVnYWwgcmVxdWlyZW1lbnQsIGFuZCBtdXN0IG5vdCBiZSBhbHRlcmVkXG4gICAqIHdpdG91dCBhcHByb3ZhbC5cbiAgICovXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IEFXU19USElOS0JPWF9FVUxBX01FU1NBR0U6IHN0cmluZyA9IGBcblRoZSBUaGlua2JveERvY2tlckltYWdlcyB3aWxsIGluc3RhbGwgRGVhZGxpbmUgb250byBvbmUgb3IgbW9yZSBFQzIgaW5zdGFuY2VzLlxuXG5EZWFkbGluZSBpcyBwcm92aWRlZCBieSBBV1MgVGhpbmtib3ggdW5kZXIgdGhlIEFXUyBUaGlua2JveCBFbmQgVXNlciBMaWNlbnNlXG5BZ3JlZW1lbnQgKEVVTEEpLiBCeSBpbnN0YWxsaW5nIERlYWRsaW5lLCB5b3UgYXJlIGFncmVlaW5nIHRvIHRoZSB0ZXJtcyBvZiB0aGlzXG5saWNlbnNlLiBGb2xsb3cgdGhlIGxpbmsgYmVsb3cgdG8gcmVhZCB0aGUgdGVybXMgb2YgdGhlIEFXUyBUaGlua2JveCBFVUxBLlxuXG5odHRwczovL3d3dy5hd3N0aGlua2JveC5jb20vZW5kLXVzZXItbGljZW5zZS1hZ3JlZW1lbnRcblxuQnkgdXNpbmcgdGhlIFRoaW5rYm94RG9ja2VySW1hZ2VzIHRvIGluc3RhbGwgRGVhZGxpbmUgeW91IGFncmVlIHRvIHRoZSB0ZXJtcyBvZlxudGhlIEFXUyBUaGlua2JveCBFVUxBLlxuXG5QbGVhc2Ugc2V0IHRoZSB1c2VyQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZSBwcm9wZXJ0eSB0b1xuVVNFUl9BQ0NFUFRTX0FXU19USElOS0JPWF9FVUxBIHRvIHNpZ25pZnkgeW91ciBhY2NlcHRhbmNlIG9mIHRoZSB0ZXJtcyBvZiB0aGVcbkFXUyBUaGlua2JveCBFVUxBLlxuYDtcblxuICAvKipcbiAgICogQSB7QGxpbmsgRG9ja2VySW1hZ2VBc3NldH0gdGhhdCBjYW4gYmUgdXNlZCB0byBidWlsZCBUaGlua2JveCdzIERlYWRsaW5lIFJDUyBEb2NrZXIgUmVjaXBlIGludG8gYVxuICAgKiBjb250YWluZXIgaW1hZ2UgdGhhdCBjYW4gYmUgZGVwbG95ZWQgaW4gQ0RLLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgVGhlIHBhcmVudCBzY29wZVxuICAgKiBAcGFyYW0gaWQgVGhlIGNvbnN0cnVjdCBJRFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlbW90ZUNvbm5lY3Rpb25TZXJ2ZXI6IENvbnRhaW5lckltYWdlO1xuXG4gIC8qKlxuICAgKiBBIHtAbGluayBEb2NrZXJJbWFnZUFzc2V0fSB0aGF0IGNhbiBiZSB1c2VkIHRvIGJ1aWxkIFRoaW5rYm94J3MgRGVhZGxpbmUgTGljZW5zZSBGb3J3YXJkZXIgRG9ja2VyIFJlY2lwZSBpbnRvIGFcbiAgICogY29udGFpbmVyIGltYWdlIHRoYXQgY2FuIGJlIGRlcGxveWVkIGluIENESy5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgc2NvcGVcbiAgICogQHBhcmFtIGlkIFRoZSBjb25zdHJ1Y3QgSURcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBsaWNlbnNlRm9yd2FyZGVyOiBDb250YWluZXJJbWFnZTtcblxuICAvKipcbiAgICogVGhlIHZlcnNpb24gb2YgRGVhZGxpbmUgaW5zdGFsbGVkIGluIHRoZSBjb250YWluZXIgaW1hZ2VzXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IHZlcnNpb24/OiBJVmVyc2lvbjtcblxuICAvKipcbiAgICogVGhlIGJhc2UgVVJJIGZvciBBV1MgVGhpbmtib3ggcHVibGlzaGVkIERlYWRsaW5lIEVDUiBpbWFnZXMuXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IGVjckJhc2VVUkk6IHN0cmluZztcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgdXNlciBoYXMgYWNjZXB0ZWQgdGhlIEFXUyBUaGlua2JveCBFVUxBXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IHVzZXJBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlOiBBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBUaGlua2JveERvY2tlckltYWdlc1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMudXNlckF3c1RoaW5rYm94RXVsYUFjY2VwdGFuY2UgPSBwcm9wcy51c2VyQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZTtcbiAgICB0aGlzLnZlcnNpb24gPSBwcm9wcz8udmVyc2lvbjtcblxuICAgIGNvbnN0IGxhbWJkYUNvZGUgPSBDb2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4nLCAnLi4nLCAnbGFtYmRhcycsICdub2RlanMnKSk7XG5cbiAgICBjb25zdCBsYW1iZGFGdW5jID0gbmV3IFNpbmdsZXRvbkZ1bmN0aW9uKHRoaXMsICdWZXJzaW9uUHJvdmlkZXJGdW5jdGlvbicsIHtcbiAgICAgIHV1aWQ6ICcwODU1MzQxNi0xZmM5LTRiZTktYTgxOC02MDlhMzFhZTFiNWInLFxuICAgICAgZGVzY3JpcHRpb246ICdVc2VkIGJ5IHRoZSBUaGlua2JveERvY2tlckltYWdlcyBjb25zdHJ1Y3QgdG8gbG9vayB1cCB0aGUgRUNSIHJlcG9zaXRvcmllcyB3aGVyZSBBV1MgVGhpbmtib3ggcHVibGlzaGVzIERlYWRsaW5lIGNvbnRhaW5lciBpbWFnZXMuJyxcbiAgICAgIGNvZGU6IGxhbWJkYUNvZGUsXG4gICAgICBydW50aW1lOiBSdW50aW1lLk5PREVKU18xNl9YLFxuICAgICAgaGFuZGxlcjogJ2Vjci1wcm92aWRlci5oYW5kbGVyJyxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLnNlY29uZHMoMzApLFxuICAgICAgbG9nUmV0ZW50aW9uOiBSZXRlbnRpb25EYXlzLk9ORV9XRUVLLFxuICAgIH0pO1xuXG4gICAgY29uc3QgZWNyUHJvdmlkZXIgPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ1RoaW5rYm94RWNyUHJvdmlkZXInLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IGxhbWJkYUZ1bmMuZnVuY3Rpb25Bcm4sXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIC8vIGNyZWF0ZSBhIHJhbmRvbSBzdHJpbmcgdGhhdCB3aWxsIGZvcmNlIHRoZSBMYW1iZGEgdG8gXCJ1cGRhdGVcIiBvbiBlYWNoIGRlcGxveW1lbnQuIENoYW5nZXMgdG8gaXRzIG91dHB1dCB3aWxsXG4gICAgICAgIC8vIGJlIHByb3BhZ2F0ZWQgdG8gYW55IENsb3VkRm9ybWF0aW9uIHJlc291cmNlIHByb3ZpZGVycyB0aGF0IHJlZmVyZW5jZSB0aGUgb3V0cHV0IEFSTlxuICAgICAgICBGb3JjZVJ1bjogdGhpcy5mb3JjZVJ1bigpLFxuICAgICAgfSxcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6UkZES19FY3JQcm92aWRlcicsXG4gICAgfSk7XG5cbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gZWNyUHJvdmlkZXI7XG5cbiAgICB0aGlzLmVjckJhc2VVUkkgPSBlY3JQcm92aWRlci5nZXRBdHQoJ0VjclVSSVByZWZpeCcpLnRvU3RyaW5nKCk7XG5cbiAgICB0aGlzLnJlbW90ZUNvbm5lY3Rpb25TZXJ2ZXIgPSB0aGlzLmVjckltYWdlRm9yUmVjaXBlKFRoaW5rYm94TWFuYWdlZERlYWRsaW5lRG9ja2VyUmVjaXBlcy5SRU1PVEVfQ09OTkVDVElPTl9TRVJWRVIpO1xuICAgIHRoaXMubGljZW5zZUZvcndhcmRlciA9IHRoaXMuZWNySW1hZ2VGb3JSZWNpcGUoVGhpbmtib3hNYW5hZ2VkRGVhZGxpbmVEb2NrZXJSZWNpcGVzLkxJQ0VOU0VfRk9SV0FSREVSKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBvblZhbGlkYXRlKCkge1xuICAgIGNvbnN0IGVycm9yczogc3RyaW5nW10gPSBbXTtcblxuICAgIC8vIFVzZXJzIG11c3QgYWNjZXB0IHRoZSBBV1MgVGhpbmtib3ggRVVMQSB0byB1c2UgdGhlIGNvbnRhaW5lciBpbWFnZXNcbiAgICBpZiAodGhpcy51c2VyQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZSAhPT0gQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZS5VU0VSX0FDQ0VQVFNfQVdTX1RISU5LQk9YX0VVTEEpIHtcbiAgICAgIGVycm9ycy5wdXNoKFRoaW5rYm94RG9ja2VySW1hZ2VzLkFXU19USElOS0JPWF9FVUxBX01FU1NBR0UpO1xuICAgIH1cblxuICAgIC8vIFVzaW5nIHRoZSBvdXRwdXQgb2YgVmVyc2lvblF1ZXJ5IGFjcm9zcyBzdGFja3MgY2FuIGNhdXNlIGlzc3Vlcy4gQ2xvdWRGb3JtYXRpb24gc3RhY2sgb3V0cHV0cyBjYW5ub3QgY2hhbmdlIGlmXG4gICAgLy8gYSByZXNvdXJjZSBpbiBhbm90aGVyIHN0YWNrIGlzIHJlZmVyZW5jaW5nIGl0LlxuICAgIGlmICh0aGlzLnZlcnNpb24gaW5zdGFuY2VvZiBWZXJzaW9uUXVlcnkpIHtcbiAgICAgIGNvbnN0IHZlcnNpb25TdGFjayA9IFN0YWNrLm9mKHRoaXMudmVyc2lvbik7XG4gICAgICBjb25zdCB0aGlzU3RhY2sgPSBTdGFjay5vZih0aGlzKTtcbiAgICAgIGlmICh2ZXJzaW9uU3RhY2sgIT0gdGhpc1N0YWNrKSB7XG4gICAgICAgIGVycm9ycy5wdXNoKCdBIFZlcnNpb25RdWVyeSBjYW4gbm90IGJlIHN1cHBsaWVkIGZyb20gYSBkaWZmZXJlbnQgc3RhY2snKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIGVycm9ycztcbiAgfVxuXG4gIHByaXZhdGUgZWNySW1hZ2VGb3JSZWNpcGUocmVjaXBlOiBUaGlua2JveE1hbmFnZWREZWFkbGluZURvY2tlclJlY2lwZXMpOiBSZXBvc2l0b3J5SW1hZ2Uge1xuICAgIGxldCByZWdpc3RyeU5hbWUgPSBgJHt0aGlzLmVjckJhc2VVUkl9JHtyZWNpcGV9YDtcbiAgICBpZiAodGhpcy52ZXJzaW9uU3RyaW5nKSB7XG4gICAgICByZWdpc3RyeU5hbWUgKz0gYDoke3RoaXMudmVyc2lvblN0cmluZ31gO1xuICAgIH1cbiAgICByZXR1cm4gQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KFxuICAgICAgcmVnaXN0cnlOYW1lLFxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBjb250YWluZXIgaW1hZ2VzIGZvciB1c2Ugd2l0aCB0aGUge0BsaW5rIFJlbmRlclF1ZXVlfSBjb25zdHJ1Y3RcbiAgICovXG4gIHB1YmxpYyBmb3JSZW5kZXJRdWV1ZSgpOiBSZW5kZXJRdWV1ZUltYWdlcyB7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBjb250YWluZXIgaW1hZ2VzIGZvciB1c2Ugd2l0aCB0aGUge0BsaW5rIFVzYWdlQmFzZWRMaWNlbnNpbmd9IGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIGZvclVzYWdlQmFzZWRMaWNlbnNpbmcoKTogVXNhZ2VCYXNlZExpY2Vuc2luZ0ltYWdlcyB7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQSBzdHJpbmcgcmVwcmVzZW50YXRpb24gb2YgdGhlIERlYWRsaW5lIHZlcnNpb24gdG8gcmV0cmlldmUgaW1hZ2VzIGZvci5cbiAgICpcbiAgICogVGhpcyBjYW4gYmUgdW5kZWZpbmVkIC0gaW4gd2hpY2ggY2FzZSB0aGUgbGF0ZXN0IGF2YWlsYWJsZSB2ZXJzaW9uIG9mIERlYWRsaW5lIGlzIHVzZWQuXG4gICAqL1xuICBwcml2YXRlIGdldCB2ZXJzaW9uU3RyaW5nKCk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgZnVuY3Rpb24gbnVtQXNTdHJpbmcobnVtOiBudW1iZXIpOiBzdHJpbmcge1xuICAgICAgcmV0dXJuIFRva2VuLmlzVW5yZXNvbHZlZChudW0pID8gVG9rZW4uYXNTdHJpbmcobnVtKSA6IG51bS50b1N0cmluZygpO1xuICAgIH1cblxuICAgIGNvbnN0IHZlcnNpb24gPSB0aGlzLnZlcnNpb247XG4gICAgaWYgKHZlcnNpb24pIHtcbiAgICAgIGNvbnN0IG1ham9yID0gbnVtQXNTdHJpbmcodmVyc2lvbi5tYWpvclZlcnNpb24pO1xuICAgICAgY29uc3QgbWlub3IgPSBudW1Bc1N0cmluZyh2ZXJzaW9uLm1pbm9yVmVyc2lvbik7XG4gICAgICBjb25zdCByZWxlYXNlID0gbnVtQXNTdHJpbmcodmVyc2lvbi5yZWxlYXNlVmVyc2lvbik7XG5cbiAgICAgIHJldHVybiBgJHttYWpvcn0uJHttaW5vcn0uJHtyZWxlYXNlfWA7XG4gICAgfVxuXG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIHByaXZhdGUgZm9yY2VSdW4oKTogc3RyaW5nIHtcbiAgICByZXR1cm4gcmFuZG9tQnl0ZXMoMzIpLnRvU3RyaW5nKCdiYXNlNjQnKS5zbGljZSgwLCAzMik7XG4gIH1cbn1cbiJdfQ==