"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotEventPluginFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const worker_configuration_1 = require("./worker-configuration");
/**
 * This construct reperesents a fleet from the Spot Fleet Request created by the Spot Event Plugin.
 * This fleet is intended to be used as input for the {@link @aws-rfdk/deadline#ConfigureSpotEventPlugin} construct.
 *
 * The construct itself doesn't create the Spot Fleet Request, but deploys all the resources
 * required for the Spot Fleet Request and generates the Spot Fleet Configuration setting:
 * a one to one mapping between a Deadline Group and Spot Fleet Request Configurations.
 *
 * ![architecture diagram](/diagrams/deadline/SpotEventPluginFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Instance Role, corresponding IAM Policy and an Instance Profile.
 * - A Fleet Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *   in the fleet.
 * - A security Group if security groups are not provided.
 * - An EC2 Launch Template for the Spot Fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *   that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *   the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *   patching strategy to keep this software current with the latest security patches. Please see
 *   https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 */
class SpotEventPluginFleet extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.defaultSubnets = !props.vpcSubnets;
        this.deadlineGroups = props.deadlineGroups.map(group => group.toLocaleLowerCase());
        this.deadlinePools = props.deadlinePools?.map(pool => pool.toLocaleLowerCase());
        this.validateProps(props);
        this.securityGroups = props.securityGroups ?? [new aws_ec2_1.SecurityGroup(this, 'SpotFleetSecurityGroup', { vpc: props.vpc })];
        this.connections = new aws_ec2_1.Connections({ securityGroups: this.securityGroups });
        this.connections.allowToDefaultPort(props.renderQueue.endpoint);
        this.fleetInstanceRole = props.fleetInstanceRole ?? new aws_iam_1.Role(this, 'SpotFleetInstanceRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
            ],
            description: `Spot Fleet instance role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.fleetInstanceRole.roleName],
        });
        this.grantPrincipal = this.fleetInstanceRole;
        this.fleetRole = props.fleetRole ?? new aws_iam_1.Role(this, 'SpotFleetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
            ],
            description: `Spot Fleet role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.blockDevices = props.blockDevices;
        this.subnets = props.vpc.selectSubnets(props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT });
        this.instanceTypes = props.instanceTypes;
        this.allocationStrategy = props.allocationStrategy ?? spot_event_plugin_fleet_ref_1.SpotFleetAllocationStrategy.LOWEST_PRICE;
        this.maxCapacity = props.maxCapacity;
        this.validUntil = props.validUntil;
        this.keyName = props.keyName;
        const imageConfig = props.workerMachineImage.getImage(this);
        this.osType = imageConfig.osType;
        this.userData = props.userData ?? imageConfig.userData;
        this.machineImage = props.workerMachineImage;
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this,
            cloudWatchLogSettings: {
                logGroupPrefix: SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: {
                groups: this.deadlineGroups,
                pools: this.deadlinePools,
                region: props.deadlineRegion,
            },
            userDataProvider: props.userDataProvider,
        });
        this.remoteControlPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + SpotEventPluginFleet.MAX_WORKERS_PER_HOST);
        this.tags = new core_1.TagManager(core_1.TagType.KEY_VALUE, 'RFDK::SpotEventPluginFleet');
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
        this.launchTemplate = this.createLaunchTemplate(props.trackInstancesWithResourceTracker ?? true);
        this._launchTemplateConfigs = this.createLaunchTemplateConfigs();
    }
    /**
     * @inheritdoc
     */
    allowRemoteControlFrom(other) {
        this.connections.allowFrom(other.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    /**
     * @inheritdoc
     */
    allowRemoteControlTo(other) {
        other.connections.allowTo(this.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    createLaunchTemplate(resourceTrackerEnabled) {
        const launchTemplate = new aws_ec2_1.LaunchTemplate(this, 'LaunchTemplate', {
            blockDevices: this.blockDevices,
            role: this.fleetInstanceRole,
            machineImage: this.machineImage,
            keyName: this.keyName,
            securityGroup: this.securityGroups[0],
            userData: this.userData,
        });
        if (this.securityGroups.length > 1) {
            launchTemplate.connections.addSecurityGroup(...this.securityGroups.slice(1));
        }
        core_1.Tags.of(launchTemplate).add(resourceTrackerEnabled ? 'DeadlineTrackedAWSResource' : 'DeadlineResourceTracker', 'SpotEventPlugin');
        return launchTemplate;
    }
    createLaunchTemplateConfigs() {
        const launchTemplateConfigs = [];
        // Create a launch template config for each instance type + subnet pair
        this.instanceTypes.forEach(instanceType => {
            this.subnets.subnetIds.forEach(subnetId => {
                launchTemplateConfigs.push({
                    LaunchTemplateSpecification: {
                        Version: aws_ec2_1.LaunchTemplateSpecialVersions.LATEST_VERSION,
                        LaunchTemplateId: this.launchTemplate.launchTemplateId,
                        LaunchTemplateName: this.launchTemplate.launchTemplateName,
                    },
                    Overrides: [{
                            InstanceType: instanceType.toString(),
                            SubnetId: subnetId,
                        }],
                });
            });
        });
        return launchTemplateConfigs;
    }
    validateProps(props) {
        this.validateFleetInstanceRole(props.fleetInstanceRole);
        this.validateInstanceTypes(props.instanceTypes);
        this.validateSubnets(props.vpc, props.vpcSubnets);
        this.validateGroups('deadlineGroups', this.deadlineGroups);
        this.validateRegion('deadlineRegion', props.deadlineRegion);
        this.validateBlockDevices(props.blockDevices);
    }
    validateFleetInstanceRole(fleetInstanceRole) {
        if (fleetInstanceRole) {
            if (core_1.Stack.of(fleetInstanceRole) !== core_1.Stack.of(this)) {
                throw new Error(`Fleet instance role should be created on the same stack as ${this.constructor.name} to avoid circular dependencies.`);
            }
        }
    }
    validateInstanceTypes(array) {
        if (array.length === 0) {
            throw new Error('At least one instance type is required for a Spot Fleet Request Configuration');
        }
    }
    validateSubnets(vpc, vpcSubnets) {
        const { subnets } = vpc.selectSubnets(vpcSubnets);
        if (subnets.length === 0) {
            core_1.Annotations.of(this).addError(`Did not find any subnets matching '${JSON.stringify(vpcSubnets)}', please use a different selection.`);
        }
    }
    validateGroups(property, array) {
        const regex = /^(?!none$)[a-z0-9-_]+$/g;
        if (array.length === 0) {
            throw new Error('At least one Deadline Group is required for a Spot Fleet Request Configuration');
        }
        array.forEach(value => {
            if (!regex.test(value)) {
                throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
            }
        });
    }
    validateRegion(property, region) {
        const regex = /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i;
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            core_1.Annotations.of(this).addWarning(`The spot-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                const { iops, volumeType } = device.volume.ebsDevice;
                if (!iops) {
                    if (volumeType === aws_ec2_1.EbsDeviceVolumeType.IO1) {
                        throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                    }
                }
                else if (volumeType !== aws_ec2_1.EbsDeviceVolumeType.IO1) {
                    core_1.Annotations.of(this).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    core_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the spot-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
}
exports.SpotEventPluginFleet = SpotEventPluginFleet;
_a = JSII_RTTI_SYMBOL_1;
SpotEventPluginFleet[_a] = { fqn: "aws-rfdk.deadline.SpotEventPluginFleet", version: "0.42.0" };
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
SpotEventPluginFleet.MAX_WORKERS_PER_HOST = 8;
//# sourceMappingURL=data:application/json;base64,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