/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { BlockDevice } from '@aws-cdk/aws-autoscaling';
import { Connections, IConnectable, IMachineImage, InstanceType, ISecurityGroup, IVpc, LaunchTemplate, OperatingSystemType, Port, SelectedSubnets, SubnetSelection, UserData } from '@aws-cdk/aws-ec2';
import { CfnInstanceProfile, IGrantable, IPrincipal, IRole } from '@aws-cdk/aws-iam';
import { Construct, Expiration, TagManager } from '@aws-cdk/core';
import { IScriptHost, LogGroupFactoryProps } from '../../core';
import { LaunchTemplateConfig } from '../../lambdas/nodejs/configure-spot-event-plugin';
import { IRenderQueue } from './render-queue';
import { SpotFleetAllocationStrategy } from './spot-event-plugin-fleet-ref';
import { IInstanceUserDataProvider } from './worker-configuration';
/**
 * Properties for the Spot Event Plugin Worker Fleet.
 */
export interface SpotEventPluginFleetProps {
    /**
     * VPC to launch the Worker fleet in.
     */
    readonly vpc: IVpc;
    /**
     * The RenderQueue that Worker fleet should connect to.
     */
    readonly renderQueue: IRenderQueue;
    /**
     * The AMI of the Deadline Worker to launch.
     */
    readonly workerMachineImage: IMachineImage;
    /**
     * The  the maximum capacity that the Spot Fleet can grow to.
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#spot-fleet-requests
     */
    readonly maxCapacity: number;
    /**
     * Types of instances to launch.
     */
    readonly instanceTypes: InstanceType[];
    /**
     * Deadline groups these workers need to be assigned to.
     *
     * Also, note that the Spot Fleet configuration does not allow using wildcards as part of the Group name
     * as described here https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#wildcards
     */
    readonly deadlineGroups: string[];
    /**
     * Deadline pools these workers need to be assigned to.
     *
     * @default - Workers are not assigned to any pool.
     */
    readonly deadlinePools?: string[];
    /**
     * Deadline region these workers needs to be assigned to.
     * Note that this is not an AWS region but a Deadline region used for path mapping.
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/cross-platform.html#regions
     *
     * @default - Worker is not assigned to any Deadline region.
     */
    readonly deadlineRegion?: string;
    /**
     * An IAM role for the spot fleet.
     *
     * The role must be assumable by the service principal `spotfleet.amazonaws.com`
     * and have AmazonEC2SpotFleetTaggingRole policy attached
     *
     * ```ts
     * const role = new iam.Role(this, 'FleetRole', {
     *   assumedBy: new iam.ServicePrincipal('spotfleet.amazonaws.com'),
     *   managedPolicies: [
     *     ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
     *   ],
     * });
     * ```
     *
     * @default - A role will automatically be created.
     */
    readonly fleetRole?: IRole;
    /**
     * An IAM role to associate with the instance profile assigned to its resources.
     * Create this role on the same stack with the SpotEventPluginFleet to avoid circular dependencies.
     *
     * The role must be assumable by the service principal `ec2.amazonaws.com` and
     * have AWSThinkboxDeadlineSpotEventPluginWorkerPolicy policy attached:
     *
     * ```ts
     * const role = new iam.Role(this, 'MyRole', {
     *   assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
     *   managedPolicies: [
     *     ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
     *   ],
     * });
     * ```
     *
     * @default - A role will automatically be created.
     */
    readonly fleetInstanceRole?: IRole;
    /**
     * Name of SSH keypair to grant access to instances.
     *
     * @default - No SSH access will be possible.
     */
    readonly keyName?: string;
    /**
     * Security Groups to assign to this fleet.
     *
     * @default - A new security group will be created automatically.
     */
    readonly securityGroups?: ISecurityGroup[];
    /**
     * User data that instances use when starting up.
     *
     * @default - User data will be created automatically.
     */
    readonly userData?: UserData;
    /**
     * The Block devices that will be attached to your workers.
     *
     * @default - The default devices of the provided ami will be used.
     */
    readonly blockDevices?: BlockDevice[];
    /**
     * Indicates how to allocate the target Spot Instance capacity
     * across the Spot Instance pools specified by the Spot Fleet request.
     *
     * @default - SpotFleetAllocationStrategy.LOWEST_PRICE.
     */
    readonly allocationStrategy?: SpotFleetAllocationStrategy;
    /**
     * Where to place the instance within the VPC.
     *
     * @default - Private subnets.
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * The end date and time of the request.
     * After the end date and time, no new Spot Instance requests are placed or able to fulfill the request.
     *
     * @default - the Spot Fleet request remains until you cancel it.
     */
    readonly validUntil?: Expiration;
    /**
     * Properties for setting up the Deadline Worker's LogGroup
     * @default - LogGroup will be created with all properties' default values and a prefix of "/renderfarm/".
     */
    readonly logGroupProps?: LogGroupFactoryProps;
    /**
     * An optional provider of user data commands to be injected at various points during the Worker configuration lifecycle.
     * You can provide a subclass of InstanceUserDataProvider with the methods overridden as desired.
     *
     * @default: Not used.
     */
    readonly userDataProvider?: IInstanceUserDataProvider;
    /**
     * Whether the instances in the Spot Fleet should be tracked by Deadline Resource Tracker.
     *
     * In addition to this property, the Spot Event Plugin must also be configured to use the Resource tracker by using the
     * [`enableResourceTracker`](https://docs.aws.amazon.com/rfdk/api/latest/docs/aws-rfdk.deadline.SpotEventPluginSettings.html#enableresourcetracker)
     * property of the `ConfigureSpotEventPlugin` construct, which is `true` by default.
     *
     * @default true
     */
    readonly trackInstancesWithResourceTracker?: boolean;
}
/**
 * Interface for Spot Event Plugin Worker Fleet.
 */
export interface ISpotEventPluginFleet extends IConnectable, IScriptHost, IGrantable {
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the
     * WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that
     * depends on this stack. If this stack depends on the other stack, use allowRemoteControlTo().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *   Adding a SecurityGroup:
     *     `workerFleet.allowRemoteControlFrom(securityGroup)`
     *
     *   Adding a CIDR:
     *     `workerFleet.allowRemoteControlFrom(Peer.ipv4('10.0.0.0/24'))`
     */
    allowRemoteControlFrom(other: IConnectable): void;
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the
     * WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this
     * stack depends on. If the other stack depends on this stack, use allowRemoteControlFrom().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *   Adding a SecurityGroup:
     *     `workerFleet.allowRemoteControlTo(securityGroup)`
     *
     *   Adding a CIDR:
     *     `workerFleet.allowRemoteControlTo(Peer.ipv4('10.0.0.0/24'))`
     */
    allowRemoteControlTo(other: IConnectable): void;
}
/**
 * This construct reperesents a fleet from the Spot Fleet Request created by the Spot Event Plugin.
 * This fleet is intended to be used as input for the {@link @aws-rfdk/deadline#ConfigureSpotEventPlugin} construct.
 *
 * The construct itself doesn't create the Spot Fleet Request, but deploys all the resources
 * required for the Spot Fleet Request and generates the Spot Fleet Configuration setting:
 * a one to one mapping between a Deadline Group and Spot Fleet Request Configurations.
 *
 * ![architecture diagram](/diagrams/deadline/SpotEventPluginFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Instance Role, corresponding IAM Policy and an Instance Profile.
 * - A Fleet Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *   in the fleet.
 * - A security Group if security groups are not provided.
 * - An EC2 Launch Template for the Spot Fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *   that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *   the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *   patching strategy to keep this software current with the latest security patches. Please see
 *   https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 */
export declare class SpotEventPluginFleet extends Construct implements ISpotEventPluginFleet {
    /**
     * Default prefix for a LogGroup if one isn't provided in the props.
     */
    private static readonly DEFAULT_LOG_GROUP_PREFIX;
    /**
     * This is the current maximum for number of workers that can be started on a single host. Currently the
     * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
     * but only the first 8 will have their ports opened in the workers' security group.
     */
    private static readonly MAX_WORKERS_PER_HOST;
    /**
     * The security groups/rules used to allow network connections.
     */
    readonly connections: Connections;
    /**
     * Indicates whether the subnets are the defaults. If `props.vpcSubnets` was passed in, this
     * will be false.
     */
    readonly defaultSubnets: boolean;
    /**
     * The principal to grant permissions to. Granting permissions to this principal will grant
     * those permissions to the spot instance role.
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The port workers listen on to share their logs.
     */
    readonly remoteControlPorts: Port;
    /**
     * Security Groups assigned to this fleet.
     */
    readonly securityGroups: ISecurityGroup[];
    /**
     * The user data that instances use when starting up.
     */
    readonly userData: UserData;
    /**
     * The operating system of the script host.
     */
    readonly osType: OperatingSystemType;
    /**
     * An IAM role associated with the instance profile assigned to its resources.
     */
    readonly fleetInstanceRole: IRole;
    /**
     * The IAM instance profile that fleet instance role is associated to.
     */
    readonly instanceProfile: CfnInstanceProfile;
    /**
     * An IAM role that grants the Spot Fleet the permission to request, launch, terminate, and tag instances on your behalf.
     */
    readonly fleetRole: IRole;
    /**
     * The Worker AMI.
     */
    readonly machineImage: IMachineImage;
    /**
     * The tags to apply during creation of instances and of the Spot Fleet Request.
     */
    readonly tags: TagManager;
    /**
     * Subnets where the instance will be placed within the VPC.
     */
    readonly subnets: SelectedSubnets;
    /**
     * Types of instances to launch.
     */
    readonly instanceTypes: InstanceType[];
    /**
     * Indicates how to allocate the target Spot Instance capacity
     * across the Spot Instance pools specified by the Spot Fleet request.
     */
    readonly allocationStrategy: SpotFleetAllocationStrategy;
    /**
     * The  the maximum capacity that the Spot Fleet can grow to.
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#spot-fleet-requests
     */
    readonly maxCapacity: number;
    /**
     * Deadline groups the workers need to be assigned to.
     *
     * @default - Workers are not assigned to any group
     */
    readonly deadlineGroups: string[];
    /**
     * Deadline pools the workers need to be assigned to.
     *
     * @default - Workers are not assigned to any pool
     */
    readonly deadlinePools?: string[];
    /**
     * Name of SSH keypair to grant access to instances.
     *
     * @default - No SSH access will be possible.
     */
    readonly keyName?: string;
    /**
     * The end date and time of the request.
     * After the end date and time, no new Spot Instance requests are placed or able to fulfill the request.
     *
     * @default - the Spot Fleet request remains until you cancel it.
     */
    readonly validUntil?: Expiration;
    /**
     * The Block devices that will be attached to your workers.
     *
     * @default - The default devices of the provided ami will be used.
     */
    readonly blockDevices?: BlockDevice[];
    /**
     * The Launch Template for this Spot Fleet. This launch template does not specify an instance type or subnet.
     */
    readonly launchTemplate: LaunchTemplate;
    /**
     * @internal
     */
    readonly _launchTemplateConfigs: LaunchTemplateConfig[];
    constructor(scope: Construct, id: string, props: SpotEventPluginFleetProps);
    /**
     * @inheritdoc
     */
    allowRemoteControlFrom(other: IConnectable): void;
    /**
     * @inheritdoc
     */
    allowRemoteControlTo(other: IConnectable): void;
    private createLaunchTemplate;
    private createLaunchTemplateConfigs;
    private validateProps;
    private validateFleetInstanceRole;
    private validateInstanceTypes;
    private validateSubnets;
    private validateGroups;
    private validateRegion;
    private validateBlockDevices;
}
