"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretsManagementIdentityRegistration = void 0;
const path = require("path");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
/**
 * Construct that configures desired Deadline Secrets Management identity registration settings.
 *
 * Resources Deployed
 * ------------------------
 * - IAM policy statements are added to the IAM policy that is attached to the IAM role of the DeploymentInstance.
 *   These statements grant the DeploymentInstance the ability to fetch the Deadline Client installer, get the value of
 *   the AWS Secrets Manager secert containing the Deadline Secrets Management administrator credentials, get the value
 *   of the AWS Secrets Manager secert containing the Deadline Repository's database credentials,
 * - Security group ingress rule to allow the DeploymentInstance to connect to the Repository's database
 * - Security group ingress rule to allow the DeploymentInstance to connect to the Repository's file-system
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The instance deployed by this construct has read/write access to the Deadline Repository (database and
 *   file-system), the AWS Secrets Manager secrets containing credentials for the Database and the Deadline Secrets
 *   Management administrator. Access to the instance permits command and control of the render farm and should be
 *   restricted.
 */
class SecretsManagementIdentityRegistration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.subnetRegistrations = new Map();
        if (!props.repository.secretsManagementSettings.enabled) {
            throw new Error('Secrets management is not enabled on repository');
        }
        /* istanbul ignore next */
        if (!props.repository.secretsManagementSettings.credentials) {
            throw new Error('Repository does not contain secrets management credentials');
        }
        this.adminCredentials = props.repository.secretsManagementSettings.credentials;
        this.deploymentInstance = props.deploymentInstance;
        this.renderQueueSubnets = props.renderQueueSubnets;
        // Download and install the Deadline Client
        this.installDeadlineClient(props);
        // Configure the Deadline Client to direct-connect to the repository
        props.repository.configureClientInstance({
            host: props.deploymentInstance,
            mountPoint: '/mnt/repository',
        });
        // Install python dependencies
        const localScriptFile = this.preparePythonScript(props);
        this.runPythonScript(props, localScriptFile);
        props.deploymentInstance.addExecutionDependency(props.repository);
    }
    addSubnetIdentityRegistrationSetting(addSubnetProps) {
        if (addSubnetProps.role === _1.SecretsManagementRole.ADMINISTRATOR) {
            throw new Error('The Administrator role cannot be set using a Deadline identity registration setting');
        }
        const { vpc, vpcSubnets } = addSubnetProps;
        const selectedSubnets = vpc.selectSubnets(vpcSubnets);
        selectedSubnets.subnets.forEach(subnet => {
            if (this.renderQueueSubnets.subnets.some(rqSubnet => subnet == rqSubnet)) {
                core_1.Annotations.of(addSubnetProps.dependent).addWarning(`Deadline Secrets Management is enabled on the Repository and VPC subnets of the Render Queue match the subnets of ${addSubnetProps.dependent.node.path}. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components`);
            }
            const observedSubnet = this.subnetRegistrations.get(subnet.subnetId);
            if (observedSubnet) {
                if (observedSubnet.registrationStatus !== addSubnetProps.registrationStatus) {
                    throw new Error(`Subnet is already registered with registrationStatus "${observedSubnet.registrationStatus}" but another caller requested "${addSubnetProps.registrationStatus}"`);
                }
                else if (observedSubnet.role !== addSubnetProps.role) {
                    throw new Error(`Subnet is already registered with role "${observedSubnet.role}" but another caller requested "${addSubnetProps.role}"`);
                }
            }
            else {
                this.subnetRegistrations.set(subnet.subnetId, {
                    registrationStatus: addSubnetProps.registrationStatus,
                    role: addSubnetProps.role,
                });
            }
        });
        addSubnetProps.dependent.node.addDependency(this.deploymentInstance);
    }
    runPythonScript(props, localScriptFile) {
        // The command-line arguments to be passed to the script that configures the Deadline identity registration
        // settings
        const scriptArgs = core_1.Lazy.list({
            produce: () => {
                return [].concat([
                    // Region
                    '--region',
                    core_1.Stack.of(this).region,
                    // Admin credentials
                    '--credentials',
                    `"${this.adminCredentials.secretArn}"`,
                ], 
                // Subnets of the load balancer
                (props.renderQueueSubnets
                    .subnetIds
                    .map(subnetID => `--connection-subnet "${subnetID}"`)), 
                // Subnets of RFDK Deadline Client constructs
                (Array.from(this.subnetRegistrations.entries())
                    // Each setting becomes a comma (,) separated string of fields
                    //   <SUBNET_ID>,<ROLE>,<REGISTRATION_STATUS>
                    .map(subnetRegistrationEntry => {
                    const [subnetID, registrationSettingEffect] = subnetRegistrationEntry;
                    return [
                        subnetID,
                        registrationSettingEffect.role.toString(),
                        (registrationSettingEffect.registrationStatus).toString(),
                    ].join(',');
                })
                    // convert into argument key/value pair
                    .map(joinedSubnetArgValue => `--source-subnet "${joinedSubnetArgValue}"`)));
            },
        });
        // We can't use ScriptAsset.executeOn(...) because we need to run as "ec2-user".
        // This is because Repository.configureClientInstance(...) used above will store the credentials
        // in a per-user credential store that is only available to "ec2-user".
        props.deploymentInstance.userData.addCommands(`sudo --login -u ec2-user ${localScriptFile} ` + core_1.Fn.join(' ', scriptArgs));
    }
    preparePythonScript(props) {
        const script = new core_2.ScriptAsset(this, 'ConfigureIdentityRegistrationSettingScript', {
            path: path.join(__dirname, '..', 'scripts', 'python', 'configure_identity_registration_settings.py'),
        });
        // Grant access to ec2:DescribeSubnets. Subnet IPv4 CIDR ranges are not exposed through
        // CloudFormation attributes. Instead, we must query them using the EC2 API on the deployment instance
        props.deploymentInstance.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ['ec2:DescribeSubnets'],
            // ec2:DescribeSubnets does not support resource level permissions. See
            // https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2.html
            resources: ['*'],
        }));
        this.adminCredentials.grantRead(props.deploymentInstance);
        script.grantRead(props.deploymentInstance);
        const localScriptFile = props.deploymentInstance.userData.addS3DownloadCommand({
            bucket: script.bucket,
            bucketKey: script.s3ObjectKey,
            localFile: '/home/ec2-user/configure_identity_registration_settings.py',
        });
        props.deploymentInstance.userData.addCommands(`chmod +x ${localScriptFile}`, `chown ec2-user:ec2-user ${localScriptFile}`);
        return localScriptFile;
    }
    installDeadlineClient(props) {
        props.version.linuxInstallers.client.s3Bucket.grantRead(props.deploymentInstance, props.version.linuxInstallers.client.objectKey);
        const clientInstallerPath = props.deploymentInstance.userData.addS3DownloadCommand({
            bucket: props.version.linuxInstallers.client.s3Bucket,
            bucketKey: props.version.linuxInstallers.client.objectKey,
        });
        props.deploymentInstance.userData.addCommands('set -x');
        props.deploymentInstance.userData.addCommands(`chmod +x "${clientInstallerPath}"`);
        props.deploymentInstance.userData.addCommands([
            // This is required b/c USER and HOME environment variables are not defined when running
            // user-data
            'sudo', '--login',
            // Run the Deadline Client installer
            `"${clientInstallerPath}"`,
            '--mode', 'unattended',
            '--connectiontype', 'Remote',
            '--proxyrootdir', '127.0.0.1:8080',
            '--noguimode', 'true',
            '--slavestartup', 'false',
            '--launcherdaemon', 'false',
            '--restartstalled', 'true',
            '--autoupdateoverride', 'False',
        ].join(' '));
    }
}
exports.SecretsManagementIdentityRegistration = SecretsManagementIdentityRegistration;
//# sourceMappingURL=data:application/json;base64,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