"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueueConnection = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
/**
 * Helper class for connecting Clients to a render queue
 */
class RenderQueueConnection {
    /**
     * Create a RQ Connection for http traffic
     */
    static forHttp(options) {
        return new HTTPConnection(options);
    }
    /**
     * Create a RQ Connection for https traffic
     */
    static forHttps(options) {
        return new HTTPSConnection(options);
    }
    /**
     * Fetch the instance configuration python script
     * @param stack The stack that the asset should be created in
     */
    connectionAssetSingleton(stack) {
        const uuid = '3be2203f-fea1-4143-bc09-7ef079b4299c';
        const uniqueId = 'RenderQueueConnectionAsset' + uuid.replace(/[-]/g, '');
        return stack.node.tryFindChild(uniqueId) ?? new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'python', 'client-rq-connection.py'),
        });
    }
    /**
     * Executes connection asset singleton wth a given list of arguments
     */
    ExecuteConnectionAsset(host, args) {
        const hostStack = core_1.Stack.of(host);
        const connectionAsset = this.connectionAssetSingleton(hostStack);
        connectionAsset.grantRead(host);
        const configureScriptPath = host.userData.addS3DownloadCommand({
            bucket: connectionAsset.bucket,
            bucketKey: connectionAsset.s3ObjectKey,
        });
        const dlExtraCommands = [];
        if (args.tlsCaArn) {
            dlExtraCommands.push('--tls-ca', `"${args.tlsCaArn}"`);
        }
        if (host.osType === aws_ec2_1.OperatingSystemType.LINUX) {
            host.userData.addCommands('if [ -f "/etc/profile.d/deadlineclient.sh" ]; then', '  source "/etc/profile.d/deadlineclient.sh"', 'fi', `"\${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "${configureScriptPath}" --render-queue "${args.address}" ${dlExtraCommands.join(' ')}`, 
            // Cleanup
            `rm -f "${configureScriptPath}"`);
            if (args.restartLauncher ?? true) {
                host.userData.addCommands('if service --status-all | grep -q "Deadline 10 Launcher"; then', '  service deadline10launcher restart', 'fi');
            }
        }
        else if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            host.userData.addCommands('$ErrorActionPreference = "Stop"', '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value', `& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "${configureScriptPath}" --render-queue "${args.address}" ${dlExtraCommands.join(' ')} 2>&1`, `Remove-Item -Path "${configureScriptPath}"`);
            if (args.restartLauncher ?? true) {
                host.userData.addCommands('If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {', '  Restart-Service "deadline10launcherservice"', '} Else {', '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1', '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1', '}');
            }
        }
    }
}
exports.RenderQueueConnection = RenderQueueConnection;
/**
 * Specialization of {@link RenderQueueConnection} for HTTP Connections
 */
class HTTPConnection extends RenderQueueConnection {
    constructor(config) {
        super();
        this.config = config;
    }
    configureClientECS(params) {
        params.hosts.forEach(host => {
            host.connections.allowToDefaultPort(this.config.endpoint);
        });
        return {
            RENDER_QUEUE_URI: `http://${this.config.endpoint.socketAddress}`,
        };
    }
    configureClientInstance(params) {
        params.host.connections.allowToDefaultPort(this.config.endpoint);
        this.ExecuteConnectionAsset(params.host, {
            address: `http://${this.config.endpoint.socketAddress}`,
            restartLauncher: params.restartLauncher,
        });
    }
}
/**
 * Specialization of {@link RenderQueueConnection} for HTTPS Connections
 */
class HTTPSConnection extends RenderQueueConnection {
    constructor(config) {
        super();
        this.config = config;
    }
    configureClientECS(params) {
        params.hosts.forEach(host => {
            host.connections.allowToDefaultPort(this.config.endpoint);
        });
        this.config.caCert.grantRead(params.grantee);
        return {
            RENDER_QUEUE_URI: `https://${this.config.endpoint.socketAddress}`,
            RENDER_QUEUE_TLS_CA_CERT_URI: this.config.caCert.secretArn,
        };
    }
    configureClientInstance(params) {
        params.host.connections.allowToDefaultPort(this.config.endpoint);
        this.config.caCert.grantRead(params.host);
        this.ExecuteConnectionAsset(params.host, {
            address: `https://${this.config.endpoint.socketAddress}`,
            tlsCaArn: this.config.caCert.secretArn,
            restartLauncher: params.restartLauncher,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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