"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
const version_1 = require("./version");
const version_query_1 = require("./version-query");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system
 * that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * ![architecture diagram](/diagrams/deadline/Repository.svg)
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon EFS Point - If no filesystem is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 * - An RFDK PadEfsStorage - If no filesystem is provided.
 * - An AWS Secrets Manager Secret - If no Secret with admin credentials for Deadline Secrets Management is provided.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *   system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *   Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *   Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *   require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *   submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *   that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *   to only those who require it.
 * - If no file-system is provided to the Repository, then the Repository creates an EFS access point with unrestricted
 *   access to the entire EFS file-system. If you would like a single EFS file-system that is used by the Deadline
 *   Repository and other agents, you should supply the file-system and a access-restricted EFS access point to the
 *   Repository construct instead.
 */
class Repository extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.database && props.backupOptions?.databaseRetention) {
            core_1.Annotations.of(this).addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && props.removalPolicy?.filesystem) {
            core_1.Annotations.of(this).addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && props.removalPolicy?.database) {
            core_1.Annotations.of(this).addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem instanceof core_2.MountableEfs && !props.fileSystem.accessPoint) {
            throw new Error('When using EFS with the Repository, you must provide an EFS Access Point');
        }
        if ((props.secretsManagementSettings?.enabled ?? true) && props.database && !props.database.databaseConstruct) {
            throw new Error('Admin credentials for Deadline Secrets Management cannot be generated when using an imported database. For setting up your own credentials, please refer to https://github.com/aws/aws-rfdk/tree/mainline/packages/aws-rfdk/lib/deadline#configuring-deadline-secrets-management-on-the-repository.');
        }
        this.version = props.version;
        const meetsMinSecretsVersion = !this.version.isLessThan(version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION);
        const secretsManagementIsEnabled = props.secretsManagementSettings?.enabled ?? meetsMinSecretsVersion;
        if (secretsManagementIsEnabled && !meetsMinSecretsVersion) {
            throw new Error(`The supplied Deadline version (${props.version.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
        }
        this.secretsManagementSettings = {
            enabled: secretsManagementIsEnabled,
            credentials: props.secretsManagementSettings?.credentials ??
                (secretsManagementIsEnabled ? new aws_secretsmanager_1.Secret(props.database?.databaseConstruct ? core_1.Stack.of(props.database?.databaseConstruct) : this, 'SMAdminUser', {
                    description: 'Admin credentials for Deadline Secrets Management',
                    generateSecretString: {
                        excludeCharacters: '\"$&\'()/<>[\\]\`{|}',
                        includeSpace: false,
                        passwordLength: 24,
                        requireEachIncludedType: true,
                        generateStringKey: 'password',
                        secretStringTemplate: JSON.stringify({ username: Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME }),
                    },
                    removalPolicy: props.secretsManagementSettings?.credentialsRemovalPolicy ?? core_1.RemovalPolicy.RETAIN,
                }) : undefined),
        };
        this.fileSystem = props.fileSystem ?? (() => {
            const fs = new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: props.removalPolicy?.filesystem ?? core_1.RemovalPolicy.RETAIN,
                securityGroup: props.securityGroupsOptions?.fileSystem,
            });
            const paddingAccess = fs.addAccessPoint('PaddingAccessPoint', {
                createAcl: {
                    ownerGid: '0',
                    ownerUid: '0',
                    permissions: '744',
                },
                path: '/RFDK_PaddingFiles',
            });
            new core_2.PadEfsStorage(this, 'PadEfsStorage', {
                vpc: props.vpc,
                vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
                accessPoint: paddingAccess,
                desiredPadding: core_1.Size.gibibytes(40),
            });
            const accessPoint = fs.addAccessPoint('AccessPoint', {
                posixUser: {
                    uid: '0',
                    gid: '0',
                },
            });
            return new core_2.MountableEfs(this, {
                filesystem: fs,
                accessPoint,
            });
        })();
        // Set up the Database of the repository
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
                    'Please ensure that the Database provided is configured correctly.';
                core_1.Annotations.of(this).addWarning(warningMsg);
            }
        }
        else {
            const databaseAuditLogging = props.databaseAuditLogging ?? true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = props.documentDbInstanceCount ?? Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                engineVersion: '3.6.0',
                instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                vpc: props.vpc,
                vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT, onePerAz: true },
                securityGroup: props.securityGroupsOptions?.database,
                instances,
                backup: {
                    retention: props.backupOptions?.databaseRetention ?? Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: props.removalPolicy?.database ?? core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets ?? {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            minCapacity: 1,
            maxCapacity: 1,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.replacingUpdate(),
            signals: aws_autoscaling_1.Signals.waitForAll({
                timeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            }),
            securityGroup: props.securityGroupsOptions?.installer,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version, props.repositorySettings, 
        // Change ownership of the Deadline repository files if-and-only-if the mounted file-system
        // uses the POSIX permissions based on the process' user UID/GID
        this.fileSystem.usesUserPosixPermissions() ? Repository.REPOSITORY_OWNER : undefined);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof version_query_1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * @inheritdoc
     */
    configureClientECS(props) {
        const hostMountPoint = props.containerInstances.filesystemMountPoint ?? '/mnt/repo';
        const majorVersion = core_1.Token.isUnresolved(this.version.majorVersion) ?
            core_1.Token.asString(this.version.majorVersion) : this.version.majorVersion.toString();
        const containerMountPoint = props.containers.filesystemMountPoint ?? `/opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Note: pathToFileURL messes up CDK Tokens like the one in majorVersion
        const containerMountPointURL = props.containers.filesystemMountPoint ?
            url_1.pathToFileURL(props.containers.filesystemMountPoint).toString() :
            `file:///opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: containerMountPointURL,
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this.installerGroup);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * @inheritdoc
     */
    configureClientInstance(props) {
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this.installerGroup);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = stack.node.tryFindChild(uniqueId) ?? core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        const prefix = logGroupProps?.logGroupPrefix ?? Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = core_1.Names.uniqueId(this);
        core_1.Tags.of(this.installerGroup).add(tagKey, core_1.Names.uniqueId(this));
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)');
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version, settings, owner) {
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        const installerArgs = [
            '-i', `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
            '-p', `"${installPath}"`,
            '-v', version.linuxFullVersionString(),
        ];
        if (this.secretsManagementSettings.enabled) {
            installerArgs.push('-r', core_1.Stack.of(this.secretsManagementSettings.credentials ?? this).region);
            this.secretsManagementSettings.credentials.grantRead(installerGroup);
            installerArgs.push('-c', this.secretsManagementSettings.credentials.secretArn ?? '');
        }
        if (settings) {
            const repositorySettingsFilePath = installerGroup.userData.addS3DownloadCommand({
                bucket: settings.bucket,
                bucketKey: settings.s3ObjectKey,
            });
            installerArgs.push('-s', repositorySettingsFilePath);
        }
        // We can ignore this in test coverage because we always use Repository.REPOSITORY_OWNER
        /* istanbul ignore next */
        if (owner) {
            installerArgs.push('-o', `${owner.uid}:${owner.gid}`);
        }
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: installerArgs,
        });
    }
}
exports.Repository = Repository;
_a = JSII_RTTI_SYMBOL_1;
Repository[_a] = { fqn: "aws-rfdk.deadline.Repository", version: "0.42.0" };
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
/**
 * The Repository owner is 1000:1000 (ec2-user on AL2).
 */
Repository.REPOSITORY_OWNER = { uid: 1000, gid: 1000 };
/**
 * Default username for auto generated admin credentials in Secret Manager.
 */
Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME = 'admin';
//# sourceMappingURL=data:application/json;base64,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