"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path_1 = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
const deployment_instance_1 = require("../../core/lib/deployment-instance");
const runtime_info_1 = require("../../core/lib/runtime-info");
const rq_connection_1 = require("./rq-connection");
const secrets_management_1 = require("./secrets-management");
const version_1 = require("./version");
const wait_for_stable_service_1 = require("./wait-for-stable-service");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends core_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API
 * to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * ![architecture diagram](/diagrams/deadline/RenderQueue.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Amazon Elastic Container Service (ECS) cluster.
 * - An AWS EC2 auto-scaling group that provides the instances that host the ECS service.
 * - An ECS service with a task definition that deploys the Deadline Remote Connetion Server (RCS) in a container.
 * - A Amazon CloudWatch log group for streaming logs from the Deadline RCS.
 * - An application load balancer, listener and target group that balance incoming traffic among the RCS containers.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - Care must be taken to secure what can connect to the RenderQueue. The RenderQueue does not authenticate API
 *   requests made against it. You must limit access to the RenderQueue endpoint to only trusted hosts. Those hosts
 *   should be governed carefully, as malicious software could use the API to remotely execute code across the entire render farm.
 * - The RenderQueue can be deployed with network encryption through Transport Layer Security (TLS) or without it. Unencrypted
 *   network communications can be eavesdropped upon or modified in transit. We strongly recommend deploying the RenderQueue
 *   with TLS enabled in production environments and it is configured to be on by default.
 */
class RenderQueue extends RenderQueueBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        /**
         * Whether SEP policies have been added
         */
        this.haveAddedSEPPolicies = false;
        /**
         * Whether Resource Tracker policies have been added
         */
        this.haveAddedResourceTrackerPolicies = false;
        this.repository = props.repository;
        this.renderQueueSize = props?.renderQueueSize ?? { min: 1, max: 1 };
        if (props.version.isLessThan(RenderQueue.MINIMUM_LOAD_BALANCING_VERSION)) {
            // Deadline versions earlier than 10.1.10 do not support horizontal scaling behind a load-balancer, so we limit to at most one instance
            if ((this.renderQueueSize.min ?? 0) > 1) {
                throw new Error(`renderQueueSize.min for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.min}`);
            }
            if ((this.renderQueueSize.desired ?? 0) > 1) {
                throw new Error(`renderQueueSize.desired for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.desired}`);
            }
            if ((this.renderQueueSize.max ?? 0) > 1) {
                throw new Error(`renderQueueSize.max for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.max}`);
            }
        }
        this.version = props?.version;
        const externalProtocol = props.trafficEncryption?.externalTLS?.enabled === false ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        let loadBalancerFQDN;
        let domainZone;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            const tlsInfo = this.getOrCreateTlsInfo(props);
            this.certChain = tlsInfo.certChain;
            this.clientCert = tlsInfo.serverCert;
            loadBalancerFQDN = tlsInfo.fullyQualifiedDomainName;
            domainZone = tlsInfo.domainZone;
        }
        else {
            if (props.hostname) {
                loadBalancerFQDN = this.generateFullyQualifiedDomainName(props.hostname.zone, props.hostname.hostname);
                domainZone = props.hostname.zone;
            }
        }
        this.version = props.version;
        const internalProtocol = props.trafficEncryption?.internalProtocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = props.renderQueueSize?.min ?? 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        const maxCapacity = this.renderQueueSize.max ?? this.renderQueueSize?.desired;
        if (this.renderQueueSize?.desired && maxCapacity && this.renderQueueSize?.desired > maxCapacity) {
            throw new Error(`renderQueueSize.desired capacity cannot be more than ${maxCapacity}: got ${this.renderQueueSize.desired}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: props.vpcSubnets ?? RenderQueue.DEFAULT_VPC_SUBNETS_OTHER,
            instanceType: props.instanceType ?? new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: this.renderQueueSize?.desired,
            maxCapacity,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            updateType: undefined,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.rollingUpdate(),
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: props.securityGroups?.backend,
        });
        this.backendConnections = this.asg.connections;
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        if (props.enableLocalFileCaching ?? false) {
            // Has to be done before any filesystems mount.
            this.enableFilecaching(this.asg);
        }
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        if (props.repository.secretsManagementSettings.enabled) {
            const errors = [];
            if (props.version.isLessThan(version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION)) {
                errors.push(`The supplied Deadline version (${props.version.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
            }
            if (props.repository.secretsManagementSettings.credentials === undefined) {
                errors.push('The Repository does not have Secrets Management credentials');
            }
            if (internalProtocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                errors.push('The internal protocol on the Render Queue is not HTTPS.');
            }
            if (externalProtocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                errors.push('External TLS on the Render Queue is not enabled.');
            }
            if (errors.length > 0) {
                throw new Error(`Deadline Secrets Management is enabled on the supplied Repository but cannot be enabled on the Render Queue for the following reasons:\n${errors.join('\n')}`);
            }
        }
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
            runAsUser: RenderQueue.RCS_USER,
            secretsManagementOptions: props.repository.secretsManagementSettings.enabled ? {
                credentials: props.repository.secretsManagementSettings.credentials,
                posixUsername: RenderQueue.RCS_USER.username,
            } : undefined,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            vpcSubnets: props.vpcSubnetsAlb ?? RenderQueue.DEFAULT_VPC_SUBNETS_ALB,
            internetFacing: false,
            deletionProtection: props.deletionProtection ?? true,
            securityGroup: props.securityGroups?.frontend,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: this.renderQueueSize?.desired,
            domainZone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_2.ConnectableApplicationEndpoint({
            address: loadBalancerFQDN ?? this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        props.repository.secretsManagementSettings.credentials?.grantRead(this);
        this.ecsServiceStabilized = new wait_for_stable_service_1.WaitForStableService(this, 'WaitForStableService', {
            service: this.pattern.service,
        });
        this.node.defaultChild = taskDefinition;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof _1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * @inheritdoc
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * @inheritdoc
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * @inheritdoc
     */
    configureSecretsManagementAutoRegistration(props) {
        if (!this.repository.secretsManagementSettings.enabled) {
            // Secrets management is not enabled, so do nothing
            return;
        }
        this.identityRegistrationSettings.addSubnetIdentityRegistrationSetting(props);
    }
    /**
     * Adds AWS Managed Policies to the Render Queue so it is able to control Deadline's Spot Event Plugin.
     *
     * See: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html for additonal information.
     *
     * @param includeResourceTracker Whether or not the Resource tracker admin policy should also be added (Default: True)
     */
    addSEPPolicies(includeResourceTracker = true) {
        if (!this.haveAddedSEPPolicies) {
            const sepPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy');
            this.taskDefinition.taskRole.addManagedPolicy(sepPolicy);
            this.haveAddedSEPPolicies = true;
        }
        if (!this.haveAddedResourceTrackerPolicies) {
            if (includeResourceTracker) {
                const rtPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy');
                this.taskDefinition.taskRole.addManagedPolicy(rtPolicy);
                this.haveAddedResourceTrackerPolicies = true;
            }
        }
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
        child.node.addDependency(this.pattern.service);
        child.node.addDependency(this.ecsServiceStabilized);
    }
    /**
     * Adds security groups to the frontend of the Render Queue, which is its load balancer.
     * @param securityGroups The security groups to add.
     */
    addFrontendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.loadBalancer.addSecurityGroup(securityGroup));
    }
    /**
     * Adds security groups to the backend components of the Render Queue, which consists of the AutoScalingGroup for the Deadline RCS.
     * @param securityGroups The security groups to add.
     */
    addBackendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.asg.addSecurityGroup(securityGroup));
    }
    enableFilecaching(asg) {
        const script = core_2.ScriptAsset.fromPathConvention(this, 'FilecachingScript', {
            osType: asg.osType,
            baseName: 'enableCacheFilesd',
            rootDir: path_1.join(__dirname, '..', 'scripts'),
        });
        // A comment in userData to make this easier to test.
        asg.userData.addCommands('# RenderQueue file caching enabled');
        script.executeOn({
            host: asg,
        });
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_2.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_2.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        if (props.secretsManagementOptions !== undefined) {
            environment.RCS_SM_CREDENTIALS_URI = props.secretsManagementOptions.credentials.secretArn;
        }
        // We can ignore this in test coverage because we always use RenderQueue.RCS_USER
        /* istanbul ignore next */
        const user = props.runAsUser ? `${props.runAsUser.uid}:${props.runAsUser.gid}` : undefined;
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
            user,
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        if (props.secretsManagementOptions !== undefined) {
            // Create volume to persist the RSA keypairs generated by Deadline between ECS tasks
            // This makes it so subsequent ECS tasks use the same initial Secrets Management identity
            const volumeName = 'deadline-user-keypairs';
            taskDefinition.addVolume({
                name: volumeName,
                dockerVolumeConfiguration: {
                    scope: aws_ecs_1.Scope.SHARED,
                    autoprovision: true,
                    driver: 'local',
                },
            });
            // Mount the volume into the container at the location where Deadline expects it
            containerDefinition.addMountPoints({
                readOnly: false,
                sourceVolume: volumeName,
                containerPath: `/home/${props.secretsManagementOptions.posixUsername}/.config/.mono/keypairs`,
            });
        }
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
    /**
     * Checks if the user supplied any certificate to use for TLS and uses them, or creates defaults to use.
     * @param props
     * @returns TlsInfo either based on input to the render queue, or the created defaults
     */
    getOrCreateTlsInfo(props) {
        if ((props.trafficEncryption?.externalTLS?.acmCertificate !== undefined) ||
            (props.trafficEncryption?.externalTLS?.rfdkCertificate !== undefined)) {
            if (props.hostname === undefined) {
                throw new Error('The hostname for the render queue must be defined if supplying your own certificates.');
            }
            return this.getTlsInfoFromUserProps(props.trafficEncryption.externalTLS, props.hostname);
        }
        return this.createDefaultTlsInfo(props.vpc, props.hostname);
    }
    /**
     * Creates a default certificate to use for TLS and a PrivateHostedZone to put the load balancer in.
     * @param vpc
     * @param hostname
     * @returns default TlsInfo
     */
    createDefaultTlsInfo(vpc, hostname) {
        const domainZone = hostname?.zone ?? new aws_route53_1.PrivateHostedZone(this, 'DnsZone', {
            vpc: vpc,
            zoneName: RenderQueue.DEFAULT_DOMAIN_NAME,
        });
        const fullyQualifiedDomainName = this.generateFullyQualifiedDomainName(domainZone, hostname?.hostname);
        const rootCa = new core_2.X509CertificatePem(this, 'RootCA', {
            subject: {
                cn: 'RenderQueueRootCA',
            },
        });
        const rfdkCert = new core_2.X509CertificatePem(this, 'RenderQueuePemCert', {
            subject: {
                cn: fullyQualifiedDomainName,
            },
            signingCertificate: rootCa,
        });
        const serverCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', rfdkCert);
        const certChain = rfdkCert.certChain;
        return {
            domainZone,
            fullyQualifiedDomainName,
            serverCert,
            certChain,
        };
    }
    /**
     * Gets the certificate and PrivateHostedZone provided in the Render Queue's construct props.
     * @param externalTLS
     * @param hostname
     * @returns The provided certificate and domain info
     */
    getTlsInfoFromUserProps(externalTLS, hostname) {
        let serverCert;
        let certChain;
        if ((externalTLS.acmCertificate !== undefined) &&
            (externalTLS.rfdkCertificate !== undefined)) {
            throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
        }
        if (!hostname.hostname) {
            throw new Error('A hostname must be supplied if a certificate is supplied, '
                + 'with the common name of the certificate matching the hostname + domain name.');
        }
        const fullyQualifiedDomainName = this.generateFullyQualifiedDomainName(hostname.zone, hostname.hostname);
        if (externalTLS.acmCertificate) {
            if (externalTLS.acmCertificateChain === undefined) {
                throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
            }
            serverCert = externalTLS.acmCertificate;
            certChain = externalTLS.acmCertificateChain;
        }
        else { // Using externalTLS.rfdkCertificate
            if (externalTLS.rfdkCertificate.certChain === undefined) {
                throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
            }
            serverCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', externalTLS.rfdkCertificate);
            certChain = externalTLS.rfdkCertificate.certChain;
        }
        return {
            domainZone: hostname.zone,
            fullyQualifiedDomainName,
            serverCert,
            certChain,
        };
    }
    /**
     * Helper method to create the fully qualified domain name for the given hostname and PrivateHostedZone.
     * @param hostname
     * @param zone
     * @returns The fully qualified domain name
     */
    generateFullyQualifiedDomainName(zone, hostname = RenderQueue.DEFAULT_HOSTNAME) {
        if (!RenderQueue.RE_VALID_HOSTNAME.test(hostname)) {
            throw new Error(`Invalid RenderQueue hostname: ${hostname}`);
        }
        return `${hostname}.${zone.zoneName}`;
    }
    /**
     * The instance that runs commands during the deployment.
     */
    get deploymentInstance() {
        const CONFIGURE_REPOSITORY_CONSTRUCT_ID = 'ConfigureRepository';
        const deploymentInstanceNode = this.node.tryFindChild(CONFIGURE_REPOSITORY_CONSTRUCT_ID);
        if (deploymentInstanceNode === undefined) {
            return new deployment_instance_1.DeploymentInstance(this, CONFIGURE_REPOSITORY_CONSTRUCT_ID, {
                securityGroup: this.backendConnections.securityGroups[0],
                vpc: this.props.vpc,
                vpcSubnets: this.props.vpcSubnets ?? RenderQueue.DEFAULT_VPC_SUBNETS_OTHER,
            });
        }
        else if (deploymentInstanceNode instanceof deployment_instance_1.DeploymentInstance) {
            return deploymentInstanceNode;
        }
        else {
            throw new Error(`Unexpected type for ${deploymentInstanceNode.node.path}. Expected ${deployment_instance_1.DeploymentInstance.name}, but found ${typeof (deploymentInstanceNode)}.`);
        }
    }
    /**
     * The construct that manages Deadline Secrets Management identity registration settings
     */
    get identityRegistrationSettings() {
        const IDENTITY_REGISTRATION_CONSTRUCT_ID = 'SecretsManagementIdentityRegistration';
        const secretsManagementIdentityRegistration = this.node.tryFindChild(IDENTITY_REGISTRATION_CONSTRUCT_ID);
        if (!secretsManagementIdentityRegistration) {
            return new secrets_management_1.SecretsManagementIdentityRegistration(this, IDENTITY_REGISTRATION_CONSTRUCT_ID, {
                deploymentInstance: this.deploymentInstance,
                repository: this.repository,
                renderQueueSubnets: this.props.vpc.selectSubnets(this.props.vpcSubnetsAlb ?? RenderQueue.DEFAULT_VPC_SUBNETS_ALB),
                version: this.props.version,
            });
        }
        else if (secretsManagementIdentityRegistration instanceof secrets_management_1.SecretsManagementIdentityRegistration) {
            return secretsManagementIdentityRegistration;
        }
        else {
            throw new Error(`Unexpected type for ${secretsManagementIdentityRegistration.node.path}. Expected ${secrets_management_1.SecretsManagementIdentityRegistration.name}, but found ${typeof (secretsManagementIdentityRegistration)}.`);
        }
    }
}
exports.RenderQueue = RenderQueue;
_a = JSII_RTTI_SYMBOL_1;
RenderQueue[_a] = { fqn: "aws-rfdk.deadline.RenderQueue", version: "0.42.0" };
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
RenderQueue.DEFAULT_HOSTNAME = 'renderqueue';
RenderQueue.DEFAULT_DOMAIN_NAME = 'aws-rfdk.com';
RenderQueue.DEFAULT_VPC_SUBNETS_ALB = { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT, onePerAz: true };
RenderQueue.DEFAULT_VPC_SUBNETS_OTHER = { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT };
/**
* The minimum Deadline version required for the Remote Connection Server to support load-balancing
*/
RenderQueue.MINIMUM_LOAD_BALANCING_VERSION = new version_1.Version([10, 1, 10, 0]);
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
/**
 * Information for the RCS user.
 */
RenderQueue.RCS_USER = { uid: 1000, gid: 1000, username: 'ec2-user' };
//# sourceMappingURL=data:application/json;base64,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