"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseConnection = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
/**
 * Helper class for connecting Thinkbox's Deadline to a specific Database.
 */
class DatabaseConnection {
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to Amazon DocumentDB.
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     */
    static forDocDB(options) {
        return new DocDBDatabaseConnection(options);
    }
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to MongoDB.
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     */
    static forMongoDbInstance(options) {
        return new MongoDbInstanceDatabaseConnection(options);
    }
}
exports.DatabaseConnection = DatabaseConnection;
_a = JSII_RTTI_SYMBOL_1;
DatabaseConnection[_a] = { fqn: "aws-rfdk.deadline.DatabaseConnection", version: "0.42.0" };
/**
 * Specialization of {@link DatabaseConnection} targetting Amazon DocumentDB.
 */
class DocDBDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        if (!this.isCompatibleDocDBVersion()) {
            core_1.Annotations.of(props.database).addError('engineVersion must be 3.6.0 to be compatible with Deadline');
        }
        this.containerEnvironment = {
            // The container must fetch the credentials from Secrets Manager
            DB_CREDENTIALS_URI: this.props.login.secretArn,
        };
        if (props.database instanceof aws_docdb_1.DatabaseCluster) {
            this.databaseConstruct = props.database;
        }
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', `INSTALLER_DB_ARGS=( ["--dbuser"]=$DB_USERNAME ["--dbpassword"]=$DB_PASSWORD ["--dbhost"]=${this.props.database.clusterEndpoint.hostname}` +
            ` ["--dbport"]=${this.props.database.clusterEndpoint.portAsString()} ["--dbtype"]=DocumentDB )`, 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', 'sudo -u ec2-user "${deadlinecommand}" -StoreDatabasecredentials "${DB_USERNAME}" "${DB_PASSWORD}"', 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, this.props.database.connections.defaultPort);
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.login.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        // To depend on document DB it is not sufficient to depend on the Cluster. The instances are what serves data, so
        // we must add a dependency to an instance in the DocDB cluster.
        // The DocDB L2 does not expose any of its instances as properties, so we have to escape-hatch to gain access.
        const docdbInstance = this.props.database.node.tryFindChild('Instance1');
        // We won't find an instance in two situations:
        //  1) The DocDB Cluster was created from attributes. In this case, the DocDB pre-exists the stack and there's no need
        //     to depend on anything.
        //  2) The DocDB Cluster was constructed, but the internal name for the instance has been changed from 'Instance1'; this is
        //     unlikely, but not impossible.
        // We can differentiate cases (1) & (2) by looking for the defaultChild on the cluster. The version from attributes will not have one.
        if (docdbInstance) {
            child.node.addDependency(docdbInstance);
        }
        else if (this.props.database.node.defaultChild) {
            throw new Error('The internal implementation of the AWS CDK\'s DocumentDB cluster construct may have changed. Please update to a newer RFDK for an updated implementation, or file a ticket if this is the latest release.');
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        let added = false;
        const errorReasons = [];
        if (this.props.database instanceof aws_docdb_1.DatabaseCluster) {
            const resource = this.props.database.node.tryFindChild('Resource');
            // TODO: Replace this code with the addSecurityGroup method of DatabaseCluster once this PR is merged:
            // https://github.com/aws/aws-cdk/pull/13290
            if (resource instanceof aws_docdb_1.CfnDBCluster) {
                const cfnCluster = resource;
                const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
                if (cfnCluster.vpcSecurityGroupIds === undefined) {
                    cfnCluster.vpcSecurityGroupIds = securityGroupIds;
                }
                else {
                    cfnCluster.vpcSecurityGroupIds.push(...securityGroupIds);
                }
                added = true;
            }
            else {
                errorReasons.push('The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed.');
            }
        }
        else {
            errorReasons.push('The "database" property passed to this class is not an instance of AWS CDK\'s DocumentDB cluster construct.');
        }
        if (!added) {
            core_1.Annotations.of(this.props.database).addWarning(`Failed to add the following security groups to ${this.props.database.node.id}: ${securityGroups.map(sg => sg.node.id).join(', ')}. ` +
                errorReasons.join(' '));
        }
    }
    /**
     * Deadline is only compatible with MongoDB 3.6. This function attempts to determine whether
     * the given DocDB version is compatible.
     */
    isCompatibleDocDBVersion() {
        // The defaultChild of a DocDB DatabaseCluster is a CfnDBCluster, but we only have this
        // child if the customer didn't import from attributes. We can check the DB version by
        // checking the value of the engineVersion property of that object.
        if (this.props.database.node.defaultChild) {
            const cluster = this.props.database.node.defaultChild;
            return cluster.engineVersion?.startsWith('3.6') ?? false;
        }
        return true; // No information, assume it's compatible.
    }
}
/**
 * Specialization of {@link DatabaseConnection} targetting MongoDB.
 */
class MongoDbInstanceDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        this.containerEnvironment = {
            DB_TLS_CLIENT_CERT_URI: props.clientCertificate.cert.secretArn,
            DB_TLS_CLIENT_CERT_PASSWORD_URI: props.clientCertificate.passphrase.secretArn,
        };
        if (props.database instanceof core_2.MongoDbInstance) {
            if (props.database.mongoDataVolume instanceof aws_ec2_1.Volume) {
                this.databaseConstruct = props.database.mongoDataVolume;
            }
        }
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, aws_ec2_1.Port.tcp(this.props.database.port));
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 
        // Suppress -x, so no secrets go to the logs
        'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'INSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true ' +
            `["--dbhost"]="${this.props.database.fullHostname}" ["--dbport"]=${this.props.database.port} ` +
            `["--dbclientcert"]="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}" ["--dbcertpass"]=$CERT_PASSWORD )`, 'unset CERT_PASSWORD', 
        // Restore -x, if it was set.
        'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export DB_CERT_FILE="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}"`, `export DB_CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.clientCertificate.cert.grantRead(grantee);
        this.props.clientCertificate.passphrase.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        if (this.props.database.hasOwnProperty('server')) {
            const db = this.props.database;
            child.node.addDependency(db.server.autoscalingGroup.node.defaultChild);
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        this.props.database.addSecurityGroup(...securityGroups);
    }
    /**
     * Download the client PKCS#12 certificate for authenticating to the MongoDB, and place it into
     * the path defined by: DB_CERT_LOCATION
     * @param host
     */
    downloadCertificate(host) {
        const stack = core_1.Stack.of(host);
        const uuid = 'e8125dd2-ab2c-4861-8ee4-998c26b30ee0';
        const uniqueId = 'GetSecretToFile' + host.osType + uuid.replace(/[-]/g, '');
        const getSecretsScript = stack.node.tryFindChild(uniqueId) ??
            core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
                osType: host.osType,
                baseName: 'getSecretToFile',
                rootDir: path.join(__dirname, '..', 'scripts'),
            });
        getSecretsScript.executeOn({
            host,
            args: [
                this.props.clientCertificate.cert.secretArn,
                MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION,
            ],
        });
    }
}
MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION = '/opt/Thinkbox/certs/mongo_client.pfx';
//# sourceMappingURL=data:application/json;base64,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