"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigureSpotEventPlugin = exports.SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginPreJobTaskMode = exports.SpotEventPluginLoggingLevel = exports.SpotEventPluginState = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const render_queue_1 = require("./render-queue");
const secrets_management_ref_1 = require("./secrets-management-ref");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const version_1 = require("./version");
/**
 * How the event plug-in should respond to events.
 */
var SpotEventPluginState;
(function (SpotEventPluginState) {
    /**
     * The Render Queue, all jobs and Workers will trigger the events for this plugin.
     */
    SpotEventPluginState["GLOBAL_ENABLED"] = "Global Enabled";
    /**
     * No events are triggered for the plugin.
     */
    SpotEventPluginState["DISABLED"] = "Disabled";
})(SpotEventPluginState = exports.SpotEventPluginState || (exports.SpotEventPluginState = {}));
/**
 * Logging verbosity levels for the Spot Event Plugin.
 */
var SpotEventPluginLoggingLevel;
(function (SpotEventPluginLoggingLevel) {
    /**
     * Standard logging level.
     */
    SpotEventPluginLoggingLevel["STANDARD"] = "Standard";
    /**
     * Detailed logging about the inner workings of the Spot Event Plugin.
     */
    SpotEventPluginLoggingLevel["VERBOSE"] = "Verbose";
    /**
     * All Verbose logs plus additional information on AWS API calls that are used.
     */
    SpotEventPluginLoggingLevel["DEBUG"] = "Debug";
    /**
     * No logging enabled.
     */
    SpotEventPluginLoggingLevel["OFF"] = "Off";
})(SpotEventPluginLoggingLevel = exports.SpotEventPluginLoggingLevel || (exports.SpotEventPluginLoggingLevel = {}));
/**
 * How the Spot Event Plugin should handle Pre Job Tasks.
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-scripts.html
 */
var SpotEventPluginPreJobTaskMode;
(function (SpotEventPluginPreJobTaskMode) {
    /**
     * Only start 1 Spot instance for the pre job task and ignore any other tasks for that job until the pre job task is completed.
     */
    SpotEventPluginPreJobTaskMode["CONSERVATIVE"] = "Conservative";
    /**
     * Do not take the pre job task into account when calculating target capacity.
     */
    SpotEventPluginPreJobTaskMode["IGNORE"] = "Ignore";
    /**
     * Treat the pre job task like a regular job queued task.
     */
    SpotEventPluginPreJobTaskMode["NORMAL"] = "Normal";
})(SpotEventPluginPreJobTaskMode = exports.SpotEventPluginPreJobTaskMode || (exports.SpotEventPluginPreJobTaskMode = {}));
/**
 * The Worker Extra Info column to be used to display AWS Instance Status
 * if the instance has been marked to be stopped or terminated by EC2 or Spot Event Plugin.
 * See "AWS Instance Status" option at https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#event-plugin-configuration-options
 * and https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/worker-config.html#extra-info
 */
var SpotEventPluginDisplayInstanceStatus;
(function (SpotEventPluginDisplayInstanceStatus) {
    SpotEventPluginDisplayInstanceStatus["DISABLED"] = "Disabled";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_0"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_1"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_2"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_3"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_4"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_5"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_6"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_7"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_8"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_9"] = "ExtraInfo0";
})(SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginDisplayInstanceStatus || (exports.SpotEventPluginDisplayInstanceStatus = {}));
/**
 * This construct configures the Deadline Spot Event Plugin to deploy and auto-scale one or more spot fleets.
 *
 * For example, to configure the Spot Event Plugin with one spot fleet:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { InstanceClass, InstanceSize, InstanceType } from '@aws-cdk/aws-ec2';
 * import { AwsThinkboxEulaAcceptance, ConfigureSpotEventPlugin, RenderQueue, Repository, SpotEventPluginFleet, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *   version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *   version,
 *   // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *   // of the AWS Thinkbox End User License Agreement
 *   userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *   vpc,
 *   version,
 * });
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *   vpc,
 *   images: images.forRenderQueue(),
 *   repository: repository,
 * });
 *
 * const fleet = new SpotEventPluginFleet(this, 'SpotEventPluginFleet', {
 *   vpc,
 *   renderQueue,
 *   deadlineGroups: ['group_name'],
 *   instanceTypes: [InstanceType.of(InstanceClass.T3, InstanceSize.LARGE)],
 *   workerMachineImage: new GenericLinuxImage({'us-west-2': 'ami-039f0c1faba28b015'}),
 *   naxCapacity: 1,
 * });
 *
 * const spotEventPluginConfig = new ConfigureSpotEventPlugin(this, 'ConfigureSpotEventPlugin', {
 *   vpc,
 *   renderQueue: renderQueue,
 *   spotFleets: [fleet],
 *   configuration: {
 *     enableResourceTracker: true,
 *   },
 * });
 * ```
 *
 * To provide this functionality, this construct will create an AWS Lambda function that is granted the ability
 * to connect to the render queue. This lambda is run automatically when you deploy or update the stack containing this construct.
 * Logs for all AWS Lambdas are automatically recorded in Amazon CloudWatch.
 *
 * This construct will configure the Spot Event Plugin, but the Spot Fleet Requests will not be created unless you:
 * - Submit the job with the assigned Deadline Group and Deadline Pool. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-submitting.html#submitting-jobs).
 *
 * Important: Disable 'Allow Workers to Perform House Cleaning If Pulse is not Running' in the 'Configure Repository Options'
 * when using Spot Event Plugin.
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#prerequisites
 *
 * Important: Any resources created by the Spot Event Plugin will not be deleted with 'cdk destroy'.
 * Make sure that all such resources (e.g. Spot Fleet Request or Fleet Instances) are cleaned up, before destroying the stacks.
 * Disable the Spot Event Plugin by setting 'state' property to 'SpotEventPluginState.DISABLED' or via Deadline Monitor,
 * ensure you shutdown all Pulse instances and then terminate any Spot Fleet Requests in the AWS EC2 Instance Console.
 *
 * Note that this construct adds additional policies to the Render Queue's role
 * required to run the Spot Event Plugin and launch a Resource Tracker:
 *  - AWSThinkboxDeadlineSpotEventPluginAdminPolicy
 *  - AWSThinkboxDeadlineResourceTrackerAdminPolicy
 *  - A policy to pass a fleet and instance role
 *  - A policy to create tags for spot fleet requests
 *
 * The Spot Fleet Requests that this construct configures Deadline to create will always use the latest version of the
 * corresponding EC2 Launch Template that was created for them.
 *
 * ![architecture diagram](/diagrams/deadline/ConfigureSpotEventPlugin.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the render queue, and save Spot Event Plugin configurations.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 * - An IAM Policy attached to Render Queue's Role.
 * - EC2 Launch Templates for each Spot Event Plugin fleet.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda function that is created by this resource has access to both the certificates used to connect to the render queue,
 *   and the render queue port. An attacker that can find a way to modify and execute this lambda could use it to
 *   execute any requets against the render queue. You should not grant any additional actors/principals the ability to modify
 *   or execute this Lambda.
 */
class ConfigureSpotEventPlugin extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (ConfigureSpotEventPlugin.uniqueRenderQueues.has(props.renderQueue)) {
            throw new Error('Only one ConfigureSpotEventPlugin construct is allowed per render queue.');
        }
        else {
            ConfigureSpotEventPlugin.uniqueRenderQueues.add(props.renderQueue);
        }
        if (props.renderQueue instanceof render_queue_1.RenderQueue) {
            // We do not check the patch version, so it's set to 0.
            const minimumVersion = new version_1.Version([10, 1, 12, 0]);
            if (props.renderQueue.version.isLessThan(minimumVersion)) {
                throw new Error(`Minimum supported Deadline version for ${this.constructor.name} is ` +
                    `${minimumVersion.versionString}. ` +
                    `Received: ${props.renderQueue.version.versionString}.`);
            }
            if (props.spotFleets && props.spotFleets.length !== 0) {
                // Always add Resource Tracker admin policy, even if props.configuration?.enableResourceTracker is false.
                // This improves usability, as customers won't need to add this policy manually, if they
                // enable Resource Tracker later in the Spot Event Plugin configuration (e.g., using Deadline Monitor and not RFDK).
                props.renderQueue.addSEPPolicies(true);
                const fleetRoles = props.spotFleets.map(sf => sf.fleetRole.roleArn);
                const fleetInstanceRoles = props.spotFleets.map(sf => sf.fleetInstanceRole.roleArn);
                new aws_iam_1.Policy(this, 'SpotEventPluginPolicy', {
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'iam:PassRole',
                            ],
                            resources: [...fleetRoles, ...fleetInstanceRoles],
                            conditions: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                        }),
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'ec2:CreateTags',
                            ],
                            resources: [
                                'arn:aws:ec2:*:*:spot-fleet-request/*',
                                'arn:aws:ec2:*:*:volume/*',
                            ],
                        }),
                    ],
                    roles: [
                        props.renderQueue.grantPrincipal,
                    ],
                });
            }
        }
        else {
            throw new Error('The provided render queue is not an instance of RenderQueue class. Some functionality is not supported.');
        }
        const region = core_1.Construct.isConstruct(props.renderQueue) ? core_1.Stack.of(props.renderQueue).region : core_1.Stack.of(this).region;
        const timeoutMins = 15;
        const configurator = new aws_lambda_1.Function(this, 'Configurator', {
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
            description: `Used by a ConfigureSpotEventPlugin ${this.node.addr} to perform configuration of Deadline Spot Event Plugin`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {}),
            environment: {
                DEBUG: 'false',
                LAMBDA_TIMEOUT_MINS: timeoutMins.toString(),
            },
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'configure-spot-event-plugin.configureSEP',
            timeout: core_1.Duration.minutes(timeoutMins),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        configurator.connections.allowToDefaultPort(props.renderQueue);
        props.renderQueue.certChain?.grantRead(configurator.grantPrincipal);
        const pluginConfig = {
            AWSInstanceStatus: props.configuration?.awsInstanceStatus ?? SpotEventPluginDisplayInstanceStatus.DISABLED,
            DeleteInterruptedSlaves: props.configuration?.deleteEC2SpotInterruptedWorkers ?? false,
            DeleteTerminatedSlaves: props.configuration?.deleteSEPTerminatedWorkers ?? false,
            IdleShutdown: props.configuration?.idleShutdown?.toMinutes({ integral: true }) ?? 10,
            Logging: props.configuration?.loggingLevel ?? SpotEventPluginLoggingLevel.STANDARD,
            PreJobTaskMode: props.configuration?.preJobTaskMode ?? SpotEventPluginPreJobTaskMode.CONSERVATIVE,
            Region: props.configuration?.region ?? region,
            ResourceTracker: props.configuration?.enableResourceTracker ?? true,
            StaggerInstances: props.configuration?.maximumInstancesStartedPerCycle ?? 50,
            State: props.configuration?.state ?? SpotEventPluginState.GLOBAL_ENABLED,
            StrictHardCap: props.configuration?.strictHardCap ?? false,
        };
        const spotFleetRequestConfigs = this.mergeSpotFleetRequestConfigs(props.spotFleets);
        const deadlineGroups = Array.from(new Set(props.spotFleets?.map(fleet => fleet.deadlineGroups).reduce((p, c) => p.concat(c), [])));
        const deadlinePools = Array.from(new Set(props.spotFleets?.map(fleet => fleet.deadlinePools).reduce((p, c) => p?.concat(c ?? []), [])));
        const properties = {
            connection: {
                hostname: props.renderQueue.endpoint.hostname,
                port: props.renderQueue.endpoint.portAsString(),
                protocol: props.renderQueue.endpoint.applicationProtocol,
                caCertificateArn: props.renderQueue.certChain?.secretArn,
            },
            spotFleetRequestConfigurations: spotFleetRequestConfigs,
            spotPluginConfigurations: pluginConfig,
            deadlineGroups,
            deadlinePools,
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: configurator.functionArn,
            resourceType: 'Custom::RFDK_ConfigureSpotEventPlugin',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(configurator.role);
        // We need to add this dependency to avoid failures while deleting a Custom Resource:
        // 'Custom Resource failed to stabilize in expected time. If you are using the Python cfn-response module,
        // you may need to update your Lambda function code so that CloudFormation can attach the updated version.'.
        // This happens, because Route Table Associations are deleted before the Custom Resource and we
        // don't get a response from 'doDelete()'.
        // Ideally, we would only want to add dependency on 'internetConnectivityEstablished' as shown below,
        // but it seems that CDK misses dependencies on Route Table Associations in that case:
        // const { internetConnectivityEstablished } = props.vpc.selectSubnets(props.vpcSubnets);
        // resource.node.addDependency(internetConnectivityEstablished);
        resource.node.addDependency(props.vpc);
        // /* istanbul ignore next */
        // Add a dependency on the render queue to ensure that
        // it is running before we try to send requests to it.
        resource.node.addDependency(props.renderQueue);
        if (props.spotFleets && props.renderQueue.repository.secretsManagementSettings.enabled) {
            props.spotFleets.forEach(spotFleet => {
                if (spotFleet.defaultSubnets) {
                    core_1.Annotations.of(spotFleet).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
                }
                props.renderQueue.configureSecretsManagementAutoRegistration({
                    dependent: resource,
                    role: secrets_management_ref_1.SecretsManagementRole.CLIENT,
                    registrationStatus: secrets_management_ref_1.SecretsManagementRegistrationStatus.REGISTERED,
                    vpc: props.vpc,
                    vpcSubnets: spotFleet.subnets,
                });
            });
        }
        this.node.defaultChild = resource;
    }
    tagSpecifications(fleet, resourceType) {
        return core_1.Lazy.any({
            produce: () => {
                if (fleet.tags.hasTags()) {
                    const tagSpecification = {
                        ResourceType: resourceType,
                        Tags: fleet.tags.renderTags(),
                    };
                    return [tagSpecification];
                }
                return undefined;
            },
        });
    }
    /**
     * Construct Spot Fleet Configurations from the provided fleet.
     * Each configuration is a mapping between one Deadline Group and one Spot Fleet Request Configuration.
     */
    generateSpotFleetRequestConfig(fleet) {
        const spotFleetRequestTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.SPOT_FLEET_REQUEST);
        const spotFleetRequestProps = {
            AllocationStrategy: fleet.allocationStrategy,
            IamFleetRole: fleet.fleetRole.roleArn,
            LaunchTemplateConfigs: fleet._launchTemplateConfigs,
            ReplaceUnhealthyInstances: true,
            // In order to work with Deadline, the 'Target Capacity' of the Spot fleet Request is
            // the maximum number of Workers that Deadline will start.
            TargetCapacity: fleet.maxCapacity,
            TerminateInstancesWithExpiration: true,
            // In order to work with Deadline, Spot Fleets Requests must be set to maintain.
            Type: spot_event_plugin_fleet_ref_1.SpotFleetRequestType.MAINTAIN,
            ValidUntil: fleet.validUntil?.date.toISOString(),
            // Need to convert from IResolvable to bypass TypeScript
            TagSpecifications: spotFleetRequestTagsToken,
        };
        const spotFleetRequestConfigurations = fleet.deadlineGroups.map(group => {
            const spotFleetRequestConfiguration = {
                [group.toLowerCase()]: spotFleetRequestProps,
            };
            return spotFleetRequestConfiguration;
        });
        return spotFleetRequestConfigurations;
    }
    mergeSpotFleetRequestConfigs(spotFleets) {
        if (!spotFleets || spotFleets.length === 0) {
            return undefined;
        }
        const fullSpotFleetRequestConfiguration = {};
        spotFleets.map(fleet => {
            const spotFleetRequestConfigurations = this.generateSpotFleetRequestConfig(fleet);
            spotFleetRequestConfigurations.map(configuration => {
                for (const [key, value] of Object.entries(configuration)) {
                    if (key in fullSpotFleetRequestConfiguration) {
                        throw new Error(`Bad Group Name: ${key}. Group names in Spot Fleet Request Configurations should be unique.`);
                    }
                    fullSpotFleetRequestConfiguration[key] = value;
                }
            });
        });
        return fullSpotFleetRequestConfiguration;
    }
}
exports.ConfigureSpotEventPlugin = ConfigureSpotEventPlugin;
_a = JSII_RTTI_SYMBOL_1;
ConfigureSpotEventPlugin[_a] = { fqn: "aws-rfdk.deadline.ConfigureSpotEventPlugin", version: "0.42.0" };
/**
 * Only one Spot Event Plugin Configuration is allowed per render queue / repository.
 */
ConfigureSpotEventPlugin.uniqueRenderQueues = new Set();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29uZmlndXJlLXNwb3QtZXZlbnQtcGx1Z2luLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY29uZmlndXJlLXNwb3QtZXZlbnQtcGx1Z2luLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7OztHQUdHO0FBRUgsNkJBQTZCO0FBRTdCLDhDQUkwQjtBQUMxQiw4Q0FJMEI7QUFDMUIsb0RBSTZCO0FBQzdCLGdEQUFrRDtBQUNsRCx3Q0FRdUI7QUFTdkIsaURBR3dCO0FBQ3hCLHFFQUdrQztBQUVsQywrRUFHdUM7QUFDdkMsdUNBQW9DO0FBRXBDOztHQUVHO0FBQ0gsSUFBWSxvQkFVWDtBQVZELFdBQVksb0JBQW9CO0lBQzlCOztPQUVHO0lBQ0gseURBQWlDLENBQUE7SUFFakM7O09BRUc7SUFDSCw2Q0FBcUIsQ0FBQTtBQUN2QixDQUFDLEVBVlcsb0JBQW9CLEdBQXBCLDRCQUFvQixLQUFwQiw0QkFBb0IsUUFVL0I7QUFFRDs7R0FFRztBQUNILElBQVksMkJBb0JYO0FBcEJELFdBQVksMkJBQTJCO0lBQ3JDOztPQUVHO0lBQ0gsb0RBQXFCLENBQUE7SUFFckI7O09BRUc7SUFDSCxrREFBbUIsQ0FBQTtJQUVuQjs7T0FFRztJQUNILDhDQUFlLENBQUE7SUFFZjs7T0FFRztJQUNILDBDQUFXLENBQUE7QUFDYixDQUFDLEVBcEJXLDJCQUEyQixHQUEzQixtQ0FBMkIsS0FBM0IsbUNBQTJCLFFBb0J0QztBQUVEOzs7R0FHRztBQUNILElBQVksNkJBZVg7QUFmRCxXQUFZLDZCQUE2QjtJQUN2Qzs7T0FFRztJQUNILDhEQUE2QixDQUFBO0lBRTdCOztPQUVHO0lBQ0gsa0RBQWlCLENBQUE7SUFFakI7O09BRUc7SUFDSCxrREFBaUIsQ0FBQTtBQUNuQixDQUFDLEVBZlcsNkJBQTZCLEdBQTdCLHFDQUE2QixLQUE3QixxQ0FBNkIsUUFleEM7QUFFRDs7Ozs7R0FLRztBQUNILElBQVksb0NBWVg7QUFaRCxXQUFZLG9DQUFvQztJQUM5Qyw2REFBcUIsQ0FBQTtJQUNyQixtRUFBMkIsQ0FBQTtJQUMzQixtRUFBMkIsQ0FBQTtJQUMzQixtRUFBMkIsQ0FBQTtJQUMzQixtRUFBMkIsQ0FBQTtJQUMzQixtRUFBMkIsQ0FBQTtJQUMzQixtRUFBMkIsQ0FBQTtJQUMzQixtRUFBMkIsQ0FBQTtJQUMzQixtRUFBMkIsQ0FBQTtJQUMzQixtRUFBMkIsQ0FBQTtJQUMzQixtRUFBMkIsQ0FBQTtBQUM3QixDQUFDLEVBWlcsb0NBQW9DLEdBQXBDLDRDQUFvQyxLQUFwQyw0Q0FBb0MsUUFZL0M7QUF3SUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBa0dHO0FBQ0gsTUFBYSx3QkFBeUIsU0FBUSxnQkFBUztJQU9yRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9DO1FBQzVFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSx3QkFBd0IsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQ3RFLE1BQU0sSUFBSSxLQUFLLENBQUMsMEVBQTBFLENBQUMsQ0FBQztTQUM3RjthQUNJO1lBQ0gsd0JBQXdCLENBQUMsa0JBQWtCLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNwRTtRQUVELElBQUksS0FBSyxDQUFDLFdBQVcsWUFBWSwwQkFBVyxFQUFFO1lBQzVDLHVEQUF1RDtZQUN2RCxNQUFNLGNBQWMsR0FBWSxJQUFJLGlCQUFPLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRTVELElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FBQyxFQUFFO2dCQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLDBDQUEwQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksTUFBTTtvQkFDckYsR0FBRyxjQUFjLENBQUMsYUFBYSxJQUFJO29CQUNuQyxhQUFhLEtBQUssQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLGFBQWEsR0FBRyxDQUFDLENBQUM7YUFDMUQ7WUFFRCxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUNyRCx5R0FBeUc7Z0JBQ3pHLHdGQUF3RjtnQkFDeEYsb0hBQW9IO2dCQUNwSCxLQUFLLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFFdkMsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO2dCQUNwRSxNQUFNLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxDQUFDO2dCQUNwRixJQUFJLGdCQUFNLENBQUMsSUFBSSxFQUFFLHVCQUF1QixFQUFFO29CQUN4QyxVQUFVLEVBQUU7d0JBQ1YsSUFBSSx5QkFBZSxDQUFDOzRCQUNsQixPQUFPLEVBQUU7Z0NBQ1AsY0FBYzs2QkFDZjs0QkFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLFVBQVUsRUFBRSxHQUFHLGtCQUFrQixDQUFDOzRCQUNqRCxVQUFVLEVBQUU7Z0NBQ1YsVUFBVSxFQUFFO29DQUNWLHFCQUFxQixFQUFFLG1CQUFtQjtpQ0FDM0M7NkJBQ0Y7eUJBQ0YsQ0FBQzt3QkFDRixJQUFJLHlCQUFlLENBQUM7NEJBQ2xCLE9BQU8sRUFBRTtnQ0FDUCxnQkFBZ0I7NkJBQ2pCOzRCQUNELFNBQVMsRUFBRTtnQ0FDVCxzQ0FBc0M7Z0NBQ3RDLDBCQUEwQjs2QkFDM0I7eUJBQ0YsQ0FBQztxQkFDSDtvQkFDRCxLQUFLLEVBQUU7d0JBQ0wsS0FBSyxDQUFDLFdBQVcsQ0FBQyxjQUFzQjtxQkFDekM7aUJBQ0YsQ0FBQyxDQUFDO2FBQ0o7U0FDRjthQUNJO1lBQ0gsTUFBTSxJQUFJLEtBQUssQ0FBQyx5R0FBeUcsQ0FBQyxDQUFDO1NBQzVIO1FBRUQsTUFBTSxNQUFNLEdBQUcsZ0JBQVMsQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxDQUFDO1FBRXJILE1BQU0sV0FBVyxHQUFHLEVBQUUsQ0FBQztRQUN2QixNQUFNLFlBQVksR0FBRyxJQUFJLHFCQUFjLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUM1RCxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7WUFDZCxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsSUFBSSxFQUFFLFVBQVUsRUFBRSxvQkFBVSxDQUFDLGdCQUFnQixFQUFFO1lBQzNFLFdBQVcsRUFBRSxzQ0FBc0MsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLHlEQUF5RDtZQUMxSCxJQUFJLEVBQUUsaUJBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsUUFBUSxDQUFDLEVBQUUsRUFDM0UsQ0FBQztZQUNGLFdBQVcsRUFBRTtnQkFDWCxLQUFLLEVBQUUsT0FBTztnQkFDZCxtQkFBbUIsRUFBRSxXQUFXLENBQUMsUUFBUSxFQUFFO2FBQzVDO1lBQ0QsT0FBTyxFQUFFLG9CQUFPLENBQUMsV0FBVztZQUM1QixPQUFPLEVBQUUsMENBQTBDO1lBQ25ELE9BQU8sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQztZQUN0QyxZQUFZLEVBQUUsd0JBQWEsQ0FBQyxRQUFRO1NBQ3JDLENBQUMsQ0FBQztRQUVILFlBQVksQ0FBQyxXQUFXLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQy9ELEtBQUssQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFcEUsTUFBTSxZQUFZLEdBQW1CO1lBQ25DLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxhQUFhLEVBQUUsaUJBQWlCLElBQUksb0NBQW9DLENBQUMsUUFBUTtZQUMxRyx1QkFBdUIsRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLCtCQUErQixJQUFJLEtBQUs7WUFDdEYsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLGFBQWEsRUFBRSwwQkFBMEIsSUFBSSxLQUFLO1lBQ2hGLFlBQVksRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLFlBQVksRUFBRSxTQUFTLENBQUMsRUFBQyxRQUFRLEVBQUUsSUFBSSxFQUFDLENBQUMsSUFBSSxFQUFFO1lBQ2xGLE9BQU8sRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLFlBQVksSUFBSSwyQkFBMkIsQ0FBQyxRQUFRO1lBQ2xGLGNBQWMsRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLGNBQWMsSUFBSSw2QkFBNkIsQ0FBQyxZQUFZO1lBQ2pHLE1BQU0sRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLE1BQU0sSUFBSSxNQUFNO1lBQzdDLGVBQWUsRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLHFCQUFxQixJQUFJLElBQUk7WUFDbkUsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGFBQWEsRUFBRSwrQkFBK0IsSUFBSSxFQUFFO1lBQzVFLEtBQUssRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLEtBQUssSUFBSSxvQkFBb0IsQ0FBQyxjQUFjO1lBQ3hFLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLGFBQWEsSUFBSSxLQUFLO1NBQzNELENBQUM7UUFDRixNQUFNLHVCQUF1QixHQUFHLElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFcEYsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsVUFBVSxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNuSSxNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxVQUFVLEVBQUUsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN4SSxNQUFNLFVBQVUsR0FBaUM7WUFDL0MsVUFBVSxFQUFFO2dCQUNWLFFBQVEsRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxRQUFRO2dCQUM3QyxJQUFJLEVBQUUsS0FBSyxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFO2dCQUMvQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsbUJBQW1CO2dCQUN4RCxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDLFNBQVMsRUFBRSxTQUFTO2FBQ3pEO1lBQ0QsOEJBQThCLEVBQUUsdUJBQXVCO1lBQ3ZELHdCQUF3QixFQUFFLFlBQVk7WUFDdEMsY0FBYztZQUNkLGFBQWE7U0FDZCxDQUFDO1FBRUYsTUFBTSxRQUFRLEdBQUcsSUFBSSxxQkFBYyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDbkQsWUFBWSxFQUFFLFlBQVksQ0FBQyxXQUFXO1lBQ3RDLFlBQVksRUFBRSx1Q0FBdUM7WUFDckQsVUFBVTtTQUNYLENBQUMsQ0FBQztRQUVILHlDQUF5QztRQUN6QyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsSUFBSyxDQUFDLENBQUM7UUFFaEQscUZBQXFGO1FBQ3JGLDBHQUEwRztRQUMxRyw0R0FBNEc7UUFDNUcsK0ZBQStGO1FBQy9GLDBDQUEwQztRQUMxQyxxR0FBcUc7UUFDckcsc0ZBQXNGO1FBQ3RGLHlGQUF5RjtRQUN6RixnRUFBZ0U7UUFDaEUsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRXZDLDZCQUE2QjtRQUM3QixzREFBc0Q7UUFDdEQsc0RBQXNEO1FBQ3RELFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUUvQyxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsT0FBTyxFQUFFO1lBQ3RGLEtBQUssQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxFQUFFO2dCQUNuQyxJQUFJLFNBQVMsQ0FBQyxjQUFjLEVBQUU7b0JBQzVCLGtCQUFXLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFVBQVUsQ0FDbEMscVJBQXFSLENBQ3RSLENBQUM7aUJBQ0g7Z0JBQ0QsS0FBSyxDQUFDLFdBQVcsQ0FBQywwQ0FBMEMsQ0FBQztvQkFDM0QsU0FBUyxFQUFFLFFBQVE7b0JBQ25CLElBQUksRUFBRSw4Q0FBcUIsQ0FBQyxNQUFNO29CQUNsQyxrQkFBa0IsRUFBRSw0REFBbUMsQ0FBQyxVQUFVO29CQUNsRSxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7b0JBQ2QsVUFBVSxFQUFFLFNBQVMsQ0FBQyxPQUFPO2lCQUM5QixDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsUUFBUSxDQUFDO0lBQ3BDLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxLQUEyQixFQUFFLFlBQW1DO1FBQ3hGLE9BQU8sV0FBSSxDQUFDLEdBQUcsQ0FBQztZQUNkLE9BQU8sRUFBRSxHQUFHLEVBQUU7Z0JBQ1osSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFO29CQUN4QixNQUFNLGdCQUFnQixHQUE4Qjt3QkFDbEQsWUFBWSxFQUFFLFlBQVk7d0JBQzFCLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRTtxQkFDOUIsQ0FBQztvQkFDRixPQUFPLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztpQkFDM0I7Z0JBQ0QsT0FBTyxTQUFTLENBQUM7WUFDbkIsQ0FBQztTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7O09BR0c7SUFDSyw4QkFBOEIsQ0FBQyxLQUEyQjtRQUNoRSxNQUFNLHlCQUF5QixHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsbURBQXFCLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUUxRyxNQUFNLHFCQUFxQixHQUEwQjtZQUNuRCxrQkFBa0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCO1lBQzVDLFlBQVksRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLE9BQU87WUFDckMscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtZQUNuRCx5QkFBeUIsRUFBRSxJQUFJO1lBQy9CLHFGQUFxRjtZQUNyRiwwREFBMEQ7WUFDMUQsY0FBYyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQ2pDLGdDQUFnQyxFQUFFLElBQUk7WUFDdEMsZ0ZBQWdGO1lBQ2hGLElBQUksRUFBRSxrREFBb0IsQ0FBQyxRQUFRO1lBQ25DLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVSxFQUFFLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFDaEQsd0RBQXdEO1lBQ3hELGlCQUFpQixFQUFHLHlCQUFvRTtTQUN6RixDQUFDO1FBRUYsTUFBTSw4QkFBOEIsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUN0RSxNQUFNLDZCQUE2QixHQUFrQztnQkFDbkUsQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFLENBQUMsRUFBRSxxQkFBcUI7YUFDN0MsQ0FBQztZQUNGLE9BQU8sNkJBQTZCLENBQUM7UUFDdkMsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLDhCQUE4QixDQUFDO0lBQ3hDLENBQUM7SUFFTyw0QkFBNEIsQ0FBQyxVQUFtQztRQUN0RSxJQUFJLENBQUMsVUFBVSxJQUFJLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzFDLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsTUFBTSxpQ0FBaUMsR0FBa0MsRUFBRSxDQUFDO1FBQzVFLFVBQVUsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDckIsTUFBTSw4QkFBOEIsR0FBRyxJQUFJLENBQUMsOEJBQThCLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDbEYsOEJBQThCLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFO2dCQUNqRCxLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsRUFBRTtvQkFDeEQsSUFBSSxHQUFHLElBQUksaUNBQWlDLEVBQUU7d0JBQzVDLE1BQU0sSUFBSSxLQUFLLENBQUMsbUJBQW1CLEdBQUcsc0VBQXNFLENBQUMsQ0FBQztxQkFDL0c7b0JBQ0QsaUNBQWlDLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO2lCQUNoRDtZQUNILENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLGlDQUFpQyxDQUFDO0lBQzNDLENBQUM7O0FBeE9ILDREQXlPQzs7O0FBdk9DOztHQUVHO0FBQ1ksMkNBQWtCLEdBQXNCLElBQUksR0FBRyxFQUFnQixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcblxuaW1wb3J0IHtcbiAgSVZwYyxcbiAgU3VibmV0U2VsZWN0aW9uLFxuICBTdWJuZXRUeXBlLFxufSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7XG4gIFJvbGUsXG4gIFBvbGljeSxcbiAgUG9saWN5U3RhdGVtZW50LFxufSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7XG4gIENvZGUsXG4gIEZ1bmN0aW9uIGFzIExhbWJkYUZ1bmN0aW9uLFxuICBSdW50aW1lLFxufSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IFJldGVudGlvbkRheXMgfSBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQge1xuICBBbm5vdGF0aW9ucyxcbiAgQ29uc3RydWN0LFxuICBDdXN0b21SZXNvdXJjZSxcbiAgRHVyYXRpb24sXG4gIElSZXNvbHZhYmxlLFxuICBMYXp5LFxuICBTdGFjayxcbn0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmltcG9ydCB7XG4gIFBsdWdpblNldHRpbmdzLFxuICBTRVBDb25maWd1cmF0b3JSZXNvdXJjZVByb3BzLFxuICBTcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbixcbiAgU3BvdEZsZWV0UmVxdWVzdFByb3BzLFxuICBTcG90RmxlZXRUYWdTcGVjaWZpY2F0aW9uLFxufSBmcm9tICcuLi8uLi9sYW1iZGFzL25vZGVqcy9jb25maWd1cmUtc3BvdC1ldmVudC1wbHVnaW4nO1xuaW1wb3J0IHtcbiAgSVJlbmRlclF1ZXVlLFxuICBSZW5kZXJRdWV1ZSxcbn0gZnJvbSAnLi9yZW5kZXItcXVldWUnO1xuaW1wb3J0IHtcbiAgU2VjcmV0c01hbmFnZW1lbnRSZWdpc3RyYXRpb25TdGF0dXMsXG4gIFNlY3JldHNNYW5hZ2VtZW50Um9sZSxcbn0gZnJvbSAnLi9zZWNyZXRzLW1hbmFnZW1lbnQtcmVmJztcbmltcG9ydCB7IFNwb3RFdmVudFBsdWdpbkZsZWV0IH0gZnJvbSAnLi9zcG90LWV2ZW50LXBsdWdpbi1mbGVldCc7XG5pbXBvcnQge1xuICBTcG90RmxlZXRSZXF1ZXN0VHlwZSxcbiAgU3BvdEZsZWV0UmVzb3VyY2VUeXBlLFxufSBmcm9tICcuL3Nwb3QtZXZlbnQtcGx1Z2luLWZsZWV0LXJlZic7XG5pbXBvcnQgeyBWZXJzaW9uIH0gZnJvbSAnLi92ZXJzaW9uJztcblxuLyoqXG4gKiBIb3cgdGhlIGV2ZW50IHBsdWctaW4gc2hvdWxkIHJlc3BvbmQgdG8gZXZlbnRzLlxuICovXG5leHBvcnQgZW51bSBTcG90RXZlbnRQbHVnaW5TdGF0ZSB7XG4gIC8qKlxuICAgKiBUaGUgUmVuZGVyIFF1ZXVlLCBhbGwgam9icyBhbmQgV29ya2VycyB3aWxsIHRyaWdnZXIgdGhlIGV2ZW50cyBmb3IgdGhpcyBwbHVnaW4uXG4gICAqL1xuICBHTE9CQUxfRU5BQkxFRCA9ICdHbG9iYWwgRW5hYmxlZCcsXG5cbiAgLyoqXG4gICAqIE5vIGV2ZW50cyBhcmUgdHJpZ2dlcmVkIGZvciB0aGUgcGx1Z2luLlxuICAgKi9cbiAgRElTQUJMRUQgPSAnRGlzYWJsZWQnLFxufVxuXG4vKipcbiAqIExvZ2dpbmcgdmVyYm9zaXR5IGxldmVscyBmb3IgdGhlIFNwb3QgRXZlbnQgUGx1Z2luLlxuICovXG5leHBvcnQgZW51bSBTcG90RXZlbnRQbHVnaW5Mb2dnaW5nTGV2ZWwge1xuICAvKipcbiAgICogU3RhbmRhcmQgbG9nZ2luZyBsZXZlbC5cbiAgICovXG4gIFNUQU5EQVJEID0gJ1N0YW5kYXJkJyxcblxuICAvKipcbiAgICogRGV0YWlsZWQgbG9nZ2luZyBhYm91dCB0aGUgaW5uZXIgd29ya2luZ3Mgb2YgdGhlIFNwb3QgRXZlbnQgUGx1Z2luLlxuICAgKi9cbiAgVkVSQk9TRSA9ICdWZXJib3NlJyxcblxuICAvKipcbiAgICogQWxsIFZlcmJvc2UgbG9ncyBwbHVzIGFkZGl0aW9uYWwgaW5mb3JtYXRpb24gb24gQVdTIEFQSSBjYWxscyB0aGF0IGFyZSB1c2VkLlxuICAgKi9cbiAgREVCVUcgPSAnRGVidWcnLFxuXG4gIC8qKlxuICAgKiBObyBsb2dnaW5nIGVuYWJsZWQuXG4gICAqL1xuICBPRkYgPSAnT2ZmJyxcbn1cblxuLyoqXG4gKiBIb3cgdGhlIFNwb3QgRXZlbnQgUGx1Z2luIHNob3VsZCBoYW5kbGUgUHJlIEpvYiBUYXNrcy5cbiAqIFNlZSBodHRwczovL2RvY3MudGhpbmtib3hzb2Z0d2FyZS5jb20vcHJvZHVjdHMvZGVhZGxpbmUvMTAuMS8xX1VzZXIlMjBNYW51YWwvbWFudWFsL2pvYi1zY3JpcHRzLmh0bWxcbiAqL1xuZXhwb3J0IGVudW0gU3BvdEV2ZW50UGx1Z2luUHJlSm9iVGFza01vZGUge1xuICAvKipcbiAgICogT25seSBzdGFydCAxIFNwb3QgaW5zdGFuY2UgZm9yIHRoZSBwcmUgam9iIHRhc2sgYW5kIGlnbm9yZSBhbnkgb3RoZXIgdGFza3MgZm9yIHRoYXQgam9iIHVudGlsIHRoZSBwcmUgam9iIHRhc2sgaXMgY29tcGxldGVkLlxuICAgKi9cbiAgQ09OU0VSVkFUSVZFID0gJ0NvbnNlcnZhdGl2ZScsXG5cbiAgLyoqXG4gICAqIERvIG5vdCB0YWtlIHRoZSBwcmUgam9iIHRhc2sgaW50byBhY2NvdW50IHdoZW4gY2FsY3VsYXRpbmcgdGFyZ2V0IGNhcGFjaXR5LlxuICAgKi9cbiAgSUdOT1JFID0gJ0lnbm9yZScsXG5cbiAgLyoqXG4gICAqIFRyZWF0IHRoZSBwcmUgam9iIHRhc2sgbGlrZSBhIHJlZ3VsYXIgam9iIHF1ZXVlZCB0YXNrLlxuICAgKi9cbiAgTk9STUFMID0gJ05vcm1hbCcsXG59XG5cbi8qKlxuICogVGhlIFdvcmtlciBFeHRyYSBJbmZvIGNvbHVtbiB0byBiZSB1c2VkIHRvIGRpc3BsYXkgQVdTIEluc3RhbmNlIFN0YXR1c1xuICogaWYgdGhlIGluc3RhbmNlIGhhcyBiZWVuIG1hcmtlZCB0byBiZSBzdG9wcGVkIG9yIHRlcm1pbmF0ZWQgYnkgRUMyIG9yIFNwb3QgRXZlbnQgUGx1Z2luLlxuICogU2VlIFwiQVdTIEluc3RhbmNlIFN0YXR1c1wiIG9wdGlvbiBhdCBodHRwczovL2RvY3MudGhpbmtib3hzb2Z0d2FyZS5jb20vcHJvZHVjdHMvZGVhZGxpbmUvMTAuMS8xX1VzZXIlMjBNYW51YWwvbWFudWFsL2V2ZW50LXNwb3QuaHRtbCNldmVudC1wbHVnaW4tY29uZmlndXJhdGlvbi1vcHRpb25zXG4gKiBhbmQgaHR0cHM6Ly9kb2NzLnRoaW5rYm94c29mdHdhcmUuY29tL3Byb2R1Y3RzL2RlYWRsaW5lLzEwLjEvMV9Vc2VyJTIwTWFudWFsL21hbnVhbC93b3JrZXItY29uZmlnLmh0bWwjZXh0cmEtaW5mb1xuICovXG5leHBvcnQgZW51bSBTcG90RXZlbnRQbHVnaW5EaXNwbGF5SW5zdGFuY2VTdGF0dXMge1xuICBESVNBQkxFRCA9ICdEaXNhYmxlZCcsXG4gIEVYVFJBX0lORk9fMCA9ICdFeHRyYUluZm8wJyxcbiAgRVhUUkFfSU5GT18xID0gJ0V4dHJhSW5mbzAnLFxuICBFWFRSQV9JTkZPXzIgPSAnRXh0cmFJbmZvMCcsXG4gIEVYVFJBX0lORk9fMyA9ICdFeHRyYUluZm8wJyxcbiAgRVhUUkFfSU5GT180ID0gJ0V4dHJhSW5mbzAnLFxuICBFWFRSQV9JTkZPXzUgPSAnRXh0cmFJbmZvMCcsXG4gIEVYVFJBX0lORk9fNiA9ICdFeHRyYUluZm8wJyxcbiAgRVhUUkFfSU5GT183ID0gJ0V4dHJhSW5mbzAnLFxuICBFWFRSQV9JTkZPXzggPSAnRXh0cmFJbmZvMCcsXG4gIEVYVFJBX0lORk9fOSA9ICdFeHRyYUluZm8wJyxcbn1cblxuLyoqXG4gKiBUaGUgc2V0dGluZ3Mgb2YgdGhlIFNwb3QgRXZlbnQgUGx1Z2luLlxuICogRm9yIG1vcmUgZGV0YWlscyBzZWUgaHR0cHM6Ly9kb2NzLnRoaW5rYm94c29mdHdhcmUuY29tL3Byb2R1Y3RzL2RlYWRsaW5lLzEwLjEvMV9Vc2VyJTIwTWFudWFsL21hbnVhbC9ldmVudC1zcG90Lmh0bWwjZXZlbnQtcGx1Z2luLWNvbmZpZ3VyYXRpb24tb3B0aW9uc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFNwb3RFdmVudFBsdWdpblNldHRpbmdzIHtcbiAgLyoqXG4gICAqIEhvdyB0aGUgZXZlbnQgcGx1Zy1pbiBzaG91bGQgcmVzcG9uZCB0byBldmVudHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IFNwb3RFdmVudFBsdWdpblN0YXRlLkdMT0JBTF9FTkFCTEVEXG4gICAqL1xuICByZWFkb25seSBzdGF0ZT86IFNwb3RFdmVudFBsdWdpblN0YXRlO1xuXG4gIC8qKlxuICAgKiBEZXRlcm1pbmVzIHdoZXRoZXIgdGhlIERlYWRsaW5lIFJlc291cmNlIFRyYWNrZXIgc2hvdWxkIGJlIHVzZWQuXG4gICAqXG4gICAqIEluIGFkZGl0aW9uIHRvIHRoaXMgcHJvcGVydHksIHRoZSBTcG90IEluc3RhbmNlcyBkZXBsb3llZCBieSB0aGUgU3BvdCBFdmVudCBQbHVnaW4gbXVzdCBhbHNvIGJlIGNvbmZpZ3VyZWQgdG8gYmUgdHJhY2tlZCBieSB0aGUgUmVzb3VyY2UgVHJhY2tlciB1c2luZyB0aGVcbiAgICogW2B0cmFja0luc3RhbmNlc1dpdGhSZXNvdXJjZVRyYWNrZXJgXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcmZkay9hcGkvbGF0ZXN0L2RvY3MvYXdzLXJmZGsuZGVhZGxpbmUuU3BvdEV2ZW50UGx1Z2luRmxlZXQuaHRtbCN0cmFja2luc3RhbmNlc3dpdGhyZXNvdXJjZXRyYWNrZXIpXG4gICAqIHByb3BlcnR5IG9mIHRoZSBgU3BvdEV2ZW50UGx1Z2luRmxlZXRgIGNvbnN0cnVjdCwgd2hpY2ggaXMgYHRydWVgIGJ5IGRlZmF1bHQuIFlvdSBjYW4gc2V0IHRoYXQgcHJvcGVydHkgdG8gYGZhbHNlYCBmb3IgZmxlZXRzIHRoYXQgeW91IHdvdWxkIGxpa2UgdG8gb3B0IG91dCBvZiB0aGVcbiAgICogUmVzb3VyY2UgVHJhY2tlci5cbiAgICpcbiAgICogU2VlIGh0dHBzOi8vZG9jcy50aGlua2JveHNvZnR3YXJlLmNvbS9wcm9kdWN0cy9kZWFkbGluZS8xMC4xLzFfVXNlciUyME1hbnVhbC9tYW51YWwvcmVzb3VyY2UtdHJhY2tlci1vdmVydmlldy5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZVJlc291cmNlVHJhY2tlcj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFNwb3QgRXZlbnQgUGx1Z2luIGxvZ2dpbmcgbGV2ZWwuXG4gICAqIE5vdGUgdGhhdCBTcG90IEV2ZW50IFBsdWdpbiBhZGRzIG91dHB1dCB0byB0aGUgbG9ncyBvZiB0aGUgcmVuZGVyIHF1ZXVlIGFuZCB0aGUgV29ya2Vycy5cbiAgICpcbiAgICogQGRlZmF1bHQgU3BvdEV2ZW50UGx1Z2luTG9nZ2luZ0xldmVsLlNUQU5EQVJEXG4gICAqL1xuICByZWFkb25seSBsb2dnaW5nTGV2ZWw/OiBTcG90RXZlbnRQbHVnaW5Mb2dnaW5nTGV2ZWw7XG5cbiAgLyoqXG4gICAqIFRoZSBBV1MgcmVnaW9uIGluIHdoaWNoIHRvIHN0YXJ0IHRoZSBzcG90IGZsZWV0IHJlcXVlc3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IFRoZSByZWdpb24gb2YgdGhlIFJlbmRlciBRdWV1ZSBpZiBpdCBpcyBhdmFpbGFibGU7IG90aGVyd2lzZSB0aGUgcmVnaW9uIG9mIHRoZSBjdXJyZW50IHN0YWNrLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVnaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbGVuZ3RoIG9mIHRpbWUgdGhhdCBhbiBBV1MgV29ya2VyIHdpbGwgd2FpdCBpbiBhIG5vbi1yZW5kZXJpbmcgc3RhdGUgYmVmb3JlIGl0IGlzIHNodXRkb3duLlxuICAgKiBTaG91bGQgZXZlbmx5IGRpdmlkZSBpbnRvIG1pbnV0ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLm1pbnV0ZXMoMTApXG4gICAqL1xuICByZWFkb25seSBpZGxlU2h1dGRvd24/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyBpZiBEZWFkbGluZSBTcG90IEV2ZW50IFBsdWdpbiB0ZXJtaW5hdGVkIEFXUyBXb3JrZXJzIHdpbGwgYmUgZGVsZXRlZCBmcm9tIHRoZSBXb3JrZXJzIFBhbmVsIG9uIHRoZSBuZXh0IEhvdXNlIENsZWFuaW5nIGN5Y2xlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVsZXRlU0VQVGVybWluYXRlZFdvcmtlcnM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBEZXRlcm1pbmVzIGlmIEVDMiBTcG90IGludGVycnVwdGVkIEFXUyBXb3JrZXJzIHdpbGwgYmUgZGVsZXRlZCBmcm9tIHRoZSBXb3JrZXJzIFBhbmVsIG9uIHRoZSBuZXh0IEhvdXNlIENsZWFuaW5nIGN5Y2xlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVsZXRlRUMyU3BvdEludGVycnVwdGVkV29ya2Vycz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIERldGVybWluZXMgaWYgYW55IGFjdGl2ZSBpbnN0YW5jZXMgZ3JlYXRlciB0aGFuIHRoZSB0YXJnZXQgY2FwYWNpdHkgZm9yIGVhY2ggZ3JvdXAgd2lsbCBiZSB0ZXJtaW5hdGVkLlxuICAgKiBXb3JrZXJzIG1heSBiZSB0ZXJtaW5hdGVkIGV2ZW4gd2hpbGUgcmVuZGVyaW5nLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgc3RyaWN0SGFyZENhcD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBTcG90IEV2ZW50IFBsdWdpbiB3aWxsIHJlcXVlc3QgdGhpcyBtYXhpbXVtIG51bWJlciBvZiBpbnN0YW5jZXMgcGVyIEhvdXNlIENsZWFuaW5nIGN5Y2xlLlxuICAgKlxuICAgKiBAZGVmYXVsdCA1MFxuICAgKi9cbiAgcmVhZG9ubHkgbWF4aW11bUluc3RhbmNlc1N0YXJ0ZWRQZXJDeWNsZT86IG51bWJlcjtcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyBob3cgdGhlIFNwb3QgRXZlbnQgUGx1Z2luIHNob3VsZCBoYW5kbGUgUHJlIEpvYiBUYXNrcy5cbiAgICogU2VlIGh0dHBzOi8vZG9jcy50aGlua2JveHNvZnR3YXJlLmNvbS9wcm9kdWN0cy9kZWFkbGluZS8xMC4xLzFfVXNlciUyME1hbnVhbC9tYW51YWwvam9iLXNjcmlwdHMuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCBTcG90RXZlbnRQbHVnaW5QcmVKb2JUYXNrTW9kZS5DT05TRVJWQVRJVkVcbiAgICovXG4gIHJlYWRvbmx5IHByZUpvYlRhc2tNb2RlPzogU3BvdEV2ZW50UGx1Z2luUHJlSm9iVGFza01vZGU7XG5cbiAgLyoqXG4gICAqIFRoZSBXb3JrZXIgRXh0cmEgSW5mbyBjb2x1bW4gdG8gYmUgdXNlZCB0byBkaXNwbGF5IEFXUyBJbnN0YW5jZSBTdGF0dXNcbiAgICogaWYgdGhlIGluc3RhbmNlIGhhcyBiZWVuIG1hcmtlZCB0byBiZSBzdG9wcGVkIG9yIHRlcm1pbmF0ZWQgYnkgRUMyIG9yIFNwb3QgRXZlbnQgUGx1Z2luLlxuICAgKiBBbGwgdGltZXN0YW1wcyBhcmUgZGlzcGxheWVkIGluIFVUQyBmb3JtYXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IFNwb3RFdmVudFBsdWdpbkRpc3BsYXlJbnN0YW5jZVN0YXR1cy5ESVNBQkxFRFxuICAgKi9cbiAgcmVhZG9ubHkgYXdzSW5zdGFuY2VTdGF0dXM/OiBTcG90RXZlbnRQbHVnaW5EaXNwbGF5SW5zdGFuY2VTdGF0dXM7XG59XG5cbi8qKlxuICogSW5wdXQgcHJvcGVydGllcyBmb3IgQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbmZpZ3VyZVNwb3RFdmVudFBsdWdpblByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBWUEMgaW4gd2hpY2ggdG8gY3JlYXRlIHRoZSBuZXR3b3JrIGVuZHBvaW50IGZvciB0aGUgbGFtYmRhIGZ1bmN0aW9uIHRoYXQgaXNcbiAgICogY3JlYXRlZCBieSB0aGlzIGNvbnN0cnVjdC5cbiAgICovXG4gIHJlYWRvbmx5IHZwYzogSVZwYztcblxuICAvKipcbiAgICogVGhlIFJlbmRlclF1ZXVlIHRoYXQgV29ya2VyIGZsZWV0IHNob3VsZCBjb25uZWN0IHRvLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVuZGVyUXVldWU6IElSZW5kZXJRdWV1ZTtcblxuICAvKipcbiAgICogV2hlcmUgd2l0aGluIHRoZSBWUEMgdG8gcGxhY2UgdGhlIGxhbWJkYSBmdW5jdGlvbidzIGVuZHBvaW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCBUaGUgaW5zdGFuY2UgaXMgcGxhY2VkIHdpdGhpbiBhIFByaXZhdGUgc3VibmV0LlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjU3VibmV0cz86IFN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogVGhlIGFycmF5IG9mIFNwb3QgRXZlbnQgUGx1Z2luIHNwb3QgZmxlZXRzIHVzZWQgdG8gZ2VuZXJhdGUgdGhlIG1hcHBpbmcgYmV0d2VlbiBncm91cHMgYW5kIHNwb3QgZmxlZXQgcmVxdWVzdHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IFNwb3QgRmxlZXQgUmVxdWVzdCBDb25maWd1cmF0aW9ucyB3aWxsIG5vdCBiZSB1cGRhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgc3BvdEZsZWV0cz86IFNwb3RFdmVudFBsdWdpbkZsZWV0W107XG5cbiAgLyoqXG4gICAqIFRoZSBTcG90IEV2ZW50IFBsdWdpbiBzZXR0aW5ncy5cbiAgICogU2VlIGh0dHBzOi8vZG9jcy50aGlua2JveHNvZnR3YXJlLmNvbS9wcm9kdWN0cy9kZWFkbGluZS8xMC4xLzFfVXNlciUyME1hbnVhbC9tYW51YWwvZXZlbnQtc3BvdC5odG1sI2V2ZW50LXBsdWdpbi1jb25maWd1cmF0aW9uLW9wdGlvbnNcbiAgICpcbiAgICogQGRlZmF1bHQgRGVmYXVsdCB2YWx1ZXMgb2YgU3BvdEV2ZW50UGx1Z2luU2V0dGluZ3Mgd2lsbCBiZSBzZXQuXG4gICAqL1xuICByZWFkb25seSBjb25maWd1cmF0aW9uPzogU3BvdEV2ZW50UGx1Z2luU2V0dGluZ3M7XG59XG5cbi8qKlxuICogVGhpcyBjb25zdHJ1Y3QgY29uZmlndXJlcyB0aGUgRGVhZGxpbmUgU3BvdCBFdmVudCBQbHVnaW4gdG8gZGVwbG95IGFuZCBhdXRvLXNjYWxlIG9uZSBvciBtb3JlIHNwb3QgZmxlZXRzLlxuICpcbiAqIEZvciBleGFtcGxlLCB0byBjb25maWd1cmUgdGhlIFNwb3QgRXZlbnQgUGx1Z2luIHdpdGggb25lIHNwb3QgZmxlZXQ6XG4gKlxuICogYGBgdHNcbiAqIGltcG9ydCB7IEFwcCwgU3RhY2ssIFZwYyB9IGZyb20gJ0Bhd3MtcmZkay9jb3JlJztcbiAqIGltcG9ydCB7IEluc3RhbmNlQ2xhc3MsIEluc3RhbmNlU2l6ZSwgSW5zdGFuY2VUeXBlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG4gKiBpbXBvcnQgeyBBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlLCBDb25maWd1cmVTcG90RXZlbnRQbHVnaW4sIFJlbmRlclF1ZXVlLCBSZXBvc2l0b3J5LCBTcG90RXZlbnRQbHVnaW5GbGVldCwgVGhpbmtib3hEb2NrZXJJbWFnZXMsIFZlcnNpb25RdWVyeSB9IGZyb20gJ0Bhd3MtcmZkay9kZWFkbGluZSc7XG4gKiBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gKiBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdTdGFjaycpO1xuICogY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZwYycpO1xuICogY29uc3QgdmVyc2lvbiA9IG5ldyBWZXJzaW9uUXVlcnkoc3RhY2ssICdWZXJzaW9uJywge1xuICogICB2ZXJzaW9uOiAnMTAuMS4xMicsXG4gKiB9KTtcbiAqIGNvbnN0IGltYWdlcyA9IG5ldyBUaGlua2JveERvY2tlckltYWdlcyhzdGFjaywgJ0ltYWdlJywge1xuICogICB2ZXJzaW9uLFxuICogICAvLyBDaGFuZ2UgdGhpcyB0byBBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlLlVTRVJfQUNDRVBUU19BV1NfVEhJTktCT1hfRVVMQSB0byBhY2NlcHQgdGhlIHRlcm1zXG4gKiAgIC8vIG9mIHRoZSBBV1MgVGhpbmtib3ggRW5kIFVzZXIgTGljZW5zZSBBZ3JlZW1lbnRcbiAqICAgdXNlckF3c1RoaW5rYm94RXVsYUFjY2VwdGFuY2U6IEF3c1RoaW5rYm94RXVsYUFjY2VwdGFuY2UuVVNFUl9SRUpFQ1RTX0FXU19USElOS0JPWF9FVUxBLFxuICogfSk7XG4gKiBjb25zdCByZXBvc2l0b3J5ID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdSZXBvc2l0b3J5Jywge1xuICogICB2cGMsXG4gKiAgIHZlcnNpb24sXG4gKiB9KTtcbiAqIGNvbnN0IHJlbmRlclF1ZXVlID0gbmV3IFJlbmRlclF1ZXVlKHN0YWNrLCAnUmVuZGVyUXVldWUnLCB7XG4gKiAgIHZwYyxcbiAqICAgaW1hZ2VzOiBpbWFnZXMuZm9yUmVuZGVyUXVldWUoKSxcbiAqICAgcmVwb3NpdG9yeTogcmVwb3NpdG9yeSxcbiAqIH0pO1xuICpcbiAqIGNvbnN0IGZsZWV0ID0gbmV3IFNwb3RFdmVudFBsdWdpbkZsZWV0KHRoaXMsICdTcG90RXZlbnRQbHVnaW5GbGVldCcsIHtcbiAqICAgdnBjLFxuICogICByZW5kZXJRdWV1ZSxcbiAqICAgZGVhZGxpbmVHcm91cHM6IFsnZ3JvdXBfbmFtZSddLFxuICogICBpbnN0YW5jZVR5cGVzOiBbSW5zdGFuY2VUeXBlLm9mKEluc3RhbmNlQ2xhc3MuVDMsIEluc3RhbmNlU2l6ZS5MQVJHRSldLFxuICogICB3b3JrZXJNYWNoaW5lSW1hZ2U6IG5ldyBHZW5lcmljTGludXhJbWFnZSh7J3VzLXdlc3QtMic6ICdhbWktMDM5ZjBjMWZhYmEyOGIwMTUnfSksXG4gKiAgIG5heENhcGFjaXR5OiAxLFxuICogfSk7XG4gKlxuICogY29uc3Qgc3BvdEV2ZW50UGx1Z2luQ29uZmlnID0gbmV3IENvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbih0aGlzLCAnQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luJywge1xuICogICB2cGMsXG4gKiAgIHJlbmRlclF1ZXVlOiByZW5kZXJRdWV1ZSxcbiAqICAgc3BvdEZsZWV0czogW2ZsZWV0XSxcbiAqICAgY29uZmlndXJhdGlvbjoge1xuICogICAgIGVuYWJsZVJlc291cmNlVHJhY2tlcjogdHJ1ZSxcbiAqICAgfSxcbiAqIH0pO1xuICogYGBgXG4gKlxuICogVG8gcHJvdmlkZSB0aGlzIGZ1bmN0aW9uYWxpdHksIHRoaXMgY29uc3RydWN0IHdpbGwgY3JlYXRlIGFuIEFXUyBMYW1iZGEgZnVuY3Rpb24gdGhhdCBpcyBncmFudGVkIHRoZSBhYmlsaXR5XG4gKiB0byBjb25uZWN0IHRvIHRoZSByZW5kZXIgcXVldWUuIFRoaXMgbGFtYmRhIGlzIHJ1biBhdXRvbWF0aWNhbGx5IHdoZW4geW91IGRlcGxveSBvciB1cGRhdGUgdGhlIHN0YWNrIGNvbnRhaW5pbmcgdGhpcyBjb25zdHJ1Y3QuXG4gKiBMb2dzIGZvciBhbGwgQVdTIExhbWJkYXMgYXJlIGF1dG9tYXRpY2FsbHkgcmVjb3JkZWQgaW4gQW1hem9uIENsb3VkV2F0Y2guXG4gKlxuICogVGhpcyBjb25zdHJ1Y3Qgd2lsbCBjb25maWd1cmUgdGhlIFNwb3QgRXZlbnQgUGx1Z2luLCBidXQgdGhlIFNwb3QgRmxlZXQgUmVxdWVzdHMgd2lsbCBub3QgYmUgY3JlYXRlZCB1bmxlc3MgeW91OlxuICogLSBTdWJtaXQgdGhlIGpvYiB3aXRoIHRoZSBhc3NpZ25lZCBEZWFkbGluZSBHcm91cCBhbmQgRGVhZGxpbmUgUG9vbC4gU2VlIFtEZWFkbGluZSBEb2N1bWVudGF0aW9uXShodHRwczovL2RvY3MudGhpbmtib3hzb2Z0d2FyZS5jb20vcHJvZHVjdHMvZGVhZGxpbmUvMTAuMS8xX1VzZXIlMjBNYW51YWwvbWFudWFsL2pvYi1zdWJtaXR0aW5nLmh0bWwjc3VibWl0dGluZy1qb2JzKS5cbiAqXG4gKiBJbXBvcnRhbnQ6IERpc2FibGUgJ0FsbG93IFdvcmtlcnMgdG8gUGVyZm9ybSBIb3VzZSBDbGVhbmluZyBJZiBQdWxzZSBpcyBub3QgUnVubmluZycgaW4gdGhlICdDb25maWd1cmUgUmVwb3NpdG9yeSBPcHRpb25zJ1xuICogd2hlbiB1c2luZyBTcG90IEV2ZW50IFBsdWdpbi5cbiAqIFNlZSBodHRwczovL2RvY3MudGhpbmtib3hzb2Z0d2FyZS5jb20vcHJvZHVjdHMvZGVhZGxpbmUvMTAuMS8xX1VzZXIlMjBNYW51YWwvbWFudWFsL2V2ZW50LXNwb3QuaHRtbCNwcmVyZXF1aXNpdGVzXG4gKlxuICogSW1wb3J0YW50OiBBbnkgcmVzb3VyY2VzIGNyZWF0ZWQgYnkgdGhlIFNwb3QgRXZlbnQgUGx1Z2luIHdpbGwgbm90IGJlIGRlbGV0ZWQgd2l0aCAnY2RrIGRlc3Ryb3knLlxuICogTWFrZSBzdXJlIHRoYXQgYWxsIHN1Y2ggcmVzb3VyY2VzIChlLmcuIFNwb3QgRmxlZXQgUmVxdWVzdCBvciBGbGVldCBJbnN0YW5jZXMpIGFyZSBjbGVhbmVkIHVwLCBiZWZvcmUgZGVzdHJveWluZyB0aGUgc3RhY2tzLlxuICogRGlzYWJsZSB0aGUgU3BvdCBFdmVudCBQbHVnaW4gYnkgc2V0dGluZyAnc3RhdGUnIHByb3BlcnR5IHRvICdTcG90RXZlbnRQbHVnaW5TdGF0ZS5ESVNBQkxFRCcgb3IgdmlhIERlYWRsaW5lIE1vbml0b3IsXG4gKiBlbnN1cmUgeW91IHNodXRkb3duIGFsbCBQdWxzZSBpbnN0YW5jZXMgYW5kIHRoZW4gdGVybWluYXRlIGFueSBTcG90IEZsZWV0IFJlcXVlc3RzIGluIHRoZSBBV1MgRUMyIEluc3RhbmNlIENvbnNvbGUuXG4gKlxuICogTm90ZSB0aGF0IHRoaXMgY29uc3RydWN0IGFkZHMgYWRkaXRpb25hbCBwb2xpY2llcyB0byB0aGUgUmVuZGVyIFF1ZXVlJ3Mgcm9sZVxuICogcmVxdWlyZWQgdG8gcnVuIHRoZSBTcG90IEV2ZW50IFBsdWdpbiBhbmQgbGF1bmNoIGEgUmVzb3VyY2UgVHJhY2tlcjpcbiAqICAtIEFXU1RoaW5rYm94RGVhZGxpbmVTcG90RXZlbnRQbHVnaW5BZG1pblBvbGljeVxuICogIC0gQVdTVGhpbmtib3hEZWFkbGluZVJlc291cmNlVHJhY2tlckFkbWluUG9saWN5XG4gKiAgLSBBIHBvbGljeSB0byBwYXNzIGEgZmxlZXQgYW5kIGluc3RhbmNlIHJvbGVcbiAqICAtIEEgcG9saWN5IHRvIGNyZWF0ZSB0YWdzIGZvciBzcG90IGZsZWV0IHJlcXVlc3RzXG4gKlxuICogVGhlIFNwb3QgRmxlZXQgUmVxdWVzdHMgdGhhdCB0aGlzIGNvbnN0cnVjdCBjb25maWd1cmVzIERlYWRsaW5lIHRvIGNyZWF0ZSB3aWxsIGFsd2F5cyB1c2UgdGhlIGxhdGVzdCB2ZXJzaW9uIG9mIHRoZVxuICogY29ycmVzcG9uZGluZyBFQzIgTGF1bmNoIFRlbXBsYXRlIHRoYXQgd2FzIGNyZWF0ZWQgZm9yIHRoZW0uXG4gKlxuICogIVthcmNoaXRlY3R1cmUgZGlhZ3JhbV0oL2RpYWdyYW1zL2RlYWRsaW5lL0NvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbi5zdmcpXG4gKlxuICogUmVzb3VyY2VzIERlcGxveWVkXG4gKiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAqIC0gQW4gQVdTIExhbWJkYSB0aGF0IGlzIHVzZWQgdG8gY29ubmVjdCB0byB0aGUgcmVuZGVyIHF1ZXVlLCBhbmQgc2F2ZSBTcG90IEV2ZW50IFBsdWdpbiBjb25maWd1cmF0aW9ucy5cbiAqIC0gQSBDbG91ZEZvcm1hdGlvbiBDdXN0b20gUmVzb3VyY2UgdGhhdCB0cmlnZ2VycyBleGVjdXRpb24gb2YgdGhlIExhbWJkYSBvbiBzdGFjayBkZXBsb3ltZW50LCB1cGRhdGUsIGFuZCBkZWxldGlvbi5cbiAqIC0gQW4gQW1hem9uIENsb3VkV2F0Y2ggbG9nIGdyb3VwIHRoYXQgcmVjb3JkcyBoaXN0b3J5IG9mIHRoZSBBV1MgTGFtYmRhJ3MgZXhlY3V0aW9uLlxuICogLSBBbiBJQU0gUG9saWN5IGF0dGFjaGVkIHRvIFJlbmRlciBRdWV1ZSdzIFJvbGUuXG4gKiAtIEVDMiBMYXVuY2ggVGVtcGxhdGVzIGZvciBlYWNoIFNwb3QgRXZlbnQgUGx1Z2luIGZsZWV0LlxuICpcbiAqIFNlY3VyaXR5IENvbnNpZGVyYXRpb25zXG4gKiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAqIC0gVGhlIEFXUyBMYW1iZGEgdGhhdCBpcyBkZXBsb3llZCB0aHJvdWdoIHRoaXMgY29uc3RydWN0IHdpbGwgYmUgY3JlYXRlZCBmcm9tIGEgZGVwbG95bWVudCBwYWNrYWdlXG4gKiAgIHRoYXQgaXMgdXBsb2FkZWQgdG8geW91ciBDREsgYm9vdHN0cmFwIGJ1Y2tldCBkdXJpbmcgZGVwbG95bWVudC4gWW91IG11c3QgbGltaXQgd3JpdGUgYWNjZXNzIHRvXG4gKiAgIHlvdXIgQ0RLIGJvb3RzdHJhcCBidWNrZXQgdG8gcHJldmVudCBhbiBhdHRhY2tlciBmcm9tIG1vZGlmeWluZyB0aGUgYWN0aW9ucyBwZXJmb3JtZWQgYnkgdGhpcyBMYW1iZGEuXG4gKiAgIFdlIHN0cm9uZ2x5IHJlY29tbWVuZCB0aGF0IHlvdSBlaXRoZXIgZW5hYmxlIEFtYXpvbiBTMyBzZXJ2ZXIgYWNjZXNzIGxvZ2dpbmcgb24geW91ciBDREsgYm9vdHN0cmFwIGJ1Y2tldCxcbiAqICAgb3IgZW5hYmxlIEFXUyBDbG91ZFRyYWlsIG9uIHlvdXIgYWNjb3VudCB0byBhc3Npc3QgaW4gcG9zdC1pbmNpZGVudCBhbmFseXNpcyBvZiBjb21wcm9taXNlZCBwcm9kdWN0aW9uXG4gKiAgIGVudmlyb25tZW50cy5cbiAqIC0gVGhlIEFXUyBMYW1iZGEgZnVuY3Rpb24gdGhhdCBpcyBjcmVhdGVkIGJ5IHRoaXMgcmVzb3VyY2UgaGFzIGFjY2VzcyB0byBib3RoIHRoZSBjZXJ0aWZpY2F0ZXMgdXNlZCB0byBjb25uZWN0IHRvIHRoZSByZW5kZXIgcXVldWUsXG4gKiAgIGFuZCB0aGUgcmVuZGVyIHF1ZXVlIHBvcnQuIEFuIGF0dGFja2VyIHRoYXQgY2FuIGZpbmQgYSB3YXkgdG8gbW9kaWZ5IGFuZCBleGVjdXRlIHRoaXMgbGFtYmRhIGNvdWxkIHVzZSBpdCB0b1xuICogICBleGVjdXRlIGFueSByZXF1ZXRzIGFnYWluc3QgdGhlIHJlbmRlciBxdWV1ZS4gWW91IHNob3VsZCBub3QgZ3JhbnQgYW55IGFkZGl0aW9uYWwgYWN0b3JzL3ByaW5jaXBhbHMgdGhlIGFiaWxpdHkgdG8gbW9kaWZ5XG4gKiAgIG9yIGV4ZWN1dGUgdGhpcyBMYW1iZGEuXG4gKi9cbmV4cG9ydCBjbGFzcyBDb25maWd1cmVTcG90RXZlbnRQbHVnaW4gZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuXG4gIC8qKlxuICAgKiBPbmx5IG9uZSBTcG90IEV2ZW50IFBsdWdpbiBDb25maWd1cmF0aW9uIGlzIGFsbG93ZWQgcGVyIHJlbmRlciBxdWV1ZSAvIHJlcG9zaXRvcnkuXG4gICAqL1xuICBwcml2YXRlIHN0YXRpYyB1bmlxdWVSZW5kZXJRdWV1ZXM6IFNldDxJUmVuZGVyUXVldWU+ID0gbmV3IFNldDxJUmVuZGVyUXVldWU+KCk7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENvbmZpZ3VyZVNwb3RFdmVudFBsdWdpblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmIChDb25maWd1cmVTcG90RXZlbnRQbHVnaW4udW5pcXVlUmVuZGVyUXVldWVzLmhhcyhwcm9wcy5yZW5kZXJRdWV1ZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBvbmUgQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luIGNvbnN0cnVjdCBpcyBhbGxvd2VkIHBlciByZW5kZXIgcXVldWUuJyk7XG4gICAgfVxuICAgIGVsc2Uge1xuICAgICAgQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luLnVuaXF1ZVJlbmRlclF1ZXVlcy5hZGQocHJvcHMucmVuZGVyUXVldWUpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5yZW5kZXJRdWV1ZSBpbnN0YW5jZW9mIFJlbmRlclF1ZXVlKSB7XG4gICAgICAvLyBXZSBkbyBub3QgY2hlY2sgdGhlIHBhdGNoIHZlcnNpb24sIHNvIGl0J3Mgc2V0IHRvIDAuXG4gICAgICBjb25zdCBtaW5pbXVtVmVyc2lvbjogVmVyc2lvbiA9IG5ldyBWZXJzaW9uKFsxMCwgMSwgMTIsIDBdKTtcblxuICAgICAgaWYgKHByb3BzLnJlbmRlclF1ZXVlLnZlcnNpb24uaXNMZXNzVGhhbihtaW5pbXVtVmVyc2lvbikpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBNaW5pbXVtIHN1cHBvcnRlZCBEZWFkbGluZSB2ZXJzaW9uIGZvciAke3RoaXMuY29uc3RydWN0b3IubmFtZX0gaXMgYCArXG4gICAgICAgIGAke21pbmltdW1WZXJzaW9uLnZlcnNpb25TdHJpbmd9LiBgICtcbiAgICAgICAgYFJlY2VpdmVkOiAke3Byb3BzLnJlbmRlclF1ZXVlLnZlcnNpb24udmVyc2lvblN0cmluZ30uYCk7XG4gICAgICB9XG5cbiAgICAgIGlmIChwcm9wcy5zcG90RmxlZXRzICYmIHByb3BzLnNwb3RGbGVldHMubGVuZ3RoICE9PSAwKSB7XG4gICAgICAgIC8vIEFsd2F5cyBhZGQgUmVzb3VyY2UgVHJhY2tlciBhZG1pbiBwb2xpY3ksIGV2ZW4gaWYgcHJvcHMuY29uZmlndXJhdGlvbj8uZW5hYmxlUmVzb3VyY2VUcmFja2VyIGlzIGZhbHNlLlxuICAgICAgICAvLyBUaGlzIGltcHJvdmVzIHVzYWJpbGl0eSwgYXMgY3VzdG9tZXJzIHdvbid0IG5lZWQgdG8gYWRkIHRoaXMgcG9saWN5IG1hbnVhbGx5LCBpZiB0aGV5XG4gICAgICAgIC8vIGVuYWJsZSBSZXNvdXJjZSBUcmFja2VyIGxhdGVyIGluIHRoZSBTcG90IEV2ZW50IFBsdWdpbiBjb25maWd1cmF0aW9uIChlLmcuLCB1c2luZyBEZWFkbGluZSBNb25pdG9yIGFuZCBub3QgUkZESykuXG4gICAgICAgIHByb3BzLnJlbmRlclF1ZXVlLmFkZFNFUFBvbGljaWVzKHRydWUpO1xuXG4gICAgICAgIGNvbnN0IGZsZWV0Um9sZXMgPSBwcm9wcy5zcG90RmxlZXRzLm1hcChzZiA9PiBzZi5mbGVldFJvbGUucm9sZUFybik7XG4gICAgICAgIGNvbnN0IGZsZWV0SW5zdGFuY2VSb2xlcyA9IHByb3BzLnNwb3RGbGVldHMubWFwKHNmID0+IHNmLmZsZWV0SW5zdGFuY2VSb2xlLnJvbGVBcm4pO1xuICAgICAgICBuZXcgUG9saWN5KHRoaXMsICdTcG90RXZlbnRQbHVnaW5Qb2xpY3knLCB7XG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgICAnaWFtOlBhc3NSb2xlJyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbLi4uZmxlZXRSb2xlcywgLi4uZmxlZXRJbnN0YW5jZVJvbGVzXSxcbiAgICAgICAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgICAgICAgIFN0cmluZ0xpa2U6IHtcbiAgICAgICAgICAgICAgICAgICdpYW06UGFzc2VkVG9TZXJ2aWNlJzogJ2VjMi5hbWF6b25hd3MuY29tJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgICdlYzI6Q3JlYXRlVGFncycsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgICdhcm46YXdzOmVjMjoqOio6c3BvdC1mbGVldC1yZXF1ZXN0LyonLFxuICAgICAgICAgICAgICAgICdhcm46YXdzOmVjMjoqOio6dm9sdW1lLyonLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgICByb2xlczogW1xuICAgICAgICAgICAgcHJvcHMucmVuZGVyUXVldWUuZ3JhbnRQcmluY2lwYWwgYXMgUm9sZSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICAgZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoZSBwcm92aWRlZCByZW5kZXIgcXVldWUgaXMgbm90IGFuIGluc3RhbmNlIG9mIFJlbmRlclF1ZXVlIGNsYXNzLiBTb21lIGZ1bmN0aW9uYWxpdHkgaXMgbm90IHN1cHBvcnRlZC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCByZWdpb24gPSBDb25zdHJ1Y3QuaXNDb25zdHJ1Y3QocHJvcHMucmVuZGVyUXVldWUpID8gU3RhY2sub2YocHJvcHMucmVuZGVyUXVldWUpLnJlZ2lvbiA6IFN0YWNrLm9mKHRoaXMpLnJlZ2lvbjtcblxuICAgIGNvbnN0IHRpbWVvdXRNaW5zID0gMTU7XG4gICAgY29uc3QgY29uZmlndXJhdG9yID0gbmV3IExhbWJkYUZ1bmN0aW9uKHRoaXMsICdDb25maWd1cmF0b3InLCB7XG4gICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICAgIHZwY1N1Ym5ldHM6IHByb3BzLnZwY1N1Ym5ldHMgPz8geyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEVfV0lUSF9OQVQgfSxcbiAgICAgIGRlc2NyaXB0aW9uOiBgVXNlZCBieSBhIENvbmZpZ3VyZVNwb3RFdmVudFBsdWdpbiAke3RoaXMubm9kZS5hZGRyfSB0byBwZXJmb3JtIGNvbmZpZ3VyYXRpb24gb2YgRGVhZGxpbmUgU3BvdCBFdmVudCBQbHVnaW5gLFxuICAgICAgY29kZTogQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJy4uJywgJ2xhbWJkYXMnLCAnbm9kZWpzJyksIHtcbiAgICAgIH0pLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgREVCVUc6ICdmYWxzZScsXG4gICAgICAgIExBTUJEQV9USU1FT1VUX01JTlM6IHRpbWVvdXRNaW5zLnRvU3RyaW5nKCksXG4gICAgICB9LFxuICAgICAgcnVudGltZTogUnVudGltZS5OT0RFSlNfMTZfWCxcbiAgICAgIGhhbmRsZXI6ICdjb25maWd1cmUtc3BvdC1ldmVudC1wbHVnaW4uY29uZmlndXJlU0VQJyxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXModGltZW91dE1pbnMpLFxuICAgICAgbG9nUmV0ZW50aW9uOiBSZXRlbnRpb25EYXlzLk9ORV9XRUVLLFxuICAgIH0pO1xuXG4gICAgY29uZmlndXJhdG9yLmNvbm5lY3Rpb25zLmFsbG93VG9EZWZhdWx0UG9ydChwcm9wcy5yZW5kZXJRdWV1ZSk7XG4gICAgcHJvcHMucmVuZGVyUXVldWUuY2VydENoYWluPy5ncmFudFJlYWQoY29uZmlndXJhdG9yLmdyYW50UHJpbmNpcGFsKTtcblxuICAgIGNvbnN0IHBsdWdpbkNvbmZpZzogUGx1Z2luU2V0dGluZ3MgPSB7XG4gICAgICBBV1NJbnN0YW5jZVN0YXR1czogcHJvcHMuY29uZmlndXJhdGlvbj8uYXdzSW5zdGFuY2VTdGF0dXMgPz8gU3BvdEV2ZW50UGx1Z2luRGlzcGxheUluc3RhbmNlU3RhdHVzLkRJU0FCTEVELFxuICAgICAgRGVsZXRlSW50ZXJydXB0ZWRTbGF2ZXM6IHByb3BzLmNvbmZpZ3VyYXRpb24/LmRlbGV0ZUVDMlNwb3RJbnRlcnJ1cHRlZFdvcmtlcnMgPz8gZmFsc2UsXG4gICAgICBEZWxldGVUZXJtaW5hdGVkU2xhdmVzOiBwcm9wcy5jb25maWd1cmF0aW9uPy5kZWxldGVTRVBUZXJtaW5hdGVkV29ya2VycyA/PyBmYWxzZSxcbiAgICAgIElkbGVTaHV0ZG93bjogcHJvcHMuY29uZmlndXJhdGlvbj8uaWRsZVNodXRkb3duPy50b01pbnV0ZXMoe2ludGVncmFsOiB0cnVlfSkgPz8gMTAsXG4gICAgICBMb2dnaW5nOiBwcm9wcy5jb25maWd1cmF0aW9uPy5sb2dnaW5nTGV2ZWwgPz8gU3BvdEV2ZW50UGx1Z2luTG9nZ2luZ0xldmVsLlNUQU5EQVJELFxuICAgICAgUHJlSm9iVGFza01vZGU6IHByb3BzLmNvbmZpZ3VyYXRpb24/LnByZUpvYlRhc2tNb2RlID8/IFNwb3RFdmVudFBsdWdpblByZUpvYlRhc2tNb2RlLkNPTlNFUlZBVElWRSxcbiAgICAgIFJlZ2lvbjogcHJvcHMuY29uZmlndXJhdGlvbj8ucmVnaW9uID8/IHJlZ2lvbixcbiAgICAgIFJlc291cmNlVHJhY2tlcjogcHJvcHMuY29uZmlndXJhdGlvbj8uZW5hYmxlUmVzb3VyY2VUcmFja2VyID8/IHRydWUsXG4gICAgICBTdGFnZ2VySW5zdGFuY2VzOiBwcm9wcy5jb25maWd1cmF0aW9uPy5tYXhpbXVtSW5zdGFuY2VzU3RhcnRlZFBlckN5Y2xlID8/IDUwLFxuICAgICAgU3RhdGU6IHByb3BzLmNvbmZpZ3VyYXRpb24/LnN0YXRlID8/IFNwb3RFdmVudFBsdWdpblN0YXRlLkdMT0JBTF9FTkFCTEVELFxuICAgICAgU3RyaWN0SGFyZENhcDogcHJvcHMuY29uZmlndXJhdGlvbj8uc3RyaWN0SGFyZENhcCA/PyBmYWxzZSxcbiAgICB9O1xuICAgIGNvbnN0IHNwb3RGbGVldFJlcXVlc3RDb25maWdzID0gdGhpcy5tZXJnZVNwb3RGbGVldFJlcXVlc3RDb25maWdzKHByb3BzLnNwb3RGbGVldHMpO1xuXG4gICAgY29uc3QgZGVhZGxpbmVHcm91cHMgPSBBcnJheS5mcm9tKG5ldyBTZXQocHJvcHMuc3BvdEZsZWV0cz8ubWFwKGZsZWV0ID0+IGZsZWV0LmRlYWRsaW5lR3JvdXBzKS5yZWR1Y2UoKHAsIGMpID0+IHAuY29uY2F0KGMpLCBbXSkpKTtcbiAgICBjb25zdCBkZWFkbGluZVBvb2xzID0gQXJyYXkuZnJvbShuZXcgU2V0KHByb3BzLnNwb3RGbGVldHM/Lm1hcChmbGVldCA9PiBmbGVldC5kZWFkbGluZVBvb2xzKS5yZWR1Y2UoKHAsIGMpID0+IHA/LmNvbmNhdChjID8/IFtdKSwgW10pKSk7XG4gICAgY29uc3QgcHJvcGVydGllczogU0VQQ29uZmlndXJhdG9yUmVzb3VyY2VQcm9wcyA9IHtcbiAgICAgIGNvbm5lY3Rpb246IHtcbiAgICAgICAgaG9zdG5hbWU6IHByb3BzLnJlbmRlclF1ZXVlLmVuZHBvaW50Lmhvc3RuYW1lLFxuICAgICAgICBwb3J0OiBwcm9wcy5yZW5kZXJRdWV1ZS5lbmRwb2ludC5wb3J0QXNTdHJpbmcoKSxcbiAgICAgICAgcHJvdG9jb2w6IHByb3BzLnJlbmRlclF1ZXVlLmVuZHBvaW50LmFwcGxpY2F0aW9uUHJvdG9jb2wsXG4gICAgICAgIGNhQ2VydGlmaWNhdGVBcm46IHByb3BzLnJlbmRlclF1ZXVlLmNlcnRDaGFpbj8uc2VjcmV0QXJuLFxuICAgICAgfSxcbiAgICAgIHNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9uczogc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3MsXG4gICAgICBzcG90UGx1Z2luQ29uZmlndXJhdGlvbnM6IHBsdWdpbkNvbmZpZyxcbiAgICAgIGRlYWRsaW5lR3JvdXBzLFxuICAgICAgZGVhZGxpbmVQb29scyxcbiAgICB9O1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0RlZmF1bHQnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IGNvbmZpZ3VyYXRvci5mdW5jdGlvbkFybixcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6UkZES19Db25maWd1cmVTcG90RXZlbnRQbHVnaW4nLFxuICAgICAgcHJvcGVydGllcyxcbiAgICB9KTtcblxuICAgIC8vIFByZXZlbnRzIGEgcmFjZSBkdXJpbmcgYSBzdGFjay11cGRhdGUuXG4gICAgcmVzb3VyY2Uubm9kZS5hZGREZXBlbmRlbmN5KGNvbmZpZ3VyYXRvci5yb2xlISk7XG5cbiAgICAvLyBXZSBuZWVkIHRvIGFkZCB0aGlzIGRlcGVuZGVuY3kgdG8gYXZvaWQgZmFpbHVyZXMgd2hpbGUgZGVsZXRpbmcgYSBDdXN0b20gUmVzb3VyY2U6XG4gICAgLy8gJ0N1c3RvbSBSZXNvdXJjZSBmYWlsZWQgdG8gc3RhYmlsaXplIGluIGV4cGVjdGVkIHRpbWUuIElmIHlvdSBhcmUgdXNpbmcgdGhlIFB5dGhvbiBjZm4tcmVzcG9uc2UgbW9kdWxlLFxuICAgIC8vIHlvdSBtYXkgbmVlZCB0byB1cGRhdGUgeW91ciBMYW1iZGEgZnVuY3Rpb24gY29kZSBzbyB0aGF0IENsb3VkRm9ybWF0aW9uIGNhbiBhdHRhY2ggdGhlIHVwZGF0ZWQgdmVyc2lvbi4nLlxuICAgIC8vIFRoaXMgaGFwcGVucywgYmVjYXVzZSBSb3V0ZSBUYWJsZSBBc3NvY2lhdGlvbnMgYXJlIGRlbGV0ZWQgYmVmb3JlIHRoZSBDdXN0b20gUmVzb3VyY2UgYW5kIHdlXG4gICAgLy8gZG9uJ3QgZ2V0IGEgcmVzcG9uc2UgZnJvbSAnZG9EZWxldGUoKScuXG4gICAgLy8gSWRlYWxseSwgd2Ugd291bGQgb25seSB3YW50IHRvIGFkZCBkZXBlbmRlbmN5IG9uICdpbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkJyBhcyBzaG93biBiZWxvdyxcbiAgICAvLyBidXQgaXQgc2VlbXMgdGhhdCBDREsgbWlzc2VzIGRlcGVuZGVuY2llcyBvbiBSb3V0ZSBUYWJsZSBBc3NvY2lhdGlvbnMgaW4gdGhhdCBjYXNlOlxuICAgIC8vIGNvbnN0IHsgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZCB9ID0gcHJvcHMudnBjLnNlbGVjdFN1Ym5ldHMocHJvcHMudnBjU3VibmV0cyk7XG4gICAgLy8gcmVzb3VyY2Uubm9kZS5hZGREZXBlbmRlbmN5KGludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQpO1xuICAgIHJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeShwcm9wcy52cGMpO1xuXG4gICAgLy8gLyogaXN0YW5idWwgaWdub3JlIG5leHQgKi9cbiAgICAvLyBBZGQgYSBkZXBlbmRlbmN5IG9uIHRoZSByZW5kZXIgcXVldWUgdG8gZW5zdXJlIHRoYXRcbiAgICAvLyBpdCBpcyBydW5uaW5nIGJlZm9yZSB3ZSB0cnkgdG8gc2VuZCByZXF1ZXN0cyB0byBpdC5cbiAgICByZXNvdXJjZS5ub2RlLmFkZERlcGVuZGVuY3kocHJvcHMucmVuZGVyUXVldWUpO1xuXG4gICAgaWYgKHByb3BzLnNwb3RGbGVldHMgJiYgcHJvcHMucmVuZGVyUXVldWUucmVwb3NpdG9yeS5zZWNyZXRzTWFuYWdlbWVudFNldHRpbmdzLmVuYWJsZWQpIHtcbiAgICAgIHByb3BzLnNwb3RGbGVldHMuZm9yRWFjaChzcG90RmxlZXQgPT4ge1xuICAgICAgICBpZiAoc3BvdEZsZWV0LmRlZmF1bHRTdWJuZXRzKSB7XG4gICAgICAgICAgQW5ub3RhdGlvbnMub2Yoc3BvdEZsZWV0KS5hZGRXYXJuaW5nKFxuICAgICAgICAgICAgJ0RlYWRsaW5lIFNlY3JldHMgTWFuYWdlbWVudCBpcyBlbmFibGVkIG9uIHRoZSBSZXBvc2l0b3J5IGFuZCBWUEMgc3VibmV0cyBoYXZlIG5vdCBiZWVuIHN1cHBsaWVkLiBVc2luZyBkZWRpY2F0ZWQgc3VibmV0cyBpcyByZWNvbW1lbmRlZC4gU2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLXJmZGsvYmxvYnMvcmVsZWFzZS9wYWNrYWdlcy9hd3MtcmZkay9saWIvZGVhZGxpbmUvUkVBRE1FLm1kI3VzaW5nLWRlZGljYXRlZC1zdWJuZXRzLWZvci1kZWFkbGluZS1jb21wb25lbnRzJyxcbiAgICAgICAgICApO1xuICAgICAgICB9XG4gICAgICAgIHByb3BzLnJlbmRlclF1ZXVlLmNvbmZpZ3VyZVNlY3JldHNNYW5hZ2VtZW50QXV0b1JlZ2lzdHJhdGlvbih7XG4gICAgICAgICAgZGVwZW5kZW50OiByZXNvdXJjZSxcbiAgICAgICAgICByb2xlOiBTZWNyZXRzTWFuYWdlbWVudFJvbGUuQ0xJRU5ULFxuICAgICAgICAgIHJlZ2lzdHJhdGlvblN0YXR1czogU2VjcmV0c01hbmFnZW1lbnRSZWdpc3RyYXRpb25TdGF0dXMuUkVHSVNURVJFRCxcbiAgICAgICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICAgICAgICB2cGNTdWJuZXRzOiBzcG90RmxlZXQuc3VibmV0cyxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gcmVzb3VyY2U7XG4gIH1cblxuICBwcml2YXRlIHRhZ1NwZWNpZmljYXRpb25zKGZsZWV0OiBTcG90RXZlbnRQbHVnaW5GbGVldCwgcmVzb3VyY2VUeXBlOiBTcG90RmxlZXRSZXNvdXJjZVR5cGUpOiBJUmVzb2x2YWJsZSB7XG4gICAgcmV0dXJuIExhenkuYW55KHtcbiAgICAgIHByb2R1Y2U6ICgpID0+IHtcbiAgICAgICAgaWYgKGZsZWV0LnRhZ3MuaGFzVGFncygpKSB7XG4gICAgICAgICAgY29uc3QgdGFnU3BlY2lmaWNhdGlvbjogU3BvdEZsZWV0VGFnU3BlY2lmaWNhdGlvbiA9IHtcbiAgICAgICAgICAgIFJlc291cmNlVHlwZTogcmVzb3VyY2VUeXBlLFxuICAgICAgICAgICAgVGFnczogZmxlZXQudGFncy5yZW5kZXJUYWdzKCksXG4gICAgICAgICAgfTtcbiAgICAgICAgICByZXR1cm4gW3RhZ1NwZWNpZmljYXRpb25dO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBTcG90IEZsZWV0IENvbmZpZ3VyYXRpb25zIGZyb20gdGhlIHByb3ZpZGVkIGZsZWV0LlxuICAgKiBFYWNoIGNvbmZpZ3VyYXRpb24gaXMgYSBtYXBwaW5nIGJldHdlZW4gb25lIERlYWRsaW5lIEdyb3VwIGFuZCBvbmUgU3BvdCBGbGVldCBSZXF1ZXN0IENvbmZpZ3VyYXRpb24uXG4gICAqL1xuICBwcml2YXRlIGdlbmVyYXRlU3BvdEZsZWV0UmVxdWVzdENvbmZpZyhmbGVldDogU3BvdEV2ZW50UGx1Z2luRmxlZXQpOiBTcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbltdIHtcbiAgICBjb25zdCBzcG90RmxlZXRSZXF1ZXN0VGFnc1Rva2VuID0gdGhpcy50YWdTcGVjaWZpY2F0aW9ucyhmbGVldCwgU3BvdEZsZWV0UmVzb3VyY2VUeXBlLlNQT1RfRkxFRVRfUkVRVUVTVCk7XG5cbiAgICBjb25zdCBzcG90RmxlZXRSZXF1ZXN0UHJvcHM6IFNwb3RGbGVldFJlcXVlc3RQcm9wcyA9IHtcbiAgICAgIEFsbG9jYXRpb25TdHJhdGVneTogZmxlZXQuYWxsb2NhdGlvblN0cmF0ZWd5LFxuICAgICAgSWFtRmxlZXRSb2xlOiBmbGVldC5mbGVldFJvbGUucm9sZUFybixcbiAgICAgIExhdW5jaFRlbXBsYXRlQ29uZmlnczogZmxlZXQuX2xhdW5jaFRlbXBsYXRlQ29uZmlncyxcbiAgICAgIFJlcGxhY2VVbmhlYWx0aHlJbnN0YW5jZXM6IHRydWUsXG4gICAgICAvLyBJbiBvcmRlciB0byB3b3JrIHdpdGggRGVhZGxpbmUsIHRoZSAnVGFyZ2V0IENhcGFjaXR5JyBvZiB0aGUgU3BvdCBmbGVldCBSZXF1ZXN0IGlzXG4gICAgICAvLyB0aGUgbWF4aW11bSBudW1iZXIgb2YgV29ya2VycyB0aGF0IERlYWRsaW5lIHdpbGwgc3RhcnQuXG4gICAgICBUYXJnZXRDYXBhY2l0eTogZmxlZXQubWF4Q2FwYWNpdHksXG4gICAgICBUZXJtaW5hdGVJbnN0YW5jZXNXaXRoRXhwaXJhdGlvbjogdHJ1ZSxcbiAgICAgIC8vIEluIG9yZGVyIHRvIHdvcmsgd2l0aCBEZWFkbGluZSwgU3BvdCBGbGVldHMgUmVxdWVzdHMgbXVzdCBiZSBzZXQgdG8gbWFpbnRhaW4uXG4gICAgICBUeXBlOiBTcG90RmxlZXRSZXF1ZXN0VHlwZS5NQUlOVEFJTixcbiAgICAgIFZhbGlkVW50aWw6IGZsZWV0LnZhbGlkVW50aWw/LmRhdGUudG9JU09TdHJpbmcoKSxcbiAgICAgIC8vIE5lZWQgdG8gY29udmVydCBmcm9tIElSZXNvbHZhYmxlIHRvIGJ5cGFzcyBUeXBlU2NyaXB0XG4gICAgICBUYWdTcGVjaWZpY2F0aW9uczogKHNwb3RGbGVldFJlcXVlc3RUYWdzVG9rZW4gYXMgdW5rbm93bikgYXMgU3BvdEZsZWV0VGFnU3BlY2lmaWNhdGlvbltdLFxuICAgIH07XG5cbiAgICBjb25zdCBzcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbnMgPSBmbGVldC5kZWFkbGluZUdyb3Vwcy5tYXAoZ3JvdXAgPT4ge1xuICAgICAgY29uc3Qgc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb246IFNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9uID0ge1xuICAgICAgICBbZ3JvdXAudG9Mb3dlckNhc2UoKV06IHNwb3RGbGVldFJlcXVlc3RQcm9wcyxcbiAgICAgIH07XG4gICAgICByZXR1cm4gc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb247XG4gICAgfSk7XG5cbiAgICByZXR1cm4gc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25zO1xuICB9XG5cbiAgcHJpdmF0ZSBtZXJnZVNwb3RGbGVldFJlcXVlc3RDb25maWdzKHNwb3RGbGVldHM/OiBTcG90RXZlbnRQbHVnaW5GbGVldFtdKTogU3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb24gfCB1bmRlZmluZWQge1xuICAgIGlmICghc3BvdEZsZWV0cyB8fCBzcG90RmxlZXRzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICBjb25zdCBmdWxsU3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb246IFNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9uID0ge307XG4gICAgc3BvdEZsZWV0cy5tYXAoZmxlZXQgPT4ge1xuICAgICAgY29uc3Qgc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25zID0gdGhpcy5nZW5lcmF0ZVNwb3RGbGVldFJlcXVlc3RDb25maWcoZmxlZXQpO1xuICAgICAgc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25zLm1hcChjb25maWd1cmF0aW9uID0+IHtcbiAgICAgICAgZm9yIChjb25zdCBba2V5LCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXMoY29uZmlndXJhdGlvbikpIHtcbiAgICAgICAgICBpZiAoa2V5IGluIGZ1bGxTcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbikge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBCYWQgR3JvdXAgTmFtZTogJHtrZXl9LiBHcm91cCBuYW1lcyBpbiBTcG90IEZsZWV0IFJlcXVlc3QgQ29uZmlndXJhdGlvbnMgc2hvdWxkIGJlIHVuaXF1ZS5gKTtcbiAgICAgICAgICB9XG4gICAgICAgICAgZnVsbFNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9uW2tleV0gPSB2YWx1ZTtcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICByZXR1cm4gZnVsbFNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9uO1xuICB9XG59XG4iXX0=