"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const x509_certificate_1 = require("../lib/x509-certificate");
test('Generate cert', () => {
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    // Expect the custom resource for cert generation
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    }));
    // Cannot have a CertificateValidFor property if not given one. Adding one
    // would cause existing certificates to be re-generated on re-deploy, and thus
    // risk breaking customer's setups.
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        CertificateValidFor: assert_1.anything(),
    }));
    // Expect the resource for converting to PKCS #12 not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
    // Expect the DynamoDB table used for custom resource tracking
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table'));
    // Expect a Secret used to store the cert
    assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret'));
    // Expect a policy that can interact with DynamoDB and SecretsManager
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-F53F5427': 'f53f5427b2e9eb4739661fcc0b249b6e',
                        },
                    },
                },
            ],
        },
    }));
    // Expect no KMS key for encryption
    assert_1.expect(stack).notTo(assert_1.haveResource('AWS::KMS::Key'));
    // Expect Lambda for doing the cert generation to use the generate() handler and openssl layer
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', (props, error) => {
        if (!props.Handler || props.Handler !== 'x509-certificate.generate') {
            error.failureReason = 'x509-certificate.generate handler not found';
            error.resource = props.Handler;
            return false;
        }
        // Our test for the correct openssl lambda layer does not include the version, so we use a filter
        // function to do a partial match
        const filterOpensslArn = (value) => {
            return value.toString().includes('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:');
        };
        if (!props.Layers
            || !Array.isArray(props.Layers)
            || Array.of(props.Layers).filter(filterOpensslArn).length === 0) {
            error.failureReason = 'openssl Lambda Layer missing';
            error.resource = props.Layers;
            return false;
        }
        if (!props.Environment
            || !props.Environment.Variables
            || !props.Environment.Variables.DATABASE) {
            error.failureReason = 'DATABASE environment variable not set';
            error.resource = props.Environment?.Variables?.DATABASE;
            return false;
        }
        return true;
    }));
    // Should not be any errors.
    expect(cert.node.metadataEntry.length).toBe(0);
});
test('Generate cert, all options set', () => {
    const stack = new core_1.Stack();
    const subject = {
        cn: 'testCN',
        o: 'testO',
        ou: 'testOu',
    };
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    const signingCertificate = new x509_certificate_1.X509CertificatePem(stack, 'SigningCert', { subject });
    const signingCertPassphraseID = stack.getLogicalId(signingCertificate.passphrase.node.defaultChild);
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        encryptionKey,
        signingCertificate,
        validFor: 3000,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    // Expect the custom resource for cert generation
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'testO',
            OU: 'testOu',
        },
        SigningCertificate: {
            Cert: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: signingCertPassphraseID,
            },
            CertChain: '',
        },
        CertificateValidFor: '3000',
    }));
    // Expect the resource for converting to PKCS #12 not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
    // Expect the DynamoDB table used for custom resource tracking
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table'));
    // Expect a Secret used to store the cert
    assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret'));
    // Expect a KMS key for encryption
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
    // Expect a policy that can interact with DynamoDB and SecretsManager for the signing cert's custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'kms:Encrypt',
                        'kms:ReEncrypt*',
                        'kms:GenerateDataKey*',
                    ],
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-B2B09A60': 'b2b09a6086e87fe14005f4e0b800e4f0',
                        },
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: 'SigningCertPassphrase42F0BC4F',
                    },
                },
            ],
        },
    }));
    // Expect a policy that can interact with DynamoDB and SecretsManager for the cert's custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: signingCertPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-BA0FA489': 'ba0fa4898b2088c5b25f15075f605300',
                        },
                    },
                },
            ],
        },
    }));
    // Expect Lambda for doing the cert generation to use the generate() handler
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'x509-certificate.generate',
    }));
});
test('Grant cert read', () => {
    const stack = new core_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantCertRead(grantable);
    // Expect the custom resource to be created
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    }));
    // Expect the grantCertRead() to add this policy
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
            ],
        },
    }));
    // Expect the grantCertRead() not to add this full read policy
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
            ],
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
});
test('Grant full read', () => {
    const stack = new core_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantFullRead(grantable);
    // Expect the custom resource to be created
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    }));
    // Expect the grantFullRead() to add this policy
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
            ],
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
});
test('Validating expiry', () => {
    // GIVEN
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    // WHEN
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        validFor: 0,
    });
    // THEN
    expect(cert.node.metadataEntry.length).toBe(1);
});
test('Validating expiry with token', () => {
    // GIVEN
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    // A numeric CDK token (see: https://docs.aws.amazon.com/cdk/latest/guide/tokens.html#tokens_number)
    const CDK_NUMERIC_TOKEN = -1.8881545897087626e+289;
    // WHEN
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        validFor: CDK_NUMERIC_TOKEN,
    });
    // THEN
    expect(cert.node.metadataEntry.length).toBe(0);
});
test('Convert to PKCS #12', () => {
    const stack = new core_1.Stack();
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    const pkcs12Cert = new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', { sourceCertificate: cert });
    const pkcs12CertPassphraseID = stack.getLogicalId(pkcs12Cert.passphrase.node.defaultChild);
    // Expect the PKCS #12 custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    }));
    // Expect the source certificate (custom resource)
    assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_X509Generator'));
    // Expect the PKCS #12 to have a password secret
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    }));
    // Expect the PKCS #12 resource to have a policy with access to the X.509 resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: pkcs12CertPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-71090F78': '71090f7809ce64f7c970cb645d4d473c',
                        },
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
            ],
        },
    }));
    // Expect no KMS key
    assert_1.expect(stack).notTo(assert_1.haveResource('AWS::KMS::Key'));
    // Expect the Lambda for converting the PEM to PKCS 12
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    }));
});
test('Convert to PKCS #12, use KMS', () => {
    const stack = new core_1.Stack();
    const subject = { cn: 'testCN' };
    const sourceCertificate = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(sourceCertificate.passphrase.node.defaultChild);
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', {
        sourceCertificate,
        encryptionKey,
    });
    // Expect the PKCS #12 custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    }));
    // Expect the source certificate (custom resource)
    assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_X509Generator'));
    // Expect the PKCS #12 to have a password secret
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    }));
    // Expect a KMS key for encryption
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
    // Expect the Lambda for converting the PEM to PKCS #12
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    }));
});
//# sourceMappingURL=data:application/json;base64,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